;; Eulisp Module
;; Author: pete broadbery
;; File: dist-task.em
;; Date: 21/sep/1991
;;
;; Project:
;; Description: 
;; distributed tasks
;;   really a simple linda pool.
;;

(defmodule dist-task 
  (standard0
   list-fns
   
   pvm-support

   rshow
   )
  ()
  
  ;; server side
  (defstruct task-pool ()
    ((clients initarg clients accessor task-pool-clients)
     (tasks initform () accessor task-pool-tasks)
     (waiters initform () accessor task-pool-waiters))
    constructor (make-task-pool clients))

  ;; client side

  (defstruct task-server ()
    ((location initarg location reader task-server-location)
     (completed-tasks initform '((-2 ())) accessor server-completed-tasks)
     (spare initform () accessor server-spare-req)
     (count initform 0 accessor server-count)
     (flow-state initform 'on accessor server-flow-state))
    constructor (make-task-server location))
  
  ;; global gonstants
  (defconstant *client-msg* 102)
  (defconstant *server-msg* 103)
  (defconstant *result-msg* 104)

  (defconstant *x-control-msg* 105)

  ;; client globals

  (deflocal *the-server* ())

  (defun the-server () *the-server*)

  ((setter setter) the-server 
   (lambda (x) (setq *the-server* x)))

  ;; functions:
  ;;  getting results...
  ;;   add-task (ts task)     add. -> taskid
  ;;   task-result (ts task)  query.

  ;; making functions
  ;;   define-task (name function)
  ;; a task is a pair (task-name, argument)

  (defstruct remote-task ()
    ((id initarg id reader remote-task-id)
     (value initform 'ugh 
	    accessor remote-task-value))
    constructor (make-unevaluated-task id)
    predicate is-remote-p)

  (defun do-task (task arg)
    (if (make-task-p task)
	(let ((t-id (make-task-id task)))
	  (register-task (the-server) (list task arg t-id))
	  (make-unevaluated-task t-id))
      (start-task (list task arg))))
  
  (defun task-result (thing)
    (if (is-remote-p thing)
	(if (eq (remote-task-value thing) 'ugh)
	    (let ((res (internal-task-result (the-server)
					     (remote-task-id thing))))
	      ((setter  remote-task-value) thing res)
	      res)
	  (remote-task-value thing))
      thing))

  (defconstant find-task-fn (mk-finder))

  (defun define-task (name function)
    ((setter find-task-fn) name function))

  (export define-task task-result do-task)
  ;; internal functions 

  ;; client-loop (server)
  ;; get-task    (server) {returns result to server}
  ;; start-task  (task) {begins a task returning to caller}
  ;; register-task (server task)
  ;; make-task-id
  ;; make-task-p (task)
  
  (defun remote-client-startup (server-id)
    (format t "Client at: ~a~%" server-id)
    (let ((s (make-task-server (pvm-make-id-from-pair server-id))))
      ((setter the-server) s)
      (client-loop s -1)))

  (export remote-client-startup)
    
  (defconstant task-dispatcher (mk-finder))

  (defun client-loop (server waitfor)
    (let ((xx (grab-result server waitfor)))
      (cond ((eq (car xx) 'ok) 
	     (format t "Client: Done: ~a ~%" 
		     (cadr xx))
	     (cadr xx))
	    ((pvm-probe *result-msg*)
	     (let ((msg (pvm-recv *result-msg* ())))
	       (add-result server (cdr msg))
	       (client-loop server waitfor)))
	    (t (if (server-spare-req server)
		   ()
		 (progn (pvm-send (task-server-location server)
				  *server-msg*
				  (list 'task-request (pvm-whoami)))
			((setter server-spare-req) server t)))
	       (let/cc cont
		       (with-handler
			(lambda (c1 c2) 
			  (backtrace)
			  (rshow c1)
			  (format t "Error Message: ~a~%Error Value: ~a~%" 
				  (condition-message c1)
				  (condition-error-value c1))
			  (flush (standard-output-stream))
			  (cont))
			(let ((msg (pvm-recv -1 t)))
			  (cond ((= (caddr msg) *result-msg*)
				 (add-result server (cdr (car msg))))
				((= (caddr msg) *client-msg*)
				 ((setter server-spare-req) server nil)
				 ((task-dispatcher (car (car msg)))
				  server (cdr (car msg))))
				(t (format t "handler[~a]: new state: ~a~%"
					   (server-flow-state server)
					   (car msg))
				   ((setter server-flow-state) server (car msg)))))))
	       (client-loop server waitfor)))))

  ;; task is: (fn arg id)
  (defun get-task (server taskbit)
    (let ((task (car taskbit)))
      (format t "Got Task: ~a~%" task)
      (let ((result ((find-task-fn (car task)) (cadr task)))
	    (where (car (caddr task))))
	(pvm-send where 
		  *result-msg*
		  (list 'set-result (caddr task) result)))))

  (defun start-task (task)
    ((find-task-fn (car task)) (cadr task)))

  (defun handle-results (server)
    (if (pvm-probe *result-msg*)
	(let ((msg (pvm-recv  *result-msg* ())))
	  (add-result server (cdr msg))
	  (handle-results server))
      nil)
    (if (pvm-probe *x-control-msg*)
	(let ((msg (pvm-recv *x-control-msg* ())))
	  (format t "handler[~a]: new state: ~a~%"
		  (server-flow-state server)
		  msg)
	  ((setter server-flow-state) server msg)
	  (handle-results server))
      nil))
	

  ;; result is (id result)
  (defun add-result (server resultbit)
    (flush (standard-output-stream))
    (let ((id (cdar resultbit))
	  (result (cadr resultbit)))
      ((setter server-count) server (- (server-count server) 1))
      (add-res-aux (server-completed-tasks server) id result)))
  
  (defun wake-up (server msg)
    nil)

  (defun add-res-aux (lst id result)
    (cond ((null lst) 
	   (error "Cant happen 1" clock-tick))
	  ((= id (caar lst))
	   ((setter cdr) (car lst) (list result)))
	  (t (add-res-aux (cdr lst) id result))))

  (defun grab-result (server id) 
    (grab-res-aux (server-completed-tasks server) id))

  (defun grab-res-aux (lst id)
    (cond ((null lst) '(()))
	  ((= id (caar lst))
	   (if (eq (cadar lst) 'undone)
	       '(())
	     (let ((res (cadar lst)))
	       ((setter car) lst
		(cadr lst))
	       ((setter cdr) lst
		(cddr lst))
	       (list 'ok res))))
	  (t (grab-res-aux (cdr lst) id))))
  
  ;; begin a remote-process
  (defun internal-task-result (server id)
    (client-loop server (cdr id)))

  ((setter task-dispatcher) 'set-result add-result)
  ((setter task-dispatcher) 'task get-task)

  (defun quit (server junk)
    ())

  (defconstant task-count (mk-counter 0))

  ;; task id is (where . number)

  (defun make-task-id (task)
    (cons (pvm-whoami) (task-count)))

  (defun make-task-p (task) 
    (handle-results (the-server))
    (and (eq (server-flow-state (the-server)) 'on)
	 (< (server-count (the-server)) 7)))
    
  (defun register-task (server task)
    ((setter server-completed-tasks) server
     (cons (list (cdr (caddr task)) 'undone) 
	   (server-completed-tasks server)))
    ((setter server-count) server (+ (server-count server) 1))
    (pvm-send (task-server-location server)
	      *server-msg*
	      (list 'add-task task)))

  ;; server functions:
  ;;
  ;; server-loop
  ;; add-new-task
  ;; get-next-task
  
  ;; NB. This is all supposed to be atomic...

  (defun remote-start-server ()
    (server-loop (make-flow-pool ())))

  (defconstant server-dispatcher (mk-finder))

  (defun server-loop (server)
    (let ((req (pvm-recv *server-msg* ())))
      ((server-dispatcher (car req)) server (cdr req))
      (server-loop server)))
  
  (defgeneric note-new-task (pool))
  (defgeneric note-alloc-task (pool))

  ;; waiters: FIFI
  ;; tasks: FILO
  (defun server-add-new-task (pool task)
    (if (null (task-pool-waiters pool))
	(progn (note-new-task pool)
	       ((setter task-pool-tasks) pool 
		(cons (car task) (task-pool-tasks pool))))
      (let ((nxt (car (t-val (task-pool-waiters pool)))))
	((setter task-pool-waiters) pool
	 (t-cdr (task-pool-waiters pool)))
	(pvm-send nxt
		  *client-msg*
		  (list 'task (car task))))))

  (defun server-get-next-task (pool where)
    (if (null (task-pool-tasks pool))
	((setter task-pool-waiters) pool 
	 (tconc (task-pool-waiters pool)
		(car where)))
      (let ((nxt (car (task-pool-tasks pool))))
        ((setter task-pool-tasks) pool
         (cdr (task-pool-tasks pool)))
	(note-alloc-task pool)
        (pvm-send (car where)
		  *client-msg*
		  (list 'task nxt)))))
  
  ;; should be in list-fns...

  (defun tconc (ptr elem)
    (let ((new-pair (list elem)))
      (cond ((null ptr)
	     (cons new-pair new-pair))
	    (t
	     ((setter cdr) (cdr ptr) new-pair)
	     ((setter cdr) ptr new-pair)
	     ptr))))

  (defun t-cdr (x)
    (if (null x)
	(error "oops" clock-tick)
      (if (null (cdr (car x)))
	  ()
	(cons (cdr (car x)) (cdr x)))))

  (defun t-val (x)
    (if (null x) x (car x)))

  (defun t-append (x list)
    (cond ((and (null x)
		(null list))
	   nil)
	  ((null x)
	   (cons list
		 (last-pair list)))
	  ((null list)
	   x)
	  (t ((setter cdr) (cdr x) list)
	     ((setter cdr) x (last-pair list))
	     x)))


  ((setter server-dispatcher) 'add-task server-add-new-task)
  ((setter server-dispatcher) 'task-request server-get-next-task)

  ;; More complex server protocol
  (defconstant *high-threshold* 6)
  (defconstant *low-threshold* 3)
  (defstruct flow-pool task-pool
    ((msg-count initform (mk-counter 0)
		reader pool-out-msg-count)
     (x-state initform 'on 
	      accessor task-pool-x-state)
     (task-count initform 0 
		 accessor pool-task-count))
    constructor (make-flow-pool clients))

  (defmethod note-new-task ((pool flow-pool))
    ((setter pool-task-count) pool 
     (+ (pool-task-count pool) 1))
    (if (and (eq (task-pool-x-state pool) 'on)
	     (> (pool-task-count pool) *high-threshold*))
	(progn (pvm-send (make-pvm-id "pvm-feel") *x-control-msg*
			 'off)
	       ((setter task-pool-x-state) pool 'off))
      (format t "server: ~a~%" (pool-task-count pool))))

  (defmethod note-alloc-task ((pool flow-pool))
    ((setter pool-task-count) pool 
     (- (pool-task-count pool) 1))
    (if (and (eq (task-pool-x-state pool) 'off)
	     (< (pool-task-count pool) *low-threshold*))
	(progn (pvm-send (make-pvm-id "pvm-feel") *x-control-msg*
		     'on)
	       ((setter task-pool-x-state) pool 'on))
      ()))
    
	      
  ;; startup
  ;;
  ;; Initialise-network (server names)
  ;; 
  (defun initialise-network (load-path initmod server-name client-names)
    (if (the-server) () (start-local-pvm))
    (let ((server-id (start-server server-name)))
      ((setter the-server) 
       (make-task-server server-id))
      (let ((clients (mapcar (lambda (name) 
			       (start-client server-id name load-path initmod))
			     client-names)))
	clients)))

  (defun as-string (x) (format nil "~a" x))

  (defun start-server (name)
    (let ((obj (pvm-initiate-by-hostname (as-string name) "pvm-feel")))
      (pvm-start obj 'dist-task 'remote-start-server ())
      obj))


  (defun start-client (server-id name path initmod)
    (let ((obj (pvm-initiate-by-hostname (as-string name) "pvm-feel")))
      (pvm-set-load-path obj path)
      (pvm-start obj initmod 'remote-client-startup (list server-id))
      obj))

  (export initialise-network)
  ;; end module
  )
