;
;; EuLisp (FEEL) Module                            (C) University Of Bath 1991 
;

;
;; utils
;

(defmodule utils

  (standard0) ()

  ;
  ;;
  ;;; Thread utilities...
  ;;
  ;

  ;
  ;; Blocking wait for an expression to become true
  ;

  (defun await-function (fn) 
    (let ((result (fn)))
      (if (null result) (progn (thread-reschedule) (await-function fn)) result)))

  (defmacro await exp
    `(await-function (lambda () ,@exp)))

  (export await await-function)

  ;
  ;;
  ;;; Semaphoring utilities
  ;;
  ;

  (defmacro critical-progn (sem . body)
    (let ((semvar (gensym))
	  (resvar (gensym)))
      `(let/cc critical-return
	 (let ((,semvar ,sem)
	       (,resvar ()))
	   (open-semaphore ,semvar)
	   (unwind-protect
	     (setq ,resvar (progn ,@body)) ; Protected form
	     (close-semaphore ,semvar))    ; On any exit
	   ,resvar))))

  (defun exclusive (fn)
    (let ((sem (make-semaphore)))
      (lambda a
	(open-semaphore sem)
	(let ((res (apply fn a)))
	  (close-semaphore sem)
	  res))))
  
  (defmacro defexfun (name args . rest)
    `(defconstant ,name (exclusive
			 (lambda ,args ,@rest))))


  (export critical-progn exclusive defexfun) 

  ;
  ;;
  ;;; TELOSesque utilities...
  ;;
  ;
  
  (defcondition key-list-error () key-list () key ())

  (defconstant *key-list-ref-failed* (gensym))

  (defun key-list-ref (l key)
    (cond ((null l) *key-list-ref-failed*)
	  ((null (cdr l)) 
	    (error 
	      "key-list-ref: unbalanced key list"
	      key-list-error 'key-list l 'key key))
	  ((eq (car l) key) (car (cdr l)))
	  (t
	    (key-list-ref (cdr (cdr l)) key))))

  (export key-list-error *key-list-ref-failed* key-list-ref)

  ;
  ;;
  ;;; Control utilities...
  ;;
  ;

  ;
  ;; Vector mas
  ;

  (defun mapv (fn . vecs)    
    (let ((len (apply min (mapcar vector-length vecs))))
      (labels
        ((iterate (i)
	   (if (= i len) ()
	     (progn
	       (apply fn (mapcar (lambda (v) (vector-ref v i)) vecs))
	       (iterate (+ i 1))))))
	(iterate 0))))

  (export mapv)

  ;
  ;; List maps
  ;

  ;; This definition does not allow for arbitary numbers of args
  (defun mapcan (*fn* . x)
    (let ((len (list-length x)))
      (cond ((= len 1) (mapcan1 *fn* (car x)))
	    ((= len 2) (mapcan2 *fn* (car x) (cadr x)))
	    ((= len 3) (mapcan3 *fn* (car x) (cadr x) (caddr x)))
	    (t (error 0 "mapcan unfinished")))))

  (defun mapcan1 (*fn* x)
    (if (not (consp x)) nil
      (nconc (*fn* (car x)) (mapcan1 *fn* (cdr x)))))

  (defun mapcan2 (*fn* x y)
    (if (or (not (consp x)) (not (consp y))) nil
      (nconc (*fn* (car x) (car y)) (mapcan2 *fn* (cdr x) (cdr y)))))

  (defun mapcan3 (*fn* x y z)
    (if (or (not (consp x)) (not (consp y)) (not (consp z))) nil
      (nconc (*fn* (car x) (car y) (car z))
	     (mapcan3 *fn* (cdr x) (cdr y) (cdr z)))))

  (export mapcan)

  ;
  ;; Simple linear maps over integral ranges...
  ;

  (defun map-indices (fn n last inc)
    (if (= n last) (progn (fn n) ())
      (progn
	(fn n)
	(map-indices fn (+ n inc) last inc))))

  (defun map-indices-collecting (fn n last inc)
    (if (= n last) (cons (fn n) ())
      (cons (fn n) (map-indices-collecting fn (+ n inc) last inc))))

  (defmacro for-index (var start end inc . body)
    `(map-indices (lambda (,var) ,@body) ,start ,end ,inc))

  (defmacro collect-index (var start end inc . body)
    `(map-indices-collecting (lambda (,var) ,@body) ,start ,end ,inc))

  (export map-indices map-indices-collecting for-index collect-index)

  (defun map-stream (code endp inc)
    (unless (endp) (code) (inc) (map-stream code endp inc)))

  (defmacro for-stream (var start endp inc tidy . body)
    `(let/cc return
       (let ((,var ,start))
	 (map-stream 
	  (lambda (,var) ,@body) 
	  (lambda () ,endp)
	  (lambda () (setq ,var ,inc)))
	 ,tidy)))

  (export map-stream for-stream)

  (defmacro for-list (var list . body)
    `(mapc (lambda (,var) ,@body) ,list))

  (defmacro collect-list (var list . body)
    `(mapc (lambda (,var) ,@body) ,list))

  (export for-list collect-list)

  (defmacro for-table (keyvar valvar tab . body)
    `(map-table (lambda (,keyvar ,valvar) ,@body) ,tab))

  (export for-table)

  (defun assq-list-ref (l k)
    (cond ((null l) ())
	  ((eq (car (car l)) k) (cdr (car l)))
	  (t (assq-list-ref (cdr l) k))))

  (defun map-psl-for (code endp inc)
    (unless (endp) (code) (inc) (map-psl-for code endp inc)))

  (defmacro psl-for forms
    (let ((forexp (assq-list-ref forms 'for))
	  (untilexp (assq-list-ref forms 'until))
	  (finallyexp (assq-list-ref forms 'finally))
	  (doexp (assq-list-ref forms 'do)))
      `(let/cc return
	 (let ((,(car forexp) (car (cdr forexp))))
	   (map-psl-for
	     (lambda () ,@doexp)
	     (lambda () ,@untilexp)
	     (lambda () (progn ,@(car (cdr (cdr forexp))))))
	   ,@finallyexp))))

  (export assq-list-ref map-psl-for psl-for)

)
