<;; pvm-to dvsm interface

(defmodule pvm-iface
  (standard0
   list-fns

   page-mapper
   handler
   genread
   swait
   ;;pvm-support
   )

  ()

  ;; message types 
  ;; read-req: (page dest req-id)
  ;;   if this host owns the page, then send it to dest
  ;;   o/w fwd to requester
  ;; write-req: (page dest req-id)
  ;;   
  ;; read-rep: (page-id req-id)
  (defconstant *read-request* 200)
  (defconstant *write-request* 201)

  (defconstant *read-reply* 202)
  (defconstant *write-reply* 203)
  (defconstant *page-loc* 1023)

  (defstruct dist-mapper page-mapper
    ((my-id initform 'unknown ;; (pvm-whoami)
	    accessor dist-mapper-host-id)
     (my-num initarg number
	     reader dist-mapper-number)
     (page-count initform (mk-counter 0)
		 reader dist-mapper-count)
     (waiter initform (make-wait-handler)
	     reader wait-list))
    constructor make-dist-mapper)

  (export make-dist-mapper)

  (defstruct dist-page-info page-info
    ;; Should be a host-identifier
    ((owner initarg owner
	    accessor dist-page-owner))
    constructor make-dist-page-info)

  (defmethod mapper-info-class ((x dist-mapper))
    dist-page-info)
  
  (defun my-page-p (mapper info)
    (equal (dist-mapper-host-id mapper)
	   (dist-page-owner info)))

  (defmethod initialize-instance ((proto dist-mapper) lst)
    (let ((new (call-next-method)))
      (register-handler *read-request*
			(read-req-handler new))
      (register-handler *write-request*
			(write-req-handler new))
      (register-handler *read-reply*
			(read-reply-handler new))
      (register-handler *write-reply* 
			(write-reply-handler new))
      new))

  
  ;; Handlers
  (defun read-req-handler (mapper)
    (lambda (msg-info)
      (let ((page-id (car (car msg-info)))
	    (dest (cadr (car msg-info)))
	    (rest (cddr (car msg-info))))
	(let ((pg-info (get-page-info mapper page-id)))
	  (if (my-page-p mapper pg-info)
	      (pvm-send *read-reply*
			dest
			(append (list (dist-mapper-host-id mapper)
				      (page-content pg-info))
				rest)
			(default-reader))
	    (pvm-send *read-request*
		      (dist-page-owner page)
		      (car msg-info)
		      (default-reader)))))))
  
  (defun write-req-handler (mapper)
    (lambda (msg-info)
      (let ((page-id (car (car msg-info)))
            (dest (cadr (car msg-info)))
            (rest (cddr (car msg-info))))
        (let ((pg-info (get-page-info mapper page-id)))
          (cond ((not (my-page-p mapper pg-info))
		 ;; send it along...
		 (pvm-send *read-request*
			   (dist-page-owner page)
			   (car msg-info)
			   (default-reader)))
		((page-locked-p pg-info)
		 ;; XXX won't happen..
		 (add-wait-request pg-info dest rest))
		(t (set-page-owner pg-info dest)
		   (pvm-send *write-reply*
			     (append (list (dist-mapper-host-id mapper)
					   (page-content pg-info))
				     rest)
			     (default-reader))))))))

  (defun read-rep-handler (mapper)
    (lambda (msg-info)
      (let ((id (car (cddr (car msg-info)))))
	(activate-wait-thread (wait-list mapper)
			      id (car msg-info)))))

  (defun write-rep-handler (mapper)
    (lambda (msg-info)
      (let ((id (car (cddr (car msg-info)))))
	(activate-wait-thread (wait-list mapper)
			      id (car msg-info)))))

		 
  ;; must reread the page to make sure of consistency
  ;;
  (defmethod map-page-write ((mapper dist-mapper) id)
    (let ((pg-info (get-page-info mapper id)))
      (if (my-page-p mapper pg-info)
	  (page-2-real pg-info id)
	(page-write-fault mapper id pg-info))))
  
  (defun page-write-fault (mapper id info)
    (let ((req-id (make-wait-id (wait-list mapper))))
      (pvm-send *write-request* (dist-page-owner info)
		(list req-id id))
      (let ((new-page (wait-4-id (wait-list mapper) req-id)))
	new-page)))

  (defmethod map-page-read ((mapper dist-mapper) id)
    (let ((pg-info (get-page-info mapper id)))
      (if (is-mapped pg-info)
	  (page-2-real pg-info id)
	(page-read-fault mapper id pg-info))))
  
  (defun page-read-fault (mapper id info)
    (let ((req-id (make-wait-id (wait-list mapper))))
      (pvm-send *read-request* (dist-page-owner info)
		(list req-id id))
      (let ((new-page (wait-4-id (wait-list mapper) req-id)))
	new-page)))

  ;; initialisation

  (deflocal *mapper*)

  (defun the-mapper () *the-mapper*)

  ((setter setter) the-mapper 
   (lambda (x) (setq *the-mapper* x)))


  (defun start-mapper-handler (mapper)
    (register-handler *read-req* 
		      (read-req-handler mapper))
    (register-handler *write-req*
		      (write-req-handler mapper)))

  ;; we use the prime number multiples to guarentee that no two 
  ;; hosts will allocate the same page
  (defconstant pg-count (mk-counter 0))

  (defmethod allocate-page-id ((mapper dist-mapper))
    (gensym))


  ;; Writing address 
  
  (defgeneric write-pg-ref (page location class writer))
  
  (export write-address)

  (defgeneric write-address (address ptr rdr))

  (defmethod write-address ((addr local-address) ptr rdr)
    ;; move the object into external space
    (let ((new-add (forward-object addr)))
      (write-address new-addr ptr rdr)))

  (defun write-address (address value reader)
    (write-pg-ref (address-page address)
		  (address-loc address)
		  (address-class address)
		  (lambda (obj)
		    (write-next obj value reader))))

  ;; reading addresses
  
  )
