/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1986. */

#include "b.h"
#include "feat.h"
#include "bmem.h"
#include "getc.h"
#include "keys.h"
#include "args.h"

char *getenv();

/* struct tabent {int code; string name, def, rep;} in getc.h */

/* Table of key definitions, filled by the following defaults
   and by reading definitions from a file.

   For the code field the following holds:
   code > 0:
       definitions for editor operations,
       new defs from keydefs file will be added in bed/e1getc.c,
        eliminating conflicting ones;
   code < 0:
       strings to be send to the terminal,
       any new defs from keydefs file overwrite the old ones

   Not all control characters can be freely used:
   ^Q and ^S are used by the Unix operating system
   for output flow control, and ^Z is used by BSD
   Unix systems for `job control'.
   Also note that ^H, ^I and ^M (and somtimes ^J) have their
   own keys on most keyboards and thus usually have a strong
   intuitive meaning.

   'def' fields initialized with a string starting with '=' are termcap names,
   and are replaced by the corresponding termcap entry (NULL if none);
   
   'def' fields initialized with a string starting with "&" are
   special characters for unix, and taken from tty structures.

*/

Visible struct tabent deftab[MAXDEFS] = {
	{IGNORE,	S_IGNORE,	NULL,		NULL},
		/* Entry to ignore a key */

	/* if there are no or too few function or arrow keys: */
	{WIDEN,		S_WIDEN,	"\033w",	"ESC w"},
	{EXTEND,	S_EXTEND,	"\033e",	"ESC e"},
	{FIRST,		S_FIRST,	"\033f",	"ESC f"},
	{LAST,		S_LAST,		"\033l",	"ESC l"},
	{PREVIOUS,	S_PREVIOUS,	"\033p",	"ESC p"},
	{NEXT,		S_NEXT,		"\033n",	"ESC n"},
	{UPARROW,	S_UPARROW,	"\033k",	"ESC k"},
	{DOWNARROW,	S_DOWNARROW,	"\033j",	"ESC j"},
	{LEFTARROW,	S_LEFTARROW,	"\033,",	"ESC ,"},
		/* , below < */
	{RITEARROW,	S_RITEARROW,	"\033.",	"ESC ."},
		/* . below > */
	{UPLINE,	S_UPLINE,	"\033u",	"ESC u"},
	{DOWNLINE,	S_DOWNLINE,	"\033d",	"ESC d"},
	{COPY,		S_COPY,		"\033c",	"ESC c"},
		/* in case ^C is interrupt */

	/* function and arrow keys as in termcap;
	 * these must follow, because the first key in the helpblurb
	 * will be the last one */
	{WIDEN,		S_WIDEN,	"=k1",		"F1"},
	{EXTEND,	S_EXTEND,	"=k2",		"F2"},
	{FIRST,		S_FIRST,	"=k3",		"F3"},
	{LAST,		S_LAST,		"=k4",		"F4"},
	{PREVIOUS,	S_PREVIOUS,	"=k5",		"F5"},
	{NEXT,		S_NEXT,		"=k6",		"F6"},
	{UPLINE,	S_UPLINE,	"=k7",		"F7"},
	{DOWNLINE,	S_DOWNLINE,	"=k8",		"F8"},
	{COPY,		S_COPY,		"=k9",		"F9"},
	{UPARROW,	S_UPARROW,	"=ku",		"^"},
	{DOWNARROW,	S_DOWNARROW,	"=kd",		"v"},
	{LEFTARROW,	S_LEFTARROW,	"=kl",		"<-"},
	{RITEARROW,	S_RITEARROW,	"=kr",		"->"},
#ifdef GOTOCURSOR
	{GOTO,		S_GOTO,		"\033g",	"ESC g"},
	{GOTO,		S_GOTO,		"\007",		"Ctrl-g"},
#endif
	{ACCEPT,	S_ACCEPT,	"\011",		"TAB"},
	{NEWLINE,	S_NEWLINE,	"\015",		"RETURN"},
	{UNDO,		S_UNDO,		"\010",		"BACKSP"},
	{REDO,		S_REDO,		"\025",		"Ctrl-U"},
	{COPY,		S_COPY,		"\003",		"Ctrl-C"},
	{DELETE,	S_DELETE,	"\004",		"Ctrl-D"},
#ifdef RECORDING
	{RECORD,	S_RECORD,	"\022",		"Ctrl-R"},
	{PLAYBACK,	S_PLAYBACK,	"\020",		"Ctrl-P"},
#endif
	{REDRAW,	S_LOOK,		"\014",		"Ctrl-L"},
#ifdef HELPFUL
	{HELP,		S_HELP,		"\033?",	"ESC ?"},
	{HELP,		S_HELP,		"=k0",		"F10"},
#endif
	{EXIT,		S_EXIT,		"\030",		"Ctrl-X"},
	{EXIT,		S_EXIT,		"\033\033",	"ESC ESC"},
	
	/* These three are taken from stty settings: */
	
	{CANCEL,	S_INTERRUPT,	"&\003",	NULL},
		/* take from intr char */
	{SUSPEND,	S_SUSPEND,	"&\032",	NULL},
		/* take from susp char */
	{UNDO,		S_UNDO,		"&\b",		NULL},
		/* take from erase char */
	
	/* These two are not key defs but string-valued options: */
	
	{TERMINIT,	S_TERMINIT,	"=ks",		NULL},
	{TERMDONE,	S_TERMDONE,	"=ke",		NULL},
	{0,		NULL,		NULL,		NULL}
};

/* Merge key definitions from termcap into the default table. */

Hidden Procedure readtermcap() {
	string tgetstr();
	char buffer[1024]; /* Constant dictated by termcap manual entry */
	static char area[1024];
	string endarea= area;
	string anentry;
	struct tabent *d, *last;

	switch (tgetent(buffer, getenv("TERM"))) {

	default:
		putmess(errfile, MESS(6800, "*** Bad tgetent() return value.\n"));
		/* Fall through */
	case -1:
		putmess(errfile, MESS(6801, "*** Can't read termcap.\n"));
		/* Fall through again */
	case 0:
	putmess(errfile, MESS(6802, "*** No description for your terminal.\n"));
		immexit(1);

	case 1:
		break;
	}

	last= deftab+ndefs;
	for (d= deftab; d < last; ++d) {
		if (d->def != NULL && d->def[0] == '=') {
			anentry= tgetstr(d->def+1, &endarea);
			if (anentry != NULL && anentry[0] != '\0') {
				undefine(d->code, anentry);
				d->def= anentry;
			}
			else
				d->def= d->rep= NULL;
		}
	}
}

/* Code to get the defaults for interrupt, suspend and undo/erase_char
 * from tty structs.
 */

#ifndef KEYS
Hidden char *intr_char= NULL;
Hidden char *susp_char= NULL;
#else
Visible char *intr_char= NULL;
Visible char *susp_char= NULL;
#endif

Hidden char *erase_char= NULL;

#ifndef TERMIO
#include <sgtty.h>
#else
#include <termio.h>
#endif
#ifdef SIGNAL
#include <signal.h>
#endif

Hidden char *getspchars() {
#ifndef TERMIO
	struct sgttyb sgbuf;
#ifdef TIOCGETC
	struct tchars tcbuf;
#endif
	static char str[6];
	
	if (gtty(0, &sgbuf) == 0) {
		if ((int)sgbuf.sg_erase != -1 
		    &&
		    !(isprint(sgbuf.sg_erase) || sgbuf.sg_erase == ' ')
		) {
			str[0]= sgbuf.sg_erase;
			erase_char= &str[0];
		}
	}
#ifdef TIOCGETC
	if (ioctl(0, TIOCGETC, (char*)&tcbuf) == 0) {
		if ((int)tcbuf.t_intrc !=  -1) {
			str[2]= tcbuf.t_intrc;
			intr_char= &str[2];
		}
	}
#endif
#if defined(TIOCGLTC) && defined(SIGTSTP)
	{
		struct ltchars buf;
		SIGTYPE (*handler)();

		handler= signal(SIGTSTP, SIG_IGN);
		if (handler != SIG_IGN) {
			/* Shell has job control */
			signal(SIGTSTP, handler); /* Reset original handler */
			if (ioctl(0, TIOCGLTC, (char*) &buf) == 0 &&
					(int)buf.t_suspc != -1) {
				str[4]= buf.t_suspc;
				susp_char= &str[4];
			}
		}
	}
#endif /* TIOCGLTC && SIGTSTP */
#else /* TERMIO */
	struct termio sgbuf;
	static char str[6];
	
	if (ioctl(0, TCGETA, (char*) &sgbuf) == 0) {
		if ((int) sgbuf.c_cc[VERASE] != 0377
		    &&
		    !(isprint(sgbuf.c_cc[VERASE]))
		) {
			str[0]= sgbuf.c_cc[VERASE];
			erase_char= &str[0];
		}
		if ((int) sgbuf.c_cc[VINTR] != 0377) {
			str[2]= sgbuf.c_cc[VINTR];
			intr_char= &str[2];
		}
	}
	/* TODO: susp_char (c_cc[VSWTCH]) #ifdef VSWTCH && SIGTSTP_EQUIVALENT */
#endif /* TERMIO */
}

Visible bool is_spchar(c) char c; {
	if (intr_char != NULL && *intr_char == c)
		return Yes;
	else if (susp_char != NULL && *susp_char == c)
		return Yes;
	return No;
}

Hidden Procedure sig_undef(c) char c; {
	struct tabent *d, *last= deftab+ndefs;
	string p;

	for (d= deftab; d < last; ++d) {
		if (d->code > 0 && d->def != NULL) {
			for (p= d->def; *p != '\0'; ++p) {
				if (*p == c) {
					d->def= d->rep= NULL;
					break;
				}
			}
		}
	}
}

/* The following is needed for the helpblurb */

#ifndef KEYS
Hidden string reprchar(c) int c; {
#else
Visible string reprchar(c) int c; {
#endif /* KEYS */

	static char str[20];

	c&= 0377;

	if ('\000' <= c && c < '\040') {		/* control char */
		switch (c) {
			case '\010':
				return "BACKSP";
			case '\011':
				return "TAB";
			case '\012':
				return "LINEFEED";
			case '\015':
				return "RETURN";
			case '\033':
				return "ESC";
			default:
				sprintf(str, "Ctrl-%c", c|0100);
				return str;
			}
		}
	else if (c == '\040') {				/* space */
		return "SPACE";
	}
	else if ('\041' <= c && c < '\177') {		/* printable char */
		str[0]= c; str[1]= '\0';
		return str;
	}
	else if (c == '\177') {				/* delete */
		return "DEL";
	}
	else if (c == 0200) {				/* conv null char */
		return "NULL";
	}
	else {
		sprintf(str, "\\%03o", c);		/* octal value */
		return str;
	}
}

Hidden Procedure get_special_chars() {
	string anentry;
	struct tabent *d, *last;
	
	getspchars();
	last= deftab+ndefs;
	for (d= deftab; d < last; ++d) {
		if (d->def != NULL && d->def[0] == '&') {
			if (d->def[1] == '\003') /* interrupt */
				anentry= intr_char;
			else if (d->def[1] == '\b') /* undo/backspace */
				anentry= erase_char;
			else if (d->def[1] == '\032') /* suspend */
				anentry= susp_char;
			else
				anentry= NULL;
			if (anentry != NULL && anentry[0] != '\0') {
				if (anentry == erase_char)
					undefine(d->code, anentry);
				else
					sig_undef(anentry[0]);
				d->def= anentry;
				d->rep= (string) savestr(reprchar(anentry[0]));
#ifdef MEMTRACE
				fixmem((ptr) d->rep);
#endif
			}
			else
				d->def= d->rep= NULL;
		}
	}
}

Visible Procedure initkeys() {
	countdefs();
#ifdef DUMPKEYS
	if (kflag)
		dumpkeys("before termcap");
#endif
	readtermcap();
#ifdef DUMPKEYS
	if (kflag)
		dumpkeys("after termcap");
#endif
	get_special_chars();
#ifdef DUMPKEYS
	if (kflag)
		dumpkeys("after special chars");
#endif
	rd_keysfile();
}

#ifdef UNUSED

Visible int kbchar() {
/* Strip high bit from input characters (matters only on PWB systems?) */
	return getchar() & 0177;
}

#endif

Visible int cvchar(c) int c; {
#ifdef KEYS
	if (c == 0)
		return 0200;
#endif
	return c;
}
