(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Thu Apr  9 09:40:59 PDT 1992 by kalsow     *)
(*      modified on Thu Mar 12 12:12:36 PST 1992 by muller     *)

UNSAFE MODULE RTTypeFP;

IMPORT RT0, RT0u, RTMisc, RTType, FPrint, Word;

TYPE
 Info = UNTRACED REF RT0.TypeInfo;

VAR
  next_seq := 0;
  mu       := NEW(MUTEX);
  info     : UNTRACED REF ARRAY OF Info;

PROCEDURE FromFingerprint (READONLY fp: Fingerprint): Typecode =
  VAR t: RT0.TypeDefinition;  x: Info;
  BEGIN
    (* first, scan the types that have already been done *)
    FOR tc := 0 TO RT0u.nTypes-1 DO
      x := RT0u.types[tc].fpInfo;
      IF (x # NIL) AND (x.fp = fp) THEN  RETURN tc  END;
    END;

    (* then, try it again forcing fingerprints to be computed *)
    FOR tc := 0 TO RT0u.nTypes-1 DO
      t := RT0u.types[tc];
      x := t.fpInfo;
      IF (x = NIL) OR ((x.fp[0] = 0) AND (x.fp[1] = 0)) THEN
        EVAL ToFingerprint (tc);
        x := t.fpInfo;
      END;
      IF (x.fp = fp) THEN  RETURN tc  END;
    END;

    (* otherwise, fail *)
    RETURN RTType.NoSuchType;
  END FromFingerprint;

PROCEDURE ToFingerprint (tc: Typecode): Fingerprint =
  VAR x: Info;  t: RT0.TypeDefinition;
  BEGIN
    IF (tc >= RT0u.nTypes) THEN
      RTMisc.FatalErrorI ("improper typecode: ", tc);
    END;

    (* check the type table first *)
    t := RT0u.types [tc];
    x := t.fpInfo;
    IF (x # NIL) AND ((x.fp[0] # 0) OR (x.fp[1] # 0)) THEN RETURN x.fp END;

    (* otherwise, use the general hash table *)
    LOCK mu DO
      IF (info = NIL) THEN Init() END;
      x := Locate (t.selfID);
      IF (x.fp[0] = 0) AND (x.fp[1] = 0) THEN
        (* found it, but this is the first time it's been needed *)
        ComputeFullFingerprint (x);
      END;
      t.fpInfo := x;
      RETURN x.fp;
    END;
  END ToFingerprint;

PROCEDURE UIDToFingerprint (id: INTEGER): Fingerprint =
  VAR x: Info;
  BEGIN
    LOCK mu DO
      IF (info = NIL) THEN Init() END;
      x := Locate (id);
      IF (x.fp[0] = 0) AND (x.fp[1] = 0) THEN
        (* found it, but this is the first time it's been needed *)
        ComputeFullFingerprint (x);
      END;
      RETURN x.fp;
    END;
  END UIDToFingerprint;

TYPE FPBuf = REF ARRAY OF INTEGER;
VAR fp_buf: FPBuf; (* protected by 'mu' *)

PROCEDURE ComputeFullFingerprint (x: Info) =
  (* called while mu is held *)
  CONST Zero = Fingerprint { 0, 0 };
  VAR i: INTEGER;
  BEGIN
    IF (fp_buf = NIL) THEN fp_buf := NEW (FPBuf, 250) END;
    INC (next_seq);
    i := AppendPartials (next_seq, 0, x);
    INC (next_seq, i);
    x.fp := FPrint.Extend (Zero, ADR (fp_buf[0]), i * BYTESIZE (INTEGER));
  END ComputeFullFingerprint;

PROCEDURE AppendPartials (base, len: INTEGER;  x: Info) : INTEGER =
  VAR ip: UNTRACED REF INTEGER;
  BEGIN
    (* make sure there's room for at least 2 more words *)
    IF (len >= LAST (fp_buf^)) THEN ExpandBuf () END;

    IF (x.seq >= base) THEN
      (* we've already visited this type *)
      fp_buf [len] := x.seq - base;
      RETURN len+1;
    ELSE
      (* record that we've visited this type *)
      x.seq := base + len;

      (* add x's partial to the buffer *)
      ip := x.data;
      fp_buf [len]   := ip^;  INC (ip, ADRSIZE (ip^));
      fp_buf [len+1] := ip^;  INC (ip, ADRSIZE (ip^));
      INC (len, 2);

      (* and finally, add the types it depends on *)
      FOR j := 2 TO x.d_len-1 DO
        len := AppendPartials (base, len, Locate (ip^));
        INC (ip, ADRSIZE (ip^));
      END;

      RETURN len;
    END;
  END AppendPartials;

PROCEDURE ExpandBuf () =
  VAR new := NEW (FPBuf, 2 * NUMBER (fp_buf^));
  BEGIN
    FOR i := 0 TO LAST (fp_buf^) DO new[i] := fp_buf[i] END;
    fp_buf := new;
  END ExpandBuf;

PROCEDURE Init () =
  (* called while 'mu' is held *)
  VAR
    j, n: INTEGER;
    x, y: Info;
    tp: RT0.TypeInfoList;
    t: RT0.TypeDefinition;
  BEGIN
    (* count the number of types that exist *)
    n := 0;
    FOR i := 0 TO RT0u.nModules - 1 DO
      tp := RT0u.modules[i].type_info;
      IF (tp # NIL) THEN
        j := 0;
        WHILE (tp.id # 0) DO INC (tp, ADRSIZE (tp^));  INC (j)  END;
        INC (n, j);
      END;
    END;

    (* allocate the global array of Info pointers *)
    info := NEW (UNTRACED REF ARRAY OF Info, 3 * n);

    (* for each type, insert its info entry into the global array *)
    FOR i := 0 TO RT0u.nModules - 1 DO
      tp := RT0u.modules[i].type_info;
      IF (tp # NIL) THEN
        WHILE (tp.id # 0) DO
          Insert (tp^);
          INC (tp, ADRSIZE (tp^));
        END;
      END;
    END;

    (* for each opaque type, fixup data pointer & length *)
    FOR i := 0 TO LAST (info^) DO
      x := info[i];
      IF (x # NIL) THEN
        IF (x.class = ORD (RT0.TypeClass.Opaque)) THEN
          t := LOOPHOLE (x.data, RT0.TypeDefinitionPtr)^;
          y := Locate (t^.selfID);
          t.fpInfo := x; (* remember this lookup *)
          x.data := y.data;
          x.d_len := y.d_len;
        END;
        <* ASSERT x.d_len >= NUMBER (Fingerprint) *>
      END;
    END;
  END Init;

(** CONST Multiplier = 1052824; **)
CONST Multiplier = 2 * 2 * 3 * 5 * 7 * 11 * 13 * 17 * 19 * 23 + 1;
(* See Knuth Vol. 2, Theorem A, page 16. *)

PROCEDURE Insert (VAR xx: RT0.TypeInfo) =
  (* called while 'mu' is held *)
  VAR x: Info;  hash, index: INTEGER;
  BEGIN
    hash := xx.id;
    LOOP
      index := Word.Mod (hash, NUMBER (info^));
      x := info [index];
      IF (x = NIL)   THEN  info [index] := ADR (xx); RETURN  END;
      IF (x.id = xx.id) THEN  RETURN  END;
      hash := Word.Plus (1, Word.Times (hash, Multiplier));
    END;
  END Insert;

PROCEDURE Locate (id: INTEGER): Info =
  (* called while 'mu' is held *)
  VAR x: Info;  hash, index: INTEGER;
  BEGIN
    hash := id;
    LOOP
      index := Word.Mod (hash, NUMBER (info^));
      x := info [index];
      IF (x = NIL) THEN RTMisc.FatalErrorI ("unknown type id: ", id) END;
      IF (x.id = id) THEN RETURN x END;
      hash := Word.Plus (1, Word.Times (hash, Multiplier));
    END;
  END Locate;

BEGIN
  <* ASSERT NUMBER (Fingerprint) = 2 *>
END RTTypeFP.

