/***********************************************************************\ 
*									* 
*   File: scorpion/src/xref/interpreter/main.c 
*				 					* 
*   Copyright (C) 1991 Dean Throop
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									*
*   Function: This module contains the xref interpreter.             	*
*           The interpreter reads a decorated xref syntax tree and       *
*           an IDL instance file (in IDLREAD format) and interprets     *
*           the directives to produce the output.                       *
*									*
*	Garbage collection of memory_constructed_sequences is partial.  *
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: /phi/softlab2/IDLToolkit/distribution/4.0/idlsystem2/src/xref/interpreter/RCS/main.c,v 4.0 89/04/12 08:29:52 cheung Exp Locker: cheung $";
#endif 

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	main.c,v $
 * Revision 4.0  89/04/12  08:29:52  cheung
 * main.c  Ver 4.0
 * 
 * Revision 3.9  89/03/26  01:06:57  cheung
 * main.c  Ver 3.9
 * 
 * Revision 3.9  89/03/02  13:58:04  cheung
 * main.c  Ver 3.9
 * 
 * Revision 3.8  89/03/02  13:42:53  cheung
 * main.c  Ver 3.9
 * 
*									*
*   Edit Log:								*
*     DEC 21 1987 (rts) Modified to be consistent with changes to IDL   *
*                       specification.                                  *
*     May 2 1986 (ddt) Created revision history.			*
*									*
\* ******************************************************************* */

/*
 * xref_interpreter usage:
 *
 * 	xref_interpreter directives_file idl_reader_data_file
 *
 * Both arguments are required, however an argument of "-" means
 * read stdin.  All output goes to stdout, except errors to stderr.
 */

#include "stdio.h"
#include "interpreter.h"
#include <ctype.h>
#define MAXSTRLEN 10000
#define MAXRECURSIVE 20		/* number of recursive calls to Reorder */

Global global_data;
int current_type_invocation = 0;
int numrecursive = 0;
predicate_declOrPredef globalpredicate[MAXRECURSIVE];

/* cast for passing NULL */
constructed_sequence IDLcons_seq;
# define NULLToconstructed_sequence \
        (IDLcons_seq.IDLinternal = 0,IDLcons_seq)


/* Defined types of forward defined functions. */
constructed_sequence memory_constructed_sequence();
IDLVALUE evaluate_atom();
Boolean evaluate_binarypredicate();
Boolean evaluate_unarypredicate();

main(argc,argv)
int argc;
char **argv;
{

  nodeDesc IDLread();
  FILE * idl_data_file;
  FILE * directives_file;
  constructed_sequence thisseq;
  void exit();

  if(argc != 3) {
    (void) fprintf(stderr,
	    "ERROR Wrong usage, usage is: %s directives_file idl_reader_data_file\n",
	    argv[0]);
    exit(1);
  }

  if(strcmp("-",argv[1]) == 0){
    directives_file = stdin;
  } else {
    directives_file = fopen(argv[1],"r");
  }
  if(NULL == directives_file){
    (void) fprintf(stderr,"ERROR can't access directives file %s\n",
	    argv[1]);
    exit(1);
  }

  if(strcmp("-",argv[2]) == 0){
    idl_data_file = stdin;
  } else {
    idl_data_file = fopen(argv[2],"r");
  }
  if(NULL == idl_data_file){
    (void) fprintf(stderr,"ERROR can't access idl data file %s\n",
	    argv[2]);
    exit(1);
  }

  global_data = NGlobal;
  initializeSEQcache(global_data->int_caches);
  global_data->int_body = read_syntax_tree(directives_file);
  global_data->int_data = IDLread(fileno(idl_data_file));

  (void) fclose(directives_file);
  (void) fclose(idl_data_file);

  thisseq.Vvirtual_sequence = Nvirtual_sequence;
  evaluate_sequence(global_data->int_body->syn_sequence, thisseq);
  return 0;
}

evaluate_expression(thisexpression)
expression thisexpression;
{
  switch((typeof(thisexpression))){
  case Kunary_op:
    {
      int sub_result =
	evaluate_expression(thisexpression.Vunary_op->syn_operand);
      /* Should check that type is not but will just assume it. */
      return !sub_result;
    }
  case Kbinary_op:
    {
      int left_result =
	evaluate_expression(thisexpression.Vbinary_op->syn_left_operand);
      int right_result =
	evaluate_expression(thisexpression.Vbinary_op->syn_right_operand);
      switch((typeof(thisexpression.Vbinary_op->syn_op_code))){ 
      case Kand_op:
	{
	  return (left_result && right_result);
	}
      case Kor_op:
	{
	  return (left_result || right_result);
	}
      default:
	{
	  (void) fprintf(stderr,"ERROR unknown Binary_op code type\n");
	}
      }
      break;
    }
  case Krelational_expression:
    {
      IDLVALUE left_operand  = 
	evaluate_atom(thisexpression.Vrelational_expression->syn_left_operand, NULLToconstructed_sequence);
      IDLVALUE right_operand = 
	evaluate_atom(thisexpression.Vrelational_expression->syn_right_operand, NULLToconstructed_sequence);

      if((left_operand.VnodeDesc == NULL) || (right_operand.VnodeDesc == NULL)){
	(void) fprintf(stderr,"ERROR Can't resolve reference for expression\n");
	exit(1);
      }

      switch((typeof(thisexpression.Vrelational_expression->syn_op_code))){
      case Kequal_op:
	{
	  return compare_values(left_operand, right_operand) == 0;
	}
      case Knot_equal_op:
	{
	  return compare_values(left_operand, right_operand) != 0;
	}
      case Kless_than_op:
	{
	  return compare_values(left_operand, right_operand) < 0;
	}
      case Kless_than_or_equal_op:
	{
	  return compare_values(left_operand, right_operand) <= 0;
	}
      case Kgtr_than_op:
	{
	  return compare_values(left_operand, right_operand) > 0;
	}
      case Kgtr_than_or_equal_op:
	{
	  return compare_values(left_operand, right_operand) >= 0;
	}
      case Ksame_op:
	{
	  if (((typeof(left_operand) != KnodeDesc)
	      || (typeof(right_operand) != KnodeDesc))) {
	    (void) fprintf(stderr, "ERROR: Same may be applied only to nodes.\n");
	    exit(1);
	  }
	  return (left_operand.VnodeDesc == right_operand.VnodeDesc);
	}
      case Kin_op:
	{
	  return compare_values(left_operand, right_operand) == 0;
	}
      default:
	{
	  (void) fprintf(stderr,"ERROR Unkown relation op code type\n");
	}
      }
      break;
    }
  case Kpredicate_call:
    {
      predicate_call thisp_c = thisexpression.Vpredicate_call;
      IDLVALUE arg1, arg2;
      atom aatom;

      retrievefirstSEQatom(thisp_c->syn_actuals, aatom);
      arg1 = evaluate_atom(aatom, NULLToconstructed_sequence);
/*    if (arg1==NULL) {
	(void) fprintf(stderr, "Error: argument to predicate evaluates to NULL\n");
	exit(1);
      }
*/

      if (lengthSEQatom(thisp_c->syn_actuals)==1)
	return (evaluate_unarypredicate(thisp_c->syn_predicate->sem_entity,arg1));
      else {
	ithinSEQatom(thisp_c->syn_actuals, 2, aatom);
	arg2 = evaluate_atom(aatom, NULLToconstructed_sequence);
/*f (arg2==NULL) {
	  (void) fprintf(stderr, "Error: second argument to predicate evaluates to NULL\n");
	  exit(1);
	}
*/
	return (evaluate_binarypredicate(thisp_c->syn_predicate->sem_entity,arg1,arg2));
      }
    }
  case Kempty_call:
    {
      constructed_sequence memory_sequence;

      memory_sequence = memory_constructed_sequence();
      evaluate_sequence(thisexpression.Vempty_call->syn_sequence, memory_sequence);
      return (emptySEQIDLVALUE(memory_sequence.Vphysical_sequence->int_values));
    }
  default:
      (void) fprintf(stderr,"ERROR Unknown expression type\n");
    }
  return FALSE;
}

int Upperstrcmp(first, second)
char *first, *second;
{
  char newone[MAXSTRLEN];
  char newtwo[MAXSTRLEN];
  char *src;
  char *dest;
  char temp;

  src = first;
  dest = newone;
  while (*dest++ = (temp = *src++, (islower(temp)) ? toupper(temp) : temp)) ;
  src = second;
  dest = newtwo;
  while (*dest++ = (temp = *src++, (islower(temp)) ? toupper(temp) : temp)) ;
  return(strcmp(newone, newtwo));
}

compare_values(left_operand,right_operand)
IDLVALUE left_operand, right_operand;
{
  /*
    Returns -2, -1, 0, or 1
    -2:	undefined values
    -1: right > left (or left NOT In right, if right is a set or sequence)
    0: values were equal (or left In right, if right is a set or sequence)
    1: left > right
    */

  if (typeof(right_operand) == KsetDesc) {
    SEQIDLVALUE sIDLV;
    IDLVALUE thisIDLV;

    foreachinSEQIDLVALUE((right_operand.VsetDesc->value), sIDLV, thisIDLV) {
      if (typeof(left_operand)==typeof(thisIDLV)) {
	switch (typeof(thisIDLV)) {
	case KintegerDesc:
	  if (thisIDLV.VintegerDesc->value==left_operand.VintegerDesc->value)
	    return 0;
	  break;
	case KrationalDesc:
	  if (thisIDLV.VrationalDesc->value==left_operand.VrationalDesc->value)
	    return 0;
	  break;
	case KbooleanDesc:
	  if (thisIDLV.VbooleanDesc->value==left_operand.VbooleanDesc->value)
	    return 0;
	  break;
	case KstringDesc:
	  if (thisIDLV.VstringDesc->value==left_operand.VstringDesc->value)
	    return 0;
	  break;
	}
      }
    }
    return(-1);
  }
  else if (typeof(right_operand) == KsequenceDesc) {
    SEQIDLVALUE sIDLV;
    IDLVALUE thisIDLV;

    foreachinSEQIDLVALUE((right_operand.VsequenceDesc->value), sIDLV, thisIDLV) {
      if (typeof(left_operand)==typeof(thisIDLV))
	switch (typeof(thisIDLV)) {
	case KintegerDesc:
	  if (thisIDLV.VintegerDesc->value==left_operand.VintegerDesc->value)
	    return 0;
	  break;
	case KrationalDesc:
	  if (thisIDLV.VrationalDesc->value==left_operand.VrationalDesc->value)
	    return 0;
	  break;
	case KbooleanDesc:
	  if (thisIDLV.VbooleanDesc->value==left_operand.VbooleanDesc->value)
	    return 0;
	  break;
	case KstringDesc:
	  if (thisIDLV.VstringDesc->value==left_operand.VstringDesc->value)
	    return 0;
	  break;
	}
    }
    return(-1);
  }
  else if( (typeof(left_operand)) != (typeof(right_operand)) ) {
    (void) fprintf(stderr,"ERROR arguments of predicate not of same type\n");
    return -2;
  }

  switch((typeof(left_operand))){
  case KintegerDesc:
    {
      if (left_operand.VintegerDesc->value == right_operand.VintegerDesc->value){
	return 0;
      } else { 
	if (left_operand.VintegerDesc->value > right_operand.VintegerDesc->value){
	  return 1;
	} else {
	  return -1;
	}
      }
    }
  case KrationalDesc:
    {
      if (left_operand.VrationalDesc->value == right_operand.VrationalDesc->value){
	return 0;
      } else { 
	if (left_operand.VrationalDesc->value > right_operand.VrationalDesc->value){
	  return 1;
	} else {
	  return -1;
	}
      }
    }
  case KbooleanDesc:
    {
      if (left_operand.VbooleanDesc->value == right_operand.VbooleanDesc->value){
	return 0;
      } else { 
	if (left_operand.VbooleanDesc->value > right_operand.VbooleanDesc->value){
	  return 1;
	} else {
	  return -1;
	}
      }
    }
  case KstringDesc:
    {
      int result = Upperstrcmp(left_operand.VstringDesc->value,
			       right_operand.VstringDesc->value);
      if (result == 0){
	return 0;
      } else { 
	if ( result > 0){
	  return 1;
	} else {
	  return -1;
	}
      }
    }
  case KsetDesc:
  case KsequenceDesc:
    {
      (void) fprintf(stderr,"ERROR can't compare sets or sequences\n");
      return -2;
    }
  case KnodeDesc:
    {				/* perform a recursive comparison of the
				   nodes' attributes */
      attrDesc l_attribute;
      attrDesc r_attribute;
      SEQattrDesc l_state;
      SEQattrDesc r_state;
      int r_index;
      int l_index;
      int r_max_index;
      int l_max_index;

/* This is incorrect, because we may have loops, and there is no checking
   for loops */
      (void) fprintf(stderr, "ERROR: = for nodes is not implemented yet.\n");
      exit(1);
      if (left_operand.VnodeDesc->name != right_operand.VnodeDesc->name){
	return -1;
      }
      r_max_index = 0;
      l_max_index = 0;
      l_index = 0;
      foreachinSEQattrDesc(left_operand.VnodeDesc->attributes,l_state,l_attribute){
	r_index = 0;
	foreachinSEQattrDesc(right_operand.VnodeDesc->attributes,r_state,r_attribute){
	  if(r_index == l_index){
	    if(r_attribute->name != l_attribute->name){
	      return -1;
	    }
	  }
	  if(r_max_index < r_index){
	    r_max_index = r_index;
	  }
	}
	if(l_max_index < l_index){
	  l_max_index = l_index;
	}
      }

      if(r_max_index != l_max_index){
	return -1;
      }
    }
  default:
    {
      (void) fprintf(stderr,"ERROR Unknown type of argument for relation\n");
      return -2;
    }
  }
}

evaluate_sequence(thissequence,thisseq)
sequence thissequence;
constructed_sequence thisseq;
{
  sequence_element element;
  SEQsequence_element state;
  foreachinSEQsequence_element(thissequence->syn_values,state,element){
    switch (typeof(element)){
    case Ktype_construct:
      {
	evaluate_type_construct(element.Vtype_construct,thisseq);
	break;
      }
    case Kfunction_call:
      {
	evaluate_function_call(element.Vfunction_call,thisseq);
	break;
      }
    case Ktail:
      {
	evaluate_tail(element.Vtail,thisseq);
	break;
      }
    case Klet_construct:
      {
	evaluate_let_construct(element.Vlet_construct, thisseq);
	break;
      }
    case Kcase_statement:
      {
	evaluate_case_statement(element.Vcase_statement,thisseq);
	break;
      }
    case Kfor_statement:
      {
	evaluate_for_statement(element.Vfor_statement,thisseq);
	break;
      }
    case Kreorder:
      {
	evaluate_reorder(element.Vreorder,thisseq);
	break;
      }
    case Kfilter:
      {
	evaluate_filter(element.Vfilter,thisseq);
	break;
      }
    case Kunique:
      {
	evaluate_unique(element.Vunique,thisseq);
	break;
      }
    case Kintersection:
      {
	evaluate_intersection(element.Vintersection,thisseq);
	break;
      }
    case Kdifference:
      {
	evaluate_difference(element.Vdifference,thisseq);
	break;
      }
    case Kextra:
      {
	evaluate_extra(element.Vextra,thisseq);
	break;
      }
    case Kreference:
    case Kstring_constant:
    case Khead:
      {
	IDLVALUE resolved_atom;

	resolved_atom = evaluate_atom(element.Vatom, thisseq);
	break;
      }
    default:
      {
	(void) fprintf(stderr,
		"ERROR Unknown sequence element type (%d)\n",
		typeof(element));
	break;
      }
    }
  }
}

evaluate_type_construct(thistype_construct,thisseq)
type_construct thistype_construct;
constructed_sequence thisseq;
{
  constructed_sequence memory_sequence;
  SEQIDLVALUE state;
  IDLVALUE element;
  SEQcache cache_state;
  cache cache_element, found_element;
  SEQIDLVALUE result_state;
  IDLVALUE result_element;

  current_type_invocation++;
  memory_sequence = memory_constructed_sequence();
  evaluate_sequence(thistype_construct->syn_sequence, memory_sequence);

  foreachinSEQIDLVALUE(memory_sequence.Vphysical_sequence->int_values,state,element){
    /* Search cache for sequence of elements. */
    found_element = NULL;
    foreachinSEQcache(global_data->int_caches, cache_state, cache_element) {
      if(((cache_element->int_element.VnodeDesc)==(element.VnodeDesc))
	 && (cache_element->int_node_name == thistype_construct->syn_name->lex_value)
	 && ((cache_element->int_attribute.Vuser_name) == (thistype_construct->syn_attribute.Vuser_name)))
	{
	  found_element = cache_element;
	  break;
	}
    }
    cache_element = found_element;

    /* If not in cache, do the work */
    if(NULL == cache_element){
      cache_element = Ncache;
      cache_element->int_element = element;
      cache_element->int_node_name = thistype_construct->syn_name->lex_value;
      cache_element->int_attribute = thistype_construct->syn_attribute;
      cache_element->int_result = memory_constructed_sequence();
      appendfrontSEQcache(global_data->int_caches, cache_element);

      if (typeof(thistype_construct->syn_attribute) == Kuser_name) {
	select_nodes_with_attribute(cache_element->int_element,
				    cache_element->int_node_name,
				    cache_element->int_attribute.Vuser_name->lex_value,
				    cache_element->int_result);
      }
      else {
	select_nodes(cache_element->int_element,
		     cache_element->int_node_name,
		     cache_element->int_result);
      }
    }				/* test to search node for types */

    /* block to copy from cache to sequence */

    foreachinSEQIDLVALUE(cache_element->int_result.Vphysical_sequence->int_values, result_state,result_element){
      add_to_constructed_sequence(thisseq, result_element);
    }				/* Loop to copy from cache to sequence */
  }				/* Loop to get types for each element of the sequence */
  free_constructed_sequence(memory_sequence);
}

select_nodes(node,name,thisseq)
IDLVALUE node;
String name;
constructed_sequence thisseq;
{
				/* have we seen this value this time around? */
  if(node.IDLclassCommon->int_type_invocation==current_type_invocation) 
    return;

  node.IDLclassCommon->int_type_invocation = current_type_invocation;

  switch (typeof(node)){
  case KnodeDesc:
    {  attrDesc element;
       SEQattrDesc state;

      if(node.VnodeDesc->name == name){
	add_to_constructed_sequence(thisseq, node);
      }
      foreachinSEQattrDesc(node.VnodeDesc->attributes,state,element){
	select_nodes(element->value, name, thisseq);
      }
      break;
    }
  case KsetDesc:
    {
      IDLVALUE element;
      SEQIDLVALUE state;

      foreachinSEQIDLVALUE(node.VsetDesc->value,state,element){
	select_nodes(element, name, thisseq);
      }
      break;
    }
  case KsequenceDesc:
    {
      IDLVALUE element;
      SEQIDLVALUE state;
      foreachinSEQIDLVALUE(node.VsequenceDesc->value,state,element){
	select_nodes(element, name, thisseq);
      }
      break;
    }
  }
}

select_nodes_with_attribute(node,name,attr,thisseq)
IDLVALUE node;
String name;
String attr;
constructed_sequence thisseq;
{
				/* have we seen this value this time around? */
  if(node.IDLclassCommon->int_type_invocation==current_type_invocation) 
    return;

  node.IDLclassCommon->int_type_invocation = current_type_invocation;

  switch (typeof(node)) {
  case KnodeDesc:
    {
      attrDesc element;
      SEQattrDesc state;

      if(node.VnodeDesc->name == name){
	add_to_constructed_sequence(thisseq, node);
      }
      foreachinSEQattrDesc(node.VnodeDesc->attributes,state,element){
	if (element->name == attr) {
	  select_nodes_with_attribute(element->value, name, attr, thisseq);
	  break;
	}
      }
      break;
    }
  case KsetDesc:
    {
      IDLVALUE element;
      SEQIDLVALUE state;

      foreachinSEQIDLVALUE(node.VsetDesc->value,state,element){
	select_nodes_with_attribute(element, name, attr, thisseq);
      }
      break;
    }
  case KsequenceDesc:
    {
      IDLVALUE element;
      SEQIDLVALUE state;
      foreachinSEQIDLVALUE(node.VsequenceDesc->value,state,element){
	select_nodes_with_attribute(element, name, attr, thisseq);
      }
      break;
    }
  }
}

evaluate_function_call(thisfunction_call,thisseq)
function_call thisfunction_call;
constructed_sequence thisseq;
{
  atom actual_atom;
  SEQatom state;
  int i = 0;
  local thisformal;
    
  foreachinSEQatom(thisfunction_call->syn_actuals,state, actual_atom) {
    i++;
    ithinSEQlocal(thisfunction_call->syn_function->sem_entity->syn_formals,i,thisformal);
    if (thisformal != NULL) {
      thisformal->int_value = evaluate_atom(actual_atom, NULLToconstructed_sequence);
    }
  }
  evaluate_sequence(thisfunction_call->syn_function->sem_entity->syn_value->syn_sequence,
		    thisseq);
}

evaluate_tail(thistail,thisseq)
tail thistail;
constructed_sequence thisseq;
{
  constructed_sequence new_sequence;

  new_sequence = memory_constructed_sequence();
  evaluate_sequence(thistail->syn_sequence, new_sequence);

  if (lengthSEQIDLVALUE(new_sequence.Vphysical_sequence->int_values) > 0) {
    add_to_constructed_sequence(thisseq,(tailSEQIDLVALUE(new_sequence.Vphysical_sequence->int_values))->value);
  }
  else {
    (void) fprintf(stderr, "Tail called on a sequence with no elements.\n");
    exit(1);
  }
}

evaluate_case_statement(thiscase_statement,thisseq)
case_statement thiscase_statement;
constructed_sequence thisseq;
{
  IDLVALUE actual;
  branch thisbranch;
  SEQbranch state;

  actual = evaluate_atom(thiscase_statement->syn_atom, NULLToconstructed_sequence);
  if(typeof(actual) != KnodeDesc){
    (void) fprintf(stderr,"ERROR Case statement atom not a node\n");
    return;
  }
  thiscase_statement->syn_local->int_value = actual;

  foreachinSEQbranch(thiscase_statement->syn_branches, state,thisbranch)
    {
      switch((typeof(thisbranch->syn_key))){
      case Kuser_name:
	{
	  if(actual.VnodeDesc->name == thisbranch->syn_key.Vuser_name->lex_value){
	    evaluate_sequence(thisbranch->syn_value->syn_sequence, thisseq);
	    goto found_branch;
	  }
	  break;
	}
      case Kdefault_construct:
	{
	  evaluate_sequence(thisbranch->syn_value->syn_sequence,thisseq);
	  goto found_branch;
	}
      default:
	{
	  (void) fprintf(stderr,"ERROR Unknown case branch type\n");
	}
      }
    } found_branch: 0;
}

evaluate_for_statement(thisfor_statement,thisseq)
for_statement thisfor_statement;
constructed_sequence thisseq;
{
  IDLVALUE element;
  SEQIDLVALUE state;
  constructed_sequence memory_sequence;

  memory_sequence = memory_constructed_sequence();
  evaluate_sequence(thisfor_statement->syn_sequence,memory_sequence);
  foreachinSEQIDLVALUE(memory_sequence.Vphysical_sequence->int_values,state,element){
    thisfor_statement->syn_local->int_value = element;
    evaluate_sequence(thisfor_statement->syn_body->syn_sequence,thisseq);
  }
}
		
evaluate_let_construct(thislet_construct,thisseq)
let_construct thislet_construct;
constructed_sequence thisseq;
{
  SEQlet_variable Slet_variable;
  let_variable Alet_variable;
  constructed_sequence memory_sequence;

  foreachinSEQlet_variable(thislet_construct->syn_assignments,
			   Slet_variable, Alet_variable) {
    memory_sequence = memory_constructed_sequence();
    evaluate_sequence(Alet_variable->syn_sequence, memory_sequence);
    Alet_variable->int_value = memory_sequence;
  }
  evaluate_sequence(thislet_construct->syn_body->syn_sequence, thisseq);
}

Boolean ReorderComparison(this, that)
IDLVALUE this, that;
{
  return(evaluate_binarypredicate(globalpredicate[numrecursive], that, this));
}

evaluate_reorder(thisreorder,thisseq)
reorder thisreorder;
constructed_sequence thisseq;
{
  constructed_sequence input_sequence;
  IDLVALUE input_element;
  SEQIDLVALUE input_state;

  input_sequence = memory_constructed_sequence();
  evaluate_sequence(thisreorder->syn_sequence,input_sequence);
  numrecursive++;
  globalpredicate[numrecursive] = thisreorder->syn_predicate->sem_entity;
  if (numrecursive > MAXRECURSIVE) {
    (void) fprintf(stderr, "Reorder called recursively too many times.\n");
    exit(1);
  }
  sortSEQIDLVALUE(input_sequence.Vphysical_sequence->int_values, ReorderComparison);
  numrecursive--;
  foreachinSEQIDLVALUE(input_sequence.Vphysical_sequence->int_values,
		       input_state, input_element){
    add_to_constructed_sequence(thisseq, input_element);
  }
  free_constructed_sequence(input_sequence);
}

evaluate_filter(thisfilter,thisseq)
filter thisfilter;
constructed_sequence thisseq;
{
  constructed_sequence input_sequence;
  IDLVALUE input_element;
  SEQIDLVALUE input_state;

  input_sequence = memory_constructed_sequence();
  evaluate_sequence(thisfilter->syn_sequence,input_sequence);
  foreachinSEQIDLVALUE(input_sequence.Vphysical_sequence->int_values, input_state,input_element) {
    if(evaluate_unarypredicate(thisfilter->syn_predicate->sem_entity, input_element)){
      add_to_constructed_sequence(thisseq, input_element);
    }
  }
  free_constructed_sequence(input_sequence);
}

evaluate_unique(thisunique,thisseq)
unique thisunique;
constructed_sequence thisseq;
{
  constructed_sequence input_sequence;
  constructed_sequence temp_sequence;
  IDLVALUE input_element;
  IDLVALUE temp_element;
  SEQIDLVALUE input_state;
  SEQIDLVALUE temp_state;
  int add_flag;
  predicate_declOrPredef thispredicate;

  input_sequence = memory_constructed_sequence();
  evaluate_sequence(thisunique->syn_sequence,input_sequence);
  thispredicate = thisunique->syn_predicate->sem_entity;
  temp_sequence = memory_constructed_sequence();
  foreachinSEQIDLVALUE(input_sequence.Vphysical_sequence->int_values,input_state,input_element){
    add_flag = TRUE;
    foreachinSEQIDLVALUE(temp_sequence.Vphysical_sequence->int_values,temp_state,temp_element) {
      if(add_flag){
	if(evaluate_binarypredicate(thispredicate,temp_element,input_element)){
	  add_flag = FALSE;
	}
      }
    }
    if(add_flag){
      add_to_constructed_sequence(temp_sequence, input_element);
    }
  }
  foreachinSEQIDLVALUE(temp_sequence.Vphysical_sequence->int_values,temp_state,temp_element){
    add_to_constructed_sequence(thisseq,temp_element);
  }
  free_constructed_sequence(temp_sequence);
  free_constructed_sequence(input_sequence);
}

evaluate_intersection(thisintersection,thisseq)
intersection thisintersection;
constructed_sequence thisseq;
{
  constructed_sequence left_sequence;
  constructed_sequence right_sequence;
  IDLVALUE left_element;
  IDLVALUE right_element;
  SEQIDLVALUE left_state;
  SEQIDLVALUE right_state;
  int add_flag;
  predicate_declOrPredef thispredicate;

  left_sequence = memory_constructed_sequence();
  right_sequence = memory_constructed_sequence();
  evaluate_sequence(thisintersection->syn_left_sequence,left_sequence);
  evaluate_sequence(thisintersection->syn_right_sequence,right_sequence);
  thispredicate = thisintersection->syn_predicate->sem_entity;
  foreachinSEQIDLVALUE(left_sequence.Vphysical_sequence->int_values,left_state,left_element){
    add_flag = FALSE;
    foreachinSEQIDLVALUE(right_sequence.Vphysical_sequence->int_values,right_state,right_element){
      if(add_flag == FALSE){
	if(evaluate_binarypredicate(thispredicate,left_element,right_element)){
	  add_flag = TRUE;
	}
      }
    }
    if(add_flag){
      add_to_constructed_sequence(thisseq, left_element);
    }
  }
  free_constructed_sequence(left_sequence);
  free_constructed_sequence(right_sequence);
}

evaluate_difference(thisdifference,thisseq)
difference thisdifference;
constructed_sequence thisseq;
{
  constructed_sequence left_sequence;
  constructed_sequence right_sequence;
  IDLVALUE left_element;
  IDLVALUE right_element;
  SEQIDLVALUE left_state;
  SEQIDLVALUE right_state;
  int add_flag;
  predicate_declOrPredef thispredicate;
  
  left_sequence = memory_constructed_sequence();
  right_sequence = memory_constructed_sequence();
  evaluate_sequence(thisdifference->syn_left_sequence,left_sequence);
  evaluate_sequence(thisdifference->syn_right_sequence,right_sequence);
  thispredicate = thisdifference->syn_predicate->sem_entity;
  foreachinSEQIDLVALUE(left_sequence.Vphysical_sequence->int_values,left_state,left_element){
    add_flag = TRUE;
    foreachinSEQIDLVALUE(right_sequence.Vphysical_sequence->int_values,right_state,right_element){
      if(add_flag == TRUE){
	if(evaluate_binarypredicate(thispredicate,left_element,right_element)){
	  add_flag = FALSE;
	}
      }
    }
    if(add_flag){
      add_to_constructed_sequence(thisseq, left_element);
    }
  }
  free_constructed_sequence(left_sequence);
  free_constructed_sequence(right_sequence);
}

evaluate_extra(thisextra,thisseq)
extra thisextra;
constructed_sequence thisseq;
{
  /*
    create memory_nodeDesc_sequence rmns lmns
    call evalute_sequence on extra.left_sequence
    call evalute_sequence on extra.right_sequence
    
    Count elements in right sequence.
    skip elements in left sequence and add remainder to result_sequence
    
    free temp_node_sequence
    */
  constructed_sequence left_sequence;
  constructed_sequence right_sequence;

  left_sequence = memory_constructed_sequence();
  right_sequence = memory_constructed_sequence();
  evaluate_sequence(thisextra->syn_left_sequence,left_sequence);
  evaluate_sequence(thisextra->syn_right_sequence,right_sequence);
  {
    {
      int count;
      IDLVALUE element;
      SEQIDLVALUE state;

      count = 0;
      foreachinSEQIDLVALUE(right_sequence.Vphysical_sequence->int_values,state,element){
	count ++;
      }
      foreachinSEQIDLVALUE(left_sequence.Vphysical_sequence->int_values,state,element){
	if ( count > 0){
	  count --;
	} else {
	  add_to_constructed_sequence(thisseq, element);
	}
      }
    }
  }
  free_constructed_sequence(left_sequence);
  free_constructed_sequence(right_sequence);
}

IDLVALUE evaluate_atom(thisatom, thisseq)
atom thisatom;
constructed_sequence thisseq;
{				/* if thisseq is NULL, then return value */
				/* else append value to thisseq */
  IDLVALUE thisvalue;
  constructed_sequence newseq;

  if (thisseq.IDLinternal == 0) {
    newseq = (memory_constructed_sequence());
  }
  else newseq = thisseq;
  switch (typeof(thisatom)) {
  case Kreference: {
      evaluate_reference(thisatom.Vreference, newseq);
      break;
    }
  case Kstring_constant: {
    IDLVALUE astring;
    
    astring.VstringDesc = NstringDesc;
    astring.VstringDesc->value = thisatom.Vstring_constant->lex_text;
    add_to_constructed_sequence(newseq, astring);
    break;
  }
  case Khead: {
    constructed_sequence new_sequence;
    IDLVALUE thisIDLVALUE;

    new_sequence = memory_constructed_sequence();
    evaluate_sequence(thisatom.Vhead->syn_sequence, new_sequence);

    if (lengthSEQIDLVALUE(new_sequence.Vphysical_sequence->int_values) == 0) {
      (void) fprintf(stderr, "Head called on a sequence with no values.\n");
      exit(1);
    }
    else {
      retrievefirstSEQIDLVALUE((new_sequence.Vphysical_sequence->int_values), thisIDLVALUE);
      add_to_constructed_sequence(newseq, thisIDLVALUE);
    }
    break;
  }
  default: {
    (void) fprintf(stderr, "ERROR in evaluation of atom\n");
    exit(1);
  }
  }
  if (thisseq.IDLinternal == 0) {
    retrievefirstSEQIDLVALUE(newseq.Vphysical_sequence->int_values, thisvalue);
    return(thisvalue);
  }
  else {
    thisvalue.VnodeDesc =NULL;
    return(thisvalue);
  }
}

evaluate_reference(thisreference, thisseq)
reference thisreference;
constructed_sequence thisseq;
{
  IDLVALUE desired_value;	/* value to be returned */
  SEQIDLVALUE SIDLVALUE;
  user_name current_name;
  user_name previous_name;	/* previous name in reference */
  SEQuser_name state;
  attrDesc attribute;
  SEQattrDesc ast;		/* Attribute state for for */
  IDLVALUE new_desired_value;

  if (typeof(thisreference->syn_local)==KRoot_construct)
    desired_value.VnodeDesc = global_data->int_data;
  else {
    localOrletRef thisRef = thisreference->syn_local.VlocalOrletRef;

    if (typeof(thisRef->sem_entity)==Klocal)
      desired_value = thisRef->sem_entity.Vlocal->int_value;
    else {
      foreachinSEQIDLVALUE(thisRef->sem_entity.Vlet_variable->int_value.Vphysical_sequence->int_values,
			   SIDLVALUE,
			   desired_value) {
	add_to_constructed_sequence(thisseq, desired_value);
      }
      return;
    }
  }

  /* Check we found a node. */
  if(desired_value.VnodeDesc == NULL){
    (void) fprintf(stderr,"ERROR null node for reference ");
    if (lengthSEQuser_name(thisreference->syn_attributes) > 0) {
      retrievefirstSEQuser_name(thisreference->syn_attributes, current_name);
      (void) fprintf(stderr,"with first attribute \"%s\"\n", current_name->lex_value);
    }
    else (void) fprintf(stderr,"with no attributes\n");
  }

  previous_name = current_name;	/* primarily to eliminate lint error */

  /* Go down the list of user names finding the associated attribute. */
  foreachinSEQuser_name(thisreference->syn_attributes,state,current_name){

    new_desired_value.VnodeDesc = NULL;
    
    if(typeof(desired_value) != KnodeDesc){
      (void) fprintf(stderr,"ERROR Reference %s not a node\n", previous_name->lex_value);
    }
    foreachinSEQattrDesc(desired_value.VnodeDesc->attributes, ast,attribute)
      {
	if(attribute->name == current_name->lex_value){
	  new_desired_value = attribute->value;
	  break;
	}
      }
    if(new_desired_value.VnodeDesc == NULL){
      (void) fprintf(stderr,"ERROR Can't find attribute %s in node %s\n",
	      current_name->lex_value,
	      desired_value.VnodeDesc->name);
    }
    desired_value = new_desired_value;
    previous_name = current_name;
  }
  add_to_constructed_sequence(thisseq, desired_value);
}

/* constructed_sequence  is an abstract type with following operations: */

add_to_constructed_sequence(thisseq,value)
constructed_sequence thisseq;
IDLVALUE value;
	/* Adds another node to the result */
	/* Invoker can not reference arg */
{
  if (typeof(thisseq)==Kphysical_sequence){
    switch (typeof(value)){	/* Add to memory sequence */
    case KbooleanDesc:
    case KintegerDesc:
    case KrationalDesc:
    case KstringDesc:
    case KnodeDesc:
      {	appendrearSEQIDLVALUE(thisseq.Vphysical_sequence->int_values,
			      value);
	break;
      }
    case KsequenceDesc:
      {	IDLVALUE element;
	SEQIDLVALUE state;
	foreachinSEQIDLVALUE(value.VsequenceDesc->value,state,element){
	  add_to_constructed_sequence(thisseq,element);
	}
	break;
      }
    case KsetDesc:
      {	IDLVALUE element;
	SEQIDLVALUE state;
	foreachinSEQIDLVALUE(value.VsetDesc->value,state,element){
	  add_to_constructed_sequence(thisseq,element);
	}
	break;
      }
    default:
      {	(void) fprintf(stderr,"ERROR: Unknown IDLVALUE type (%d).\n", typeof(value));
      }
    }
  } else {
    switch (typeof(value)){
    case KintegerDesc:
    case KrationalDesc:
      {	(void) printf( value.VIDLnumber.IDLclassCommon->stringRep);
	break;
      }
    case KstringDesc:
      {	(void) printf( value.VstringDesc->value);
	break;
      }
    case KnodeDesc:
      {	(void) printf("Printing node %s",value.VnodeDesc->name);
	break;
      }
    case KsequenceDesc:
      {	(void) printf( "Printing sequences unimplemented\n");
	break;
      }
    case KsetDesc:
      {	(void) printf( "Printing sets unimplemented\n");
	break;
      }
    default:
      {	(void) fprintf(stderr,"ERROR: Unknown IDLVALUE type\n");
      }
    }
  }
}

constructed_sequence memory_constructed_sequence()
	/* Creates sequence which is memory resident */
	/* Can be iterated over */
	/* Must be deleted when finished */
{
  constructed_sequence new_sequence;

  new_sequence.Vphysical_sequence = Nphysical_sequence;
  initializeSEQIDLVALUE(new_sequence.Vphysical_sequence->int_values);

  return (new_sequence);
}

/*ARGSUSED*/
free_constructed_sequence(thisseq)
constructed_sequence thisseq;
	/* Frees the memory used by a memory_sequence */
	/* sequence is the sequence to free */
{
/*  if (typeof(thisseq)==Kphysical_sequence){
    while(! emptySEQIDLVALUE(thisseq.Vphysical_sequence->int_values)){
      removefirstSEQIDLVALUE(thisseq.Vphysical_sequence->int_values);
    }
  }
  free((char *)thisseq.Vphysical_sequence);
*/
}

Boolean evaluate_binarypredicate(thispredicate, arg1, arg2)
predicate_declOrPredef thispredicate;
IDLVALUE arg1, arg2;
{
  local thisformal;

  if (typeof(thispredicate)==KIsPredicate)
    {
      if ((typeof(arg1)==KnodeDesc) && (typeof(arg2)==KstringDesc))
	return(arg1.VnodeDesc->name == arg2.VstringDesc->value);
      else return FALSE;
    }
  else {			/* user-defined predicate */
    retrievefirstSEQlocal(thispredicate.Vpredicate_declaration->syn_formals, thisformal);
    thisformal->int_value = arg1;
    ithinSEQlocal(thispredicate.Vpredicate_declaration->syn_formals, 2, thisformal);
    thisformal->int_value = arg2;
    return evaluate_expression(thispredicate.Vpredicate_declaration->syn_value);
  }
}

Boolean evaluate_unarypredicate(thispredicate, arg)
predicate_declOrPredef thispredicate;
IDLVALUE arg;
{
  local thisformal;

  if (typeof(thispredicate)==KIsPredicate) {
    (void) fprintf(stderr, "ERROR: Illegal unary predicate\n");
    exit(1);
  }
  retrievefirstSEQlocal(thispredicate.Vpredicate_declaration->syn_formals, thisformal);
  thisformal->int_value = arg;
  return evaluate_expression(thispredicate.Vpredicate_declaration->syn_value);
}
