/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/utilities.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	Utilities.c is a collection of functions which all
 *			functions may use as menial tasks.  They are generally
 *			low level, and thus not appropriate for coding within
 *			a module.
 */

/**** INCLUDE FILES ****/

#include <stdio.h>
#include <ctype.h>
#include "global.h"
#include "history.h"
#include "utilities.h"


void exit();
char *malloc(),*calloc();

/**** DECLARATIONS ****/

static int	Instance_has_been_modified = 0;


/**** CODE ****/

/*
 *	Routine :	exception()
 *
 *	Description :	Exception() is the error message printer for
 *			treewalk.  In most cases, the control passes back
 *			to the calling function, but there are critical
 *			errors which causes a premature exit().
 *
 *	Arguments :	error 	- (IN) the parameter of the error message
 *				  to print.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
/*VARARGS*/
void exception(error, arg1, arg2, arg3)
int	 error;
char	*arg1, *arg2, *arg3;
{
	char	*s;

    switch ( error )
    {
	case BAD_COMMAND_LINE:
	  (void)fprintf(stderr, "USAGE: %s <filename>\n", arg1);
	  exit(-1);

	case AT_ROOT_NODE:
	  s = "Unable to traverse beyond the Root node\n";
	  break;

	case BAD_COMMAND:
	  s = "'%s' is not a valid command\n";
	  break;

	case NUM_OF_ARGS:
	  s = "Incorrect number of arguments to %s command\n";
	  break;

	case OUT_OF_BOUNDS:
	  s = "The given %s position is out of range for the %s\n";
	  break;

	case BAD_OPTION:
	  s = "Unable to %s a %s type\n";
	  break;

	case ILLEGAL_CHARACTER:
	  s = "Invalid character in input line\n";
	  break;

	case FILE_NONEXISTENT:
	  (void)fprintf(stderr, "Input file '%s' does not exist\n", arg1);
	  exit(-1);

	case BAD_LABEL:
	  s = "The name '%s' is not a valid identifier\n";
	  break;

	case UNABLE_TO_ACCESS:
	  s = "Unable to %s file %s\n";
	  break;

	case EOF_FOUND:
	  (void)fprintf(stderr, "\nEnd of File encountered - Exiting %s\n", arg1);
	  exit(-1);

	case NONNUM_ARG:
	  s = "The %s command requires numerical arguments\n";
	  break;

	case NOT_HERE:
	  s = "The '%s' command has not been implmented\n";
	  break;

	case TYPE_MISMATCH:
	  s =  "The new item must be of %s type\n";
	  break;

	case BAD_WRITE:
	  s = "in IDLwrite. Write aborted\n";
	  break;

	case OUT_OF_MEMORY:
	  (void)fprintf(stderr, "ERROR - OUT OF MEMORY. EXITING %s.\n", Program_name);
	  exit(-1);

	case FILE_OPEN_ERROR:
	  s = "Unable to open %s file\n";
	  break;

	case BAD_READ:
	  s = "in IDLread; Read aborted.\n";
	  break;

	case LIST_STACK_EMPTY:
	  s = "There is no current set or sequence on List Stack\n";
	  break;

	case BAD_RELABEL:
	  s = "Labels must be in the form of 'L<digit><digit>...'\n";
	  break;

	case DUPLICATE_LABEL:
	  s = "The label '%s' is currently in use by another node in the instance\n";
	  break;

	case EMPTY_SYM_TABLE:
	  s = "There are no names defined in indentifier table\n";
	  break;

	case CRITICAL_ERROR:
	  (void)fprintf(stderr, "CRITICAL ERROR - TREEWALK ABORTED.\n");
	  exit(-1);

    }  /* end of switch */

    (void)fprintf(stderr, "***Error: ");
    (void)fprintf(stderr, s, arg1, arg2, arg3);
    (void)fflush(stderr);

}  /* end of exception() */


/*
 *	Routine :	myfgets()
 *
 *	Description :	This is my own input function which accepts input
 *			from stdin, and returns the number of characters
 *			read from stdin.  The argument n is a maximum
 *			allowable string length.  The function does not
 *			return a carriage return.
 *
 *	Arguments :	line	-- (OUT) the buffer to store characters read.
 *			n	-- (IN) parameter for maximum line length.
 *
 *	Return Value :	number of characters read.
 *
 *	Side Effects :	None.
 */
int myfgets(line, n)
char	*line;
int	 n;
{
	int	 c, m = 0;
	char	*cs;

	cs = line;
	while ( ( c = getc(stdin) ) != EOF )
	{
		if (  c == '\n' )
			break;

		/*
		 *	Run through this filter to ignore 'nasty'
		 *	characters.
		 */

		if ( ! isalnum(c) && ! isspace(c) && ! isprint(c) &&
		     c != '\0' )
		{
			continue;
		}

		if ( m < n )
		{
			m++;
			*cs++ = c;
		}
	}
	if ( c == EOF )
		exception(EOF_FOUND, Program_name);

	*cs = '\0';

	return m;

}  /* end of myfgets() */


/*
 *	Routine :	myfgetc()
 *
 *	Description :	Another one of the my functions which retrieves
 *			input from the keyboard one character at a time,
 *			while aborting treewalk in event of EOF.  Upon a
 *			carriage return, myfgetc() returns a 0.
 */
int myfgetc()
{
	int	c;

	if ( ( c = getchar() ) == EOF )
		exception(EOF_FOUND, Program_name);

	if ( c == '\n' )
		return 0;

	(void)fflush(stdin);
	return c;

}  /* end of myfgetc() */ 


/*
 *	Next are a set of generic stack functions to manipulate
 *	the various linked lists that are used in treewalk.
 */

void initlist(element)
qnode *element;
{
	element->prev = element;
	element->next = element;
	return;
}


qnode *remlist(list)
qnode *list;
{

	qnode *temp;

	if ( isempty(list) )
		return NULL;

	temp = list->next;
	list->next = temp->next;
	temp->next->prev = list;

	return temp;

}


int isempty(list)
qnode *list;
{
	return ( ( list->next == list && list->prev == list ) ? 1 : 0 );
}


void inslist(list, element)
qnode *list;
qnode *element;
{
	element->next = list->next;
	element->prev = list;
	list->next->prev = element;
	list->next = element;
	return;
}


char *malloc_(size)
int	size;
{
	char	*ret;

	/*
	 *	This is to align on an 8-byte boundary.
	 */

	if ( ( size % 8 ) != 0 )
		size = ((int)(size/8) + 1) * 8;

	if ( ( ret = (char *)malloc((unsigned) size) ) == NULL )
		exception(OUT_OF_MEMORY);

	return ret;

}  /* end of malloc_() */


char *calloc_(number, size)
int	number;
int	size;
{
	char	*ret;

	if ( ( ret = (char *)calloc((unsigned) number, (unsigned)size*8) ) == NULL )
		exception(OUT_OF_MEMORY);

	return ret;

}  /* end of calloc_() */


/*
 *	Routine :	Set_changed_flag()
 *
 *	Description :	This routine will set the flag that indicates
 *			that the IDL instance has changed.
 */
void Set_changed_flag()
{

	Instance_has_been_modified = 1;

}  /* end of Set_changed_flag() */


/*
 *	Routine :	Clear_changed_flag()
 *
 *	Description :	This routine will clear the flag that indicates
 *			that the IDL instance has changed.
 */
void Clear_changed_flag()
{

	Instance_has_been_modified = 0;

}  /* end of Clear_changed_flag() */


/*
 *	Routine :	Test_changed_flag()
 *
 *	Description :	This will return the status of the
 *			changed flag.
 */
int Test_changed_flag()
{

	return Instance_has_been_modified;

}  /* end of Test_change_flag() */


/*
 *	Routine :	Query_upon_change()
 *
 *	Description :	This call will query the user if they would still
 *			want to continue the current operation since
 *			the instance has been changed.
 */
int Query_upon_change(operation)
char	*operation;
{
	char	buf[80];

	if ( Instance_has_been_modified == 0 )
		return 1;

	(void)printf("Data has changed. Do you still wish to %s? ( y or n ) [n] : ", operation);

	(void)myfgets(buf, 80);

	if ( ! strcmp(buf, "y") || ! strcmp(buf, "Y") )
		return 1;
	else
		return 0;

}  /* end of Query_upon_change() */


/*
 *	Routine :	Get_default_arg()
 *
 *	Description :	Routines which need to ask the user for additional
 *			arguments to a command should use this call.  
 *
 *	Arguments :	message		(IN) -- The message to prompt
 *					the user to enter argument.
 *			buf		(OUT) -- The buffer to store
 *					the reply.
 *			len		(IN) -- The length of the buffer.
 */
int Get_default_arg(message, buf, len)
char	*message;
char	 buf[];
int	 len;
{

	(void)printf("%s", message);

	(void)myfgets(buf, len);

	if ( buf[0] == '\0' )
		return 0;

	return 1;

}  /* end of Get_default_arg() */


/*
 *	Routine :	Create_history_node()
 *
 *	Description :	All this routine does is allocates a structure
 *			of type history as defined in history.h and
 *			assigns the values to the fields of the structure.
 *			The order of the arguments should be the same
 *			as the order the fields are listed in the definition
 *			of structure history.
 */
history *Create_history_node(lastitem, elementnum, listsize, gotoname,
			     attrname, searchkey, searchvalue)
IDLVALUE	 lastitem;
int		 elementnum, listsize;
char		*gotoname, *attrname;
char		*searchkey, *searchvalue;
{
	history		*newhist;

	newhist = (history *)malloc_(sizeof(history));

	newhist->lastitem = lastitem;
	newhist->elementnum = elementnum;
	newhist->listsize = listsize;
	newhist->gotoname = (gotoname == NULL ? NULL : StringToChar(gotoname));
	newhist->attrname = (attrname == NULL ? NULL : StringToChar(attrname));
	newhist->searchkey = (searchkey == NULL ? NULL :
				StringToChar(searchkey));
	newhist->searchvalue = (searchvalue == NULL ? NULL :
				StringToChar(searchvalue));

	inslist((qnode *)Historylist, (qnode *)newhist);

	return newhist;

}  /* end of Create_history_node() */


/*
 *	Routine :	Destroy_history_node()
 *
 *	Description :	This is a destructor for a structure of type history.
 */
void Destroy_history_node(node)
history	*node;
{
	if ( node->gotoname != NULL )
		free((char *)node->gotoname);

	if ( node->attrname != NULL )
		free((char *)node->attrname);

	if ( node->searchkey != NULL )
		free((char *)node->searchkey);

	if ( node->searchvalue != NULL )
		free((char *)node->searchvalue);

	free((char *)node);

}  /* end of Destroy_history_node() */


/*
 *	Routine :	Validate_label()
 *
 *	Description :	This routine takes a node label and determines if
 *			if of the form 'L<digit>+' in the form of
 *			regular expressions.
 */
int Validate_label(newlabel)
char	*newlabel;
{
	int	i, j;

	/*
	 *	Now it must be checked that the label is of the form
	 *	L<digit>+
	 */

	if ( newlabel[0] != 'L' )
	{
		exception(BAD_RELABEL);
		return 0;
	}
	j = strlen(newlabel);
	for ( i = 1; i < j; i++ )
	{
		if ( ! isdigit(newlabel[i]) )
		{
			exception(BAD_RELABEL);
			return 0;
		}
	}

	/*
	 *	The label is good, so return a positive status to the 
	 *	caller.
	 */

	return 1;

}  /* end of relabel() */
