/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/symbol.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	This file contains the routines necessary to
 *			manipulate the symbol table used to label items.
 *			The 'name', and 'goto' commands are
 *			implemented here.
 *
 *	Functions :	name()
 *			printSymboltable()
 *			go_to()
 */

/**** INCLUDE FILES ****/

#include <stdio.h>
#include <string.h>
#include "global.h"
#include "symbol.h"
#include "history.h"
#include "list.h"
#include "utilities.h"



/**** CODE ****/

/*
 *	Routine :	name()
 *
 *	Description :	This is a feature of treewalk to give an identifier
 *			to the current item, to be used for later reference
 *			by the goto command.  If name is given an argument
 *			then the list of other defined names is searched
 *			for a match.  If one exists, then the current item
 *			will replace the old entry, otherwise a new entry in
 *			the table is created.  If name is given an argument
 *			of NULL, then the list of currently defined names
 *			is printed to the stdout in alphabetical order.  Also
 *			the type of item that the name references is printed
 *			alongside the name.
 *
 *	Arguments :	ident	- (IN) the label to give to the current item.
 *				  if ident == NULL, then print the list
 *		  		  of defined identifiers.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void name(ident)
char	*ident;
{
	void	printSymboltable();

	symbol	*symptr, *temp1;
	int	 notequal = 1;

	/*
	 *	If no argument was given, then by default, print out the
	 *	contents of the symbol table.
	 */

	if ( ident == NULL )
	{
		printSymboltable();
		return;
	}

	/*
	 *	Search through the symbol table, looking for either
	 *	a match or an entry which is lexically greater than
	 *	ident. 
	 */

	symptr = Symboltable->next;
	while ( symptr != Symboltable )
	{
		if ( ( notequal = strcmp(ident, symptr->label) ) <= 0 )
			break;
		else
			symptr = symptr->next;
	}
	
	/*
 	 *	If ident does not already exist, then create a new
	 *	entry and insert it into the Symboltable.
	 */

	if ( notequal )
	{ 
		temp1 = (symbol *)malloc_(sizeof(symbol));
		temp1->label = (char *)malloc_(strlen(ident) + 1);
		(void)strcpy(temp1->label, ident);
		temp1->dest = Currentitem;
		temp1->next = symptr;
		temp1->prev = symptr->prev;
		symptr->prev->next = temp1;
		symptr->prev = temp1;
	}
	else 
	{
		/*
		 * 	There was an existing entry of the same name, so
		 *	just replace the information.
		 */

		free(symptr->label);
		symptr->label = (char *)malloc_(strlen(ident) + 1);
		(void)strcpy(symptr->label, ident);
		symptr->dest = Currentitem;
	}

	return;

}  /* end of name() */


/*
 *	Routine :	printSymboltable()
 *
 *	Description :	This routine simply dumps the contents of the 
 *			symbol table to the screen.
 */
void printSymboltable()
{
	symbol	*symptr;
	int	 i = 0;

	/*
	 *	First scan the symbol table to count the total number of
	 *	entries in the symbol table.
	 */

	for ( symptr = Symboltable->next; symptr != Symboltable;
		symptr = symptr->next )
	{
		i++;
	}

	/*
	 *	If there are no entries in the table, then do not attempt
	 *	to print the contents of the table.
	 */

	if ( i == 0 )
	{
		(void)printf("There are no names currently declared.\n");
		return;
	}

	(void)printf("There are %d names currently declared:\n", i);

	symptr = Symboltable->next;
	while ( symptr != Symboltable )
	{
		(void)printf("\t%s  ", symptr->label);
		 myprintf(symptr->dest);
		symptr = symptr->next;
	}

}  /* end of printSymboltable() */


/*
 *	Routine :	go_to()
 *
 *	Description :	This is the function which will make the current item
 *			whatever is referenced by the name specified.  The
 *			naming of items if assumed to be done by function
 *			name().  If the name does not exist as a label, then
 *			an error message is produced, and no traversal is
 *			made.  When the current item changes, then a record
 *			of the movement must be appended to the history
 *			structure.
 *
 *	Arguments :	place	- (IN) the name of the label of the object
 *				  to travel to.
 *
 *	Return Value :  None.
 *
 *	Side Effects :	None.
 */
void go_to(place)
char 	*place;
{

	symbol	*symptr;
	int	 notequal = 1;
	char	 buf[80];
	void	 list();

	/*
	 *	If the Symboltable is empty, then obviously nothing
	 *	can be done.
	 */

	if ( isempty((qnode *)Symboltable) )
	{
		exception(EMPTY_SYM_TABLE);
		return;
	}

	/*
	 *	If no arguments were given, then obtain a name from the
	 *	user.
	 */

	if ( place == NULL )
	{
		if ( ! Get_default_arg("Destination item : ", buf, 80) )
			return;

		place = buf;
	}

	/*
	 *	Scan through the Symboltable for a duplicate entry.
	 *	The search can terminate if an entry that is lexically
	 *	greater than place exists, since the list is in
	 *	alphabetical order.
	 */

	symptr = Symboltable->next;
	while ( symptr != Symboltable )
	{
		if (  ( notequal = strcmp(symptr->label, place) ) >= 0 )
			break;
		else
			symptr = symptr->next;
	}

	/*
	 *	If the label is not valid in the symbol table.
	 */

	if ( notequal )
	{
		exception(BAD_LABEL, place);
		return;
	}

	/*
	 *	Update the Historylist to handle the change of the current
	 *	item.
	 */

	(void)Create_history_node(Currentitem, 0, 0, place, place,
				      (char *)NULL, (char *)NULL);
	Currentitem = symptr->dest;
	
	list(1, NULL);

}  /* end of go_to() */
