/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/search.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	This is a feature of treewalk to search through
 *			an IDL instance to find specific values for a
 *			given attribute.
 *	Functions :
 */


/**** INCLUDE FILES ****/

#include <stdio.h>
#include <ctype.h>
#include "global.h"
#include "search.h"
#include "list.h"
#include "history.h"
#include "utilities.h"
#include "C/IDLIO.h"

/**** DECLARATIONS ****/

IDLVALUE	SearchList;	/*  This is for constructing a list of */
				/*  matched nodes		       */


/**** CODE ****/

/*
 *	Routine :	search()
 *
 *	Description :	This command feature will search the entire IDL
 *			instance starting from the root node for occurrences
 *			of a specific value from a given attribute name.  The
 *			set of matches, if any, should be placed on the list
 *			stack so that they may be reviewed by the user at thier
 *			convienence.
 *
 *	Arguments :	attribute_name	     (IN) -- The name of the attribute
 *					     to search under each node.
 *			value		     (IN) -- The value to look for
 *					     if an attribute name is matched.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	If any matches are found, they will be added to the
 *			list stack for further inspection by the user.
 */
int search(attribute_name, value, interactive_mode)
char	*attribute_name;
char	*value;
int	 interactive_mode;
{
	int		go_find();
	void		list(),inslist();
   	setseqhist      *listptr;
	int		 listsize,
			 length;
	int		 search_flag;
	char		 buf1[81], buf2[81];
	char		*charptr = buf2;
	int		 i;
	int		 ret_val;


	IDLVALUE	start;

	/*
	 *	If the attribute name given was null, then
	 *	query the user for a search key
	 */

	if ( attribute_name == NULL )
	{
	      if ( ! Get_default_arg("name of attribute to be searched : ", 
				     buf1, 80) )
	      {
			return 0;
	      }

	      attribute_name = buf1;
	}

	/*
	 *	Now search for one of the keywords "Type" or "For".  But
	 *	first convert to lower case, so that the comparision is
	 *	not case sensitive.
	 */

	for ( tempbuf = attribute_name, i = 0; attribute_name[i] != '\0'; i++ )
	{
		char	c;

		if ( isupper(attribute_name[i]) )
			c = tolower(attribute_name[i]);
		else
			c = attribute_name[i];

		tempbuf[i] = c;

	}

	if ( ! strcmp(tempbuf, "for") )
	{
		search_flag = LABEL_SEARCH;
		attribute_name = StringToChar("Label");
	}
	else if ( ! strcmp(tempbuf, "type") )
	{
		search_flag = NODE_SEARCH;
		attribute_name = "Type";
	}
	else
		search_flag = ATTR_SEARCH;


	/*
	 *	If the value given is NULL, then query the user for
	 *	a search value.
	 */

	if ( value == NULL )
	{
	      char	*s;

	      switch ( search_flag )
	      {
		case NODE_SEARCH:
			s = "Node type : ";
			break;

		case LABEL_SEARCH:
			s = "Label : L";
			buf2[0] = 'L';
			charptr = buf2 + 1;
			break;

		case ATTR_SEARCH:
			s = "Value : ";
			break;
	      }

	      if ( ! Get_default_arg(s, charptr, 80) )
			return 0;

	      value = buf2;
	}

	/*	If the value is quoted, then remove the quotes. */
	length = strlen(value);
	if ( (value[0]=='"') && (value[length-1]=='"')  ) {
	  for (i=0; i<length; i++)
	    value[i] = value[i+1];
	  value[length - 2] = '\0';
	}

	/*
	 *	If the search is for a label, then the label needs to
	 *	be checked to ensure that it is indeed a valid label.
	 */

	if ( search_flag == LABEL_SEARCH )
	{
		if ( ! Validate_label(value) )
			return 0;
	}

	/*
	 *	If this search is being done internally, then do not
	 *	create a search list, or print anything to the screen
	 */

	if ( interactive_mode == YES )
	{
		SearchList.VsequenceDesc = NsequenceDesc; 
		initializeSEQIDLVALUE(SearchList.VsequenceDesc->value);

		(void)printf("SEARCHING ...\n");
	}
	else
		SearchList.VsequenceDesc = NULL;

	/*
	 *	Start the search at the Root node.
	 */

	start.VnodeDesc = Root;
	IDLepoch++;		/* to detect reachability */

	/*
	 *	Make the recursive call to perfrom the search.
	 */

	ret_val = go_find(attribute_name, value, start, search_flag, 0);

	/*
	 *	If the search was internal, then just check the return
	 *	value of the recursive call.
	 */

	if ( interactive_mode == NO )
		return ret_val;

	/*
	 *	If the search was interactive, then add the found items
	 *	to the List stack.
	 */

	if ( ( interactive_mode == YES ) &&
	     ! emptySEQIDLVALUE(SearchList.VsequenceDesc->value) )
	{
		/*
		 *	Add the sequence to the list stack
		 */

		listsize = lengthSEQIDLVALUE(SearchList.VsequenceDesc->value);
		(void)Create_history_node(Currentitem, 1, listsize,
				      (char *)NULL, "SEARCH", attribute_name,
				      value);

		ithinSEQIDLVALUE(SearchList.VsequenceDesc->value, 1, start);
		Currentnode = start.VnodeDesc;

	        listptr = (setseqhist *)malloc_(sizeof(setseqhist));
       		listptr->setorseq = SearchList;
        	listptr->begin = Historylist->next;
        	inslist((qnode *)Liststack, (qnode *)listptr);

		(void)printf("Found %d Items\n\n",
			lengthSEQIDLVALUE(SearchList.VsequenceDesc->value));

		list(1, NULL);

		ret_val = 1;

	}
	else
	{
		(void)printf("No matches found\n");
		ret_val = 0;
	}

	return ret_val;

}  /* end of search() */


/*
 *	Routine :	go_find()
 *
 *	Description :	This is the recursive call of the treewalk search
 *			routine.  If a match is found on both the attribute
 *			name and the corresponding value, then the node
 *			from which the attribute came is inserted into a list
 *			of nodes for later review.
 *
 *	Arguments :	attribute_name	   (IN) -- The name of the attribute
 *					   to search under.
 *			value		   (IN) -- The value of the attribute
 *					   to search for.
 *			item		   (IN) -- The object which is to be
 *					   scanned.
 *			search_flag	   (IN) -- The type of search :
 *					    NODE_SEARCH
 *					    LABEL_SEARCH
 *					    ATTR_SEARCH
 *			ret_val		   (IN) -- This is so that
 *					   the return value can bubble
 *					   up through the search.
 *
 *
 *	Return Value :	1 if a match was found, 0 if no match was found.
 *
 *	Side Effects :	If a match is found, then the item is added to a
 *			list of perviously located items.
 */
int go_find(attribute_name, value, item, search_flag, ret_val)
char	 *attribute_name;
char	 *value;
IDLVALUE  item;
int	  search_flag;
int	  ret_val;
{
	SEQattrDesc	remAttr;
	SEQIDLVALUE	remSEQ;
	IDLVALUE	item2;
	attrDesc	nodeAttribute;
	char		buf[256];


	/*
	 *	Depending of what type of IDL entity we have will determine
	 *	how the recursion is handled.
	 */

  	switch ( typeof(item) )
  	{
	  case KnodeDesc:

		/*
	 	 *	If this item has been scanned previously on this search,
	 	 *	then do not search it again, else mark the node as being
		 *	searched.
	 	 */

 		if ( item.VnodeDesc->touched == IDLepoch )
			return ret_val;
		else
			item.VnodeDesc->touched = IDLepoch;

		/*
		 *	If the search is for a node type then check it
		 *	here.
		 */

		if ( search_flag == LABEL_SEARCH )
		{
			if ( ! strcmp(item.VnodeDesc->label, value) )
			{
				add_to_search_sequence(item);
				ret_val = 1;
			}
		}
		else if ( search_flag == NODE_SEARCH )
		{
			if ( ! strcmp(item.VnodeDesc->name, value) )
			{
				add_to_search_sequence(item);
				ret_val = 1;
			}
		}

		/*
	 	 *	Loop through each of the attributes of the node,
		 *	looking for potential matches.  If the attribute
 		 *	is not a basic type, then make a recursive call
		 *	on the attribute.
		 */

		foreachinSEQattrDesc(item.VnodeDesc->attributes, remAttr,
				     nodeAttribute)
		{
		  switch ( typeof(nodeAttribute->value) )
		  {
		     case KintegerDesc:
			if ( search_flag != ATTR_SEARCH )
				break;
			(void)IDLlongToExternal((long)nodeAttribute->value.
					  VintegerDesc->value, buf, sizeof buf);

			if ( ! strcmp(attribute_name, nodeAttribute->name) &&
			     ! strcmp(value, buf) )
			{
				add_to_search_sequence(item);
				ret_val = 1;
			}	

			break;


		     case KrationalDesc:
			if ( search_flag != ATTR_SEARCH )
				break;
			(void)IDLdoubleToExternal(nodeAttribute->value.
					  VrationalDesc->value, buf,sizeof buf,10);

			if ( ! strcmp(attribute_name, nodeAttribute->name) &&
			     ! strcmp(value, buf) )
			{
				add_to_search_sequence(item);
				ret_val = 1;
			}	

			break;


		     case KbooleanDesc:
			if ( search_flag != ATTR_SEARCH )
				break;
			(void)IDLBooleanToExternal(nodeAttribute->value.
					  VbooleanDesc->value, buf, sizeof buf);

			if ( ! strcmp(attribute_name, nodeAttribute->name) &&
			     ! strcmp(value, buf) )
			{
				add_to_search_sequence(item);
				ret_val = 1;
			}	

			break;


		     case KstringDesc:
			if ( search_flag != ATTR_SEARCH )
				break;
			(void)IDLStringToExternal(nodeAttribute->value.
					  VstringDesc->value, buf, sizeof buf);

			/*
			 *	This hack is necessary to remove the double
			 *	quote at the end of every string.
			 */

			buf[strlen(buf) - 1] = '\0';

			if ( ! strcmp(attribute_name, nodeAttribute->name) &&
			     ! strcmp(value, buf + 1) )
			{
				add_to_search_sequence(item);
				ret_val = 1;
			}	

			break;

		     default:

			/*
			 *	The item was not a basic type, so make a 
			 *	recursive call on the attribute.
			 */

			ret_val = go_find(attribute_name, value,
					  nodeAttribute->value, search_flag,
					  ret_val);
			break;

		  }  /* end of switch */

		}  /* end of foreachin */
			
		break;


	  case KsequenceDesc:

		/*
		 *	If this sequence has been traversed already, then
		 *	do not attempt it again, else mark it as being
		 *	traveled.
		 */

 		if ( item.VsequenceDesc->touched != IDLepoch ) {
		  item.VnodeDesc->touched = IDLepoch;
		  
		  foreachinSEQIDLVALUE(item.VsequenceDesc->value, remSEQ, item2)
		    if (1==go_find(attribute_name, value, item2,
				   search_flag, ret_val))
		      ret_val = 1;
		}
		break;


	  case KsetDesc:

		/*
		 *	If this set has been traversed already, then
		 *	do not attempt it again, else mark it as being
		 *	traveled.
		 */

 		if ( item.VsetDesc->touched != IDLepoch ) {
		  item.VnodeDesc->touched = IDLepoch;

		  foreachinSEQIDLVALUE(item.VsetDesc->value, remSEQ, item2)
			if (1==go_find(attribute_name, value, item2,
					  search_flag, ret_val))
			  ret_val = 1;
		}
		break;

	  default:
		break;

  	} /* end of switch */

	return ret_val;

}  /* end of go_find() */
		

add_to_search_sequence(item)
IDLVALUE	item;
{
	if ( SearchList.VsequenceDesc == NULL )
		return;

	if ( ! inSEQIDLVALUE(SearchList.VsequenceDesc->value, item) )
		appendrearSEQIDLVALUE(SearchList.VsequenceDesc->value, item);

}  /* end of add_to_search_sequence() */


