/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/move.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	This is a function to traverse the current SET or
 *			SEQ  in treewalk.
 */


/**** INCLUDE FILES ****/

#include <stdio.h>
#include <ctype.h>
#include "global.h"
#include "history.h"
#include "list.h"
#include "utilities.h"



/**** CODE ****/

/*
 *	Routine :	move()
 *
 *	Description :	If the current item is a member of a set or sequence
 *			then it will move to a new element of that particular
 *			structure as given by a displacement.  A negative
 *			displacement will indicate traversal toward the
 *			beginning of the set or seq, while a positive
 *			number will direct motion toward the rear.  A 
 *			displacement of 0 indicates no change in position.
 *			A record of the motion must be maintained for
 *			historical purposes.
 *
 *	Arguments :	displacement	- (IN) the number of elements to
 *					traverse.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void move(argflag, displacement)
int	argflag;
int	displacement;
{

	history		*temp1;
	setseqhist	*listptr;
	IDLVALUE	 set_or_seq_ele;
	SEQIDLVALUE	 remIDLVALUE;
	qnode 		*remlist();
	int		 setln, newpos;
	char		 argbuf[80], *charptr = argbuf;
	void		 list();

	/*
	 *	If no arguments were given, then obtain the argument
	 *	from the user.
	 */

	if ( argflag == 0 )
	{
		if ( ! Get_default_arg("Displacement : ", argbuf, 80) )
			return;

		/*
		 *	Scan the input for numerical arguments.
		 */

		while ( *charptr != '\0' )
		{
			if ( ! isdigit(*charptr++) )
			{
				exception(NONNUM_ARG, "m");
				return;
			}
		}

		/*
		 *	The argument is numeric, so convert it to an
		 *	integer, and continue processing.
		 */

		displacement = atoi(argbuf);
		
	}

	/*
	 *	If the list stack is empty then do not allow the move to
	 *	happen.
	 */

	if ( isempty((qnode *)Liststack) )
	{
		exception(LIST_STACK_EMPTY);
		return;
	}

	/*
	 *	Obtain the SET/SEQ from the top element of the Liststack
	 */

	listptr = Liststack->next;
	newpos = listptr->begin->elementnum + displacement;
	
	/*
	 *	Now determine if the given displacement is within bounds
	 *	of the current SET/SEQ.
	 */

	switch ( typeof(listptr->setorseq) )
	{
		case KsequenceDesc:
		  ithinSEQIDLVALUE(listptr->setorseq.VsequenceDesc->value,
				   newpos, set_or_seq_ele);

		  if ( set_or_seq_ele.IDLclassCommon == NULL )
		  {
			exception(OUT_OF_BOUNDS, "element", "sequence");
			return;
		  }
		  else
			break;

		  case KsetDesc:
		    setln = lengthSEQIDLVALUE(listptr->setorseq.VsetDesc->value);
		    if ( newpos <= 0 || newpos > setln )
		    {
			exception(OUT_OF_BOUNDS, "element", "set");
			return;
		    }
		    else
		    {
			setln = 1;
			foreachinSEQIDLVALUE(listptr->setorseq.VsetDesc->value,
					     remIDLVALUE, set_or_seq_ele)
			{
				if ( setln++ == newpos )
					break;
			}
		    }
	
	}

	/*
	 *	Now what needs to be done is to forget everything which
	 *	was traversed from the other item in the SET/SEQ.
	 */

	while ( Historylist->next != listptr->begin )
	{
		temp1 = (history *)remlist((qnode *)Historylist);
		Destroy_history_node(temp1);
	}

	/*
	 *	Now that the past has been removed, adjust the top of the
	 *	history list to reflect the new position.  This also changes
	 *	The liststack information, since they point to the same
	 *	objects.
	 */

	Historylist->next->elementnum = newpos;
	Currentitem = set_or_seq_ele;	

	/*
	 *	Display the new item to the screen
	 */

	list(1, NULL);

	return;

}  /* end of move() */


/*
 *	Routine :	head()
 *
 *	Description :	If the current item is a member of a set or sequence
 *			then this function will go to the head, or beginning
 *			of the set or seq.  If the current item is not a
 *			member of either of the two structures, then an
 *			error message is printed, and no traversal is per-
 *			formed.
 *
 *	Arguments :	None.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void head()
{
	if ( isempty((qnode *)Liststack) )
	{
		exception(LIST_STACK_EMPTY);
		return;
	}

	move(1, 1 - Liststack->next->begin->elementnum);

	return;

}  /* end of head() */


/*
 *	Routine :	tail()
 *
 *	Description :	If the current item is a member of a set or sequence,
 *			then make the current item the last element of that
 *			set or sequence.  A record of the traversal must be
 *			maintained in the history of traversals.
 *
 *	Arguments :	None.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void tail()
{
	int newpos;

	if ( isempty((qnode *)Liststack) )
	{
		exception(LIST_STACK_EMPTY);
		return;
	}

	switch ( typeof(Liststack->next->setorseq) )
	{
	  case KsequenceDesc:
		newpos = lengthSEQIDLVALUE(Liststack->next->setorseq.
					   VsequenceDesc->value);
		break;

	  case KsetDesc:
		newpos = lengthSEQIDLVALUE(Liststack->next->setorseq.VsetDesc->
					 value);
		break;
	}

	move(1, newpos - Liststack->next->begin->elementnum);
	return;

}  /* end of tail() */ 
