/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/modify.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	This is the file which allows one to modify an
 *			IDL instance.  This sub-module contains the commands
 *			'change' and 'label'.
 *
 *	Functions :	change()
 *			node_attr()
 *			stringscan()
 *			relabel()
 */


/**** INCLUDE FILES ****/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "global.h"
#include "modify.h"
#include "list.h"



/**** CODE ****/

/*
 *	Routine :	change()
 *
 *	Description :	This function is set to change the information of the
 *			given item.  Typechecking is performed upon the new
 *			item before replacing the old one.  The current item
 *			is retained as being current.  Changes are not
 *			allowed for items which are of type SET or SEQ.
 *
 *	Arguments :	attrnum		- (IN) if the current item is a
 *					  node, then this is the attribute
 *				 	  number to be changed.
 *			nth_element	- (IN) if the attribute is a 
 *					set or sequence, then this is the
 *					element in that sequence to modify
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void change(attrnum, nth_element)
int 	attrnum;
int	nth_element;
{
	IDLVALUE	node_attr();
	int		stringscan();
	double		atof();
	IDLVALUE 	chngitem;
	char		newvalue[81];
	char		*charptr;
	void		list();

	/*
	 *	Check for invalid input response.
	 */

	if ( attrnum == 0 && Currenttype == KnodeDesc )
	{
		exception(BAD_OPTION, "change", "node");
		return;
	}

	/*
	 *	If the current item is a node, then call node_attr() to
	 *	get a handle on the attribute as requested by the attrnum,
	 *	else assume that the current item is to be changed.
	 */

	if ( Currenttype == KnodeDesc )
	{
		chngitem = node_attr(attrnum, nth_element); 
		if (  chngitem.IDLclassCommon == NULL )
				return;
	}
	else if ( attrnum != 0 )
	{
		exception(OUT_OF_BOUNDS, "attribute", "non-node type");
		return;
	}
	else
		chngitem = Currentitem;

	if ( typeof(chngitem) == KnodeDesc )
	{
		exception(BAD_OPTION, "change", "node");
		return;
	}

	/*
	 *	Prompt the user to input a new value
	 */

	(void)printf("Old Value : ");
	myprintf(chngitem);
	(void)printf("New Value (<cr> to retain current value) : ");;
	(void)myfgets(newvalue, 80); 
	
	/*
	 *	If only a carriage return is given, then make no
	 *	changes.  Also print the current item, to ensure
	 *	the user that no changes were made.
	 */

	if ( newvalue[0] == '\0' )
	{
		list(1, NULL);
		return;
	}

	switch ( typeof(chngitem) )
	{
		case KstringDesc:
		  chngitem.VstringDesc->value = StringToChar(newvalue);
		  Set_changed_flag();
		  break;

		case KbooleanDesc:

		  /*
		   *	First go through newvalue and change it all
		   *	to lower case.
		   */

		  for ( charptr = newvalue; *charptr != '\0'; charptr++ )
		  {
			if ( isupper(*charptr) )
				*charptr = tolower(*charptr);
		  }

		  /*
		   *	If the change is neither TRUE or FALSE, then give
		   *	an error message and abort the change.
		   */

		  if ( !strcmp(newvalue, "false") )
		  {
			chngitem.VbooleanDesc->value = (Boolean)FALSE;
			Set_changed_flag();
		  }
		  else if ( !strcmp(newvalue, "true") )
		  {
			chngitem.VbooleanDesc->value = (Boolean)TRUE;
			Set_changed_flag();
		  }
		  else
			exception(TYPE_MISMATCH, "boolean");
		  break;	

		case KintegerDesc:

		  /*
		   *	Call stringscan in the case of an integer to type
		   *	check the new integer value.
		   */

		  if ( !stringscan(newvalue, KintegerDesc) )
		  {
			exception(TYPE_MISMATCH, "integer");
			break;
		  }
		  chngitem.VintegerDesc->stringRep = StringToChar(newvalue);
		  chngitem.VintegerDesc->value = atoi(newvalue);
		  Set_changed_flag();
		  break;

		case KrationalDesc:

		  /*
		   *	Call stringscan() in the case of a rational number
		   *	to type check the new decimal value.
		   */

		  if ( !stringscan(newvalue, KrationalDesc) )
		  {
			exception(TYPE_MISMATCH, "rational");
			break;
		  }
		  chngitem.VrationalDesc->stringRep = StringToChar(newvalue);
		  chngitem.VrationalDesc->value = atof(newvalue);
		  Set_changed_flag();
		  break;

		default:
			exception(CRITICAL_ERROR);

	}  /* end of switch */

	list(1, NULL);

}  /* end of change() */


/*
 *	Routine :	node_attr()
 *
 *	Description :	Given the attribute number of the current item,
 *			which is assumed to be a node, node_attr() will return
 *			a handle to the attribute so that it may be reviewed
 *			and/or modified.  If the attribute is a SET or SEQ,
 *			then an error is given.
 */
IDLVALUE node_attr(attrnum, nth_element)
int 	attrnum;
int	nth_element;
{
	attrDesc	theattr;
	IDLVALUE	theitem;

	/*
	 *	Pull the desired attribute from the node.  If the
	 *	attribute number is invalid, then print an error
	 *	message.
	 */

	ithinSEQattrDesc(Currentnode->attributes, attrnum, theattr);
	if ( theattr == NULL )
	{
		exception(OUT_OF_BOUNDS, "attribute", "node");
		theitem.IDLclassCommon = NULL;
		return theitem;
	}

	/*
	 *	Assign the value of the attribute as being the return
	 *	value of this function.
	 */

	theitem = theattr->value;

	/*
	 *	If the attribute was a set or seq, then pull the
	 *	desired element from the set/seq and set that as the
	 *	return value.
	 */

	if ( typeof(theattr->value) == KsetDesc ||
	     typeof(theattr->value) == KsequenceDesc )
	{
		/*
		 *	If no element number was given, assume it is
		 *	the first element of the set/seq.
		 */

		if ( nth_element == 0 )
			nth_element = 1;

		/*
		 *	Attempt to retrieve the correct element from
		 *	the set/sequence.
		 */

		if ( typeof(theattr->value) == KsequenceDesc )
		{
			ithinSEQIDLVALUE(theattr->value.VsequenceDesc->value,
					 nth_element, theitem);
		}
		else
		{
			ithinSEQIDLVALUE(theattr->value.VsetDesc->value,
					 nth_element, theitem);
		}

		if ( theitem.IDLclassCommon == NULL )
		{
			exception(OUT_OF_BOUNDS, "element",
				  ((typeof(theattr->value) == KsetDesc) ?
				   "set" : "sequence"));
			return theitem;
		}
	}
	else if ( nth_element != 0 )
	{
		exception(OUT_OF_BOUNDS, "element",
			 "non-set or non-sequence type");
		theitem.IDLclassCommon = NULL;
		return theitem;
	}


	return theitem;

}  /* end of node_attr() */


/*
 *	Routine :	stringscan()
 *
 *	Description :	This function takes a string and an indicator
 *			if the string is to represent an integer or a rational
 *			number and scans it to ensure that the string really
 *			does represent one of these two constructs.
 *
 *	Arguments :	str		(IN/OUT) -- The string to be scanned.
 *			type		(IN) -- The type of number the string
 *					is to represent :
 *						KintegerDesc
 *						KrationalDesc
 */
int stringscan(str, type)
char *str;
int   type;
{
	int negflag = 0, num = 0;
	int decimal = 0, zero = 0;
	char	*buf, *strbegin;
	int 	j = 0;

	/*
	 *	Make a new string which will hold the syntatically correct
	 *	digits.
	 */

	buf = (char *)malloc_(strlen(str) + 1);
	strbegin = str;

	/*
	 *	Scan the input string
	 */

	while ( *str != '\0' )
	{
		/*
		 *	If the is leading white space, then ignore it,
		 *	else what we have is not a number, ie 5 7.
		 */

		if ( *str == ' ' || *str == '\t' ) 
		{
			if ( negflag || num || decimal )
				return 0;
			else
			{
				*str++;
				continue;
			}

		}

		/*
		 *	If a negative sign is found, then check to see if one
		 *	was encountered.  If so, then we have an invalid number,
		 *	ie. -59-8
		 */

		if ( *str == '-' )
		{
			if ( negflag || decimal || num || zero )
				return 0;
			else
			{
				negflag = 1;
				*(buf + j++) = '-';
				str++;
				continue;
			}
		}

		/*
		 *	If a decimal point is found, check to see if the number
		 *	is indeed a rational.   If it is not, then there is
		 *	an error, otherwise, see if a decimal was found already.
		 *	If this is the case, again there is an error. ie. 43.9.7
		 */

		if ( *str == '.' )
		{
			if ( type == KintegerDesc || decimal )
				return 0;
			else
			{
				decimal = 1;
				*(buf + j++) = '.';
				str++;
				continue;
			}
		}

		/*
		 *	After all the above checking, if the character is not a
		 *	digit, then this is not a valid number, else copy it
		 *	into the string.
		 */

		if ( !isdigit(*str) )
			return 0;
		else
		{
			/*
			 *	If the digit is a 0, then do not necessarily
			 *	add it to the output string, since it could
			 *	be a meaningless leading zero, ie. 0562
			 */

			if ( *str == '0' && !num && !decimal )
			{
				str++;
				zero = 1;
				continue;
			}

			/*
			 *	Copy the digit into the output string.
			 */

			*(buf + j++) = *str;
			if ( *(str++) == '0' && decimal && !num )
				num = 0;
			else
				num = 1;
			continue;
		}

	} /* end while */
	
	/*
	 *	Terminate the output string.
	 */

	*(buf + j) = '\0';

	/*
	 *	If we got a negative sign followed by just a zero, then just
	 *	place a zero into the buffer without the negative sign.
	 */

	if ( (zero || negflag) && !num )
	{
		*(buf) = '0';
		*(buf + 1) = '\0';
		num = 1;
	}

	/*
	 *	Copy the output buffer into the input buffer.  This should be
 	 *	guaranteed to work, since at most the output buffer is the
	 *	same size as the input buffer.
	 */

	(void)strcpy(strbegin, buf);
/***
	free(buf);
***/

	return num;

}  /* end of stringscan() */


/*
 *	Routine :	relabel()
 *
 *	Description :	The current item must be of type nodeDesc for this
 *			command to work.  This will replace the label of the
 *			current node with the new label specified.
 *
 *	Arguments :	newlabel		- (IN) the new label.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void relabel(newlabel)
char *newlabel;
{
	char	 buf[81];
	void	 list();

	if ( Currenttype != KnodeDesc )
	{
		exception(BAD_OPTION, "relabel", "non-node");
		return;
	}

	if ( newlabel == NULL )
	{
		(void)printf("Old label : %s\n", (Currentnode->label[0] != '\0') ?
		       Currentnode->label : "None");
		buf[0] = 'L';
		if ( ! Get_default_arg("New label (<cr> to retain previous label): L", buf + 1, 79) )
		{
			list(1, NULL);
			return;
		}

		newlabel = buf;
	}

	/*
	 *	Check to see if the label is indeed a valid label.
	 */

	if ( ! Validate_label(newlabel) )
		return;

	/*
	 *	This is a minor optimization to see if the user just
	 *	gave back the same label which was there.  If so,
	 *	then don't change it.
	 */

	if ( ! strcmp(Currentnode->label, newlabel) )
	{
		list(1, NULL);
		return;
	}

	/*
	 *	Now scan through the IDL instance looking for a duplicate
	 *	copy of this label on another node.
	 */

	if ( search("for", newlabel, NO) )
	{
		exception(DUPLICATE_LABEL, newlabel);
		return;
	}
	 

	/*
	 *	At this point, the label is good, so copy it into the
	 *	Currentnode
	 */

	Currentnode->label = NewString(newlabel);
	Set_changed_flag();

	list(1, NULL);

}  /* end of relabel() */
