/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/list.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	The function list() below will print out the
 *			attributes of the current node.
 */


/**** INCLUDE FILES ****/

#include <stdio.h>
#include <string.h>
#include "global.h"
#include "list.h"


/**** MACROS ****/

#define lnpos(X)	( (unsigned int)(X) >>  12 )


/**** DECLARATIONS ****/

char	numbuf[30];


/**** CODE ****/

/*
 *	Routine :	list()
 *
 *	Description :	Probably the most abused of all the commands, list()
 *			displays the information of the current item.  List()
 *			requires 2 arguments, which must be positive or
 *			NULL.  If any of the arguments are NULL, then default
 *			value for that argument is used.  All the information
 *			printed to stdout should either fit onto the screen
 *			or be terminated before wrapping around the screen.
 *
 *	Arguments :	start	- (IN) the attribute number to begin printing
 *				  the current node.  If start is 0, then
 *				  begin printing from the last line printed.
 *			finish	- (IN) the last line to be printed in the
 *				  current node.  If finish == 0, then default
 *				  to the maximum number of attributes that
 *				  fit on one screen.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void list(start, finish)
int	 start;
int 	finish;
{
	void listnode();

	if ( Currenttype == KnodeDesc )
	{
		listnode(start, finish);
		return;
	}
	else
		myprintf(Currentitem);

}  /* end of list() */



/*
 *	Routine :	listnode()
 *
 *	Description :	This routine will print out the contents of a
 *			node type in the format as given in the requirement
 *			specification.
 *
 *	Arguments :	start		(IN) -- The position in the node at
 *					which to start printing.
 *			finish		(IN) -- The position in the node at
 *					which to stop printing.
 */
void listnode(start, finish)
int	start;
int	finish;
{
	void 	printnodeheader(), printprintbuf();
	char	*itoa(), *utoa();

	int	node_length;
	static int last_attr = 1;
	attrDesc	next_attr;

	/*
	 *	Since the currenttype is a node, then the arguments
	 *	to list() must be taken into consideration.  It is
	 *	at this time where the number of attributes to fit on one
	 *	screen is calculated.
	 */

	node_length = lengthSEQattrDesc(Currentnode->attributes);
	if ( node_length == 0 && start <= 1 )
	{
		printnodeheader(Currentnode);
		return;
	}

	last_attr++;
	if ( start == 0 )
		start = last_attr;
	if ( finish == 0 )
	{
		if ( node_length > LINES - 2 )
		{
			finish = ( start + LINES - 2 > node_length ?
				   node_length : start + LINES - 2);
		}
		else
			finish = node_length;
	}

	/*
	 * 	If the arguments are not within bounds of the node then
 	 * 	print the error message and return.
	 */

	if ( start > node_length || finish > node_length || finish < start )
	{
		exception(OUT_OF_BOUNDS, "attribute", "node");
		return;
	}

	Current_line = start;

	/*
	 *	Now sequence through the attributes, and print them to the
	 *	screen.  The buffering is used to maintain a proper display
	 *	for the screen width as defined by COLS.
	 */

	printnodeheader(Currentnode);
	while( start <= finish )
	{
		ithinSEQattrDesc(Currentnode->attributes, start, next_attr);
		catprintbuf("  ");
		catprintbuf(itoa(start++));
		catprintbuf(" ");
		catprintbuf(next_attr->name);
		catprintbuf(" (");
		catprintbuf(utoa(lnpos(next_attr->srcpos)));
		catprintbuf(") ");

	        myprintf(next_attr->value);
	}

	last_attr = finish;
	
	return;
} 


/*
 *	Routine :	printnodeheader()
 *
 *	Description :	The information like type, label, and srcpos about
 *			a node are printed to the stdout.
 *
 *	Arguments :	node	- (IN) the node whose node header is to be
 *				  printed.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void printnodeheader(node)
nodeDesc	node;
{
	char *utoa();
	void printprintbuf();

	catprintbuf("Node of type ");
	catprintbuf(node->name);
	catprintbuf(" (");
	catprintbuf(utoa(lnpos(node->srcpos)));
	catprintbuf(")");

	if ( *node->label != '\0' )
	{
		catprintbuf(": ");
		catprintbuf(node->label);
	}

	catprintbuf("\n");

	printprintbuf();
	printbuf[0] = '\0';

}  /* end of printnodeheader() */


/*
 *	Routine :	myprintf()
 *
 *	Description :	One of my famous 'my' functions, this guy will
 *			given an attribute, print the attribute information
 *			to the screen.  The buffering is needed to maintain
 *			an acceptable display as parameterized by COLS.
 *
 *	Arguments :	attr	- (IN)	the attribute whose information is
 *				  to be displayed.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void myprintf(attr)
IDLVALUE	attr;
{
    char *itoa(), *IDLstring();
    void printprintbuf();
	
    switch ( typeof(attr) )
    {
	case KnodeDesc:
	  printnodeheader(attr.VnodeDesc);
	  break;

	case KbooleanDesc:
	  if ( attr.VbooleanDesc->value == TRUE )
		catprintbuf("TRUE\n");
	  else
		catprintbuf("FALSE\n");
	  break;

	case KsetDesc:
	  catprintbuf("Set with ");
	  catprintbuf(itoa(lengthSEQIDLVALUE(attr.VsetDesc->value)));
	  catprintbuf(" elements\n");
	  break;

	case KsequenceDesc:
	  catprintbuf("Sequence with ");
	  catprintbuf(itoa(lengthSEQIDLVALUE(attr.VsequenceDesc->value)));
	  catprintbuf(" elements\n");
	  break;

	case KstringDesc:
	  catprintbuf(IDLstring(attr.VstringDesc->value) );
	  catprintbuf("\n");
	  break;

	case KintegerDesc: case KrationalDesc:
	  catprintbuf(attr.VIDLnumber.IDLclassCommon->stringRep);
	  catprintbuf("\n");
	  break;

    }

	printprintbuf();
	printbuf[0] = '\0';

    return;

}  /* end of myprintf() */


/*
 *	Routine :	catprintbuf()
 *
 *	Description :	The name is short for concatenate print buffer.  Any
 *			time something is printed ot the screen, a call to
 *			call to catprintbuf() is made to assure that the text
 *			will fit onto the screen as per the design in the
 *			requirement specification.
 */
int catprintbuf(nextitem)
char *nextitem;
{
if ( strlen(printbuf) < COLS + 2 )
	(void)strncat(printbuf, nextitem, COLS - strlen(printbuf) + 1 );
}


char *itoa(num)
int num;
{

	(void)IDLintToExternal(num, numbuf, 29);

	return numbuf;

}  /* end of itoa() */


char *utoa(num)
unsigned int num;
{

	(void)IDLUnsignedIntToExternal(num, numbuf, 29);

	return numbuf;

}  /* end of utoa() */


char *IDLstring(idlstring)
char *idlstring;
{

	(void)IDLStringToExternal(idlstring, tempbuf, COLS - 1);

	return tempbuf;

}  /* end of IDLstring() */


/*
 *	Routine :	printprintbuf()
 *
 *	Description :	This routine is called whenever a string constructed by
 *			catprintbuf() needs to be printed to the screen.  If the
 *			string is excessively long, then the substring ".." is
 *			appended to the end of the string to indicate that it
 *			has been clipped in order to fit onto the screen.
 */
void printprintbuf()
{
	int j;

	if ( ( j =  strlen(printbuf) ) > COLS )
	{
		printbuf[j-4] = '.';
		printbuf[j-3] = '.';
		printbuf[j-2] = '\n';
		printbuf[j-1] = '\0';
	}

	(void)printf("%s", printbuf);

}  /* end of printprintbuf() */
