/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/inputoutput.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	Read in a new IDL instance from a file with a
 *			given filename into treewalk.
 */

/**** INCLUDE FILES ****/

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <string.h>
#include "global.h"
#include "C/IDLIO.h"
#include "input.h"
#include "history.h"
#include "symbol.h"
#include "list.h"
#include "utilities.h"



/**** CODE ****/

/*
 *	Routine :	input_new_instance()
 *
 *	Description :	The user wants a new instance in memory, so this
 *			function will give it to him.  If the specified file
 *			does not exist, or there is an error in the IDL
 *			reading, then an error message should be printed
 *			but the old instance remains current.  The old
 *			instance need not be erased from memory upon reading
 *			the new instance.  All the data structures for
 *			past history, list stacks, etc. must be reset to
 *			initial values.
 *
 *	Arguments :	filename	- (IN) the filename to be read in.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void input_new_instance(filename)
char	*filename;
{

	void resetlists(),list();

	nodeDesc	 temp;
	FILE		*fn;
	char		 buf[80];


	if ( ! Query_upon_change("input a new file") )
		return;

	/*
	 *	If the user did not specify an input file, now is the time
	 *	to ask.
	 */

	if ( filename == NULL )
	{
		if ( ! Get_default_arg("Input filename : ", buf, 80) )
			return;

		filename = buf;
	}

	/*
	 *	See if the given file exists, if not print error
	 *	message, but do not exit treewalk.
	 */

	if ( access(filename, R_OK) == -1 )
	{
		exception(UNABLE_TO_ACCESS, "read", filename);
		return;
	}

	(void)printf("Reading IDL instance in %s...\n", filename);


	if ( ( fn = fopen(filename, "r") ) == NULL )
	{
		exception(FILE_OPEN_ERROR, "input");
		return;
	}

	if ( ( temp = IDLread(fileno(fn)) ) == NULL )
	{
		exception(BAD_READ);
		(void)fclose(fn);
		return;
	}

	(void)fclose(fn);

	/*
	 *	Now replace the old Root with the new root of the new
	 *	instance.  Note : here, if it were available, one would
	 *	deallocate all the memory from the old instance.  As it
	 *	is, the old instance is resident in memory.
	 */

	Root = temp;
	Currentitem.VnodeDesc = Root;
	free((char *)Defaultfile);
	Defaultfile = (char *)malloc_(strlen(filename) + 1);
	(void)strcpy(Defaultfile, filename);

	resetlists();

	/*
	 *	List the root node of the new instance
	 */

	list(1, NULL);

}  /* end of input_new_instance() */


/*
 *	Routine :	resetlists()
 *
 *	Description :	After reading in the new instance, then all of
 *			the history and names of the last instance are
 *			forgotten.
 *
 *	Arguments :	None.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void resetlists()
{

	history		*histptr;
	symbol		*symptr;
	setseqhist	*listptr;
	void 		initlist();

	/*
	 *	Reset the history list
	 */

	histptr = Historylist->next;
	while ( histptr != Historylist )
	{
		histptr = histptr->next;
		Destroy_history_node(histptr->prev);
	}
	initlist((qnode *)Historylist);

	/*
	 *	Clear the symboltable
	 */

	symptr = Symboltable->next;
	while ( symptr != Symboltable )
	{
		symptr = symptr->next;
		free((char *)symptr->prev);
	}
	initlist((qnode *)Symboltable);

	/*
	 *	Clear the Liststack
	 */

	listptr = Liststack->next;
	while ( listptr != Liststack )
	{
		listptr = listptr->next;
		free((char *)listptr->prev);
	}
	initlist((qnode *)Liststack);

	return;

}  /* end of resetlists() */


/*
 *	Routine : 	output_idl_instance()
 *
 *	Description :	The current IDL instance in memory will be saved to the' *			given filename, and control will be given back to the
 *			treewalk interpreter.  If no filename is given, then the *			default is the filename used to input the file.
 *
 *	Arguments :	filename	- (IN) the name of the file to be
 *					  saved.
 *
 *	Return Value :	None.
 *
 *	Side Effects :	None.
 */
void output_idl_instance(filename)
char	*filename;
{
	int	Overwrite_input_file();

	char 	*thefile;
	char  	 buf[80];
	FILE 	*thefiledesc;
	struct stat	statbuf1, statbuf2;

	/*
	 *	If no filename was given use the default filename.
	 */

	if ( filename == NULL )
	{
		(void)printf("No filename specified. ");
		if ( Overwrite_input_file() )
			thefile = Defaultfile;
		else if ( ! Get_default_arg("Output filename : ", buf, 80) )
		{
			return;
		}
		else
			thefile = buf;
	}
	else
		thefile = filename;

	/*
	 *	Determine a priori if the file is available for the
	 *	write operation.
	 */

	if ( access(thefile, W_OK) && !access(thefile, F_OK) )
	{
		exception(UNABLE_TO_ACCESS, "write", thefile);
		return;
	}

	/*
	 *	If the filename given was from the command line, then
	 *	check to see if the file is actually the input file.
	 *	If it is, prompt the user to check if they really
 	 *	wish to proceed.
	 */

	if ( thefile != Defaultfile )
	{
		if ( ! stat(Defaultfile, &statbuf1) &&
		     ! stat(thefile, &statbuf2) )
		{
			if ( ( statbuf1.st_dev == statbuf2.st_dev ) &&
			     ( statbuf1.st_ino == statbuf2.st_ino ) )
			{
				if ( ! Overwrite_input_file() )
					return;
			}
		}
	}

	(void)printf("Writing IDL instance to %s...\n", thefile);
	if ( ( thefiledesc = fopen(thefile, "w") ) == NULL )
	{
		exception(FILE_OPEN_ERROR, "output");
		return;
	}

	/*
 	 *	Now write the IDL instance to the designated file.
	 */

	if ( IDLwrite(fileno(thefiledesc), Root) )
	{
		exception(BAD_WRITE);
		(void)fclose(thefiledesc);
		return;
	}

	(void)fclose(thefiledesc);
	(void)printf("Write successful.\n");

	Clear_changed_flag();

	return;

}  /* end of output_idl_instance() */


/*
 *	Routine :	Overwrite_input_file()
 *
 *	Description :	This function will query the user to determine
 *			if the input file should be written over by the
 *			output operation.  The call returns a 1 if yes
 *		  	and a 0 for no.
 */
int Overwrite_input_file()
{
	char	buf[81];

	(void)printf("Overwrite current input file (y or n) [n]? ");

	(void)myfgets(buf, 80);

	if ( ! strcmp(buf, "Y") || ! strcmp(buf, "y") )
		return 1;

	return 0;

}  /* end of Overwrite_input_file() */
