/***********************************************************************\ 
*									* 
*   File: scorpion/src/treewalk/ReceiveUserCommands.c 
*				 					* 
*   Copyright (C) 1991 Aaron Cavender
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *	Description :	This is a submodule of ProcessUserCommands.c.
 *			The function of this module is to receive characters
 *			from standard input and buffer them in a reasonable 
 *			fashion.
 *
 *	Functions :	Receive_User_Commands()
 *			clean_up_input()
 *			get_input_line()
 *			remove_white_blanks()
 */


/**** INCLUDE FILES ****/
#include <stdio.h>
#include <ctype.h>
#include "global.h"
#include "ReceiveUserCommands.h"
#include "command.h"


/**** CODE ****/

/*
 *	Routine :	Receive_User_Commands()
 *
 *	Description :	This function provides the interface between treewalk
 *			and the user input.  Receive_User_Commands() reads
 *			in characters from stdin and will then proceed to
 *			do the following:
 *				1.  Check for nonalphanumeric characters.
 *				2.  Remove whitespaces and tabs.
 *				3.  Take the input and place it in a buffer
 *				    where each word in the buffer is sep-
 *				    arated by exactly one space.
 *				4.  The first word in the buffer is con-
 *				     verted to lower case.
 *
 *	Arguments :	inputline   - (OUT) the buffer which to store
 *				      the input stream.
 *
 *	Return Value :  A value of 0 is returned for any of the errors
 *			mentioned above are encountered.  1 for sucess.
 *
 *	Side Effects :  None.
 */
int Receive_User_Commands(refined_input_buf)
char	*refined_input_buf;
{
	int	get_input_line(), clean_up_input();

	char	raw_input_buf[MAXLINE+1];
	int	errflg;

	(void)printf("-> ");  /* prompt */

	/*
 	 *	Recieve the input from the input stream, but perform no
  	 *	analysis of the input yet.
	 */

	if ( ! get_input_line(raw_input_buf) )
		return 0;

	/*
	 *	Now scan the buffer to  remove
  	 *	unnecessary spaces and tabs.  Also take the first word
 	 *	in the buffer and convert it to all lower case.
 	 */

	if ( ( errflg = clean_up_input(raw_input_buf, refined_input_buf) ) < 1 )
	{
		if ( errflg == 0 )	/* Empty input line */
			return 0;
		if ( errflg == -1 )
			exception(ILLEGAL_CHARACTER);

		return 0;
	}

	return 1; 

}  /* end of Receive_User_Commands() */


/*
 *	Routine :	clean_up_input()
 *
 *	Description :	After reading the user's command, this function will
 *			take the specified buffer and do the following :
 *			1.   put the first word of the buffer into lower case.
 *			2.   remove unnessary tabs and spaces.
 *
 *	Arguments :	inbuf		- (IN) this is the direct input from
 *					  the user.
 *			outbuf		- (OUT) buffer to store the modified
 *					  input.
 *
 *	Return Value :	If there are no complications with modifing the
 *			input line, the length of the modified line is
 *			returned.  If an error occurrs, a NULL is the
 *			return value.
 *
 *	Side Effects :	None.
 */
int clean_up_input(inbuf, outbuf)
char	*inbuf;
char	*outbuf;
{
	int	j, k = 0, l = 0;

	j = strlen(inbuf) - 1;

	/*
 	 *  Assuming the first word in the input string is a command,
	 *  this loop will find the first word and convert it to lower
	 *  case for easier error checking.
	 */

	 if ( ( k = remove_white_blanks(inbuf, k) ) > j )
		return 0;
	 while ( inbuf[k] != ' ' && inbuf[k] != '\t' )
	 {
		if ( k > j )
		{
			outbuf[l] = '\0';
			return l;
		}

		/*
		 *	Convert first word to lower case.
		 */

		if ( isupper(inbuf[k]) )
			outbuf[l++] = tolower(inbuf[k++]);
		else
			outbuf[l++] = inbuf[k++];
	}

	/*
	 *  Now the rest of the line is rid of extra spaces and tabs.
	 */

	while ( k <= j )
	{

		if ( inbuf[k] != ' ' && inbuf[k] != '\t' )
			outbuf[l++] = inbuf[k++];
		else
		{
			if ( ( k = remove_white_blanks(inbuf, k) ) > j )
				break;
			outbuf[l++] = ' ';
		}
	}

	outbuf[l] = '\0';

	return l;

}  /* end of clean_up_input() */


/*
 *	Routine :	get_input_line()
 *
 *	Description :	This function will store characters from stdin into
 *			a specified buffer, which is passed as an argument.
 *			The routine returns when one of the following
 *			conditions are found :
 *			1.   A carriage return is found
 *			2.   An EOF occurrs.
 *			The characters are only buffered up to the length
 *			specified in MAXLINE
 *
 *	Arguments :	line		- (OUT) this is where the input
 *					  characters are buffered.
 *
 *	Return Value :	NULL if the operation failed.  Otherwise the number
 *			of characters read in is returned.
 *
 *	Side Effects :	None.
 */
int get_input_line(line)
char	line[];
{

	return myfgets(line, MAXLINE);
	
}  /* end of get_input_line() */


/*
 *	Routine :	remove_white_blanks()
 *
 *	Description :	This call takes a string and an offset into the string
 *			as input.  The function will then return an offset into
 *			the same string which is the first non-white space
 *			character in the string after the initial position.
 */
int remove_white_blanks(buf, pos)
char	*buf;
int	 pos;
{
	int	foundwhite = 1;

	while ( foundwhite )
	{
		if ( *(buf + pos) == '\0' )
			break;
			
		if ( *(buf + pos) == ' ' || *(buf + pos) == '\t' )
		{
			pos++;
			continue;
		}
		else
			foundwhite = 0;
	}

	return pos;

} /* end of remove_white_blanks() */
			
