{***********************************************************************\ 
*									* 
*   File: scorpion/src/treepr/dSetXcoords.p 
*				 					* 
*   Copyright (C) 1991 Nancy Butler, Joan Curry, Steven Konstant,
*	Dore Rosenblum, and Greg Bollella
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************} 

const

#include "globalconst.h"

type
 
#include "globaltypes.h"

#include "SetXcoords.h"

procedure SetXcoords;

(***************************************************************************)
(*                                                                         *)
(* procedure SetXcoords ( T: TreeNodeList;  var minxpos,maxxpos:integer )  *)
(*                                                                         *)
(* Input parameter                                                         *)
(*   T - pointer to the data structure created by MakeTree to represent    *)
(*       the IDL data structure                                            *)
(*                                                                         *)
(* Output parameters                                                       *)
(*   minxpos - minimum x coordinate assigned as a node position.           *)
(*   maxxpos - maximum ( x coordinate + boxwidth ) over all nodes of the   *)
(*             tree.                                                       *)
(*                                                                         *)
(* Result                                                                  *)
(*   An x coordinate value for the upper left corner of a node's box       *)
(*   is computed and stored in the Xcoord field of each TreeNode of T.     *)
(*									   *)
(*   SetXcoords initializes minxpos, maxxpos, and the Last field of each   *)
(*   Headernode.  It then calls internal procedure Position on the root    *)
(*   node of the tree, T.  The Xcoord fields of each node are set within   *)
(*   Position.								   *) 
(*                                                                         *)
(* Programmer: Nancy Butler                                                *)
(* Date      : 3/18/84                                                     *)
(*                                                                         *)
(***************************************************************************)

const between = 6;          (* number of spaces between nodes, where a space
                               is the size of a character.                 *)
      betweenseq = 4;       (* number of spaces between nodes of a sequence 
                               or set                                      *)
var head: TreeNodeList;     (* pointer to a HeaderNode used to step through
                               the list of Header nodes of T               *)
    rightshif: boolean;     (* if true then oldpos may have to be shifted  
			       right to avoid being too close to left neighbor*)

procedure Seqposition(seqnodeptr: TreeNodePtr;
                      head: TreeNodeList); forward;

procedure Position( node:TreeNodePtr;head:TreeNodeList;oldpos:integer;
                    rightsh:boolean);

(***************************************************************************)
(*                                                                         *)
(* procedure Position                                                      *)
(*                                                                         *)
(* Input parameters                                                        *)
(*   node - pointer to a node of the tree.                                 *)
(*   head - pointer to the Header node of the level which 'node' is on.    *)
(*   oldpos - a tentative xcoordinate position to be assigned to 'node'.   *)
(*   rightsh - if true may have to shift node left to avoid left neighbor.  *)
(*                                                                         *)
(* Result                                                                  *)
(*     The x coordinate position of the upper left corner of the bounding  *)
(*   box of 'node' is determined and stored at node^.Xcoord.               *)
(*     Minxpos is updated if the position computed for the node is         *)
(*   less than minxpos.  Maxpos is set to node^.Xcoord + node^.BoxWidth if *)
(*   this quantity is greater than maxpos.                                 *)
(*                                                                         *)
(* Procedures called: Seqposition, Position.                               *) 
(* Nonlocal constant used: between and betweenseq, from parent procedure   *)
(*   SetXcoord.   					       	           *)
(* Nonlocal variables: minxpos, maxxpos, from parent procedure SetXcoord.  *)
(*                                                                         *)
(* Algorithm                                                               *)
(*     The nodes of the tree are assigned x coordinate positions 	   *)
(* in a left-to-right postorder traversal.  The final position assigned    *)
(* a node is such that the node is centered above its sons.                *)
(*     Oldpos is a tentative node position for the node.  If oldpos        *)
(* puts the node too close to its left neighbor, oldpos is set to place    *)
(* the node the required distance from its neighbor.  A node with          *)
(* no sons is positioned at oldpos.  When Position is called on the first  *)
(* son of a node, the oldpos passed as the son's tentative position  is    *)
(* computed such that the father will not be moved left when centered over *)
(* the sons.  Subsequent sons are given tentative positions that leave     *)
(* the required number of spaces between a node.  When a sequence or set   *)
(* is encountered, it is positioned as if each node in it where a son of   *)
(* the parent node. 							   *) 
(*                                                                         *)
(* Programmer: Nancy Butler                                                *)
(* Date      : 3/18/84            Modified: 1/28/85                        *)
(*                                                                         *)
(***************************************************************************)

   var oldposson: integer;  (* tentative oldpos of a son of 'node'.        *)
       sum: real;           (* number of positions taken by sons of node   *)
       posson: real;        (* oldposson is computed as a real value then  
                               truncated accordingly                       *)
       sonptr: PattrList;   (* a Pattr record whose Value field points to  
                               the current son                             *)
       seqptr: TreeNodePtr; (* points to a member of a sequence or set     *)
       rtson: TreeNodePtr;  (* the rightmost son of 'node'                 *)
       rightshif:boolean;   (* passed to subsequent calls to Position, to  
                               tell Position that oldpos may need to be 
			       shifted right.  True for all calls from Position
			       (but false for calls from SeqPosition)     *)

   begin
   rightshif:=true;
   with head^ do
      begin
      (* shift oldpos right if node is currently too close to its left     
         neighbor, which is head^.Last.                                    *)
      if (Last <> nil) and (rightsh)  then
         if Last^.Xcoord + Last^.BoxWidth + between > oldpos then
            oldpos:= Last^.Xcoord + Last^.BoxWidth + between;
      (* set Last to the last node of a level that has been assigned a
         position.                                                         *)
      Last:=node;
      end;

   (* Determine x coordinate position of node based on position of son(s). *)

   with node^ do
      begin

      (* Case 1: node has no sons *)
      if ( tag = SeqValue ) or ( tag = SetValue ) or ( Duplicate ) then
         Xcoord:=oldpos
      else if Pattribs = nil then
         Xcoord:=oldpos

      (* Case 2: node has one son which if it is a sequence node is the
         only node of that sequence *)
      else if (Pattribs^.Next = nil) and
              ((Pattribs^.Value^.tag = RegularNode)  or 
               (Pattribs^.Value^.Next = nil) or
               (Pattribs^.Value^.Next^.ArcSource <> node)) then

         begin  
         (* compute leftmost xcoord of son that will not cause father to be
           shifted left when father is repositioned based on son's position*)

         oldposson:=oldpos + ((BoxWidth - Pattribs^.Value^.BoxWidth) div 2);
         Position(Pattribs^.Value,head^.Next,oldposson,rightshif);

         (* position node directly above its one son *)
         Xcoord:=Pattribs^.Value^.Xcoord - ((BoxWidth -
              Pattribs^.Value^.BoxWidth) div 2); 
         end

      (* Case 3: node has more than one son or a multinode sequence as a son *)
      else
         begin
         (* compute a tentative position for first son that will not cause
            father to be shifted left when father is positioned based on
            position of sons.                                               *)

         sonptr:=Pattribs;
         sum:=0;
	 while sonptr <> nil do
	    begin
	    sum:=sum + sonptr^.Value^.BoxWidth + between;
	    if sonptr^.Value^.tag <> RegularNode then
	       begin
	       seqptr:=sonptr^.Value;
	       while not (last in seqptr^.Seqsetpos) do
                  begin
		  sum:=sum + seqptr^.Next^.BoxWidth + betweenseq;
		  seqptr:=seqptr^.Next;
		  end;
               end;
            sonptr:=sonptr^.Next;
            end;
	 sum:=sum - between;

	 posson:=(2 * oldpos + BoxWidth - sum)/2;

         if (trunc(posson) < 0) or (trunc(posson) = posson) then
            oldposson:=trunc(posson)
         else
            oldposson:=trunc(posson + 1);

         (* position sons of node  *)
         sonptr:=Pattribs;
         while sonptr <> nil do
            begin
            Position(sonptr^.Value,head^.Next,oldposson,rightshif);
            if sonptr^.Value^.tag <> RegularNode then
               Seqposition(sonptr^.Value,head^.Next);
            sonptr:=sonptr^.Next;
            oldposson:=head^.Next^.Last^.Xcoord + head^.Next^.Last^.BoxWidth
                       + between;
            end;
  
         (* position node between leftmost and rightmost sons  *)
         rtson:=head^.Next^.Last;
         Xcoord:=round ((( rtson^.Xcoord + rtson^.BoxWidth + 
                       Pattribs^.Value^.Xcoord )
                       / 2) - BoxWidth/2);
         end;

      (* update minxpos and maxxpos *)
      if Xcoord + BoxWidth > maxxpos then
         maxxpos:=Xcoord + BoxWidth;
      if Xcoord < minxpos then
         minxpos:=Xcoord;
      end;    (*  with node^  *)
   end;       (*  procedure Postition  *)

procedure Seqposition;

(***************************************************************************)
(*                                                                         *)
(* procedure Seqposition (seqnodeptr:TreeNodePtr; head: TreeNodeList )     *)
(*                                                                         *)
(* Input parameters                                                        *)
(*   seqnodeptr - first node of a sequence.                                *)
(*   head -       Header node for the level the sequence starting with     *)
(*                  'seqnodeptr' is on.                                    *)
(*                                                                         *)
(* Result                                                                  *)
(*   The x coordinate of the upper left corner of the bounding box of each *)
(*   node in the sequence starting with 'seqnodeptr' is computed and       *)
(*   stored in the Xcoord field of the node.                               *)
(*                                                                         *)
(* Procedures called: Position.                                            *)
(* Nonlocal constant used: betweenseq, from parent procedure SetXcoord.    *)
(*                                                                         *)
(* Programmer: Nancy Butler                                                *)
(* Date      : 3/18/84            Modified: 1/28/85                        *)
(*                                                                         *)
(***************************************************************************)

   var rightshif:boolean;   (* to indicate to Position that a right shift will
                             not be needed.                                *)
   begin
   rightshif:=false;
   (* while next node to right is part of the current sequence or set,
      position it, leaving 'betweenseq' spaces between nodes.         *)
   while not (last in seqnodeptr^.Seqsetpos) do
      with seqnodeptr^ do
         begin
         Position(Next,head,Xcoord + BoxWidth + betweenseq,rightshif);
         seqnodeptr:=Next;
         end;
   end;

(*  Body of procedure SetXcoords    *)

 begin
   minxpos:=0;
   maxxpos:=0;
   rightshif:=true;

   (* initialize all Last pointers from Header nodes to nil indicating that
      no nodes have been positioned yet.                                  *)
   head:=T;
   while head <> nil do
      begin
      head^.Last:=nil;
      head:=head^.Next;
      end;

   (* call Position on root node with tentative position 0 to begin 
      positioning nodes *)
   Position(T^.First, T, 0,rightshif);

end;
