{***********************************************************************\ 
*									* 
*   File: scorpion/src/treepr/Printdriver.p 
*				 					* 
*   Copyright (C) 1991 Nancy Butler, Joan Curry, Steven Konstant,
*	Dore Rosenblum, and Greg Bollella
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************} 

const
#include "globalconst.h"

type
#include "globaltypes.h"

#include "Printdriver.h"
#include "plot.h"
#include "iArccross.h"
#include "lArccross.h"

var

#include "globals.h"

procedure Printdriver; 

(***************************************************************************)
(*                                                                         *)
(* procedure Printdriver (start:TreeNodeList; minpos, maxpos, pagewidth,   *)
(*                  pagelength:integer; device:devicetype; cond:boolean);  *)
(*                                                                         *)
(* Input parameters                                                        *)
(*    start - pointer to the data structure created to represent the IDL   *)
(*            data structure.                                              *)
(*    minpos - minimum Xcoord of the nodes in the tree.                    *)
(*    maxpos - maximum of Xcoord + BoxWidth over all nodes in the tree.    *)
(*    pagewidth - number of characters on an output page.                  *)
(*    pagelength - number of lines on an output page.                      *)
(*    device - type of device output will be printed on: lineprinter or    *)
(*             the standard plot lob routines.                             *)
(*    cond - flag; true for condensed mode, false for full mode            *)
(*                                                                         *)
(* Result                                                                  *)
(*    If device = plot, commands to the plot to print the tree         *)
(*    are written to standard out.  If device = lineprinter, the tree,     *)
(*    broken into pages if necessary, is written to standard out.          *)
(* 									   *)
(* Procedures called:							   *)
(*    Leveler: to determine last level on current page  		   *)
(*    lArccross, iArccross: to draw arcs crossing the page that connect    *)
(*       two nodes not on the page. (l: lineprinter, i: plot)              *)
(*    lDrawbox, iDrawbox: to draw a node, arcs from it, and if parent is   *)
(*       off the page, the arc to it.					   *)
(*									   *)
(* Algorithm								   *)
(*    The tree data structure is output a row of pages at a time.          *)
(*  Each row of pages is output a page at a time.  Within a page the       *)
(*  nodes are output level by level starting at the top.  The Xcoord       *)
(*  position of a node is checked to decide if the node is on the current  *)
(*  page.  								   *)
(*    Arcs from nodes are drawn when the node is output.  If a node's      *)
(*  father is on a different page, the arc to the node is also drawn when  *)
(*  the node is drawn.  Checking for arcs that fall on the current page,   *)
(*  but connect two nodes not on the page, is done at appropriate times.   *)
(*                                                                         *)
(* Programmer : Nancy Butler                                               *)
(* Written    : 4/6/84            Modified : 1/29/85                       *)
(*                                                                         *)
(***************************************************************************)
   const
	notlastl = false;   (* signal to Box drawing routines not to print any
			       subtree information   *)

   var row,                 (* page number of output page in vertical 
                               direction *)
       column,              (* page number of output page in horizontal 
                               direction *)
       numpages,            (* number of pages in horizontal direction that
                               are needed to print the tree               *)
       initposadj,          (* position adjustment for first page of each 
			       row of pages of output                     *)
       posadj: integer;     (* position adjustment added to Xcoord of each 
                               node to get its absolute position on a page *)
       Boxptr: TreeNodePtr; (* pointer to current node                *)
       level,               (* pointer to HeaderNode of current node      *)
       topnextrow,          (* pointer to HeaderNode of first level on next
			       row of pages                               *)
       topofpage,           (* pointer of HeaderNode of top level of nodes
                               for current row of pages                   *)
       prevlevel:TreeNodeList; (* pointer to HeaderNode of level above    
                                  current node                            *)
       prevbottom :TreeNodeList; (* pointer to HeaderNode of level at bottom
                                  of previous page 			  *)
       cknode: TreeNodePtr; (* used to step through a level of nodes      *)
       Parray: PageArray;   (* page of lineprinter output                 *)
       top: boolean;        (* true if currently processing the top line  *)
                            (* of nodes on a page                         *)
       left:boolean;        (* passed to arcdrawing procedures, true if looking
                               for arcs across left side of the page, false
                               if arcs across right side of page.         *)
       abovepage:boolean;   (* passed to iArccross to tell it source is above
			       the page *)
       continue:boolean;     (* continue iteration                        *)
       cont1: boolean;	     (* continue iteration			  *)
       firsttime : boolean;  (* if true PostScript and plot
                                initializations are done                  *)
      


(***************************************************************************)
(*                                                                         *)
(* procedure Leveler;						           *)
(*									   *)
(* Input/Output parameter						   *)
(*     curlevel - on input this points to the first level of the tree to   *)
(*                be on this row of pages.  When returned to caller it     *)
(*                points to the first level to be placed on the next row   *)
(* 		  of pages.                                                *)
(*									   *)
(*  This procedure is used to determine the levels of the tree that can be *)
(*    printed on the current row of pages.                                 *) 
(*									   *)
(*  Programmer: Steven Konstant						   *)
(*  Written   : 3/16/84							   *)
(*									   *)
(***************************************************************************)

procedure Leveler(var curlevel:TreeNodeList); 

var pageht: integer;
    temp: integer;

begin
   pageht:=4; (*allow 3 lines at top of page for incoming arcs*)
   while (pageht < pagelength) and (curlevel <> nil) do  
   begin
      temp := (curlevel^.MaxBoxHt * 2) + 1; (*This is the space required*)
					   (*for lineprinter printing    *) 
      if (pageht + temp) <= pagelength then
      begin
	 curlevel^.Ycoord := pageht; (*set y value on the page *)
         pageht:= pageht + temp + 3; (*box will fit plus 3 lines for arcs*)
         curlevel:= curlevel^.Next
      end (*of if statement*)

      else pageht:= pagelength;  (*when another level won't fit, break *)
				(*while condition.                    *)

   end; (*of while loop*)


end; (*of procedure Leveler*)


(*  Body of procedure Printdriver  *)
   
   begin
   firsttime := true;
   level:=start;
   prevlevel:=nil;
   prevbottom:=nil;
   while level <> nil do
      begin
      level^.Last:=level^.First;
      level:=level^.Next;
      end;
   row:=0;
   level:=start; 

   (* calculate posadj for first page of output  *)

   if (maxpos - minpos) mod pagewidth = 0 then
      numpages := (maxpos - minpos) div pagewidth
   else
      numpages := (maxpos - minpos) div pagewidth + 1;
   initposadj:=((numpages * pagewidth) - (maxpos - minpos)) div 2 - minpos + 1;

   (* Main loop which is repeated once for each row of pages produced. *)
   while level <> nil do
      begin
      column:=0;
      posadj:=initposadj;
      topofpage:=level;
      row:=row + 1;
      topnextrow := level;
      Leveler(topnextrow);  (* topnextrow is first level on next page   *)
   
      (* produce a page in this row of pages  *)
      while column < numpages do
	 begin
	 column:=column + 1;
	 if device = lineprinter then
	    ArrayInitialize(pagelength,pagewidth)
         else if firsttime then begin
	    openplDvi;
	    spaceDvi(0,0,MaxXY,MaxXY);
	    eraseDvi;
	    firsttime := false;
            end
	 else
	    eraseDvi;
	 top:=true;

	 (* produce a level of nodes on this page *)
	 while (level <> topnextrow) and (level <> nil) do
	    begin
	    Boxptr:=level^.Last; 
            left:=true;

	    (* check for arcs across the page connecting nodes not on the page*)
	    if (Boxptr <> nil) then
	       begin
	          if top then
		  begin
	          if (Boxptr^.ArcSource <> nil)  then 
                  if (Boxptr^.ArcSource^.Xcoord + posadj > 0) and 
                     (Boxptr^.Xcoord + posadj < 0) then 
                     (* check if arcs from Boxptr^.ArcSource to some of its
                       sons cross the upper left-hand corner of this page  *)
                     case device of
                       lineprinter:lArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                         pagewidth,left);
                       plot:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                         top,cond);
                       PostScript:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                         top,cond);
                     end;
                  end
               else
	          if (Boxptr^.ArcSource <> nil) then 
                  if (Boxptr^.ArcSource^.Xcoord + posadj > pagewidth) 
                      and (Boxptr^.Xcoord + posadj < 0) then 

                     (* arc connecting 2 nodes not on this page crosses the
			page *)
                     case device of
                       lineprinter:lArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 pagewidth,left);
                       plot:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 top,cond);
                       PostScript:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                         top,cond);
                     end;

	   if Boxptr^.Xcoord + posadj < 0 then 
               if Boxptr^.Xcoord + posadj + Boxptr^.BoxWidth < 0 then  
	          (* none of Boxptr node extends onto the current page *)
	          Boxptr:=Boxptr^.Next; 
               end;

	    (* output a node's contents and arcs from it   *)
	    if Boxptr <> nil then
		if Boxptr^.Xcoord + posadj > pagewidth then
		    cont1 :=false
		else cont1:= true;
            while (Boxptr <> nil) and cont1 do
	       begin
               case device of
                 lineprinter:lDrawBox(Boxptr,level,prevlevel,posadj,top,
                                 Boxptr^.Seqsetpos,cond,notlastl,pagewidth,
			       	 pagelength);
	         plot:iDrawBox(Boxptr,level,prevlevel,posadj,top,
                                 Boxptr^.Seqsetpos,cond,notlastl);
	         PostScript:iDrawBox(Boxptr,level,prevlevel,posadj,top,
                                 Boxptr^.Seqsetpos,cond,notlastl);
               end;
	       level^.Last:=Boxptr;
	       Boxptr:=Boxptr^.Next;
	       if Boxptr <> nil then
	          if Boxptr^.Xcoord + posadj > pagewidth then
		     cont1:=false;
	       end;

            (* at right side of page, check for arcs across page *)
            left:=false;     
            if top then
	       begin
	       if (Boxptr <> nil) then
                  if (Boxptr^.ArcSource <> nil) then
		     if (Boxptr^.ArcSource^.Xcoord + posadj < pagewidth) and
		        (Boxptr^.Xcoord + posadj > pagewidth) then  

                  (* check if arcs from Boxptr^.ArcSource cross upper right-
                     hand corner of this page *)
                     case device of
                       lineprinter:lArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 pagewidth,left);
                       plot:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 top,cond);
                       PostScript:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 top,cond);
                     end;
               end                 
            else 
	       if (Boxptr <> nil) then             
                  if (Boxptr^.ArcSource <> nil) then 
		     if (Boxptr^.ArcSource^.Xcoord + posadj < 0) and
		        (Boxptr^.Xcoord + posadj > pagewidth) then    

                     (* arc from Boxptr^.ArcSource to Boxptr crosses page but
                        neither node is on the page *)
                     case device of
                       lineprinter:lArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 pagewidth,left);
                       plot:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 top,cond);
                       PostScript:iArccross(Boxptr^.ArcSource,prevlevel,posadj,
                                 top,cond);
                  end; 

            (* check for arcs crossing the lower left or right corner 
               of the page if at page bottom *)

	    if (level^.Next = topnextrow) and (device = plot) and
	       (topnextrow <> nil) then

		begin
		cknode:=topnextrow^.First;
		if cknode <> nil then 
		   continue:=true
		else 
  		   continue:=false;
		abovepage:=false;
		while continue do
		   begin
		   if (cknode^.ArcSource^.Xcoord + posadj < 0)
		      and (cknode^.Xcoord + posadj > 0) then
		      (* arcs across left corner of page *)
		      iArccross(cknode^.ArcSource,level,posadj,abovepage,cond)

		   else
		      if (cknode^.ArcSource^.Xcoord + posadj > pagewidth)
		        and (cknode^.Xcoord + posadj < pagewidth) then
			(* arcs across right corner of page *)
		      iArccross(cknode^.ArcSource,level,posadj,abovepage,cond);

		   cknode:=cknode^.Next;
		   if cknode <> nil then 
			if (not ((cknode^.ArcSource^.Xcoord+posadj > pagewidth) 
                           and (cknode^.Xcoord + posadj > pagewidth))) then
		 	   continue:=true
			else
			   continue:=false
		   else continue:=false;
		   end;
		end;

            prevlevel:=level;
	    level:=level^.Next;
            top:=false;
	    end;   (*  while loop that prints a level of nodes  *)

         (* Complete processing of a page of output *)
	 if device = lineprinter then   
	    Printout(pagelength,pagewidth,row,column)
         else begin
	    iPrintHeader(row,column);
	    closeplDvi;
            end;
	 posadj:=posadj - pagewidth;

         if column < numpages then 
            begin
            (* reset level and previous level to proper values for this row 
               of pages *)
	    prevlevel := prevbottom;		
            level:=topofpage;
	    end
	 else  (* start a new row of pages *)
	    prevbottom := prevlevel;

         end;    (* loop to print a page of output *)
      end;    (*  Main loop which prints a row of pages *)
   end;   (* procedure Print *)

