{***********************************************************************\ 
*									* 
*   File: scorpion/src/treepr/MakeTree.p 
*				 					* 
*   Copyright (C) 1991 Nancy Butler, Joan Curry, Steven Konstant,
*	Dore Rosenblum, and Greg Bollella
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************} 

const

#include "globalconst.h"

type

#include "globaltypes.h"
#include "MakeTree.h"
#include "MsgChar.h"

procedure MakeTree ;

(***************************************************************************)
(*                                                                         *)
(* procedure MakeTree ( idlStruct       : nodeDesc ;                       *)
(*                      var ValidStruct : boolean  ;                       *)
(*                      var Tr          : TreeNodeList )                   *)
(*                                                                         *)
(* Input parameter                                                         *)
(*   idlStruct - data structure built by treein.                           *)
(*                                                                         *)
(* Output parameters                                                       *)
(*   ValidStruct - true if the structure meets the restrictions on numbers *)
(*                 of different kinds of attributes, specified in          *)
(*                 globalconst.h, and if it contains no sequences or sets  *)
(*                 containing values that are sequences or sets.           *)
(*   Tr - the main data structure of the treepr program, represents same   *)
(*        information as idlStruct.                                        *)
(*        The following fields are set by MakeTree:                        *)
(*          in HeaderNodes -                                               *)
(*               First, Last, & Next are set.                              *)
(*               Ycoord is initialized to 0.                               *)
(*               MaxBoxHt is left uninitialized.                           *)
(*          in TreeNodes -                                                 *)
(*               Duplicate, Next, ArcSource, Level, Seqsetpos,             *)
(*               Subtrnum, and SeqSetnum are set.  		           *)
(*               All the variant fields are set.                           *)
(*               Xcoord, BoxHt, and BoxWidth are left uninitialized.       *)
(*                                                                         *)
(* Calls only subroutines declared internal to MakeTree, except for a      *)
(*        C function called by internal procedure PrintMessage.            *)
(*                                                                         *)
(* Programmer : Joan Curry                                                 *)
(* Written    : 4/19/84               Modified: 1/28/85  4/7/85  nlb       *)
(*                                                                         *)
(***************************************************************************)

  type
    ErrType       = (SetSeqNested, SetSeqType, NPattrNum) ;
                          {3 kinds of errors checked for}

    TreeValueKind = (Simple, Node, Seq, Set) ;
                          {Simple includes the 4 basic IDL simple types, also}
                          {unattributed nodes, which represent values of enum}
                          {types.  Seq and Set, here, includes nodes that are}
                          {members of seqs and sets, as well as simple seq}
                          {and set values.}

  var
    ThisLevel   ,                  {list of tree nodes at current level}
    NextLevel   : TreeNodeList ;   { "   "   "     "   "   next     "  }

    CurrentNode ,
    NewNode     : TreeNodePtr ;

    NewPattr    : PattrList   ;   {A Pattr is a "pointer" attribute}
    NewNPattr   : NPattrList  ;   {"non-pointer" or simple attribute}

    NPattrCount : integer ;       {Number of NPattributes of current node}

    idlAttrList  : LAttrDesc ;  {list of attributes in idlStruct}
    idlValueList : LIDLVALUE ;  { "   "   values    "     "     }

    SValueKind    : TreeValueKind ; {the kind of value a seq or set contains}
    MixedKinds    : boolean       ; {true if set or seq contains >1 value kind}

    posinseq    : seqset ;      {position in seq or set}
    level,                      {level in tree }
    subnum,                     {number of the current son on its level}
    seqnum      : integer;      {number of node in its sequence}

  procedure PrintMessage (WhichErr : ErrType ; BadNode : TreeNodePtr) ;
  (*******************************************************************)
  (* procedure PrintMessage                                          *)
  (*                                                                 *)
  (* Input parameters                                                *)
  (*   WhichErr - which kind of error it is.                         *)
  (*   BadNode - the node at fault.                                  *)
  (*                                                                 *)
  (* Prints error message to stderr.  Calls a C function, MsgChar,   *)
  (*   to print a char to stderr.                                    *)
  (*                                                                 *)
  (* Programmer: Joan Curry                                          *)
  (* Written   : 4/22/84                                             *)
  (*******************************************************************)
    var
      i : integer ;
    begin
      case WhichErr of
        SetSeqNested :
          message('Error: Sets and sequences may not be nested.');
        SetSeqType   :
          message('Error: Set or sequence members should all have same type');
        NPattrNum    :
          message('Error: Max number of simple attributes/node exceeded.');
      end ;

      { Try to identify the bad node }
      if BadNode^.Nlabel.length = 0 then
          message('Node whose attributes caused error is unlabeled.')
      else
        begin
          for i := 1 to BadNode^.Nlabel.length do
              MsgChar (BadNode^.Nlabel.value[i]);
          message(' is the label of the node whose attributes are in error');
        end
    end ;

  function WhichValueKind (idlValuePtr : IDLVALUE) : TreeValueKind ;
  (*******************************************************************)
  (* function WhichValueKind                                         *)
  (*                                                                 *)
  (* Input parameter                                                 *)
  (*   idlValuePtr - a value in the IDL structure, no restrictions.  *)
  (* Returns Simple, Node, Set, or Seq, depending on what kind of    *)
  (*   value it is, as documented in the type def'n of TreeValueKind.*)
  (*                                                                 *)
  (* Programmer: Joan Curry                                          *)
  (* Written   : 3/31/84                                             *)
  (*******************************************************************)
    begin
      case idlValuePtr^.TypeOf of
        KIntegerDesc, KBooleanDesc, KStringDesc, KRationalDesc :
            WhichValueKind := Simple ;
        KNodeDesc :
            if idlValuePtr^.VnodeDesc^.attributes = nil then
                WhichValueKind := Simple
            else
                WhichValueKind := Node ;
        KSequenceDesc :
            WhichValueKind := Seq ;
        KSetDesc :
            WhichValueKind := Set
      end
    end ;

  procedure CheckValList (idlValueList   : LIDLVALUE ;
                          var MixedKinds : boolean ;
                          var ValueKind  : TreeValueKind) ;
  (*******************************************************************)
  (* procedure CheckValList                                          *)
  (*                                                                 *)
  (* Input parameters                                                *)
  (*   idlValueList - non-empty list of IDL values, expected to be a *)
  (*                  sequence or set.                               *)
  (* Output parameters                                               *)
  (*   MixedKinds - true if idlValueList contains values of more     *)
  (*                than one TreeValueKind.                          *)
  (*   ValueKind - returns nothing meaningful if MixedKinds is true. *)
  (*               Otherwise returns the kind of values that         *)
  (*               idlValueList contains.                            *)
  (* Calls WhichValueKind.                                           *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/31/84                                            *)
  (*******************************************************************)
    begin
      MixedKinds := false ;
      ValueKind := WhichValueKind (idlValueList^.value) ;
      idlValueList := idlValueList^.next ;
      while idlValueList <> nil do
        begin
          if WhichValueKind (idlValueList^.value) <> ValueKind then
              MixedKinds := true ;
          idlValueList := idlValueList^.next
        end
    end ;

  procedure GetStringValue ( idlValuePtr : IDLVALUE ;
                             var S       : rString   ) ;
  (*******************************************************************)
  (* procedure GetStringValue                                        *)
  (*                                                                 *)
  (* Input parameter                                                 *)
  (*   idlValuePtr - a simple value in the IDL structure, where an   *)
  (*                 unattributed node is taken to be a simple value.*)
  (* Output parameter                                                *)
  (*   S - string repr'n of the value in the IDL structure, where the*)
  (*       string rep'n of a node value is the IDL node type.        *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/10/84                                            *)
  (*******************************************************************)
    var
      idlString : String  ;
      i         : integer ;
    begin
      case idlValuePtr^.TypeOf of
        KIntegerDesc :
            S := idlValuePtr^.VintegerDesc^.stringRep^ ;
        KRationalDesc :
            S := idlValuePtr^.VrationalDesc^.stringRep^ ;
        KStringDesc :            {String value will be printed with quotes}
          begin                             {around it, so store it that way.}
            idlString := idlValuePtr^.VstringDesc^.value ;
            S.value[1] := '"' ;    
            i := 2 ;
            while ( i < StringMax ) and ( i-1 <= idlString^.length ) do
              begin
                S.value[i] := idlString^.value[i-1] ;
                i := i+1
              end ;
              S.value[i] := '"' ; 
            S.length := i 
          end ;
        KBooleanDesc :
            if idlValuePtr^.VbooleanDesc^.value then
              begin                  {trying to keep flexibility to change}
                S.value[1] := 'T' ;  {StringMax without using non-standard}
                S.value[2] := 'R' ;  {string padding}
                S.value[3] := 'U' ;
                S.value[4] := 'E' ; 
                S.length := 4
              end
            else
              begin
                S.value[1] := 'F' ;
                S.value[2] := 'A' ;
                S.value[3] := 'L' ;
                S.value[4] := 'S' ;
                S.value[5] := 'E' ;
                S.length := 5
              end ;
        KNodeDesc :       {should only get here if the node is unattributed}
            S := idlValuePtr^.VnodeDesc^.name^ ;
      end
    end ;

  function MakeTreeValueNode ( WhichBoxKind : BoxKind ;
                               idlValuePtr  : IDLVALUE ;
                               Father       : TreeNodePtr;
			       level	    : integer;
                               posinseq     : seqset ) : TreeNodePtr ;

  (*******************************************************************)
  (* function MakeTreeValueNode                                      *)
  (*                                                                 *)
  (* Input parameters                                                *)
  (*   WhichBoxKind - which kind of node it will be, must be either  *)
  (*                  SeqValue or SetValue.                          *)
  (*   idlValuePtr - value in the IDL structure that corresponds to  *)
  (*                 this node in the tree.                          *)
  (*   Father - father of this node in the tree.                     *)
  (*   posinseq - position in seq or set.                            *)
  (* Returns a new TreeNode with fields initialized as follows:      *)
  (*   tag       - to WhichKind.                                     *)
  (*   Duplicate - to false (can't duplicate a value).               *)
  (*   Next      - to nil.                                           *)
  (*   ArcSource - to Father.                                        *)
  (*   Seqsetpos - position in seq or set if node is in seq or set.  *)
  (*   Empty     - to true iff idlValuePtr is nil.                   *)
  (*   StrRepVal - to string repr'n of value in IDL structure if     *)
  (*               idlValuePtr not nil, otherwise to null string.    *)
  (*                                                                 *)
  (* Calls GetStringValue                                            *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/30/84                                            *)
  (*******************************************************************)

    var
      P : TreeNodePtr ;
    begin
      new (P) ;
      P^.tag := WhichBoxKind ;
      P^.ArcSource := Father ;
      P^.Next := nil ;
      P^.Duplicate := false ;
      P^.Level := level;
      P^.Seqsetpos := posinseq ;
      if idlValuePtr = nil then  {empty sequence or set}
        begin
          P^.Empty := true ;
          P^.StrRepVal.length := 0
        end
      else
        begin
          P^.Empty := false ;
          GetStringValue (idlValuePtr, P^.StrRepVal)
        end ;
      MakeTreeValueNode := P
    end ;

  function MakeTreeNode ( WhichBoxKind  : BoxKind  ;
                          idlNodePtr    : nodeDesc ;
                          Father        : TreeNodePtr ;
			  level		: integer;
                          posinseq      : seqset ) : TreeNodePtr ;

  (*******************************************************************)
  (* function MakeTreeNode                                           *)
  (*                                                                 *)
  (* Input parameters                                                *)
  (*   WhichKind  - which kind of tree node it will be, must be a    *)
  (*                RegularNode, SeqNode, or SetNode.                *)
  (*   idlNodePtr - node in the IDL structure that corresponds to    *)
  (*                to this node in the tree.                        *)
  (*   Father - father of this node in the tree structure.           *)
  (*   posinseq - position in seq or set.                            *)
  (*                                                                 *)
  (* Returns a new TreeNode with the following fields set:           *)
  (*     idlRef   ArcSource   Seqsetpos  tag   Nlabel   Ntype        *)
  (*     Duplicated - to 'Touched' field of idlNodePtr^ - should be  *)
  (*                  true if that node has been encountered before  *)
  (*                  by MakeTree.                                   *)
  (*     Pattribs, NPattribs, and Next - to nil.                     *)
  (* Xcoord, BoxHt, BoxWidth, & PadPattrLine are left uninitialized. *)
  (*                                                                 *)
  (* SIDE EFFECT : Sets 'Touched' field of idlNodePtr^ to true.      *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/10/84                                            *)
  (*******************************************************************)

    var
      P : TreeNodePtr ;
    begin
      new (P) ;
      P^.tag := WhichBoxKind ;
      P^.Nlabel := idlNodePtr^.Flabel^ ;
      P^.Ntype := idlNodePtr^.name^ ;
      P^.Duplicate := idlNodePtr^.Touched ;
      P^.idlRef := idlNodePtr ;
      P^.ArcSource := Father ;
      P^.Level := level;
      P^.Seqsetpos := posinseq ;
      P^.Pattribs := nil ;
      P^.NPattribs := nil ;
      P^.Next := nil ;
      idlNodePtr^.Touched := true ;
      MakeTreeNode := P
    end ;

  function MakeEmptyList : TreeNodeList ;

  (*******************************************************************)
  (* function MakeEmptyList                                          *)
  (*                                                                 *)
  (* Returns an empty list of TreeNodes.  Used to initialize lists   *)
  (*   and to reset used lists to empty.                             *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/10/84                                            *)
  (*******************************************************************)

    var
      L : TreeNodeList ;
    begin
      new(L) ;
      L^.First := nil ;
      L^.Last := nil ;
      L^.Next := nil ;
      L^.Ycoord := 0;
      MakeEmptyList := L
    end ;


  procedure AddTreeNode ( var TrNode     : TreeNodePtr ;
                          var TrNodeList : TreeNodeList;
			  var subcount, setseqnum : integer) ;

  (******************************************************************)
  (*  procedure AddTreeNode                                         *)
  (*                                                                *)
  (*  Update parameter                                              *)
  (*    TrNode - a tree node, no restrictions.  The Subtrnum and    *)
  (*             Setseqnum fields are set.                          *)
  (*    TrNodeList - list of tree nodes.  TrNode is added to the    *)
  (*                 end of the list.                               *)
  (*    subcount - the count of the number of sons at this level in *)
  (*               the tree.					    *)
  (*    setseqnum - the count of the number of nodes in the current *)
  (*                sequence or set.				    *)
  (*                                                                *)
  (*  Programmer : Joan Curry                                       *)
  (*  Written    : 3/10/84       Modified: 4/7/85  by butler        *)
  (******************************************************************)

    begin
      if TrNodeList^.First = nil then
	  begin
          TrNodeList^.First := TrNode;
	  subcount:=1;
	  TrNode^.Subtrnum:= 1;
	  if TrNode^.tag <> RegularNode then
	     begin
	     TrNode^.SetSeqnum:=1;
	     setseqnum:=1;
	     end;
	  end
      else
	  begin
          TrNodeList^.Last^.Next := TrNode ;
	  if TrNode^.tag = RegularNode then
	     begin
	     subcount := subcount +1;
	     TrNode^.Subtrnum := subcount;
	     end
	  else
	     if (TrNodeList^.Last^.tag = RegularNode) or 
		(last in TrNodeList^.Last^.Seqsetpos) then
		begin
		subcount:=subcount + 1;
		TrNode^.Subtrnum:= subcount;
		TrNode^.SetSeqnum:= 1;
		setseqnum:= 1;
		end
	     else
		begin
		setseqnum:=setseqnum +1;
		TrNode^.SetSeqnum:=setseqnum;
		TrNode^.Subtrnum:=subcount;
		end;
	   end;
      TrNodeList^.Last := TrNode;
    end ;

  function MakeNPattr ( idlAttrPtr : attrDesc ) : NPattrList ;
  (*******************************************************************)
  (* function MakeNPattr                                             *)
  (*                                                                 *)
  (* Input parameter                                                 *)
  (*   idlAttrPtr - an attribute in the IDL structure, must not      *)
  (*                have a sequence or set value.                    *)
  (* Returns a corresponding attribute for the tree structure,       *)
  (*   with fields initialized as follows:                           *)
  (*     Next  - to nil.                                             *)
  (*     Name  - to name of the attribute in the IDL structure.      *)
  (*     StrRepVal - for simple attributes, to the string repr'n of  *)
  (*                 the attribute's value in the IDL structure. For *)
  (*                 node attributes, to the node type in the IDL    *)
  (*                 structure (should be called only for            *)
  (*                 unattributed nodes).                            *)
  (* Calls GetStringValue                                            *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/10/84                                            *)
  (*******************************************************************)
    var
      P : NPattrList ;
    begin
      new (P) ;
      P^.Next := nil ;
      P^.Name := idlAttrPtr^.name^ ;
      GetStringValue (idlAttrPtr^.value, P^.StrRepVal) ;
      MakeNPattr := P
    end ;

  function MakePattr ( idlAttrPtr : attrDesc ;
                       val        : TreeNodePtr ) : PattrList ;
  (*******************************************************************)
  (* function MakePattr                                              *)
  (*                                                                 *)
  (* Input parameters                                                *)
  (*   idlAttrPtr - attribute in the IDL structure, must have node   *)
  (*                value.                                           *)
  (*   val - TreeNode that will be the value of the attribute.       *)
  (* Returns a corresponding attribute for the tree structure, with  *)
  (*   fields initialized as follows:                                *)
  (*     Next  - to nil.                                             *)
  (*     Name  - to name of attribute in IDL structure.              *)
  (*     Value - to val.                                             *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/10/84                                            *)
  (*******************************************************************)
    var 
      P : PattrList ;
    begin
      new (P) ;
      P^.Next := nil ;
      P^.Name := idlAttrPtr^.name^ ;
      P^.Value := val ;
      MakePattr := P
    end ;

  procedure AddNPattr ( Attr         : NPattrList ;
                        var AttrList : NPattrList  ) ;
  (*******************************************************************)
  (* procedure AddNPattr                                             *)
  (*                                                                 *)
  (* Input parameter                                                 *)
  (*   Attr - a "non-pointer" attribute for the tree structure.      *)
  (*                                                                 *)
  (* Update parameter                                                *)
  (*   AttrList - list of "non-pointer" attributes in tree structure.*)
  (*              Attr is added to the end of the list.              *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/10/84                                            *)
  (*******************************************************************)
    var
      P : NPattrList ;
    begin
      if AttrList = nil then
          AttrList := Attr
      else
        begin
          P := AttrList ;
          while P^.Next <> nil do
              P := P^.Next ;
          P^.Next := Attr
        end
    end ;


  procedure AddPattr ( Attr         : PattrList ;
                       var AttrList : PattrList  ) ;
  (*******************************************************************)
  (* procedure AddPattr                                              *)
  (*                                                                 *)
  (* Input parameter                                                 *)
  (*   Attr - a "pointer" attribute for the tree structure.          *)
  (*                                                                 *)
  (* Update parameter                                                *)
  (*   AttrList - list of "pointer" attributes in tree structure.    *)
  (*              Attr is added to the end of the list.              *)
  (*                                                                 *)
  (* Programmer : Joan Curry                                         *)
  (* Written    : 3/10/84                                            *)
  (*******************************************************************)
    var
      P: PattrList ;
    begin
      if AttrList = nil then
          AttrList := Attr
      else
        begin
          P := AttrList ;
          while P^.Next <>nil do
              P := P^.Next ;
          P^.Next := Attr
        end
    end ;

begin                                    (* body of MakeTree *)

  { Initialize variables, & put root node into data structure }
  ValidStruct := true ;

  ThisLevel := MakeEmptyList ;
  NextLevel := MakeEmptyList ;
  level := 1;
  subnum:= 1;
  seqnum:= 1;

  NewNode := MakeTreeNode (RegularNode, idlStruct, nil, level,[]) ;
  AddTreeNode (NewNode, ThisLevel, subnum, seqnum) ;
  Tr := ThisLevel ;
  level:= 2;

 {Invariant:  All tree nodes above this level are completely processed.  If}
 {this level is empty, processing is finished.  Otherwise, nodes at this level}
 {are already linked into the tree, and all fields that MakeTree sets are set}
 {except for the attributes}

  while ThisLevel^.First <> nil do
    begin               {a new level of the tree}

      CurrentNode := ThisLevel^.First ;
      while CurrentNode <> nil do
        begin                {a new node at this level}

          if (CurrentNode^.tag = SeqValue) or
             (CurrentNode^.tag = SetValue) or
             (CurrentNode^.Duplicate) then
                  {no attributes to process - a value doesn't have attributes,}
                  {and a duplicate node's attributes should not be put in tree}
                  {a second time.}
          else
            begin                       {CurrentNode has attributes to process}
 
              NPattrCount := 0 ;
              idlAttrList := CurrentNode^.idlRef^.attributes ;

              while idlAttrList <> nil do
                begin                   {a new attribute for this node}
 
                  case WhichValueKind (idlAttrList^.value^.value) of

                    Simple :
                      begin
                        NPattrCount := NPattrCount + 1 ;
                        NewNPattr := MakeNPattr (idlAttrList^.value) ;
                        AddNPattr (NewNPattr, CurrentNode^.NPattribs)
                      end ;
  
                    Node :
                      begin
                        NewNode := MakeTreeNode (RegularNode,
                                    idlAttrList^.value^.value^.VnodeDesc,
                                    CurrentNode, level, [] ) ;
                        NewPattr := MakePattr (idlAttrList^.value, NewNode) ;
                        AddPattr (NewPattr, CurrentNode^.Pattribs) ;
                        AddTreeNode (NewNode, NextLevel, subnum, seqnum)
                      end ;
  
                    Seq :
                      begin
                        idlValueList :=
                              idlAttrList^.value^.value^.VsequenceDesc^.value ;

                        if idlValueList = nil then            {empty sequence}
                          begin
                            posinseq := [first,last];
                            NewNode := MakeTreeValueNode (SeqValue, nil,
                                           CurrentNode, level, posinseq) ;
                            NewPattr := MakePattr(idlAttrList^.value, NewNode);
                            AddPattr (NewPattr, CurrentNode^.Pattribs) ;
                            AddTreeNode (NewNode, NextLevel, subnum, seqnum)
                          end

                        else                               {non-empty sequence}
                          begin
                            CheckValList(idlValueList, MixedKinds, SValueKind);

                            if MixedKinds then                          {error}
                              begin
                                PrintMessage (SetSeqType, CurrentNode) ;
                                ValidStruct := false
                              end

                            else             {seq values are all the same kind}
                              case SValueKind of

                                Seq, Set :         {nested set or seq is error}
                                  begin
                                    PrintMessage (SetSeqNested, CurrentNode) ;
                                    ValidStruct := false
                                  end ;

                                Simple :     {1st seq value gets linked in the}
                                             {same way as any value, except it}
                                             {has a SeqValue tag}
                                  begin
                                    if idlValueList^.next = nil then
				       posinseq := [first,last] 
				    else 
				       posinseq := [first];
                                    NewNode := MakeTreeValueNode (SeqValue,
                                      idlValueList^.value,CurrentNode,
				      level, posinseq);
                                    NewPattr := MakePattr (idlAttrList^.value,
                                                           NewNode) ;
                                    AddPattr(NewPattr, CurrentNode^.Pattribs) ;
                                    AddTreeNode (NewNode,NextLevel,
						 subnum,seqnum) ;

                                            {Remaining seq values get inserted}
                                            {into next tree level, but aren't}
                                            {separate attributes}
                                    idlValueList := idlValueList^.next ;
                                    while idlValueList <> nil do
                                      begin
                                        if idlValueList^.next = nil then
				           posinseq := [last] 
			                else posinseq := [] ;		
                                        NewNode := MakeTreeValueNode (SeqValue,
                                            idlValueList^.value, CurrentNode,
					    level,posinseq) ;
                                        AddTreeNode (NewNode, NextLevel,subnum,
						     seqnum) ;
                                        idlValueList := idlValueList^.next
                                      end
                                  end ;

                                Node :               {Logic here is same as in}
                                                            {Simple case above}
                                  begin
                                    if idlValueList^.next = nil then
				       posinseq := [first,last] 
				    else 
				       posinseq := [first];
                                    NewNode := MakeTreeNode (SeqNode,
                                                idlValueList^.value^.VnodeDesc,
                                                CurrentNode, level, posinseq) ;
                                    NewPattr := MakePattr (idlAttrList^.value,
                                                           NewNode) ;
                                    AddPattr(NewPattr, CurrentNode^.Pattribs) ;
                                    AddTreeNode (NewNode, NextLevel,subnum,
						 seqnum) ;

                                    idlValueList := idlValueList^.next ;
                                    while idlValueList <> nil do
                                      begin
                                        if idlValueList^.next = nil then
				           posinseq := [last] 
			                else posinseq := [] ;		
                                        NewNode := MakeTreeNode (SeqNode,
                                                idlValueList^.value^.VnodeDesc,
                                                CurrentNode, level, posinseq) ;
                                        AddTreeNode (NewNode, NextLevel,
						     subnum, seqnum) ;
                                        idlValueList := idlValueList^.next
                                      end
                                  end     {case for Node}

                              end     {case SValueKind}
                          end     {else -- non-empty sequence}
                      end ;   {case for Seq}
  
                    Set :     {Code for Set case is identical to that for Seq }
                              {case, except for (1) path of assignment to     }
                              {idlValueList, and (2) first parameter passed to}
                              {MakeTreeNode and MakeTreeValueNode.}
                      begin
                        idlValueList :=
                              idlAttrList^.value^.value^.VsetDesc^.value ;

                        if idlValueList = nil then
                          begin
			    posinseq := [first,last] ;
                            NewNode := MakeTreeValueNode (SetValue, nil,
                                               CurrentNode,level,posinseq) ;
                            NewPattr := MakePattr(idlAttrList^.value, NewNode);
                            AddPattr (NewPattr, CurrentNode^.Pattribs) ;
                            AddTreeNode (NewNode, NextLevel, subnum, seqnum)
                          end

                        else                         {non-empty set}
                          begin
                            CheckValList(idlValueList, MixedKinds, SValueKind);

                            if MixedKinds then
                              begin
                                PrintMessage (SetSeqType, CurrentNode) ;
                                ValidStruct := false
                              end

                            else       {all values in set are the same kind}
                              case SValueKind of

                                Seq, Set :
                                  begin
                                    PrintMessage (SetSeqNested, CurrentNode) ;
                                    ValidStruct := false
                                  end ;

                                Simple :
                                  begin
                                    if idlValueList^.next = nil then
				       posinseq := [first,last] 
				    else 
				       posinseq := [first];
                                    NewNode := MakeTreeValueNode (SetValue,
                                           idlValueList^.value, CurrentNode,
					   level, posinseq) ;
                                    NewPattr := MakePattr (idlAttrList^.value,
                                                           NewNode) ;
                                    AddPattr(NewPattr, CurrentNode^.Pattribs) ;
                                    AddTreeNode (NewNode, NextLevel,
						 subnum,seqnum) ;

                                    idlValueList := idlValueList^.next ;
                                    while idlValueList <> nil do
                                      begin
                                        if idlValueList^.next = nil then
				           posinseq := [last] 
			                else posinseq := [] ;		
                                        NewNode := MakeTreeValueNode (SetValue,
                                            idlValueList^.value, CurrentNode,
					    level, posinseq) ;
                                        AddTreeNode (NewNode, NextLevel,
					    subnum, seqnum) ;
                                        idlValueList := idlValueList^.next
                                      end
                                  end ;

                                Node :
                                  begin
                                    if idlValueList^.next = nil then
				       posinseq := [first,last] 
				    else 
				       posinseq := [first];
                                    NewNode := MakeTreeNode (SetNode,
                                                idlValueList^.value^.VnodeDesc,
                                                CurrentNode, level, posinseq) ;
                                    NewPattr := MakePattr (idlAttrList^.value,
                                                           NewNode) ;
                                    AddPattr(NewPattr, CurrentNode^.Pattribs) ;
                                    AddTreeNode (NewNode, NextLevel, subnum,
						 seqnum) ;

                                    idlValueList := idlValueList^.next ;
                                    while idlValueList <> nil do
                                      begin
                                        if idlValueList^.next = nil then
				           posinseq := [last] 
			                else posinseq := [] ;		
                                        NewNode := MakeTreeNode (SetNode,
                                                idlValueList^.value^.VnodeDesc,
                                                CurrentNode, level, posinseq) ;
                                        AddTreeNode (NewNode, NextLevel,
						 subnum,seqnum) ;
                                        idlValueList := idlValueList^.next
                                      end
                                  end     {case for Node}
                              end     {case SValueKind}
                          end    {else -- non-empty set}
                      end    {case for Set}
                  end ;   {case WhichValueKind -- have processed one attribute}

                  idlAttrList := idlAttrList^.next
                end          {while idlAttrList <> nil}
            end ;        {else begin -- CurrentNode has attributes to process}
   
          if NPattrCount > MaxNPAttribs then                     {error check}
            begin
              PrintMessage (NPattrNum, CurrentNode) ;
              ValidStruct := false
            end ;

          CurrentNode := CurrentNode^.Next
        end ;            {while CurrentNode <> nil}
 
     {Link in next level if it isn't empty, & set up for next time thru loop}
      if NextLevel^.First <> nil then
          ThisLevel^.Next := NextLevel ;
      ThisLevel := NextLevel ;
      NextLevel := MakeEmptyList;
      level := level + 1;
    end      {while ThisLevel^.First <> nil}

end ;                                  (* body of MakeTree *)
