/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlc/semantic/without.c 
*				 					* 
*   Copyright (C) 1991 Karen Shannon
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Function: routines used in processing without clauses		*
*									*
*	     ProcessWithoutClause  DeleteAssertion DeleteDefine  	*
*	     DeleteProduction DeleteAttribute DeleteSubclass		*
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: without.c,v 1.1 89/07/05 16:32:16 kps Locked $";
#endif

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	without.c,v $
 * Revision 1.1  89/07/05  16:32:16  kps
 * Initial revision
 * 
 * Revision 4.0  89/04/12  02:50:35  cheung
 * without.c  Ver 4.0
 * 
 * Revision 3.9  89/03/30  13:24:42  cheung
 * without.c  Ver 3.9
 * 
 * Revision 3.9  89/03/26  15:05:23  cheung
 * without.c  Ver 3.9
 * 
 * Revision 3.2  88/02/19  18:22:42  rajan
 * Version 3.2
 * 
 * Revision 1.1  87/04/18  11:40:56  shannon
 * Initial revision
 * 
 * Revision 1.1  86/06/06  14:47:38  shannon
 * Initial revision
 * 
 * Revision 2.0  86/01/22  07:22:26  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.7  86/01/22  07:10:12  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.6  85/07/26  10:02:24  shannon
 * added without assertions
 * 
 * Revision 1.5  85/07/25  10:36:49  shannon
 * changed code for new null and empty class rep
 * 
 * Revision 1.4  85/07/18  19:38:56  shannon
 * deleted some unused variables
 * 
 * Revision 1.3  85/07/04  14:55:21  shannon
 * changed set of symbols to sequence
 * 
 * Revision 1.2  85/07/04  10:07:10  shannon
 * expanded the without routines to delete classes when seeing
 * a clause of the form Without ::=; and to delete nodes when
 * seeing a clause of the form Without node =>;
 * 
 * Revision 1.1  85/06/11  10:19:31  shannon
 * Initial revision
 * 
*									*
*   Edit Log:								*
*     Jan 9 1985 (shannon) Created.					*
*									*
\* ******************************************************************* */
 
#include "Semantic.h"
#include "macros.h"
#include <stdio.h>

/***********************************************************************
 *
 * Procedure ProcessWithoutClause
 *
 * Purpose: Process the without clause given by 'woclause'
 *
 * Algorithm: Call the appropriate routine depending on type of 'woclause'
 *
 * Errors Checked For: invalid type of 'woclause'
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

ProcessWithoutClause(st, woclause)
StructureEntity st;	/* the structure */
withoutClause woclause;	/* the without clause */
{
	SEQwithoutItem tl;	/* traversal of list of withoutClause */
	withoutItem lval;	/* value in list */

	Assume((IsStructureEntity(st) && IswithoutClause(woclause)),
		"ProcessWithoutClause");

	foreachinSEQwithoutItem(woclause->syn_list, tl, lval) {

		switch (typeof(lval)) {

			case KwithoutAssert:
				DeleteAssertion(st, lval.VwithoutAssert);
				break;

			case KwithoutDefine:
				DeleteDefine(st, lval.VwithoutDefine);
				break;

			case KwithoutType:
				DeleteType(st, lval.VwithoutType);
				break;

			case KwithoutAttribute:
			case KwithoutSubclass:
				DeleteProduction(st, lval.VwithoutProduction);
				break;
			
			default:
				Fatal1(1001, 0, "ProcessWithoutClause");
				break;
		}
	}
}


/***********************************************************************
 *
 * Procedure DeleteAssertion
 *
 * Purpose: Delete the assertion from the structure
 *
 * Algorithm: Find the assertion in the assertion set of the structure
 *	      with the same name as the assertion in the without clause
 *	      and remove it from the set.
 *
 * Errors Checked For: Non-existent assertion
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
DeleteAssertion(st, woassert)
StructureEntity st;	
withoutAssert woassert;
{
	SETAssertion tAs;		/* traversal of assertions set */
	Assertion As;			/* an assertion in the set */
	Boolean found;			/* indicator if assertion is found */

	Assume((IsStructureEntity(st)&&IswithoutAssert(woassert)),
	       "DeleteAssertion");

	found = FALSE;
	foreachinSETAssertion(st->sem_assertions, tAs, As) {
	    if (strequal(As->lex_name, woassert->syn_item->lex_name)) {
		found = TRUE;
		break;
	    }
	}


	/* if found, remove from sequence but *do not* de-allocate storage
	   since this assertion may be pointed to by other structures    */

	if (found) {
	    SetAssertEntity(woassert->syn_item, As);
	    removeSETAssertion(st->sem_assertions, As);
	}
	else {
	    SetErrorEntity(woassert->syn_item);
	    Warning0(300, woassert->syn_item->lex_namepos);
	}
}

/***********************************************************************
 *
 * Procedure DeleteDefine
 *
 * Purpose: Delete the definition from the structure
 *
 * Algorithm: Find the definition in the definition set of the structure
 *	      with the same name as the definition in the without clause
 *	      and remove it from the set.
 *
 * Errors Checked For: Non-existent definition
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
DeleteDefine(st, wodef)
StructureEntity st;	
withoutDefine wodef;
{
	SETDefinition tdef;		/* traversal of definitions set */
	Definition Adef;		/* a definition in the set */
	Boolean found;			/* indicator if definition is found */

	Assume((IsStructureEntity(st)&&IswithoutDefine(wodef)),
	       "DeleteDefine");

	found = FALSE;
	foreachinSETDefinition(st->sem_definitions, tdef, Adef) {
	    if (strequal(Adef->sem_name, wodef->syn_item->lex_name)) {
		found = TRUE;
		break;
	    }
	}


	/* if found, remove from sequence but *do not* de-allocate storage
	   since this definition may be pointed to by other structures    */

	if (found) {
	    SetDefineEntity(wodef->syn_item, Adef);
	    removeSETDefinition(st->sem_definitions, Adef);
	}
	else {
	    SetErrorEntity(wodef->syn_item);
	    Warning0(304, wodef->syn_item->lex_namepos);
	}
}

/***********************************************************************
 *
 * Procedure DeleteType
 *
 * Purpose: Delete the type from structure 'st'
 *
 * Algorithm: Find and delete the type from structure sem_types
 *	      Delete each attribute with the type
 *	      Delete each subclass of this type
 * 	      Delete each assertion and definition with the type.
 *
 * Errors Checked For: Non-existent type
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
		
DeleteType(st, wotype)
StructureEntity st;
withoutType wotype;
{
	SETAssertion SAssert;	/* set/seq traverals and values */
	Assertion AnAssert;
	SETAttribute SAtt;
	Attribute AnAtt;
	SETDefinition SDef;
	Definition ADef;
	SETClass SClass;
	Class AClass;
	SETTypeEntity SType;
	TypeEntity AType,AType1;
	SetOf setdeletetype;
	SeqOf seqdeletetype;
	TypeEntity deletetype;	/* the type to be deleted */

	Assume((IsStructureEntity(st)&&IswithoutType(wotype)),
		"DeleteType");

	if (FindType(st, wotype->syn_item) == FOUND) {
	    deletetype = GetTypeEntity(wotype->syn_item);

	    /* set the reference to the copied from ancestor */
	    SetTypeEntity(wotype->syn_item, 
		deletetype.IDLclassCommon->sem_copiedfrom.VTypeEntity);

	    /* remove the type from the structure */
	    removeSETTypeEntity(st->sem_types, deletetype);

	    /* delete any attributes with this type */
	    foreachinSETAttribute(deletetype.IDLclassCommon->inv_attributes,
				  SAtt, AnAtt) {
		removeSETAttribute(AnAtt->inv_parent.VClass->sem_allattributes, 
				   AnAtt);
	    }

	    /* delete any sets or sequences of this type */
	    /* delete any attributes with sets or seqs of this type */
	    if (typeof(deletetype) != KSetOf && typeof(deletetype) != KSeqOf) {
		setdeletetype = NULL;
		seqdeletetype = NULL;
		foreachinSETTypeEntity(st->sem_types, SType, AType) {
		    switch (typeof(AType)) {
			case KSetOf:
			    if (SameType(deletetype.VNamedType,
					 AType.VSetOf->sem_component)) {
				setdeletetype = AType.VSetOf;
			    }
			    break;
			case KSeqOf:
			    if (SameType(deletetype.VNamedType,
					 AType.VSeqOf->sem_component)) {
				seqdeletetype = AType.VSeqOf;
			    }
			    break;
		    }
		}
		if (setdeletetype != NULL) {
		    AType1.VSetOf = setdeletetype;
		    removeSETTypeEntity(st->sem_types, AType1);
		    foreachinSETAttribute(setdeletetype->inv_attributes, 
								SAtt, AnAtt) {
			removeSETAttribute(AnAtt->inv_parent.VClass->
					sem_allattributes, AnAtt);
		    }
		}
		if (seqdeletetype != NULL) {
		    AType1.VSeqOf = seqdeletetype;
		    removeSETTypeEntity(st->sem_types, AType1);
		    foreachinSETAttribute(seqdeletetype->inv_attributes, 
								SAtt, AnAtt) {
			removeSETAttribute(AnAtt->inv_parent.VClass->
					sem_allattributes, AnAtt);
		    }
		}
	    }

	    if (typeof(deletetype) == KClass) {
		foreachinSETClass(deletetype.VClass->sem_ancestors, SClass, 
					AClass)
		    removeSETClass(AClass->sem_subclasses, deletetype.VClass);
		foreachinSETClass(deletetype.VClass->sem_subclasses, SClass,
					AClass)
		    removeSETClass(AClass->sem_ancestors, deletetype.VClass);
	    }

	    foreachinSETAssertion(deletetype.IDLclassCommon->inv_assertions,
				  SAssert, AnAssert) {
		removeSETAssertion(st->sem_assertions, AnAssert);
	    }

	    foreachinSETDefinition(deletetype.IDLclassCommon->inv_definitions,
				   SDef, ADef) {
		removeSETDefinition(st->sem_definitions, ADef);
	    }
	}
	else {
	    SetErrorEntity(wotype->syn_item.IDLclassCommon);
	    Warning0(257, GetTypePos(wotype->syn_item));
	}
}

/***********************************************************************
 *
 * Procedure DeleteProduction
 *
 * Purpose: Delete the production from the structure
 *
 * Algorithm: Find the class or classes.
 *	      Call delete attribute or delete subclass routines.
 *
 * Errors Checked For: Non-existent class
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

DeleteProduction(st, woProd)
StructureEntity st;	  /* the structure where production is deleted */
withoutProduction woProd; /* production to be deleted */
{
	SETClass allclasses;	/* all the classes */
	NamedTypeRef nameref;	/* reference to class */
	SETClass GetAllClasses();
        Boolean IsClassType();
        int FindNamedType();

	Assume((IsStructureEntity(st)&&IswithoutProduction(woProd)),
		"DeleteProduction");

	initializeSETClass(allclasses);

	if (typeof(woProd.IDLclassCommon->syn_lefthandside) == KAllClasses) {
	    allclasses = GetAllClasses(st);
	}
	else {
	    nameref = woProd.IDLclassCommon->syn_lefthandside.VNamedTypeRef;
	    if (FindNamedType(st, nameref->lex_name, (NamedType *) &(nameref->sem_entity))
			== NOTFOUND){
		SetErrorEntity(nameref);
		Warning0(153, nameref->lex_namepos);
	    }
	    else if (TypeOfEntity(nameref) != KClass) {
		SetErrorEntity(nameref);
		Warning0(229, nameref->lex_namepos);
	    }
	    else if ((typeof(woProd) == KwithoutSubclass) &&
		     (!IsClassType(GetClassEntity(nameref)))) {
		SetErrorEntity(nameref);
		Warning0(230, nameref->lex_namepos);
	    }
	    else {
		addSETClass(allclasses, GetClassEntity(nameref));
	    }
	}

	if (typeof(woProd) == KwithoutSubclass) {
	    DeleteSubclass(st, allclasses, 
			   woProd.VwithoutSubclass->syn_righthandside);
	}
	else {
	    DeleteAttribute(allclasses, 
			    woProd.VwithoutAttribute->syn_righthandside);
	}
}

/***********************************************************************
 *
 * Procedure DeleteAttribute
 *
 * Purpose: Delete the attribute or attributes from the classes
 *
 * Algorithm: 
 *
 * Errors Checked For: non-existent attribute
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
DeleteAttribute(allclasses, AttOrAll)
SETClass allclasses;                /* classes where attribute(s) should 
				       be deleted */
AttributeOrAllAttributes AttOrAll;  /* attribute or all attributes */
{
	AttributeRefs attrefs;	/* reference to attributes */
	AllAttributes alltype;
	Attribute attfound;	/* attribute found */
	SourcePosition spos;	/* position of reference or '*' */
	SETAttribute allatts;	/* all attributes deleted */
	Boolean found=FALSE;	/* indicator of attribute(s) found */
	Boolean propagated = TRUE;
	int sizeset;
	SETClass SClass;	/* set/seq traversals and values */
	Class AClass;
	SEQAttribute SAtt;
	Attribute AnAtt;
        Boolean PropagatedAttribute();

	Assume((IsAttOrAll(AttOrAll)), "DeleteAttribute");

	initializeSETAttribute(allatts);
	if (typeof(AttOrAll) == KAllAttributes) {
	    alltype = AttOrAll.VAllAttributes;
	    foreachinSETClass(allclasses, SClass, AClass) {
		foreachinSEQAttribute(AClass->sem_allattributes, SAtt, AnAtt) {
		    addSETAttribute(allatts, AnAtt);
		    addSETAttribute(alltype->sem_items, 
				    AnAtt->sem_copiedfrom.VAttribute);
		}
		initializeSEQAttribute(AClass->sem_allattributes);
	    }
	}
	else {
	    attrefs = AttOrAll.VAttributeRefs;
	    sizeset = sizeSETClass(allclasses);
	    foreachinSETClass(allclasses, SClass, AClass) {
		if (FindClassAttribute(AClass, attrefs->lex_name, &attfound)
					== FOUND) {
		    found = TRUE;
		    removeSEQAttribute(AClass->sem_allattributes, attfound);
		    addSETAttribute(attrefs->sem_items, 
				    attfound->sem_copiedfrom.VAttribute);
		    addSETAttribute(allatts, attfound);
		}
		else if (sizeset > 1) { /* *=> att */
		    continue;
		}
		else if (PropagatedAttribute(AClass, attrefs->lex_name)) {
		    Warning0(279, attrefs->lex_namepos);
		    propagated = TRUE;
		}
		else { /* not found */
		    Warning0(281, attrefs->lex_namepos);
		}
	    }
	    if (sizeset>1 && !found) {
		Warning0(281, attrefs->lex_namepos);
	    }
	}
	if (!emptySETClass(allclasses) && !propagated && 
				emptySETAttribute(allatts)) {
	    if (typeof(AttOrAll) == KAllAttributes)
		spos = AttOrAll.VAllAttributes->lex_allpos;
	    else spos = AttOrAll.VAttributeRefs->lex_namepos;
	    Warning0(281, spos);
	}
	else if (typeof(AttOrAll) == KAllAttributes)
	    AttOrAll.VAllAttributes->sem_items = allatts;
}



/***********************************************************************
 *
 * Procedure DeleteSubclass
 *
 * Purpose: Delete a subclass(es) from a class
 *
 * Algorithm: Find and delete the subclasses
 *
 * Errors Checked For: Non-existent subclass(es)
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
	
DeleteSubclass(st, allclasses, ClassOrAll)
StructureEntity st;	      /* structure where subclass(es) are deleted */
SETClass allclasses;          /* classes where subclasses should be deleted */
ClassOrAllClasses ClassOrAll; /* subclass(es) to be deleted */
{
	SETClass subclasses;	/* set of subclasses deleted */
	NamedTypeRef nameref;	/* subclass reference */
	Class classfound;	/* subclass found */
	SETClass SClass;	/* set traversals and values */
	Class AClass;
	SETClass SClass2;
	Class AClass2;
        extern int FindNamedType();

	Assume((IsStructureEntity(st)&&IsClassOrAll(ClassOrAll)),
		"DeleteSubclass");

	if (typeof(ClassOrAll) == KAllClasses) {
	    initializeSETClass(subclasses);
	    foreachinSETClass(allclasses, SClass, AClass) {
		foreachinSETClass(AClass->sem_subclasses, SClass2, AClass2) {
		    addSETClass(subclasses, AClass2);
		    addSETClass(ClassOrAll.VAllClasses->sem_items,
				AClass2->sem_copiedfrom.VClass);
		}
		initializeSETClass(AClass->sem_subclasses);
	    }
	}
	else {
	    nameref = ClassOrAll.VNamedTypeRef;
/* ClassToNamedType not needed in argument #3 because a pointer is passed */
	    if (FindNamedType(st, nameref->lex_name, (NamedType *) &classfound)==NOTFOUND){
		SetErrorEntity(nameref);
		if (!emptySETClass(allclasses))
		    Warning0(231, nameref->lex_namepos);
	    }
	    else {
		SetClassEntity(nameref, classfound->sem_copiedfrom.VClass;);
		foreachinSETClass(classfound->sem_ancestors, SClass, AClass)
		    removeSETClass(AClass->sem_subclasses, classfound);
	    }
	}
}


/***********************************************************************
 *
 * Procedure SetWithoutClauseErrors
 *
 * Purpose: Set all references in woclause to error
 *
 * Algorithm: 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
SetWithoutClauseErrors(woclause)
withoutClause woclause;
{
	SEQwithoutItem Switem;
	withoutItem Awitem;
	ClassOrAllClasses lhs;
	ClassOrAllClasses rhs;
	AttributeOrAllAttributes rhs2;

	Assume((IswithoutClause(woclause)), "SetWithoutClauseErrors");

	foreachinSEQwithoutItem(woclause->syn_list, Switem, Awitem) {
	    switch (typeof(Awitem)) {
		case KwithoutAssert:
		    SetErrorEntity(Awitem.VwithoutAssert->syn_item);
		    break;
		case KwithoutDefine:
		    SetErrorEntity(Awitem.VwithoutDefine->syn_item);
		    break;
		case KwithoutType:
		    SetErrorEntity(Awitem.VwithoutType->syn_item.IDLclassCommon);
		    break;
		case KwithoutSubclass:
		    lhs = Awitem.VwithoutSubclass->syn_lefthandside;
		    rhs = Awitem.VwithoutSubclass->syn_righthandside;
		    if (typeof(lhs)==KNamedTypeRef) {
			SetErrorEntity(lhs.VNamedTypeRef);
		    }
		    if (typeof(rhs)==KNamedTypeRef) {
			SetErrorEntity(rhs.VNamedTypeRef);
		    }
		    break;
		case KwithoutAttribute:
		    lhs = Awitem.VwithoutAttribute->syn_lefthandside;
		    if (typeof(lhs)==KNamedTypeRef) {
			SetErrorEntity(lhs.VNamedTypeRef);
		    }
		    rhs2 = Awitem.VwithoutAttribute->syn_righthandside;
		    break;
	    }
	}
}
