/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlc/semantic/sort.c 
*				 					* 
*   Copyright (C) 1991 Karen Shannon
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Function: Routines for sorting the dictionary into processing order *
*									*
*	     SortDictionary  StructureDependency  ProcessDependency	*
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: sort.c,v 1.1 89/07/05 16:32:08 kps Locked $";
#endif

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	sort.c,v $
 * Revision 1.1  89/07/05  16:32:08  kps
 * Initial revision
 * 
 * Revision 4.0  89/05/02  09:07:31  cheung
 * sort.c version 4.0
 * 
 * Revision 4.0  89/04/29  17:17:18  cheung
 * sort.c ver 4.0
 * 
 * Revision 4.0  89/04/12  02:50:12  cheung
 * sort.c  Ver 4.0
 * 
 * Revision 3.9  89/03/30  13:24:28  cheung
 * sort.c  Ver 3.9
 * 
 * Revision 3.9  89/03/26  15:05:07  cheung
 * sort.c  Ver 3.9
 * 
 * Revision 3.2  88/02/19  18:22:39  rajan
 * Version 3.2
 * 
 * Revision 1.1  87/04/18  11:40:51  shannon
 * Initial revision
 * 
 * Revision 1.1  86/06/06  14:47:25  shannon
 * Initial revision
 * 
 * Revision 2.0  86/01/22  07:22:05  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.2  86/01/22  07:10:55  shannon
 * Release 2.0 if IDL System
 * 
 * Revision 1.1  85/06/11  10:18:38  shannon
 * Initial revision
 * 
*									*
*   Edit Log:								*
*     Jan 9 1985 (shannon) Created.					*
*									*
\* ******************************************************************* */
 
#include "Semantic.h"
#include "macros.h"
#include <stdio.h>


/***********************************************************************
 *
 * Procedure SortCompilationUnit
 *
 * Purpose: Sort the compilation unit into processing order
 *
 * Algorithm: Sort the structures and processes. All structures should be
 *	      put in structures seq. Any derived or refined structure 
 *	      should be after its ancestor structure(s). All processes
 *	      should be put in processes seq. Any refined process should be 
 *  	      after its ancestor process(es).
 *	      Duplicate structures and processes are ignored.
 *
 * Errors Checked For: Duplicate structures and processes.
 *
 * Assumptions/Limitations: There are no circular dependency relations
 *
 **********************************************************************/

SortCompilationUnit(compUnit)
compilationUnit compUnit;
{
	SEQStructureEntity newSEQSE, SStr, SStr2;
	StructureEntity AStr, AStr2;
	SEQProcessEntity newSEQPE, SPr, SPr2;
	ProcessEntity APr, APr2;
	SEQprocessStatement Sprstmt;
	processStatement Aprstmt;
	SEQPort SPort;
	Port APort;
	Boolean StructureDependency();
	Boolean ProcessDependency();

	Assume((IscompilationUnit(compUnit)),
		"SortCompilationUnit");

	newSEQSE = copySEQStructureEntity(compUnit->sem_structures);
	foreachinSEQStructureEntity(newSEQSE, SStr, AStr) {
	    GetStAncestors(AStr, compUnit);

	    removefirstSEQStructureEntity(newSEQSE);
	    /* find duplicates */
	    foreachinSEQStructureEntity(newSEQSE,
					SStr2, AStr2) {
		if (AStr->lex_name == AStr2->lex_name){
		    Warning0(109, AStr2->lex_namepos);
		    AStr2->sem_duplicate = TRUE;
		    break;
		}
	    }
	}

	newSEQPE = copySEQProcessEntity(compUnit->sem_processes);
	foreachinSEQProcessEntity(newSEQPE, SPr, APr) {
	    GetPrAncestors(APr, compUnit);

	    /* mark port structures as used */
	    foreachinSEQprocessStatement(APr->syn_body, Sprstmt, Aprstmt) {
		if (typeof(Aprstmt)==KportDefinition) {
		    foreachinSEQPort(Aprstmt.VportDefinition->syn_ports, SPort,
				     APort) {
			if (FindStructure(APort->syn_data->lex_name,
					  compUnit, &AStr)==FOUND) {
			    AStr->inv_used = TRUE;
			}
		    }
		}
	    }

	    removefirstSEQProcessEntity(newSEQPE);
	    /* find duplicates */
	    foreachinSEQProcessEntity(newSEQPE, SPr2, APr2){
		if (APr->lex_name == APr2->lex_name) {
		    Warning0(408, APr2->lex_namepos);
		    APr2->sem_duplicate = TRUE;
		    break;
		}
	    }
	}
	RemoveUnused(compUnit);
	compUnit->sem_processes = sortSEQProcessEntity(compUnit->sem_processes, 
						ProcessDependency);
	compUnit->sem_structures = sortSEQStructureEntity(compUnit->sem_structures,
						StructureDependency);

}
			


/***********************************************************************
 *
 * Function StructureDependency
 *
 * Purpose: check if structure st1 is dependent on structure st2 
 *
 * Return Value: TRUE if st1 is dependent on st2, else FALSE
 *
 * Algorithm: if st2 is contained in the refines or derived list of
 *	      st1 then return TRUE else return FALSE
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
					

Boolean StructureDependency(st1, st2)
StructureEntity st1, st2;
{
	SEQStructureRef SStRef;	/* traversal of structure's 'refines' and
				   'from' seqs */
	StructureRef AStRef;	/* value in seq */

	Assume((IsStructureEntity(st1)&&IsStructureEntity(st2)),
		"StructureDependency");
	if (typeof(st1->syn_refines) == KStructureRef)
	    if (strequal(st1->syn_refines.VStructureRef->lex_name, 
			 st2->lex_name)) {
		return(TRUE);
	    }

	foreachinSEQStructureRef(st1->syn_from, SStRef, AStRef) {
	    if (strequal(AStRef->lex_name, st2->lex_name)) {
		return(TRUE);
	    }
	}

	/* if this point is reached, st1 is not derived or refined from st2 */

	return(FALSE);
}




/***********************************************************************
 *
 * Function ProcessDependency
 *
 * Purpose: check if process pr1 is dependent on process pr2 
 *
 * Return Value: TRUE if pr1 is dependent on pr2, else FALSE
 *
 * Algorithm: if pr2 is contained in the refines list of
 *	      pr1 then return TRUE else return FALSE
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
					
Boolean ProcessDependency(pr1, pr2)
ProcessEntity pr1, pr2;
{
	
	Assume((IsProcessEntity(pr1)&&IsProcessEntity(pr2)),
		"ProcessDependency");

	if (typeof(pr1->syn_refines) == KProcessRef)
	    if (strequal(pr1->syn_refines.VProcessRef->lex_name, pr2->lex_name))
	    {
		    return(TRUE);
	    }

	/* if this point is reached, pr1 is not refined from pr2 */
	return(FALSE);
}


RemoveUnused(compUnit)
compilationUnit compUnit;
{
	SEQStructureEntity SStr;
	StructureEntity AStr;
	SEQProcessEntity SPr;
	ProcessEntity APr;
	StructureRef AStRef;
	SEQStructureEntity  used_structures;
	SEQProcessEntity used_processes;

	initializeSEQStructureEntity(used_structures);
	initializeSEQProcessEntity(used_processes);

	foreachinSEQStructureEntity(compUnit->sem_structures, SStr, AStr) {
	    if (AStr->inv_used) {
		GetUsedStructures(AStr, &used_structures);
	    }
	}
	foreachinSEQProcessEntity(compUnit->sem_processes, SPr, APr) {
	    if (APr->inv_used) {
		GetUsedProcesses(APr, &used_processes);
		if (typeof(APr->syn_invariant)==KStructureRef) {
		    AStRef = APr->syn_invariant.VStructureRef;
		    if (TypeOfEntity(AStRef)== KStructureEntity) {
			AStr = GetStructureEntity(AStRef);
			AStr->inv_used = TRUE;
			GetUsedStructures(AStr, &used_structures);
		    }
		}
	    }
	}
	compUnit->sem_processes = used_processes;
	compUnit->sem_structures = used_structures;
}

GetUsedStructures(AStr, used_structures)
StructureEntity AStr;
SEQStructureEntity *used_structures;
{
	SEQStructureRef SStRef;
	StructureRef AStRef;
	StructureEntity st;

	if (inSEQStructureEntity((*used_structures), AStr))
	    return; /* already done */
	appendrearSEQStructureEntity((*used_structures), AStr);
	foreachinSEQStructureRef(AStr->syn_from, SStRef, AStRef) {
	    if (TypeOfEntity(AStRef) == KStructureEntity) {
		st = GetStructureEntity(AStRef);
		st->inv_used = TRUE;
		GetUsedStructures(st, used_structures);
	    }
	}
	if (typeof(AStr->syn_refines)==KStructureRef) {
	    AStRef = AStr->syn_refines.VStructureRef;
	    if (TypeOfEntity(AStRef) == KStructureEntity) {
		st = GetStructureEntity(AStRef);
		st->inv_used = TRUE;
		GetUsedStructures(st, used_structures);
	    }
	}
}

GetUsedProcesses(APr, used_processes)
ProcessEntity APr;
SEQProcessEntity *used_processes;
{
	ProcessRef APrRef;
	ProcessEntity pr;

	if (inSEQProcessEntity((*used_processes), APr))
	    return; /* already done */
	appendrearSEQProcessEntity((*used_processes), APr);
	if (typeof(APr->syn_refines)==KProcessRef) {
	    APrRef = APr->syn_refines.VProcessRef;
	    if (TypeOfEntity(APrRef) == KProcessEntity) {
		pr = GetProcessEntity(APrRef);
		pr->inv_used = TRUE;
		GetUsedProcesses(pr, used_processes);
	    }
	}
}

/* get the ancestors for the structure */
GetStAncestors(st, compUnit)
StructureEntity st;
compilationUnit compUnit;
{
	SEQStructureRef allancestors;
	SEQStructureRef SStRef;
	StructureRef    AStRef;
	StructureEntity ancst;

	Assume((IsStructureEntity(st)), "GetAncestors");
	
	/* add all structures from 'refines' and 'from' to 'allancestors' */
	initializeSEQStructureRef(allancestors);
	foreachinSEQStructureRef(st->syn_from, SStRef, AStRef)
	    appendrearSEQStructureRef(allancestors, AStRef);
	if (typeof(st->syn_refines) == KStructureRef)
	    appendrearSEQStructureRef(allancestors, 
					st->syn_refines.VStructureRef);

	/* for each ancestor structure, copy information into new structure,
	   resolve duplicates						     */

	foreachinSEQStructureRef(allancestors, SStRef, AStRef) {
	
	    if (FindStructure(AStRef->lex_name,compUnit,&ancst)==NOTFOUND) {
		Warning0(105, AStRef->lex_namepos);
		SetErrorEntity(AStRef);
	    }
	    
	    else {

		/* set the entity field */
		SetStructureEntity(AStRef, ancst);
	    }
	}
}

/* get the ancestors for the process */
GetPrAncestors(pr, compUnit)
ProcessEntity pr;
compilationUnit compUnit;
{
	ProcessEntity pfound;
	ProcessRef    pref;

	/* if process is refined, find parent process in compUnit */
	if (typeof(pr->syn_refines)==KProcessRef) {
	    pref = pr->syn_refines.VProcessRef;
	    if (FindProcess(compUnit, pref->lex_name, &pfound)== FOUND){
		SetProcessEntity(pref, pfound);
	    }
	    else {
		SetErrorEntity(pref);
	    }
	}
}
