/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlc/semantic/copy.c 
*				 					* 
*   Copyright (C) 1991 Karen Shannon
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Function: the high level routines involved in copying from ancestor	*
*	     structures or nonterminals					*
*									*
*	     CopyFromAncestors  CopyRoot  CopyNamedType  		*
*	     CopyClass  CopyPrivateNT  CopyAttribute			*
*	     Resolvents 						*
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: copy.c,v 1.1 89/07/05 16:31:40 kps Locked $";
#endif

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	copy.c,v $
 * Revision 1.1  89/07/05  16:31:40  kps
 * Initial revision
 * 
 * Revision 4.0  89/04/29  17:17:36  cheung
 * copy.c version 4.0
 * 
 * Revision 4.0  89/04/12  02:47:33  cheung
 * copy.c  Ver 4.0
 * 
 * Revision 3.9  89/03/30  13:22:51  cheung
 * copy.c  Ver 3.9
 * 
 * Revision 3.9  89/03/26  15:03:32  cheung
 * copy.c  Ver 3.9
 * 
 * Revision 3.2  88/02/19  18:21:48  rajan
 * Version 3.2
 * 
 * Revision 1.1  87/04/18  11:40:33  shannon
 * Initial revision
 * 
 * Revision 1.1  86/06/06  14:55:55  shannon
 * Initial revision
 * 
 * Revision 2.0  86/01/22  07:21:01  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.9  86/01/22  07:08:31  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.8  85/07/26  10:03:08  shannon
 * added copying of assertions
 * 
 * Revision 1.7  85/07/25  10:34:46  shannon
 * changed code for new null and empty class rep
 * 
 * Revision 1.6  85/07/18  19:43:27  shannon
 * fixed bug in resolvents --don't add propagated attributes
 * 
 * Revision 1.5  85/07/05  20:38:45  shannon
 * added ancestors attribute for each dependent structure
 * 
 * Revision 1.4  85/07/04  14:54:23  shannon
 * changed set of symbols to sequence
 * 
 * Revision 1.3  85/07/04  10:09:08  shannon
 * fixed bug that copied propogated attributes of classes.
 * 
 * Revision 1.2  85/06/27  19:21:35  shannon
 * deleted node representation, just using sequence of targetrep now.
 * 
 * Revision 1.1  85/06/11  10:15:45  shannon
 * Initial revision
 * 
*									*
*   Edit Log:								*
*     Jan 9 1985 (shannon) Created.					*
*									*
\* ******************************************************************* */

#include "Semantic.h"
#include "macros.h"
#include <stdio.h>

/***********************************************************************
 *
 * Procedure CopyFromAncestors
 *
 * Purpose: Copy information from ancestor structures into 'st'
 *
 * Algorithm: For each ancestor structure given in 'from' or 'refines' , 
 *	      copy information into new structure 'st'. resolve duplicates.
 *	      actually just copied references
 *
 * Errors Checked For: Non-existent ancestor structure.
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

CopyFromAncestors(st,refined)
StructureEntity st;		/* the dependent structure */
Boolean refined;
{

	SEQStructureRef SStRef;	/* set/seq traversals and values */
	StructureRef AStRef;	
	SEQStructureRef allancestors; /* all ancestor structures */
	StructureEntity ancst;	/* points to ancestor structure found */

	Assume((IsStructureEntity(st)), "CopyFromAncestors");
	
	/* add all structures from 'refines' and 'from' to 'allancestors' */
	initializeSEQStructureRef(allancestors);
	foreachinSEQStructureRef(st->syn_from, SStRef, AStRef)
	    appendrearSEQStructureRef(allancestors, AStRef);
	if (typeof(st->syn_refines) == KStructureRef)
	    appendrearSEQStructureRef(allancestors, 
					st->syn_refines.VStructureRef);

	/* for each ancestor structure, copy information into new structure,
	   resolve duplicates						     */

	foreachinSEQStructureRef(allancestors, SStRef, AStRef) {
	
	    if (EntityIsError(AStRef)) {
		Warning0(105, AStRef->lex_namepos);
	    }
	    
	    else {

		/* get the entity field */
		ancst = GetStructureEntity(AStRef);

		/* copy assertion language information of
		   ancestor structure to new structure	   */
		
		CopyAssertInfo(st, ancst, refined);
				

		/* copy types of ancestor structure, 
		   resolve duplicates */
		   
		CopyAllTypes(st, ancst->sem_types);

		/* if dependent structure does not have a root, assign 
		   it to the root of the first ancestor structure */

		if (typeof(st->sem_root) == KVoid) {
		    if (typeof(st->syn_root) == KVoid) 
			    CopyRoot(st, ancst->sem_root);
		}
	    }
	}
}


/***********************************************************************
 *
 * Procedure CopyRoot
 *
 * Purpose: Copy the root to the structure 'st'
 *
 * Algorithm: Find the root in the structure types. Attach this
 *	      to the structure root.
 *
 * Errors Checked For: Non-existent root.
 *
 * Assumptions/Limitations: structure has a root type of void
 *
 **********************************************************************/
	
CopyRoot(st, ancestor_root)
StructureEntity st;		/* the dependent structure */
TypeEntityOrVoid ancestor_root;	/* the root of the ancestor structure */
{

	NamedType newroot; 
	String rootname;

	Assume((IsStructureEntity(st)&&IsTypeEntityOrVoid(ancestor_root)),
		"CopyRoot");

	if ((typeof(ancestor_root) != KClass) &&
	    (typeof(ancestor_root) != KAtomic)){
	    return;
	}
	else rootname = ancestor_root.VNamedType.IDLclassCommon->sem_name;

	if (FindNamedType(st, rootname, &newroot) == NOTFOUND) {
	    Warning0(107, st->lex_endpos);
	}
	else st->sem_root.VNamedType = newroot;
}

/***********************************************************************
 *
 * Procedure CopyAllTypes
 *
 * Purpose: copy all types into structure
 *
 * Algorithm: 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
CopyAllTypes(st, alltypes)
StructureEntity st;
SETTypeEntity alltypes;
{
    SETTypeEntity SType;
    TypeEntity AType;
    static SetRef asetref;
    static SeqRef aseqref;
    static Boolean init = TRUE;

    Assume((IsStructureEntity(st)), "CopyAllTypes");

    if (init) {
	init = FALSE;
	asetref = NSetRef;
	asetref->syn_component = NNamedTypeRef;
	aseqref = NSeqRef;
	aseqref->syn_component = asetref->syn_component;
    }

    /* add the prelude types without making copies */
    /* copy the named types */
    foreachinSETTypeEntity(alltypes, SType, AType) {
	if (AType.IDLclassCommon->sem_isPreludeType) {
	    addSETTypeEntity(st->sem_types, AType);
	}
	else if ((typeof(AType) == KClass)  || (typeof(AType) == KAtomic))
	    CopyNamedType(st, AType.VNamedType);
    }

    /* attach the attribute types */
    /* add the structured types at the same time */
    /* attach the subclasses, and ancestors */
    AssignAttributeTypes(st);
    AssignClassRelatives(st);

    /* copy any sets/sequences not already added from attributes */
    foreachinSETTypeEntity(alltypes, SType, AType) {
	if (typeof(AType) == KSetOf) {
	    asetref->syn_component->lex_name = 
		AType.VSetOf->sem_component.IDLclassCommon->sem_name;
	    if (FindType(st, SetRefToTypeRef(asetref)) == NOTFOUND)
		AddNewType(st, SetRefToTypeRef(asetref));
	}
	else if (typeof(AType) == KSeqOf) {
	    aseqref->syn_component->lex_name = 
		AType.VSetOf->sem_component.IDLclassCommon->sem_name;
	    if (FindType(st, SeqRefToTypeRef(aseqref)) == NOTFOUND)
		AddNewType(st, SeqRefToTypeRef(aseqref));
	}
    }
    /* set the copied_from fields of the sets and sequences */
    foreachinSETTypeEntity(alltypes, SType, AType) {
	if (AType.IDLclassCommon->sem_isPreludeType) {
		continue;
	}
	if (typeof(AType)==KSetOf) {
	    asetref->syn_component->lex_name = 
		AType.VSetOf->sem_component.IDLclassCommon->sem_name;
	    if (FindType(st, SetRefToTypeRef(asetref)) == FOUND)
		asetref->sem_entity.VSetOf->sem_copiedfrom.VSetOf =
			AType.VSetOf;
	}
	else if (typeof(AType)==KSeqOf) {
	    aseqref->syn_component->lex_name = 
		AType.VSetOf->sem_component.IDLclassCommon->sem_name;
	    if (FindType(st, SeqRefToTypeRef(aseqref)) == FOUND)
		aseqref->sem_entity.VSeqOf->sem_copiedfrom.VSeqOf =
			AType.VSeqOf;
	}
    }
}

/***********************************************************************
 *
 * Procedure CopyNamedType
 *
 * Purpose: Copy type 'AType' into structure 'st'
 *
 * Algorithm: Check for type already existing in 'st'. If so, 
 *	      resolve any differences. 
 *	      If not, make a new copy and add to 'st' types
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

CopyNamedType(st, AType)
StructureEntity st;	/* structure where type is to be added */
NamedType AType;	/* type to be added */
{

	String tname;	        /* name of type */
	Class newclass;         /* new class added */
	Atomic newatomic;       /* new atomic added */
	NamedType dup_type;	/* pointer to duplicate type in structure */
	TypeEntity AType1;

	Assume((IsStructureEntity(st)&&IsNamedType(AType)),
		"CopyNamedType");

	/* de-reference name */

	tname = GetTypeName(AType);

	/* check if 'AType' is already contained in the 'types' list
	 * if so, return. If not, check to see if a type with the
	 * same name exists. If it does, resolve any differences.
	 */

	if (FindNamedType(st, tname, &dup_type)==FOUND){
	    ResolveTypes(dup_type, AType, st);  /* dup_type should be a 
						 superset of AType      */
	}
		
	/* if 'AType' is not already contained in the 'types' list add it */
	else switch (typeof(AType)) {
	    case KClass:
		newclass = NClass;
		CopyClass(st, newclass, AType.VClass);
		AType1.VClass = newclass;
		addSETTypeEntity(st->sem_types, AType1);
		break;

	    case KAtomic:
		newatomic = NAtomic;
		CopyAtomic(st, newatomic, AType.VAtomic);
	       	AType1.VAtomic = newatomic;
		addSETTypeEntity(st->sem_types, AType1);
		break;
	}
}

	
/***********************************************************************
 *
 * Procedure CopyClass
 *
 * Purpose: Copy Class 'old_class' to Class 'new_class'
 *
 * Algorithm: Copy references to all attributes. Set copiedfrom field
 *	      to old_class
 *	      Set parent of new class to structure.
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: storage is allocated for new class
 *
 **********************************************************************/
	
CopyClass(st, new_class, old_class)
StructureEntity st;	/* structure */
Class new_class,	/* Class that is copied to */
      old_class;	/* Class that is copied from */
{

	SEQAttribute SAtt;
	Attribute AnAtt;
	NamedTypeRef newnameref;
	Attribute newAtt;
	SETClass SClass;
	Class AClass;

	Assume((IsStructureEntity(st)&&IsClass(new_class)&&IsClass(old_class)),
		"CopyClass");

	/* copy information from old class to new class */
	new_class->sem_name = old_class->sem_name;
	new_class->inv_definitionPoint = old_class->inv_definitionPoint;
	new_class->sem_copiedfrom.VClass = old_class;

	/* don't copy propagated attributes */
	foreachinSEQAttribute(old_class->sem_allattributes, SAtt, AnAtt) {
	    if ((typeof(AnAtt->inv_parent)==KClass)&&
		(AnAtt->inv_parent.VClass == old_class)) {
		newAtt = NAttribute;
		CopyAttribute(new_class, newAtt, AnAtt);
		appendfrontSEQAttribute(new_class->sem_allattributes, newAtt);
	    }
	}
	foreachinSETClass(old_class->sem_subclasses, SClass, AClass) {
	    newnameref = NNamedTypeRef;
	    newnameref->lex_name = AClass->sem_name;
	    addSETNamedTypeRef(new_class->inv_subclasses, newnameref);
	}

	/* set parent of new class to structure */
	new_class->inv_parent.VStructureEntity = st;
}



/***********************************************************************
 *
 * Procedure CopyAtomic
 *
 * Purpose: Copy Atomic 'old_atom' to Atomic 'new_atom'
 *
 * Algorithm: Copy references to all attributes. Set copiedfrom field to
 *		old_atom
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: 'new_atom' has storage allocated.
 *
 **********************************************************************/

CopyAtomic(st, new_atom, old_atom)
StructureEntity st;
Atomic new_atom,	/* Atomic that is copied to */
	  old_atom;	/* Atomic that is copied from */
{

	Assume((IsStructureEntity(st)&&IsAtomic(new_atom)&&IsAtomic(old_atom)),
		"CopyAtomic");

	new_atom->sem_name = old_atom->sem_name;
	new_atom->inv_definitionPoint = old_atom->inv_definitionPoint;
	new_atom->sem_copiedfrom.VAtomic = old_atom;

	new_atom->inv_parent.VStructureEntity = st;
}


/***********************************************************************
 *
 * Procedure CopyAttribute
 *
 * Purpose: Copy Attribute 'old_att' to 'new_att'
 *
 * Algorithm: Copy references to all attributes. Set copiedfrom field
 *	      to old_att
 *	      Set parent field of new_att to class. 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: 'new_att' already has storage allocated
 *
 **********************************************************************/

CopyAttribute(AClass, new_att, old_att)
Class AClass;
Attribute new_att,	/* attribute that is copied to */
	  old_att;	/* attribute that is copied from */
{
	Assume((IsClass(AClass)&&IsAttribute(new_att)&&IsAttribute(old_att)),
		"CopyAttribute");

	new_att->lex_name = old_att->lex_name;
	new_att->lex_namepos = 0;
	new_att->syn_type = old_att->syn_type;
	/**
	CopyTypeRef(&(new_att->syn_type), old_att->syn_type);
	**/

	new_att->sem_copiedfrom.VAttribute = old_att;
	new_att->inv_parent.VClass = AClass;
}


				

/***********************************************************************
 *
 * Procedure ResolveTypes
 *
 * Purpose: Resolve the differences between types t1 and t2.
 *	    t1 should be a superset of t2.
 *
 * Algorithm: if t1 and t2 are not the same types, report error.
 *	      If they are bot atomics, check that the reps are the same.
 *	      If they are both classes,
 *	      add all attributes, subclasses, ancestors, & restrictions
 *	      of t2 not contained in t1 to t1.
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: not fully implemented yet.
 *
 **********************************************************************/

ResolveTypes(t1, t2, st)
NamedType t1, t2;
StructureEntity st;
{
	String tname;			/* name of type */
	SEQAttribute SAttribute;	/* set/seq traverals and values */
	Attribute AnAttribute;
	Attribute newAtt;
	Attribute attfound;
	NamedTypeRef ANamedTypeRef;
	SETClass SClass;
	Class AClass;
	Boolean SameAttributeAncestor();

	Assume((IsNamedType(t1)&&IsNamedType(t2)&&IsStructureEntity(st)),
		"ResolveTypes");
	tname = GetTypeName(t1);


	if (typeof(t1) != typeof(t2)){
	    Serious1(262, st->lex_endpos, tname);
	    return;
	}
	if (typeof(t1) == KClass) {
	    if (emptySETClass(t1.VClass->sem_subclasses) &&
		!emptySETClass(t2.VClass->sem_subclasses)) {
		Serious1(262, st->lex_endpos, tname);
		return;
	    }
	}
	if (!SameType(t1.IDLclassCommon->inv_definitionPoint,
		     t2.IDLclassCommon->inv_definitionPoint)) {
	    Serious1(263, st->lex_endpos, tname);
	    return;    
	}

	if (typeof(t1) == KAtomic) {
	    /* not fully implemented yet */
	    if (t1.VAtomic->sem_name != t2.VAtomic->sem_name)
		Serious1(262, st->lex_endpos, tname);
	    return;
	}

	
	/* else both are classes */
	if (typeof(t1) == KClass) {
	    /* make sure all attributes are here */
	    foreachinSEQAttribute(t2.VClass->sem_allattributes, SAttribute,
				AnAttribute) {
		/* don't copy propagated attributes */
		if (AnAttribute->inv_parent.VClass != t2.VClass)
		    continue;
		if (FindClassAttribute(t1.VClass, AnAttribute->lex_name,
			&attfound) != FOUND) {
		    newAtt = NAttribute;
		    CopyAttribute(t1.VClass, newAtt, AnAttribute);
		    appendfrontSEQAttribute(t1.VClass->sem_allattributes,
					newAtt);
		}
		else if (!SameAttributeAncestor(AnAttribute, attfound)) {
		    /* Print warning */
		    Warning2(272, st->lex_endpos, t1.VClass->sem_name,
				AnAttribute->lex_name);
		}
	    }
	    /* make sure all descendants are here */
	    foreachinSETClass(t2.VClass->sem_subclasses, SClass, AClass) {
		ANamedTypeRef = NNamedTypeRef;
		ANamedTypeRef->lex_name = AClass->sem_name;
		addSETNamedTypeRef(t1.VClass->inv_subclasses, ANamedTypeRef);
	    }
	}
}

CopyTypeRef(new_tref, old_tref)
TypeRef *new_tref, 
	old_tref;
{
	Assume((IsTypeRef(old_tref)),
		"CopyTypeRef");

	switch (typeof(old_tref)) {
	    case KSetRef:
		new_tref->VSetRef = NSetRef;
		CopyTypeRef((TypeRef *) &(new_tref->VSetRef->syn_component),
			    NamedTypeRefToTypeRef(old_tref.VSetRef->syn_component));
		break;
	    case KSeqRef:
		new_tref->VSeqRef = NSeqRef;
		CopyTypeRef((TypeRef *) &(new_tref->VSeqRef->syn_component), 
			    NamedTypeRefToTypeRef(old_tref.VSeqRef->syn_component));
		break;
	    case KNamedTypeRef:
		new_tref->VNamedTypeRef = NNamedTypeRef;
		new_tref->VNamedTypeRef->lex_name = 
				old_tref.VNamedTypeRef->lex_name;
		new_tref->VNamedTypeRef->lex_namepos = 0;
		break;
	}

}



/***********************************************************************
 *
 * Procedure CopyAssertInfo
 *
 * Purpose: copy the assertion information from the ancestor structure
 *	    to the structure
 *
 * Algorithm: 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
CopyAssertInfo(st, ancst, refined)
StructureEntity st;		/* structure */
StructureEntity ancst;		/* ancestor structure */
Boolean refined;
{
	SETAssertion SAs;	/* set traversals and values */
	Assertion AnAs;
	SETDefinition SDef;
	Definition ADef;
	Assertion newassert;
	Definition newdef;

	foreachinSETAssertion(ancst->sem_assertions, SAs, AnAs) {
	    /* only copy named assertions for derived structures */
	    if (!refined && strlen(AnAs->lex_name)==0)
		continue;
	    newassert = NAssertion;
	    CopyAssertion(newassert, AnAs); 
	    addSETAssertion(st->sem_assertions, newassert);
	}
	foreachinSETDefinition(ancst->sem_definitions, SDef, ADef) {
	    newdef = NDefinition;
	    CopyDefinition(newdef, ADef);
	    addSETDefinition(st->sem_definitions, newdef);
	}
}

CopyAllPorts(newpr, oldpr)
ProcessEntity newpr;
ProcessEntity oldpr;
{
	SETPort SPort;
	Port APort;
	Port newport;

	foreachinSETPort(oldpr->sem_ports, SPort, APort){
	    newport = NPort;
	    newport->lex_name = APort->lex_name;
	    newport->syn_data = APort->syn_data;
	    newport->sem_portType = APort->sem_portType;
	    addSETPort(newpr->sem_ports, newport);
	}
}

Boolean SameAttributeAncestor(att1, att2)
Attribute att1, att2;
{
	int code = 0;

	if (att1 == att2)
	    return(TRUE);

	if (typeof(att1->sem_copiedfrom)==KAttribute) 
	    code += 1;
	if (typeof(att2->sem_copiedfrom)==KAttribute)
	    code += 2;
	switch (code) {
	    case 0:	/* neither copied */
		return(FALSE);
	    case 1:	/* att1 copied */
		return(SameAttributeAncestor(att1->sem_copiedfrom.VAttribute, att2));
	    case 2:	/* att2 copied */
		return(SameAttributeAncestor(att1, att2->sem_copiedfrom.VAttribute));
	    case 3:  	/* both copied */
		if (SameAttributeAncestor(att1->sem_copiedfrom.VAttribute, att2))
			return(TRUE);
		else return(SameAttributeAncestor(att1, att2->sem_copiedfrom.VAttribute));
            }
/*NOTREACHED*/
}

