/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlc/semantic/class.c 
*				 					* 
*   Copyright (C) 1991 Karen Shannon
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Function: Routines concerned with classes of the structure		*
*									*
*	     IsAncestorClass  PropogateClassAttributes  		*
*	     PropogateAttributes  FindTheAttribute			*
*	     SetEmptyClasses CheckForCycles                           	*
*	     GetAllClasses						*
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: class.c,v 1.1 89/07/05 16:31:37 kps Locked $";
#endif

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	class.c,v $
 * Revision 1.1  89/07/05  16:31:37  kps
 * Initial revision
 * 
 * Revision 4.0  89/04/12  02:47:21  cheung
 * class.c  Ver 4.0
 * 
 * Revision 3.9  89/03/30  13:22:44  cheung
 * class.c  Ver 3.9
 * 
 * Revision 3.9  89/03/26  15:03:22  cheung
 * class.c  Ver 3.9
 * 
 * Revision 3.2  88/02/19  18:21:44  rajan
 * Version 3.2
 * 
 * Revision 1.1  87/04/18  11:40:30  shannon
 * Initial revision
 * 
 * Revision 1.1  86/06/06  14:55:37  shannon
 * Initial revision
 * 
 * Revision 2.0  86/01/22  07:20:51  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.6  86/01/22  07:08:15  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.5  85/07/25  12:37:05  shannon
 * changed message for invalid class hierarchy
 * 
 * Revision 1.4  85/07/25  10:32:24  shannon
 * checks for cycles and sharing in class hierarchy
 * changed code for new null and empty class rep
 * 
 * Revision 1.3  85/07/18  19:43:02  shannon
 * deleted unused variables
 * 
 * Revision 1.2  85/07/04  14:54:52  shannon
  changed set of symbols to sequence
 * 
 * Revision 1.1  85/06/11  10:15:22  shannon
 * Initial revision
 * 
*									*
*   Edit Log:								*
*     Jan 9 1985 (shannon) Created.					*
*									*
\* ******************************************************************* */
 
#include "Semantic.h"
#include "macros.h"
#include <stdio.h>

/***********************************************************************
 *
 * Function IsAncestorClass
 *
 * Purpose: check if c2 is an ancestor of cl
 *
 * Return Value: TRUE if c2 is an ancestor of c1 else FALSE
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
Boolean IsAncestorClass(c1, c2)
Class c1, c2;
{
	Assume((IsClass(c1)&&IsClass(c2)), "IsAncestorClass");

	return(inSETClass(c1->sem_ancestors, c2));
}

/***********************************************************************
 *
 * Procedure PropogateClassAttributes
 *
 * Purpose: For each class in 'st', propogate its attributes to all of
 *	    its descendants.
 *
 * Algorithm: For each class without a cycle error, call PropogateAttributes
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

PropogateClassAttributes(st)
StructureEntity st;
{
	SEQClass SClass;
	Class AClass;
	SEQClass Ssubclass;
	Class Asubclass;
	SETTypeEntity SType;
	TypeEntity AType;
	SEQClass allclasses;
	Boolean IsAncestorClass();

	Assume((IsStructureEntity(st)), "PropogateClassAttributes");

	/* extract out all classes with descendants from types */
	initializeSEQClass(allclasses);
    	foreachinSETTypeEntity(st->sem_types, SType, AType)
	    if ((typeof(AType) == KClass) &&
		(!emptySETClass(AType.VClass->sem_subclasses)))
		    appendfrontSEQClass(allclasses, AType.VClass);
	
	/* sort the classes from most-inclusive to least-inclusive */
	allclasses = (SEQClass)IDLListSort((pGenList) allclasses, IsAncestorClass);

	foreachinSEQClass(allclasses, SClass, AClass)
	    if (!emptySEQAttribute(AClass->sem_allattributes)) {
		foreachinSEQClass(AClass->sem_subclasses, Ssubclass, Asubclass){
		    PropogateAttributes(Asubclass, AClass->sem_allattributes);
		}
	    }
}



/***********************************************************************
 *
 * Procedure PropogateAttributes
 *
 * Purpose: Propogate the attributes of class_fields to descendant 'subclass'
 *
 * Algorithm: Search for each attribute in 'class_fields' in the attributes of
 *	      'subclass'. If not found, add attribute to back of 
 *            attseq. If found, check for type conflict,
 *	      delete the attribute existing in the descendant and add the
 *	      new attribute to the back of attseq to preserve order.
 *	      If the descendant is a class propogate the attributes to
 *	      the descendant's descendants also.
 *	      Add the new attseq to the front of the descendant attributes
 *
 * Errors Checked For: duplicate attributes with conflicting types.
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

PropogateAttributes(subclass, class_atts)
Class subclass;
SEQAttribute class_atts;
{

	SEQAttribute SAtt;
	Attribute AnAtt;
	SEQAttribute SAtt2;
	Attribute AnAtt2;
	Boolean found;		/* indicator if attribute is already found */

	Assume((IsClass(subclass)), "PropogateAttributes");

	foreachinSEQAttribute(class_atts, SAtt, AnAtt) {
	    found = FALSE;
	    foreachinSEQAttribute(subclass->sem_allattributes, SAtt2, AnAtt2)
		if (strequal(AnAtt->lex_name, AnAtt2->lex_name)){
			found = TRUE;
			break;
		}
		/* add the attribute whether it is found or not */
		/* unless the exact (same ptr) attribute already exists */
		if (!inSEQAttribute(subclass->sem_allattributes, AnAtt))
		    appendfrontSEQAttribute(subclass->sem_allattributes, AnAtt);
			
		/* if found, then check for conflict within the types of 
	           the attribute and delete the existing attribute. 
		   report duplicate. If attribute is the same (propagated
		   from some common ancestor) there is no error */

		if (found){
		   if (AnAtt != AnAtt2) {
		       if (CheckTypes(AnAtt->syn_type,AnAtt2->syn_type) 
					== NOTEQUAL)
			    Warning0(276, AnAtt2->lex_namepos);
		       else Warning0(273, AnAtt2->lex_namepos ); /* duplicate */
		       removeSEQAttribute(subclass->sem_allattributes, AnAtt2);
		   }
		}
	}
}
	


/***********************************************************************
 *
 * Procedure CheckForCycles
 *
 * Purpose: check for cycles in the class hierarchy relation
 *
 * Algorithm: use marking routines and check for sharing
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

CheckForCycles(st)
StructureEntity st;
{
	SETTypeEntity SType;
	TypeEntity AType;
	Boolean IsSubclass();

	Assume((IsStructureEntity(st)), "CheckForCycles");

	foreachinSETTypeEntity(st->sem_types, SType, AType) {
	    if (typeof(AType) == KClass) {
		if (IsSubclass(AType.VClass, AType.VClass)) {
		    Serious1(226, st->lex_endpos, AType.VClass->sem_name);
		}
	    }
	}
}

/***********************************************************************
 *
 * Function GetAllClasses
 *
 * Purpose: Get the set of classes in a structure
 *
 * Return Value: the set of classes
 *
 * Algorithm: Search for the classes in the types set 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
 SETClass GetAllClasses(st)
 StructureEntity st;
 {
     SETClass allclasses;	/* the classes */
     SETTypeEntity SType;	/* set traversal and value */
     TypeEntity AType;

     initializeSETClass(allclasses);
     foreachinSETTypeEntity(st->sem_types, SType, AType) {
	 if (typeof(AType) == KClass)
	     addSETClass(allclasses, AType.VClass);
     }
     return(allclasses);
}


/***********************************************************************
 *
 * Function PropagatedAttribute
 *
 * Purpose: See if an attribute is propagated
 *
 * Return Value: TRUE if attribute is propagated else FALSE
 *
 * Algorithm: Check if the attribute exists in the ancestors of the class
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: Attributes have not been propagated yet
 *
 **********************************************************************/
 Boolean PropagatedAttribute(theclass, attname)
 Class theclass;
 String attname;
 {
	SETClass SClass; 	/* set/seq traversals and values */
	Class AClass;
	SEQAttribute SAtt;
	Attribute AnAtt;
	Boolean found=FALSE;	/* indicator if attribute is found */

	foreachinSETClass(theclass->sem_ancestors, SClass, AClass) {
	    foreachinSEQAttribute(AClass->sem_allattributes, SAtt, AnAtt) {
		if (strequal(AnAtt->lex_name, attname)) {
		    found = TRUE;
		    break;
		}
	    }
	    if (!found) 
		found = PropagatedAttribute(AClass, attname);
	    if (found)
		break;
	}
	return(found);
}

/***********************************************************************
 *
 * Function IsSubclass
 *
 * Purpose: Check if class2 is a direct or indirect subclass of class1
 *
 * Return Value: TRUE if class2 is a subclass else FALSE
 *
 * Algorithm: See if class2 can be reached through class1 subclasses
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
Boolean IsSubclass(class1, class2)
Class class1, class2;
{
	SETClass SClass;	/* set traversal and values */
	Class AClass;
	Boolean found = FALSE;	/* indicator if class2 is found as a subclass */

	Assume((IsClass(class1)&&IsClass(class2)), "IsSubclass");

	if (inSETClass(class1->sem_subclasses, class2))
	    found = TRUE;
	else foreachinSETClass(class1->sem_subclasses, SClass, AClass) {
	    found = IsSubclass(AClass, class2);
	    if (found) 
		break;
	}
	return(found);
}

/***********************************************************************
 *
 * Function IsClassType
 *
 * Purpose:  Determine if a Class is an IDL class or node
 *
 * Return Value: TRUE of a class is an IDL class else FALSE
 *
 * Algorithm: 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
Boolean IsClassType(aclass)
Class aclass;
{
	Boolean isclass = FALSE;

	Assume((IsClass(aclass)), "IsClassType");

	if (!emptySETClass(aclass->sem_subclasses)) {
	    isclass = TRUE;
	}
	if (!isclass) {
	    if (typeof(aclass->sem_copiedfrom)==KClass)
		isclass = IsClassType(aclass->sem_copiedfrom.VClass);
	}
	return(isclass);
}
