/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlc/semantic/analyze3.c 
*				 					* 
*   Copyright (C) 1991 Karen Shannon
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Function: This file contains the lower level routines for analyzing *
*	      processes.						*
*									*
*  	     AddPorts CheckInvariant  CheckTypes  			*
*	     AddRestrictions AddClassRestrictions 			*
*	     AddAttributeRestrictions					*
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: analyze3.c,v 1.1 89/07/05 16:31:31 kps Locked $";
#endif

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	analyze3.c,v $
 * Revision 1.1  89/07/05  16:31:31  kps
 * Initial revision
 * 
 * Revision 4.0  89/04/12  02:46:50  cheung
 * analyze3.c  Ver 4.0
 * 
 * Revision 3.9  89/03/30  13:22:21  cheung
 * analyze3.c  Ver 3.9
 * 
 * Revision 3.9  89/03/26  15:02:55  cheung
 * analyze3.c  Ver 3.9
 * 
 * Revision 3.2  88/02/19  18:21:30  rajan
 * Version 3.2
 * 
 * Revision 1.1  87/04/18  11:40:24  shannon
 * Initial revision
 * 
 * Revision 1.1  86/06/06  14:55:30  shannon
 * Initial revision
 * 
 * Revision 2.0  86/01/22  07:20:31  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.7  86/01/22  07:07:47  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.6  85/07/25  10:29:44  shannon
 * added new restriction processing
 * changed code for new null and empty class rep
 * 
 * Revision 1.5  85/07/18  19:41:38  shannon
 * added invariant structure to symbol table scope for
 * copied structures. 
 * 
 * Revision 1.4  85/07/05  20:40:42  shannon
 * added processing of unmarker port
 * added check for duplicate port names
 * 
 * Revision 1.3  85/07/04  10:12:56  shannon
 * added Restrict To changes (ClassNT's now also have
 * restrictions and there is a Restrict * To)
 * 
 * Revision 1.2  85/06/27  19:19:37  shannon
 * fixed AddRestrictions procedure
 * 
 * Revision 1.1  85/06/11  10:14:52  shannon
 * Initial revision
 * 
*									*
*   Edit Log:								*
*     Jan 9 1985 (shannon) Created.					*
*									*
\* ******************************************************************* */

#include "Semantic.h"
#include <stdio.h>
#include "macros.h"

extern Boolean SeriousError;


/***********************************************************************
 *
 * Procedure AddPorts
 *
 * Purpose: Add the ports to process.
 *
 * Algorithm: Find the structure associated with each port and add
 *	      port to process.
 *
 * Errors Checked For: non-existent port structure
 *		       port structure without a root.
 *		       duplicate port structure.
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

AddPorts(portDef, Aprocess)
portDefinition portDef;		/* port definition */
ProcessEntity Aprocess;		/* process where ports are to be added */
{

	StructureEntity stfound;/* the structure for the input port */
	SEQPort SPort;		/* traversal of port seq */
	Port APort;		/* value of port seq */
	SETPort SPort2;		/* traversal of ports of process */
	Port APort2;		/* value of port in ports set */
	Boolean duplicate;	/* indicator of duplicate port name */


	Assume((IsportDefinition(portDef)&&IsProcessEntity(Aprocess)),
		"AddPorts");

	foreachinSEQPort(portDef->syn_ports, SPort, APort) {
	    APort->sem_portType = portDef->syn_portType;
	    if (FindStructure(APort->syn_data->lex_name, Aprocess->inv_parent, 
			      &stfound) == NOTFOUND) {
		Warning0(451, APort->syn_data->lex_namepos);
		SetErrorEntity(APort->syn_data);
	    }
	    else {
		stfound->inv_isportstructure = TRUE;
		SetStructureEntity(APort->syn_data, stfound);
		if (typeof(stfound->sem_root) == KVoid) {
		    Warning0(103, APort->syn_data->lex_namepos);
		}
		else {
		    duplicate = FALSE;
		    foreachinSETPort(Aprocess->sem_ports, SPort2, APort2) {
			if (strequal(APort->lex_name, APort2->lex_name)) {
			      duplicate = TRUE;
			      break;
			}
		    }
		    if (duplicate)
		        Warning0(452, APort->lex_namepos);
		    else {
			addSETPort(Aprocess->sem_ports, APort);
		    }
		}
	    }
	}
}



/***********************************************************************
 *
 * Procedure CheckInvariant
 *
 * Purpose: Check that every class and attribute defined in each 
 * 	    structure associated with a port exists in the invariant 
 *	    structure. If class doesn't exist, add to invariant.
 *
 * Algorithm: Check that the invariant structure is not a port structure,
 *	      	if so, make a new copy 
 *	      For each port structure, add all classes to invariant
 *	      	structure. CopyNonTerminal takes care of resolving 
 *	      	duplicates. 
 *	      For every copied nonterminal and attribute print warning 
 *	      	message. Don't print messages if default invariant is used.
 *	      	Disable messages after 15.
 *	      Add the port to ports of every attribute and Type 
 *		in invariant contained in port
 *	      Attach any descendants or attribute user types that may have
 *	      	been added.
 *	      Sort the classes from least-inclusive to most-inclusive.
 *	      Propogate field definitions from an ancestor class to all nodes
 * 	      	and classes which are members of the class. Propogated 
 *	      	attributes are marked as such. Duplicate attributes are 
 *	      	checked for type conflict with existing attribute     
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/


CheckInvariant(pr)
ProcessEntity pr;	/* the process */
{

	SETTypeEntity SType;	/* set/seq traversals and values */
	TypeEntity AType;
	SEQAttribute SAtt;
	Attribute AnAttribute;
	SETPort SPort;		
	Port APort;	
	StructureEntity portst;	/* the port structure */
	StructureEntity Invst;	/* the invariant structure */
	StructureEntity Invspec;/* invariant specified */
	Boolean copied=FALSE;	/* indicates if invariant has been copied */
	int wcount=0;		/* number of warning messages displayed */
	Boolean defaultInv;	/* indicator if default invariant is used */
	String tname;		/* name of type */
	SETString havechecked;	/* port structures checked for this process */
	ProcessEntity oldpr;

	Assume((IsProcessEntity(pr)&&IsStructureEntity(pr->sem_invariant)),
		"CheckInvariant");
	
	/* set the invariant structure for de-referencing */
	Invst = pr->sem_invariant;


	/* if an invariant was specified and was found, 
	 * copy all types to new invariant */

	if ((typeof(pr->syn_invariant)==KStructureRef)&&
	    TypeOfEntity(pr->syn_invariant.VStructureRef)==KStructureEntity){

	    Invspec = GetStructureEntity(pr->syn_invariant.VStructureRef);
	    copied = TRUE;
	    CopyAllTypes(Invst, Invspec->sem_types);
	}

	
	/* if process is refined, copy all types of old process
	 * invariant to new, copy all ports of old to new */
	if ((typeof(pr->syn_refines)==KProcessRef)&&
	    TypeOfEntity(pr->syn_refines.VProcessRef)==KProcessEntity){

	    oldpr = GetProcessEntity(pr->syn_refines.VProcessRef);
	    Invspec = oldpr->sem_invariant;
	    copied = TRUE;
	    CopyAllTypes(Invst, Invspec->sem_types);
	    CopyAllPorts(pr, oldpr);
	}
	/* Mark types already in invariant>
	 * For each port structure, add all types to the invariant
	 * structure							
	 */

	foreachinSETTypeEntity(Invst->sem_types, SType, AType) {
	    MarkClassTouched(AType);
	    if (typeof(AType) == KClass) {
		foreachinSEQAttribute(AType.VClass->sem_allattributes,
				SAtt, AnAttribute)
		    MarkTouched(AnAttribute);
	    }
	}

	initializeSETString(havechecked);
	foreachinSETPort(pr->sem_ports, SPort, APort) {

	    /* Get structure associated with port 
	     * If structure was already copied from,
	     * continue 
	     */

	    portst = GetStructureEntity(APort->syn_data);
	    if (inSETString(havechecked, portst->lex_name))
		continue;
	    else addSETString(havechecked, portst->lex_name);

	    /* add all types of port structures to invariant
	       structure. CopyAllTypes() takes care of resolving
	       duplicates						*/


	    CopyAllTypes(Invst, portst->sem_types);
	}
	/* if invariant has a root, assign it the correct type
	 */

	if (copied && (typeof(Invspec->syn_root) != KVoid)){
	    if (FindType(Invst, Invspec->syn_root.VTypeRef) == NOTFOUND){
		Warning0(106, Invst->lex_endpos);
	    }
	    else Invst->sem_root.VTypeEntity = 
		GetTypeEntity(Invspec->syn_root.VTypeRef);
	}

	/*  Print warning messages for any copied
	    nonterminals and attributes	unless invariant was not
	    given */

	wcount = 0;
	defaultInv = !strcmp(IDLDEFAULTINVARIANTNAME, Invst->lex_name);
	foreachinSETTypeEntity(Invst->sem_types, SType, AType){
	    tname = GetTypeName(AType);
	    if (ClassTouched(AType))
		UnmarkClassTouched(AType);
    	    else {
		++wcount;
		if ((!defaultInv) && (wcount<15))
		    Recoverable1(404, pr->lex_endpos, tname);
	    }

	    if (typeof(AType) == KClass) {
	        foreachinSEQAttribute(AType.VClass->sem_allattributes, 
					    SAtt, AnAttribute) {
		    /* only unmark if not propagated */
	            if (NodeTouched(AnAttribute)) {
			if (AnAttribute->inv_parent.VClass == AType.VClass)
		            UnmarkTouched(AnAttribute);
		    }
		    /* only print warning if not propagated */
		    else if (AnAttribute->inv_parent.VClass == AType.VClass){
			++wcount;
			if ((!defaultInv) && (wcount<15))
			     Recoverable2(405, pr->lex_endpos, 
					AnAttribute->lex_name, tname);
		    }
		}
	    }
	}

	/* check for cycles in class hierarchy relation */
	CheckForCycles(Invst);


	/* Sort the classes from least-inclusive to most-inclusive.
	  Check for sharing of descendants.
	  Propogate field definitions from an ancestor class to all nodes
  	  and classes which are members of the class. Propogated 
	  attributes are marked as such. Duplicate attributes are 
	  checked for type conflict with existing attribute     */


	if (!SeriousError) {
	    PropogateClassAttributes(Invst);
	}

}
	




/***********************************************************************
 *
 * Function CheckTypes
 *
 * Purpose: Compare two types for equality.
 *
 * Return Value: EQUAL if types are equivalent, else NOTEQUAL
 *
 * Algorithm: Check the types for equality. if equal, return EQUAL else
 *	      return NOTEQUAL
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
CheckTypes(t1,t2)
TypeRef t1, t2;	/* the two types compared */
{
        TypeRef ThisTypeRef;
	Assume((IsTypeRef(t1)&&IsTypeRef(t2)),
		"CheckTypes");

	if (typeof(t1) != typeof(t2))
	    return(NOTEQUAL);
	if (typeof(t1) == KNamedTypeRef) {
	    if (strequal(t1.VNamedTypeRef->lex_name, 
			 t2.VNamedTypeRef->lex_name)) 
		    return(EQUAL);
	    else return(NOTEQUAL);
	}
	else {
	  ThisTypeRef = NamedTypeRefToTypeRef(t1.VSetOrSeqRef.IDLclassCommon->syn_component);
	  return(CheckTypes(ThisTypeRef,
	   NamedTypeRefToTypeRef(t2.VSetOrSeqRef.IDLclassCommon->syn_component)
		    ));
	}
}

/***********************************************************************
 *
 * Procedure AddRestrictions
 *
 * Purpose: add the operations a class or attribute is restricted to
 *
 * Algorithm: find the class which is specified for operations
 *	      call the appropriate restriction adder depending on type
 *
 * Errors Checked For: specified class not found
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
AddRestrictions(r, pr)
restriction r;
ProcessEntity pr;
{

	SETTypeEntity SType;
	TypeEntity AType;
	SEQAttribute SAtt;
	Attribute AnAtt;
	TypeEntity typefound;	/* specified type found  */
	StructureEntity Invst;  /* invariant structure of process */
	TypeOrAllTypes typenames; /* for de-referencing */
	SETTypeEntity alltypes;	/* all types for type restrictions */
	SETClass allclasses;	/* all classes for attribute restrictions */
	attributeRestriction ar; /* for de-referencing */
	AttributeRefs attrefs;
	SETAttribute allattributes;


	Assume((Isrestriction(r)&&IsProcessEntity(pr)&&
		IsStructureEntity(pr->sem_invariant)),
		"AddRestrictions");
	/* initialize */
	initializeSETTypeEntity(alltypes);
	initializeSETClass(allclasses);

	/* determine the invariant structure */
	Invst = pr->sem_invariant;
	Assume((IsStructureEntity(Invst)), 
		"process invariant in AddRestrictions");

	/* de-reference the type name(s) for the restriction */
	typenames = r.IDLclassCommon->syn_type;

	/* if all types are specified, add all to r.syn_type.alltypes */

	if (typeof(typenames) == KAllTypes) {
	    if (typeof(r) == KtypeRestriction) {
		foreachinSETTypeEntity(Invst->sem_types, SType, AType)
		    addSETTypeEntity(alltypes, AType);
	    }
	    else {
		/* only class types for attribute restrictions */
		foreachinSETTypeEntity(Invst->sem_types, SType, AType)
		    if (typeof(AType) == KClass) {
			addSETClass(allclasses, AType.VClass);
			addSETTypeEntity(alltypes, AType);
		    }
	    }
	    typenames.VAllTypes->sem_items = alltypes;
	}

	/* find the type in the invariant structure of pr 
	   and add to Type Reference  */

	else {
	    if (FindType(Invst, typenames.VTypeRef) == NOTFOUND){
		Warning0(475, GetTypePos(typenames.VTypeRef));
		SetErrorEntity(typenames.VTypeRef.IDLclassCommon);
	    }
	    else {
		typefound = GetTypeEntity(typenames.VTypeRef);
		if (typeof(r) == KtypeRestriction) {
		    addSETTypeEntity(alltypes, typefound);
		}
		else {
		    if (typeof(typefound) != KClass) {
			Warning0(476, GetTypePos(typenames.VTypeRef));
			SetErrorEntity(typenames.VTypeRef.IDLclassCommon);
		    }
		    else {
			addSETClass(allclasses, typefound.VClass);
			addSETTypeEntity(alltypes, typefound);
		    }
		}
	    }
	}
	if (typeof(r) == KattributeRestriction) {
	    ar = r.VattributeRestriction;
	    initializeSETAttribute(allattributes);
	    if (typeof(ar->syn_attribute) == KAllAttributes) {
		foreachinSETTypeEntity(alltypes, SType, AType) {
		    Assume((IsClass(AType.VClass)), "AddRestrictions2");
		    foreachinSEQAttribute(AType.VClass->sem_allattributes, 
					SAtt, AnAtt) {
			addSETAttribute(allattributes, AnAtt);
		    }
		}
	    }
	    else {
		attrefs = ar->syn_attribute.VAttributeRefs;
		foreachinSETTypeEntity(alltypes, SType, AType) {
		    Assume((IsClass(AType.VClass)), "AddRestrictions3");
		    foreachinSEQAttribute(AType.VClass->sem_allattributes, 
					SAtt, AnAtt) {
			if (AnAtt->lex_name == attrefs->lex_name) {
			    addSETAttribute(attrefs->sem_items, AnAtt);
			}
		    }
		}
		if (emptySETAttribute(attrefs->sem_items) && 
		    !emptySETTypeEntity(alltypes)) {
		    Warning0(478, attrefs->lex_namepos);
		}
	    }
	}
}



/***********************************************************************
 *
 * Procedure AddPortAssoc
 *
 * Purpose: Add the port association rep to the port
 *
 * Algorithm: 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

 AddPortAssoc(pr, aportassoc)
 ProcessEntity pr;
 portAssociation aportassoc;
 {
	SETPort SPort;
	Port APort;
	SEQparameter Sparm;
	parameter parm;
	PortRef aportref;
	static Atomic IntType, StrType, RatType;
	static Boolean init=TRUE;
	Atomic GetIntegerType(),
	       GetStringType(),
	       GetRationalType();

	Assume((IsProcessEntity(pr) && IsportAssociation(aportassoc)),
		"AddPortAssoc");

	if (init) {
	    init = FALSE;
	    IntType = GetIntegerType();
	    StrType = GetStringType();
	    RatType = GetRationalType();
	}

	aportref = aportassoc->syn_port;
	SetErrorEntity(aportref);
	foreachinSETPort(pr->sem_ports, SPort, APort) {
	    if (strequal(APort->lex_name, aportref->lex_name)) {
		SetPortEntity(aportref, APort);
		break;
	    }
	}
	if (! EntityIsError(aportref)) {
	    appendrearSEQRepRef(APort->sem_rep, aportassoc->syn_rep);
	}
	else {
	    Warning0(453, aportref->lex_namepos);
	    SetErrorEntity(aportref);
	}
	/* set types for all integerTokens stringTokens and rationalTokens */
	foreachinSEQparameter(aportassoc->syn_rep->syn_id, Sparm, parm) {
	    switch (typeof(parm)) {
		case KintegerToken:
		    parm.VintegerToken->sem_type.VAtomic = IntType;
		    break;
		case KstringToken:
		    parm.VstringToken->sem_type.VAtomic = StrType;
		    break;
		case KrationalToken:
		    parm.VrationalToken->sem_type.VAtomic = RatType;
		    break;
	    }
	}
}
