/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlc/semantic/analyze2.c 
*				 					* 
*   Copyright (C) 1991 Karen Shannon
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Function: Routines used in the analysis process and called from the *
*           higher level routines in analyze.c				*
*									*
*	     AddClass  AddAtomic AddDescendants  AddAttributes  	*
*	     AddAssertion AddInstance					*
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: analyze2.c,v 1.1 89/07/05 16:31:28 kps Locked $";
#endif

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	analyze2.c,v $
 * Revision 1.1  89/07/05  16:31:28  kps
 * Initial revision
 * 
 * Revision 4.0  89/04/12  02:46:35  cheung
 * analyze2.c  Ver 4.0
 * 
 * Revision 3.9  89/03/30  13:22:09  cheung
 * analyze2.c  Ver 3.9
 * 
 * Revision 3.9  89/03/26  15:02:42  cheung
 * analyze2.c  Ver 3.9
 * 
 * Revision 1.1  87/04/18  11:40:21  shannon
 * Initial revision
 * 
 * Revision 1.1  86/06/06  14:55:15  shannon
 * Initial revision
 * 
 * Revision 2.0  86/01/22  07:20:18  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.7  86/01/22  07:07:35  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.6  85/07/26  10:01:04  shannon
 * added assertion processing
 * 
 * Revision 1.5  85/07/25  10:29:12  shannon
 * changed code for new null and empty class rep
 * 
 * Revision 1.4  85/07/18  19:42:43  shannon
 * deleted unused variables
 * 
 * Revision 1.3  85/07/04  14:53:50  shannon
 * changed set of symbols to seq
 * 
 * Revision 1.2  85/07/04  10:11:16  shannon
 * fixed bug that allowed duplicate descendants for classes.
 * 
 * Revision 1.1  85/06/11  10:14:27  shannon
 * Initial revision
 * 
*									*
*   Edit Log:								*
*     Jan 9 1985 (shannon) Created.					*
*									*
\* ******************************************************************* */

#include "Semantic.h"
#include "macros.h"
#include <stdio.h>


/***********************************************************************
 *
 * Procedure AddClass
 *
 * Purpose: Add the Class given by the production 'prod' to the 
 *	    structure 'st' types
 *
 * Algorithm: Check if Class already exists. If it does, combine
 *	      the two Classes by creating a new copy of the class.
 *	      If the Class doesn't already exist add to structure 
 *	      'st'. Add descendants for a class production and add 
 *	      attributes for an attribute production.
 *
 * Errors Checked For: production is correct type.
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

AddClass(st, prod)
StructureEntity st;		/* structure */
production prod; 	
{

	TypeEntity AType,AType1;	
	Class AClass;		
	String cname;		/* class name */

	Assume((IsStructureEntity(st)&&Isproduction(prod)), "AddClass");

	cname = prod.IDLclassCommon->syn_class->lex_name;

	/* check if Class already exists */

	if (FindNamedType(st, cname, &AType) == FOUND) {

  	    /* if the type found is an Atomic, print error */
	    if (typeof(AType) == KAtomic) {
		Warning0(260, prod.IDLclassCommon->syn_class->lex_namepos);
		SetErrorEntity(prod.IDLclassCommon->syn_class);
	    }

	    /* if the type is copied and is a node in the ancestor structure
	       and the production is a classProduction
	       print error --can't promote node to class */
	    else if ((typeof(AType.VClass->sem_copiedfrom)==KClass) &&
		     (emptySETClass(AType.VClass->sem_subclasses)) &&
		     (typeof(prod) == KsubclassProduction)) {
		Warning0(154, prod.IDLclassCommon->syn_class->lex_namepos);
		SetErrorEntity(prod.IDLclassCommon->syn_class);
	    }

	    /* else set the entity to the class */
	    else SetClassEntity(prod.IDLclassCommon->syn_class, AType.VClass);
	}

	/* if Class doesn't already exist add to structure st */
	else {
	    AClass = NClass;
	    AClass->sem_name = cname;
	    AClass->inv_definitionPoint.VClass = AClass;
	    AClass->inv_parent.VStructureEntity = st;
	    SetClassEntity(prod.IDLclassCommon->syn_class, AClass);
	    AType1.VClass = AClass;
	    addSETTypeEntity(st->sem_types, AType1);
	}
}


/***********************************************************************
 *
 * Procedure AddAtomic
 *
 * Purpose: Add atomic type to structure 'st'
 *
 * Algorithm: Check for duplicate type in 'st'. If not found, add the new
 *	      new type.
 *
 * Errors Checked For: duplicate type
 *
 * Assumptions/Limitations:
 *
 **********************************************************************/
AddAtomic(st, AnatomicDecl)
StructureEntity st;
atomicDecl AnatomicDecl;
{

	TypeEntity AType,AType1;	/* type found in set */
	Atomic AnAtomic;	/* the new  atomic created */

	Assume((IsStructureEntity(st)&&IsatomicDecl(AnatomicDecl)), 
		"AddAtomic");

	/* check for duplicate type */

	if (FindNamedType(st, AnatomicDecl->syn_atomic->lex_name, 
						&AType) == FOUND) {
	    Warning0(254, AnatomicDecl->syn_atomic->lex_namepos);
	    SetErrorEntity(AnatomicDecl->syn_atomic);
	}

	/* if not duplicate type, add to st->sem_types */

	else {
	    AnAtomic = NAtomic;
	    SetAtomicEntity(AnatomicDecl->syn_atomic, AnAtomic);
	    AnAtomic->sem_name = AnatomicDecl->syn_atomic->lex_name;
	    AnAtomic->inv_definitionPoint.VAtomic = AnAtomic;
	    AnAtomic->inv_parent.VStructureEntity = st;
	    AType1.VAtomic = AnAtomic;
	    addSETTypeEntity(st->sem_types, AType1);
	}
}


/***********************************************************************
 *
 * Procedure AddDescendants
 *
 * Purpose: Add the descendants given in 'class_prod' to Class
 *
 * Algorithm: Add the descendants to the subclasses set of the Class. 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

AddDescendants(class_prod)
subclassProduction class_prod;
{

	SEQNamedTypeRef SNamedTypeRef;	/* set/seq traversals and values */
	NamedTypeRef ANamedTypeRef;
	Class AClass;			/* class information is added to */
	Class subclass;
	NamedType typefound;
	
	Assume((IssubclassProduction(class_prod)), "AddDescendants");

	/* if the class ref entity is Error, set the entity of all subclass
	 *  class refs to Error also
	 */
	if (EntityIsError(class_prod->syn_class))  {
	    foreachinSEQNamedTypeRef(class_prod->syn_subclasses, SNamedTypeRef, 
				ANamedTypeRef) {
		SetErrorEntity(ANamedTypeRef);
	    }
	    return;
	}
	else AClass = GetClassEntity(class_prod->syn_class);
	    
	/* add descendants */
	foreachinSEQNamedTypeRef(class_prod->syn_subclasses, SNamedTypeRef, 
							ANamedTypeRef) {
	    if (FindNamedType(AClass->inv_parent.VStructureEntity, 
			      ANamedTypeRef->lex_name, &typefound)==NOTFOUND){
		Recoverable0(200, ANamedTypeRef->lex_namepos);

		/* create a new Class and add to st->types
		 */
		AddNewType(AClass->inv_parent.VStructureEntity, NamedTypeRefToTypeRef(ANamedTypeRef));
		subclass = GetClassEntity(ANamedTypeRef);
		SetClassEntity(ANamedTypeRef, subclass);
	    }

	    /* check that the subclass is a class */
	    else if (typeof(typefound) != KClass) {
		Warning0(259, ANamedTypeRef->lex_namepos);
		SetErrorEntity(ANamedTypeRef);
	    }
	    else {
		subclass = typefound.VClass;
		SetClassEntity(ANamedTypeRef, subclass);
	    }

	    /* add to ancestor list and subclass list
	     * check for duplicates
	     */
	    if (!EntityIsError(ANamedTypeRef)) {
		if (inSETClass(AClass->sem_subclasses, subclass)) {
		    Warning0(258, ANamedTypeRef->lex_namepos);
		}
		else {
		    addSETClass(subclass->sem_ancestors, AClass);
		    addSETClass(AClass->sem_subclasses, subclass);
		}
	    }
	}
}
		
		
/***********************************************************************
 *
 * Procedure AddAtributes
 *
 * Purpose: Add the attributes in 'att_prod' to the Class
 *
 * Algorithm: Add each attribute in att_prod  to the Class. Check for
 *	      duplicate attributes. 
 *
 * Errors Checked For: duplicate attributes.
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

AddAttributes(att_prod )
attributeProduction att_prod;	/* attribute production */
{
	SEQAttribute SAtt;	/* set/seq traversals and values */
	Attribute AnAttribute;
	Attribute attfound;
	Class AClass;		/* the Class */

	Assume((IsattributeProduction(att_prod)&&
		(EntityIsError(att_prod->syn_class)||
		(TypeOfEntity(att_prod->syn_class)==KClass))),
		"AddAttributes");

	if (EntityIsError(att_prod->syn_class)) {
	    foreachinSEQAttribute(att_prod->syn_attributes, SAtt, AnAttribute){
		SetErrorEntity(AnAttribute->syn_type.IDLclassCommon);
		AnAttribute->inv_parent.Verror = Nerror;
	    }
	    return;
	}
	else {
	    AClass = GetClassEntity(att_prod->syn_class);
	}


	/* traverse through the attribute production attributes, 
	 * add an attribute to Class.allattributes 
	 * for each unique attribute found
	 */

	foreachinSEQAttribute(att_prod->syn_attributes, SAtt, AnAttribute){

	    /* check if attribute already exists, if it does, print error 
	     */

	    if (FindClassAttribute(AClass, AnAttribute->lex_name, &attfound)
				== FOUND) {
		if (!CheckTypes(AnAttribute->syn_type, attfound->syn_type)){
		    if (typeof(AnAttribute->syn_type) == KNamedTypeRef)
			Warning0(275, 
			    AnAttribute->syn_type.VNamedTypeRef->lex_namepos);
		    else Warning0(275, 
			AnAttribute->syn_type.VSetOrSeqRef.
				IDLclassCommon->lex_keywordpos);
		}
		else Warning0(274, AnAttribute->lex_namepos);
		AnAttribute->inv_parent.Verror = Nerror;
		SetErrorEntity(AnAttribute->syn_type.IDLclassCommon);
	    }

	    /* if attribute doesn't already exist, add to nt */

	    else {
		appendrearSEQAttribute(AClass->sem_allattributes, AnAttribute);
		AnAttribute->inv_parent.VClass = AClass;
	    }
	}
}




/***********************************************************************
 *
 * Procedure AddAssertion
 *
 * Purpose: Add the assertion to the structure or process
 *
 * Algorithm: Add the assertion to the assertion sequence
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
AddAssertion(AstOrpr, AnAssertion) 
StructureOrProcess AstOrpr;
Assertion AnAssertion;
{
     addSETAssertion(AstOrpr.IDLclassCommon->sem_assertions, AnAssertion);
}

/***********************************************************************
 *
 * Procedure AddInstance
 *
 * Purpose: Add the instance to the structure or process
 *
 * Algorithm: Add the instance to a definition if one is found with the
 *	      same name, otherwise, create a new definition
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
AddInstance(AstOrpr, ADefInstance) 
StructureOrProcess AstOrpr;
DefInstance ADefInstance;
{
     SETDefinition SDef;	/* set traversal and value */
     Definition ADef;
     String defname;		/* name of definition */
     Boolean found=FALSE;	/* indicator if definition is found */

     defname = ADefInstance.IDLclassCommon->syn_def->lex_name;
     foreachinSETDefinition(AstOrpr.IDLclassCommon->sem_definitions, 
			    SDef, ADef) {
	if (strequal(ADef->sem_name, defname)) {
	    found = TRUE;
	    break;
	}
     }
     if (!found) {
	 ADef = NDefinition;
	 ADef->sem_name = defname;
	 addSETDefinition(AstOrpr.IDLclassCommon->sem_definitions, ADef);
     }
     SetDefineEntity(ADefInstance.IDLclassCommon->syn_def, ADef);
     addSETDefInstance(ADef->sem_overload, ADefInstance);
}



/***********************************************************************
 *
 * Procedure AddNewType
 *
 * Purpose: Add a new type to the structure
 *
 * Algorithm: Add a new type given by the type reference and set
 *	      the entity field of the type reference to the new type.
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: The type does not already exist in the structure.
 *
 **********************************************************************/
AddNewType(st, atyperef)
StructureEntity st;
TypeRef atyperef;
{
     SetOf newset;
     SeqOf newseq;
     NamedTypeRef component;
     NamedType compfound;
     Class newclass;
     String compname;
     String buf;

     Assume((IsStructureEntity(st)&&IsTypeRef(atyperef)), "AddNewType");
     switch (typeof(atyperef)) {
	 case KNamedTypeRef:
	     newclass = NClass;
	     newclass->sem_name = atyperef.VNamedTypeRef->lex_name;
	     newclass->inv_definitionPoint.VClass = newclass;
	     newclass->inv_parent.VStructureEntity = st;
	     SetClassEntity(atyperef.VNamedTypeRef, newclass);
	     break;
	 case KSetRef:
	     component = atyperef.VSetRef->syn_component;
	     newset = NSetOf;
	     newset->inv_definitionPoint.VSetOf = newset;
	     newset->inv_parent.VStructureEntity = st;
	     if (FindNamedType(st,component->lex_name, &(compfound))
				== NOTFOUND){
		AddNewType(st, NamedTypeRefToTypeRef(component));
	     }
	     else SetNamedTypeEntity(component, compfound);

	     newset->sem_component = GetNamedTypeEntity(component);
	     compname = newset->sem_component.IDLclassCommon->sem_name;
	     buf = (String)GetHeap(strlen(compname)+8);
	     (void)sprintf(buf, "Set Of %s", compname);
	     newset->sem_name = NewString(buf);
	     SetSetEntity(atyperef.VSetRef, newset);
	     break;
	 case KSeqRef:
	     component = atyperef.VSeqRef->syn_component;
	     newseq = NSeqOf;
	     newseq->inv_definitionPoint.VSeqOf = newseq;
	     newseq->inv_parent.VStructureEntity = st;
	     if (FindNamedType(st,component->lex_name, &(compfound))
				== NOTFOUND){
		AddNewType(st, NamedTypeRefToTypeRef(component));
	     }
	     else SetNamedTypeEntity(component, compfound);

	     newseq->sem_component = GetNamedTypeEntity(component);
	     compname = newseq->sem_component.IDLclassCommon->sem_name;
	     buf = (String)GetHeap(strlen(compname)+8);
	     (void)sprintf(buf, "Seq Of %s", compname);
	     newseq->sem_name = NewString(buf);
	     SetSeqEntity(atyperef.VSeqRef, newseq);
	     break;
	     
     }

     addSETTypeEntity(st->sem_types, GetTypeEntity(atyperef));
}



/***********************************************************************
 *
 * Procedure AssignAttributeTypes
 *
 * Purpose: assign the types to the attributes
 *
 * Algorithm: 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
AssignAttributeTypes(st)
StructureEntity st;
{
	SETTypeEntity SType; /* set/seq traversals and values */
	TypeEntity AType;
	SEQAttribute SAtt;
	Attribute AnAtt;
	NamedTypeRef component; /* component of set or seq ref */
	TypeEntity atttype;	/* type of attribute */
	TypeEntityOrError save_type;

	Assume((IsStructureEntity(st)), "AssignAttributeTypes");
	foreachinSETTypeEntity(st->sem_types, SType, AType) {
	    if (typeof(AType) == KClass)
		foreachinSEQAttribute(AType.VClass->sem_allattributes, 
					SAtt, AnAtt) {
		     save_type = AnAtt->syn_type.IDLclassCommon->sem_entity;

		     if (FindType(st, AnAtt->syn_type)== NOTFOUND){
			 if (typeof(AnAtt->syn_type)==KNamedTypeRef) {
			     Recoverable0(200, GetTypePos(AnAtt->syn_type));
			 }
			 else {
			     component = AnAtt->syn_type.VSetOrSeqRef.
					IDLclassCommon->syn_component;
			     if (EntityIsError(component)) {
				 Recoverable0(200, component->lex_namepos);
			     }
			 }
			 AddNewType(st, AnAtt->syn_type);
			 if (typeof(AnAtt->sem_copiedfrom)==KAttribute) {
			     atttype = GetTypeEntity(AnAtt->syn_type);
			     if (IsSetOrSeq(atttype)){
				 atttype.IDLclassCommon->sem_copiedfrom.
					VTypeEntity = GetTypeEntity(AnAtt->
					sem_copiedfrom.VAttribute->syn_type);
			     }
			 }
		     }
		     atttype = GetTypeEntity(AnAtt->syn_type);
		     addSETAttribute(atttype.IDLclassCommon->inv_attributes, 
				     AnAtt);
		     if (typeof(save_type) != Kerror)
			AnAtt->syn_type.IDLclassCommon->sem_entity = save_type;
		}
	}
}

/***********************************************************************
 *
 * Procedure AssignClassRelatives
 *
 * Purpose: assign the descendants and ancestors of the classes
 *
 * Algorithm: 
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

AssignClassRelatives(st)
StructureEntity st;
{
	SETTypeEntity SType;		/* set/seq traversals and values */
	TypeEntity AType;
	SETNamedTypeRef SNamedTypeRef;
	NamedTypeRef ANamedTypeRef;
	Class AClass;			/* the type if it is a class */

	Assume((IsStructureEntity(st)), "AssignClassRelatives");

	foreachinSETTypeEntity(st->sem_types, SType, AType) {
	    if (typeof(AType) == KClass) {
		AClass = AType.VClass;
		foreachinSETNamedTypeRef(AClass->inv_subclasses, 
				     SNamedTypeRef, ANamedTypeRef) {
		    if (FindType(st, NamedTypeRefToTypeRef(ANamedTypeRef)) == NOTFOUND) {
			Recoverable0(200, ANamedTypeRef->lex_namepos);
			AddNewType(st, NamedTypeRefToTypeRef(ANamedTypeRef));
		    }
		    addSETClass(AClass->sem_subclasses, 
				ANamedTypeRef->sem_entity.VClass);
		    addSETClass(ANamedTypeRef->sem_entity.VClass->sem_ancestors,
				AClass);
		}
		AType.VClass->inv_subclasses = NULL;
	    }
	}
}
