/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlc/semantic/analyze.c 
*				 					* 
*   Copyright (C) 1991 Karen Shannon
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Function: routines used in analyzing the structures and processes	*
*           of the dictionary 						*
*									*
*           Analyze  AnalyzeStructure  AnalyzeProcess			*
*	    DeterminePortStructures                 		*
*									*
\* ******************************************************************* */

#ifndef lint
    static char rcsid[] = "$Header: analyze.c,v 1.1 89/07/05 16:31:25 kps Locked $";
#endif

/* ******************************************************************* *\
*   Revision Log:							*
*	$Log:	analyze.c,v $
 * Revision 1.1  89/07/05  16:31:25  kps
 * Initial revision
 * 
 * Revision 4.0  89/04/29  17:17:53  cheung
 * analyze.c version 4.0
 * 
 * Revision 4.0  89/04/12  02:46:20  cheung
 * analyze.c  Ver 4.0
 * 
 * Revision 3.9  89/03/30  13:21:56  cheung
 * analyze.c  Ver 3.9
 * 
 * Revision 3.9  89/03/26  15:02:31  cheung
 * analyze.c  Ver 3.9
 * 
 * Revision 3.2  88/02/19  18:21:10  rajan
 * Version 3.2
 * 
 * Revision 1.1  87/04/18  11:40:16  shannon
 * Initial revision
 * 
 * Revision 1.1  86/06/06  14:55:07  shannon
 * Initial revision
 * 
 * Revision 2.0  86/01/22  07:19:55  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.9  86/01/22  07:06:49  shannon
 * Release 2.0 of IDL System
 * 
 * Revision 1.8  85/07/26  09:59:26  shannon
 * added assertion processing
 * 
 * Revision 1.7  85/07/25  10:26:48  shannon
 * added new restriction processing
 * changed code for new empty and null class rep
 * 
 * Revision 1.6  85/07/19  19:30:20  shannon
 * added check for port names the same as nonterminal names in a process.
 * 
 * Revision 1.5  85/07/18  19:40:43  shannon
 * added invariant structure to symbol table scope
 * for default structures
 * 
 * Revision 1.4  85/07/05  20:40:23  shannon
 * added processing of unmarker port
 * 
 * Revision 1.3  85/07/04  14:53:20  shannon
 * added check for unreachable nodes and classes
 * changed set of symbols to seq
 * 
 * Revision 1.2  85/07/04  13:57:20  shannon
 * added check for nonreachable nodes and classes
 * 
 * Revision 1.1  85/06/11  10:14:07  shannon
 * Initial revision
 * 
*									*
*   Edit Log:								*
*     Jan 9 1985 (shannon) Created.					*
*									*
\* ******************************************************************* */

#include "Semantic.h"
#include "macros.h"
#include <stdio.h>

#define verboseOut(str, arg)	{(void) fprintf(stderr, str, arg); (void) fflush(stderr);}
extern Boolean SeriousError;	/* indicator of semantic error */
extern Boolean V2conform;	/* indicator if version 2 conformance is on*/


/***********************************************************************
 *
 * Procedure Analyze
 *
 * Purpose: Analyze the structures and processes contained in compilationUnit 
 *
 * Algorithm: Analyze each structure. 
 *	      Analyze each process.
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: structures and processes seqs are sorted.
 *
 **********************************************************************/
Analyze(compUnit, verbosemode)
compilationUnit compUnit; /* the compilation unit */
Boolean verbosemode;	  /* indicator if mode is verbose */
{

	SEQStructureEntity SStr;   /* seq traversals and values */
	StructureEntity Astructure;  
	SEQProcessEntity SPr;
	ProcessEntity Aprocess;	     

	Assume((IscompilationUnit(compUnit)&&IsBoolean(verbosemode)),
		"Analyze");

	/* analyze all structures */
	
	foreachinSEQStructureEntity(compUnit->sem_structures, SStr, Astructure){

		/* if verbose mode, print out structure name */

		if (verbosemode)
		    verboseOut("idlc:    analyzing structure '%s'\n",
			    Astructure->lex_name);

		
		/* set parent field */
		Astructure -> inv_parent = compUnit;

		/* analyze structure */
		AnalyzeStructure(Astructure);
	}


	/* determine all port structures and mark as such */
	DeterminePortStructures(compUnit, compUnit->sem_processes);

	/* analyze all processes */

	foreachinSEQProcessEntity(compUnit->sem_processes, SPr, Aprocess) {
	
		/* if verbose mode, print out process name */

		if (verbosemode)
		    verboseOut("idlc:    analyzing process '%s'\n",
			    Aprocess->lex_name);

		/* set the parent field */

		Aprocess -> inv_parent = compUnit;

		/* analyze the process */

		AnalyzeProcess(Aprocess);

	}
}





/***********************************************************************
 *
 * Procedure AnalyzeStructure
 *
 * Purpose: Analyze the structure.
 *	    Add appropriate information to semantic attributes.
 *
 * Algorithm: Check if the structure is derived or refined. If so, copy
 * 	        the ancestor structure(s) into the new structure.
 *	      Process without clauses for derived structures. Check that 
 *	        there are no without clauses for non-derived structures.
 *	      Add Classes for left-hand sides of productions in structure. 
 *		For each left-hand side, add entries for right-hand sides.
 *	      Add atomic types and type representation information.
 *	      Add assertions and definitions
 *	      Attach types to attributes and descendants to classes.
 *	      Attach types to assertions
 *	      Check for cycles in the class hierarchy.
 *	      Sort the classes of each structure from least-inclusive to 
 *	        most-inclusive. 
 *	      Propogate attributes from an ancestor class
 *	        to all nodes and classes which are descendants.
 *	      Assign the root to the structure.
 *
 * Errors Checked For: refined only structure with additional productions
 *		       non-derived structure having without clauses.
 *		       cycles in class hierarchy
 *		       root not found
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/

AnalyzeStructure(Astructure)
StructureEntity Astructure; /* the structure declaration */
{

	SEQstructureStatement tb;	
	structureStatement bval;	
	Boolean derived = FALSE,       	/* indicator of derived structure */
		refined = FALSE;       	/* indicator of refined structure */


	Assume((IsStructureEntity(Astructure)), "AnalyzeStructure");

	/* add prelude types to structure */
	AddPreludeTypes(Astructure);


	/* check if the structure is derived or refined. If so, copy 
	 * references for all types of the ancestor structure into
	 * the new structure. 
	 */

	if (typeof(Astructure->syn_refines)==KStructureRef) refined = TRUE;
	if (!emptySEQStructureRef(Astructure->syn_from)) derived = TRUE;
	if (!V2conform && derived && refined)
	    Warning0(100, Astructure->lex_endpos);
	if (refined || derived) CopyFromAncestors(Astructure, refined);


	/* process without clauses for derived structures by deleting
	 * the appropriate types. Check that there
	 * are no without clauses for non-derived structures 
	 */

	foreachinSEQstructureStatement(Astructure->syn_body, tb, bval) {
	    if (typeof(bval) == KwithoutClause)
		if (derived)
		    ProcessWithoutClause(Astructure, bval.VwithoutClause);

		else {  /* non-derived */
		    Recoverable0(102, bval.VwithoutClause->lex_stmtpos);
		    /* set error fields */
		    SetWithoutClauseErrors(bval.VwithoutClause);
		}
	}


	/* Add Classes for left-hand sides of productions in structure.
	 */

	foreachinSEQstructureStatement(Astructure->syn_body, tb, bval) {
	    if ((typeof(bval) == KsubclassProduction) ||
		(typeof(bval) == KattributeProduction)) {
		    if (refined && !derived && !V2conform) {
			Warning0(101, 
			  bval.Vproduction.IDLclassCommon->lex_stmtpos);
			SetErrorEntity(bval.Vproduction.IDLclassCommon->syn_class);
		    }
		    else AddClass(Astructure, bval.Vproduction);
	    }
	}

	/* For each left-hand side, add entries for right-hand sides
	 */
	foreachinSEQstructureStatement(Astructure->syn_body, tb, bval) {
	    if (typeof(bval) == KsubclassProduction) {
		AddDescendants(bval.VsubclassProduction);
	    }
	    else if (typeof(bval) == KattributeProduction){
		AddAttributes(bval.VattributeProduction);
	    }
	}

	/* add private types */
	foreachinSEQstructureStatement(Astructure->syn_body, tb, bval)
	    if (typeof(bval) == KatomicDecl) {
		if (refined && !derived && !V2conform) {
		    Warning0(101, bval.VatomicDecl->lex_stmtpos);
		    SetErrorEntity(bval.VatomicDecl->syn_atomic);
		}
		else AddAtomic(Astructure, bval.VatomicDecl);
	    }

	/* attach the types of the attributes to the types in
	 * the structure. Adds sets and sequences at this time.
	 * attach the new members to classes
	 */
	AssignAttributeTypes(Astructure);
	AssignClassRelatives(Astructure);
	

	/* add the type representation information */
	foreachinSEQstructureStatement(Astructure->syn_body, tb, bval)
	    if (typeof(bval) == KtypeRepDecl)
		AddTypeRep(Astructure, bval.VtypeRepDecl);
	    else if (typeof(bval) == KattributeRep)
		AddAttributeRep(Astructure, bval.VattributeRep);

	/* add the assertions and definitions */

	foreachinSEQstructureStatement(Astructure->syn_body, tb, bval)
		if (typeof(bval) == KAssertion) 
		    AddAssertion(StructureEntityToStructureOrProcess(Astructure), bval.VAssertion);
		else if ((typeof(bval) == Knoncyclicdef) ||
			 (typeof(bval) == Kcyclicdef) ||
			 (typeof(bval) == KPrivateDefInstance))
		    AddInstance(StructureEntityToStructureOrProcess(Astructure), bval.VDefInstance);


	/* attach the types to assertions */
	AddStPrAssertionTypes(StructureEntityToStructureOrProcess(Astructure));

	Assume((TRUE), "after type attachment in AnalyzeStructure");


	/* Check for cycles
	 * Propogate attribute from an ancestor class to all 
  	 * classes which are subclasses of the class. 
	 * Duplicate attributes are 
	 * checked for type conflict with existing attribute     
	 */

	CheckForCycles(Astructure);
	if (!SeriousError)  {
	    PropogateClassAttributes(Astructure);
	}

	Assume((TRUE), "after propogation in AnalyzeStructure");

	/* assign the root */
	if (typeof(Astructure->syn_root) == KNamedTypeRef){
	    if (FindNamedType(Astructure,
			Astructure->syn_root.VNamedTypeRef->lex_name,
			&(Astructure->sem_root.VNamedType)) == NOTFOUND) {
		Warning0(106, Astructure->syn_root.VNamedTypeRef->lex_namepos);
		SetErrorEntity(Astructure->syn_root.VNamedTypeRef);
	    }
	    else SetNamedTypeEntity(Astructure->syn_root.VNamedTypeRef,
				    Astructure->sem_root.VNamedType);
	}
}



/***********************************************************************
 *
 * Procedure DeterminePortStructures
 *
 * Purpose: Determine the ports for all processes.
 *
 * Algorithm: Determine the ports for all processes.
 *
 * Errors Checked For: none
 *
 * Assumptions/Limitations: none
 *
 **********************************************************************/
 DeterminePortStructures(compUnit, allprocesses)
 compilationUnit compUnit;
 SEQProcessEntity allprocesses;
 {
	SEQProcessEntity SPr;  /* seq traversals and values */
	ProcessEntity Aprocess;
	SEQprocessStatement tb;
	processStatement bval;

	Assume((IscompilationUnit(compUnit)),
		"DeterminePortStructures");
	foreachinSEQProcessEntity(allprocesses, SPr, Aprocess) {
	    Aprocess->inv_parent = compUnit;
	    foreachinSEQprocessStatement(Aprocess->syn_body, tb, bval) {
		if (typeof(bval) == KportDefinition)
		    AddPorts(bval.VportDefinition, Aprocess);
	    }
	}
 }

/***********************************************************************
 *
 * Procedure AnalyzeProcess 
 *
 * Purpose: Analyze the process given by process declaration 'pr_decl'.
 *	    Add appropriate information to symbol table process 'pr'.
 *
 * Algorithm: Add the assertions and instances.
 *	      Find the invariant structure for the process. If structure
 *	      	doesn't exist, create a new structure.
 *	      Check that the invariant structure contains all types in
 *		port structures of the process. Add each type that 
 *		doesn't exist.
 *	      Check that are port names unique from nonterminal names.
 *	      Propogate up non-direct descendants into parent 
 *	      	classes for the invariant structure of the process.
 *	      Add restrictions and target language information.
 *	      Add assertion types to invariant structure
 *	      Add assertion types for the process		
 *
 * Errors Checked For: Non-existent invariant structure.
 *		       Port names which are the same as nonterminal names.
 *
 * Assumptions/Limitations: ports for processes are already determined
 *
 **********************************************************************/


AnalyzeProcess(Aprocess)
ProcessEntity Aprocess;		/* the process */
{

	SEQprocessStatement tb;		/* set/seq traversals and values */
	processStatement bval;	
	StructureEntity stfound;	/* the invariant structure found */
	StructureEntity newstructure;	/* new invariant structure */
	ProcessRef pref;		/* for de-referencing */
	ProcessEntity pfound;		/* process found */

	Assume((IsProcessEntity(Aprocess)), "AnalyzeProcess");

	/* create a new invariant structure */
	newstructure = NStructureEntity;
	newstructure->inv_parent = Aprocess->inv_parent;
	newstructure->lex_beginpos = Aprocess->lex_endpos;
	newstructure->lex_endpos = Aprocess->lex_endpos;
	newstructure->lex_name = NewString(IDLDEFAULTINVARIANTNAME);
	Aprocess->sem_invariant = newstructure;

	/* add the prelude types to the invariant */
	AddPreludeTypes(Aprocess->sem_invariant);

	/* if process is refined, find parent process in compUnit */
	if (typeof(Aprocess->syn_refines)==KProcessRef) {
	    pref = Aprocess->syn_refines.VProcessRef;
	    if (EntityIsError(pref)) {
		Warning0(400, pref->lex_namepos);
	    }
	    else {
		pfound = GetProcessEntity(pref);
		Aprocess->sem_target = pfound->sem_target;
	    }
	}
	

	/* copy the ports, target language, assertions, 
	   and instances for the process */

        Aprocess->sem_target.VVoid = NVoid;
	foreachinSEQprocessStatement(Aprocess->syn_body, tb, bval) {
	    switch (typeof(bval)) {

		case KAssertion:
		    AddAssertion(ProcessEntityToStructureOrProcess(Aprocess), bval.VAssertion);
		    break;

		case Kcyclicdef:
		case Knoncyclicdef:
		case KPrivateDefInstance:
		    AddInstance(ProcessEntityToStructureOrProcess(Aprocess), bval.VDefInstance);
		    break;
			
		case KtargetStmt:
		    AddTarget(Aprocess, bval.VtargetStmt);
		    break;
		case KportAssociation:
		case KtypeRestriction:	
		case KattributeRestriction:	
		case KportDefinition:
			break;		
	
		default:
			Fatal1(1001, 0, "AnalyzeProcess");
			break;
	    }
	}



	if (typeof(Aprocess->syn_invariant)==KStructureRef) {
	    if (FindStructure(Aprocess->syn_invariant.VStructureRef->lex_name, 
				Aprocess->inv_parent, &stfound) == NOTFOUND){
		    Recoverable0(401, Aprocess->syn_invariant.VStructureRef
				    ->lex_namepos);
		    SetErrorEntity(Aprocess->syn_invariant.VStructureRef);
	    }
	    else SetStructureEntity(Aprocess->syn_invariant.VStructureRef, 
				    stfound);
	}

	/* check that every type defined in each structure associated
	 * with a port exists in the invariant structure. If it doesn't, add
	 * it to the invariant
	 */

	CheckInvariant(Aprocess);

	


	/* add port associations and restrictions */

	foreachinSEQprocessStatement(Aprocess->syn_body, tb, bval) {
	    switch (typeof(bval)) {

		case KportAssociation:
		    AddPortAssoc(Aprocess, bval.VportAssociation);
		    break;
		    
		case KtypeRestriction:
		case KattributeRestriction:
		    AddRestrictions(bval.Vrestriction, Aprocess);
		    break;

		default: 
		    break;
	    }
	}

	/* add the assertion types for the process */
	AddStPrAssertionTypes(ProcessEntityToStructureOrProcess(Aprocess));
}

