/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlbrowse/windows.c 
*				 					* 
*   Copyright (C) 1991 Marco Chierotti
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *
 *  Description: module to interact with the user through the terminal. 
 *               For more details refer to the header of its 
 *               corresponding .h file.
 */


/***** INCLUDE FILES *****/


#include <curses.h>
#include <string.h>
#include "global.h"
#include "windows.h"

void wdelch();
void winsch();

extern char *this_program_name;


/*
 * definitions for key mapping
 */
#define BELL_STRING "\007"           /* string to ring the term. bell */
#define UP_CH_1      'i'
#define UP_CH_2      'p'
#define DOWN_CH_1    'm'
#define DOWN_CH_2    'n'
#define SCR_UP_CH    'u'
#define SCR_DOWN_CH  'd'
#define LEFT_CH_1    'j'
#define LEFT_CH_2    'l'
#define RIGHT_CH_1   'k'
#define RIGHT_CH_2   'r'
#define ESCAPE_CH    27      /* decimal ASCII value for <esc> char */
#define CR_CH        13      /* decimal ASCII value for <cr> char */
#define BACK_SP_CH   '\b'    /* backspace character */
#define CONTROL_C_CH 0x03    /* hex value for ^C character */
#define HELP_CH      'h'
#define C_FILE_CH    'c'
#define F_NAME_CH    'f'
#define SETUP_CH     's'
#define VERBOSE_CH   'v'
#define QUIT_CH      'q'

/*
 * definition of virtual window
 */
typedef struct {
    int top_row;     /* top row currently visible */
    int last_row;    /* last row not empty */
    int curr_row;    /* currently selected row */
    line_type body[MAX_LINE];    /* array of lines */
    } virtual_window_type;
/*
 * definition of virtual screen
 */
#define MAX_WINDOWS 4 /* max number of windows on the screen */
#define MAX_SCREEN_LINE 200 /* max buffer for typed commands */
/*
 * define the minimum value for screen dimensions
 */
#define MIN_COLS  10
#define MIN_LINES  6

typedef struct {
    char command_line[MAX_SCREEN_LINE+1]; /* buffer for typed in commands */
    char menu_line[MAX_COL+1];            /* menu with commands */
    int  col1;                            /* left/center window boundary */
    int  col2;                            /* center/right window boundary */
    int  line;                            /* top/bottom windows boundary */
    virtual_window_type win[MAX_WINDOWS]; /* left, center, right and bottom */
    w_number_type curr_window;            /* currently selected window */
    boolean zoomed;                       /* flag for future zoom mode */
    } virtual_screen_type;
/*
 * private variables
 */
static virtual_screen_type virt_screen; /* virtual screen */
/*
 * private variables for curses windows
 */
static WINDOW *browse_screen; /* screen for browse */
static WINDOW *curse_win[MAX_WINDOWS]; /* left, center, right and bottom */
static WINDOW *menu_w;        /* menu with commands */


/***** FUNCTION PROTOTYPES *****/

static void browse_win_init();
static void browse_win_redraw();
static int last_visible_line();
static int width();
static void select_item();
static void disselect_item();
static void my_mvwaddstr();
static void write_message();

/***** CODE *****/


 
/*****************************************************************************/
void init_w_system()
    {
    (void)initscr();
    crmode();
    noecho();
    nonl();          /* for faster redrawing */

    } /* init_w_system */


/*****************************************************************************/
boolean init_browse()
    {
    w_number_type window; /* index to loop through windows */

    /*
     * check if the terminal line number and column number is adequate
     */
    if ( ( COLS < MIN_COLS ) || ( LINES < MIN_LINES ) ) {
        quit();
        return(FALSE);
        }
    /*
     * initialize the virtual windows
     */
    (void)strcpy(virt_screen.command_line,"command line");
    (void)strcpy(virt_screen.menu_line,
"idlbrowse: Help Quit Prev Next Right Left Down Up Candle File Setup                         "
                                          );
    /*
     * setup to variable length for default values
     */
    virt_screen.col1 = (COLS - 2)/3;
    virt_screen.col2 = ((COLS - 2)/3) * 2 + 1;
    virt_screen.line = ( 7 * LINES ) / 18;
    /*
     * for the moment initialize to dummy arguments
     */
    for (window = CENTER_W; window <= BOTTOM_W; window++) {
        virt_screen.win[window].top_row = -1;
        virt_screen.win[window].last_row = -1;
        virt_screen.win[window].curr_row = -1; 
        }

        virt_screen.curr_window = LEFT_W;
        virt_screen.zoomed = FALSE;
/*@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
        virt_screen.win[LEFT_W].top_row = 0;
        virt_screen.win[LEFT_W].curr_row = 0;
        virt_screen.win[LEFT_W].last_row = MAX_LINE-1;

        for (i = 0 ; i <= virt_screen.win[LEFT_W].last_row; i++) {
            (void)sprintf(virt_screen.win[LEFT_W].body[i], 
                                          "P %d window %d",i,LEFT_W);
            }
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@*/
    /*
     * initialize the curse windows
     */
    browse_win_init();
    /*
     * write to the screen
     */
    overwrite(browse_screen,stdscr);  
    move(0, 0);
    refresh();
    return (TRUE);

    } /* init_browse */


/*****************************************************************************/
command_type get_command()
    {
    command_type key; /* key pressed in the keyboard */

    switch(wgetch(curse_win[virt_screen.curr_window])) {
        case UP_CH_1:
        case UP_CH_2:
            key = UP;
            break;
        case DOWN_CH_1:
        case DOWN_CH_2:
            key =DOWN;
            break;
        case SCR_UP_CH:
            key = SCR_UP;
            break;
        case SCR_DOWN_CH:
            key = SCR_DOWN;
            break;
        case LEFT_CH_1:
        case LEFT_CH_2:
            key = LEFT;
            break;
        case RIGHT_CH_1:
        case RIGHT_CH_2:
            key = RIGHT;
            break;
        case ESCAPE_CH:
            key = ESCAPE;
            break;
        case HELP_CH:
            key = HELP;
            break;
        case C_FILE_CH:
            key = C_FILE;
            break;
        case F_NAME_CH:
            key = F_NAME;
            break;
        case SETUP_CH:
            key = SETUP;
            break;
        case QUIT_CH:
        case CONTROL_C_CH:
            key = QUIT;
            break;
	case VERBOSE_CH:
	    key = VERBOSE;
	    break;
        default:
            key = NOTKNOWN;
            break;
        } /* switch */

    return(key);
    } /* get_command */


/*****************************************************************************/
boolean select_prev_item()
    {
    w_number_type curr_window;
    int curr_row, top_row;
    /*
     * make copies for brevity in reference
     */
    curr_window = virt_screen.curr_window;
    curr_row = virt_screen.win[curr_window].curr_row;
    top_row = virt_screen.win[curr_window].top_row;
    /*
     * if scroll is impossible, no further action
     */
    if ( (curr_row <= 0) || (curr_row < top_row) ) 
        return(FALSE);
    /*
     * if needed, scroll down one line:
     */
    (void)wmove(curse_win[curr_window], curr_row - top_row, 0); 
    if (curr_row == top_row) {
        /*
         * delete the bottom line in the window
         */
        (void)wmove(curse_win[curr_window], curr_row - top_row + 
                           last_visible_line(curr_window), 0); 
        (void)wdeleteln(curse_win[curr_window]);
        /*
         * insert a blank line and overwrite on it
         */
        (void)wmove(curse_win[curr_window], curr_row - top_row, 0); 
        (void)winsertln(curse_win[curr_window]);
        my_mvwaddstr(curr_window, curr_row - top_row -1, 0,
                                virt_screen.win[curr_window].body[curr_row]);
        top_row--;
        (void)wmove(curse_win[curr_window], curr_row - top_row, 0); 
        }
    /*
     * disselect current item and select previous
     */
    disselect_item(curr_window,curr_row,top_row);
    curr_row--;
    select_item(curr_window,curr_row,top_row);
    /*
     * update global values for current window
     */
    virt_screen.win[curr_window].curr_row = curr_row;
    virt_screen.win[curr_window].top_row = top_row;

    (void)wrefresh(browse_screen);   
    (void)wrefresh(curse_win[curr_window]);

    return(TRUE);

    } /* select_prev_item */



/*****************************************************************************/
boolean select_next_item()
    {
    w_number_type curr_window;
    int curr_row, top_row, last_row, last_visible;
    /*
     * make copies for brevity in reference
     */
    curr_window = virt_screen.curr_window;
    curr_row = virt_screen.win[curr_window].curr_row;
    top_row = virt_screen.win[curr_window].top_row;
    last_row = virt_screen.win[curr_window].last_row;
    last_visible = last_visible_line(curr_window);
    /*
     * if scroll is impossible, no further action
     */
    if ( (curr_row >= last_row) ||
                         (curr_row > (top_row + last_visible)) ) 
        return(FALSE);
    /*
     * if needed, scroll up one line:
     */
    (void)wmove(curse_win[curr_window], curr_row - top_row, 0); 
    if (curr_row == (top_row + last_visible)) {
        /*
         * delete the top line in the window
         */
        (void)wmove(curse_win[curr_window], 0, 0 );
        (void)wdeleteln(curse_win[curr_window]);
        /*
         * insert a blank line at the bottom and overwrite on it
         */
        (void)wmove(curse_win[curr_window], last_visible, 0); 
        (void)winsertln(curse_win[curr_window]);
        my_mvwaddstr(curr_window, last_visible, 0,
                           virt_screen.win[curr_window].body[curr_row +1]);
        top_row++;
        (void)wmove(curse_win[curr_window], curr_row - top_row, 0); 
        }
    /*
     * disselect current item and select next 
     */
    disselect_item(curr_window,curr_row,top_row);
    curr_row++;
    select_item(curr_window,curr_row,top_row);

    /*
     * update global values for current window
     */
    virt_screen.win[curr_window].curr_row = curr_row;
    virt_screen.win[curr_window].top_row = top_row;

    (void)wrefresh(browse_screen);   
    (void)wrefresh(curse_win[curr_window]);

    return(TRUE);


    } /* select_next_item */

 
/*****************************************************************************/
boolean select_top_element(window)
    {
    if ( (window < LEFT_W) || (window > BOTTOM_W) )
        return(FALSE);      /* wrong window */ 

    if ( virt_screen.win[window].last_row < 0 )
        return(FALSE);    /* empty window */
    /*
     * reset parameters and redraw
     */
    virt_screen.win[window].top_row = 0;
    virt_screen.win[window].curr_row = 0;

    browse_win_redraw(window);

    return(TRUE);

    } /* select_top_element */


/*****************************************************************************/
boolean select_prev_scroll()
    {
    int curr_row, top_row, last_row;
    w_number_type curr_window;
    int n_scroll;                 /* # of lines to scroll */
    int new_top_row;              /* new top of window */
    int last_visible;              /* index of last visible row in window */
    int i;
    /*
     * copy into local variables for brevity
     */
    curr_window = virt_screen.curr_window;
    top_row = virt_screen.win[curr_window].top_row;
    curr_row = virt_screen.win[curr_window].curr_row;
    last_row = virt_screen.win[curr_window].last_row;
    last_visible = last_visible_line(curr_window);  
    /*
     * check if empty window or already at the top
     */
    if ( (last_row < 0) || (top_row <= 0) )
	return(FALSE);
    /*
     * calculate new top and # of lines to scroll
     */
    n_scroll = (last_visible +1)/2;     /* scroll half a window */
    if ( (top_row - n_scroll) >= 0 )
	new_top_row = top_row - n_scroll;  /* regular scroll */
    else {
	new_top_row = 0;    /* cannot go beyond the top */
	n_scroll = top_row;
	}
    /*
     * move current row, if needed
     */
    if ( (curr_row > (new_top_row + last_visible)) ||
				     (curr_row < new_top_row) ) {
	disselect_item(curr_window,curr_row, top_row);
	curr_row = new_top_row + last_visible;
	select_item(curr_window, curr_row, top_row);
	} /* if */
    /*
     * scroll the curse window
     */
    for (i = 1; i <= n_scroll; i++) {
	(void)wmove(curse_win[curr_window], last_visible, 0);
	(void)wdeleteln(curse_win[curr_window]);   /* last line deleted */
	(void)wmove(curse_win[curr_window], 0, 0);
	(void)winsertln(curse_win[curr_window]);
        my_mvwaddstr(curr_window, 0, 0,
		      virt_screen.win[curr_window].body[top_row - i]);

	} /* for */

    top_row = new_top_row;
    /*
     * restore modified global values
     */
    virt_screen.win[curr_window].top_row = top_row;
    virt_screen.win[curr_window].curr_row = curr_row;

    (void)wrefresh(browse_screen);
    (void)wrefresh(curse_win[curr_window]);

    return(TRUE);

    } /* select_prev_scroll */


 
/*****************************************************************************/
boolean select_next_scroll()
    {
    int curr_row, top_row, last_row;
    w_number_type curr_window;
    int n_scroll;                 /* # of lines to scroll */
    int new_top_row;              /* new top of window */
    int last_visible;              /* index of last visible row in window */
    int i;
    /*
     * copy into local variables for brevity
     */
    curr_window = virt_screen.curr_window;
    top_row = virt_screen.win[curr_window].top_row;
    curr_row = virt_screen.win[curr_window].curr_row;
    last_row = virt_screen.win[curr_window].last_row;
    last_visible = last_visible_line(curr_window);  
    /*
     * check if empty window or already at the bottom 
     */
    if ( (last_row < 0) || ((top_row + last_visible) >= last_row) )
	return(FALSE);
    /*
     * calculate new top and # of lines to scroll
     */
    n_scroll = (last_visible +1)/2;     /* scroll half a window */
    if ( (top_row + n_scroll + last_visible) <= last_row)
	new_top_row = top_row + n_scroll;  /* regular scroll */
    else {
        n_scroll = last_row - (top_row + last_visible);
	new_top_row = top_row + n_scroll;  /* cannot go beyond the bottom */
	}
    /*
     * move current row, if needed
     */
    if ( (top_row <= curr_row) && (curr_row < new_top_row )) {
	disselect_item(curr_window,curr_row, top_row);
	curr_row = new_top_row;
	select_item(curr_window, curr_row, top_row);
	} /* if */
    /*
     * scroll the curse window
     */
    for (i = 1; i <= n_scroll; i++) {
	(void)wmove(curse_win[curr_window], 0, 0);
	(void)wdeleteln(curse_win[curr_window]);   /* top line deleted */
	(void)wmove(curse_win[curr_window], last_visible, 0);
	(void)winsertln(curse_win[curr_window]);
        my_mvwaddstr(curr_window, last_visible, 0,
             virt_screen.win[curr_window].body[top_row + last_visible + i]);
	} /* for */

    top_row = new_top_row;
    /*
     * restore modified global values
     */
    virt_screen.win[curr_window].top_row = top_row;
    virt_screen.win[curr_window].curr_row = curr_row;

    (void)wrefresh(browse_screen);
    (void)wrefresh(curse_win[curr_window]);

    return(TRUE);

    } /* select_next_scroll */


 
/*****************************************************************************/
void clear_window(window)
    w_number_type window;
    {
    if ( ( window < LEFT_W) || (window > BOTTOM_W) )
        return;   /* wrong window */
    /*
     * reset parameters for the virtual window
     */
    virt_screen.win[window].last_row = -1;   /* empty */
    virt_screen.win[window].curr_row = -1;
    virt_screen.win[window].top_row = -1;   
    
    (void)wclear(curse_win[window]);
    (void)wrefresh(browse_screen);
    (void)wrefresh(curse_win[window]);

    } /* clear_window */


 
/*****************************************************************************/
boolean move_to_next_window()
    {
    if ( virt_screen.curr_window >= BOTTOM_W )
        return(FALSE);         /* cannot move further */
    else if (virt_screen.win[virt_screen.curr_window + 1].last_row < 0)
        return(FALSE);   /* next window is empty */
    else {
        virt_screen.curr_window++;
        return(TRUE);
        }

    } /* move_to_next_window */


 
/*****************************************************************************/
boolean move_to_prev_window()
    {
        virt_screen.curr_window--;
        (void)wmove(curse_win[virt_screen.curr_window],
              virt_screen.win[virt_screen.curr_window].curr_row -
              virt_screen.win[virt_screen.curr_window].top_row, 0);
        refresh();
        return(TRUE);

    } /* move_to_prev_window */


 
/*****************************************************************************/
w_number_type current_window()
    {
    if ( (virt_screen.curr_window >= LEFT_W) && 
                          (virt_screen.curr_window <= BOTTOM_W) )
        return(virt_screen.curr_window);
    else
        return(WRONG_W);

    } /* current_window */


 
/*****************************************************************************/
w_number_type next_window(window)
    w_number_type window;
    {
    if ( window < BOTTOM_W)
        return(window +1);
    else
        return(WRONG_W);

    } /* next_window */


 
/*****************************************************************************/
w_number_type prev_window(window)
    w_number_type window;
    {
    if ( window > LEFT_W)
        return(window -1);
    else
        return(WRONG_W);

    } /*prev_window */


/*****************************************************************************/
int give_selected_item(item_name, window)
    char item_name[];
    w_number_type window;
    {
    if ( (window < LEFT_W) || (window > BOTTOM_W) )
        return (-1); /* wrong window */

    if ( (virt_screen.win[window].curr_row < 0) ||
         (virt_screen.win[window].curr_row > virt_screen.win[window].last_row))
        return(-1);  /* wrong selection or no selection */
    /*
     * copy to the output item_name
     */
    (void)strcpy(item_name,
         virt_screen.win[window].body[virt_screen.win[window].curr_row]);
    if (strlen(item_name) == 0)
        return(-1); /* empty line */

    return(virt_screen.win[window].curr_row);


    } /* give_selected_item */

 
/*****************************************************************************/
boolean selected_item_is_structure(item_name)
    char item_name[];
    {
    if ( (strlen(item_name) < 2) || (strlen(item_name) > MAX_COL) )
        return(FALSE);
    if ( ( item_name[0] == 'S') && ( item_name[1] == ' ') )
        return(TRUE);

    return(FALSE);

    } /* selected_item_is_a_structure */



/*****************************************************************************/
boolean selected_item_is_defaultprocessinvariant(item_name)
    char item_name[];
    {
    if ( (strlen(item_name) < 2) || (strlen(item_name) > MAX_COL) )
        return(FALSE);
    if ( ( item_name[0] == 'I') && ( item_name[1] == ' ') )
        return(TRUE);

    return(FALSE);

    } /* selected_item_is_a_defaultprocessinvariant */


 
/*****************************************************************************/
boolean selected_item_is_process(item_name)
    char item_name[];
    {
    if ( (strlen(item_name) < 2) || (strlen(item_name) > MAX_COL) )
        return(FALSE);
    if ( ( item_name[0] == 'P') && ( item_name[1] == ' ') )
        return(TRUE);

    return(FALSE);

    } /* selected_item_is_a_process */


 
/*****************************************************************************/
void write_list_to_window(list, window)
    list_type *list;
    w_number_type window;
    {
    int i;
    /*
     *  if window out of range, do nothing
     */
    if ( (window < LEFT_W) || (window > BOTTOM_W) )
        return;    
    /*
     * copy the list into the virtual window
     */
    for (i = 0; i <= (*list).last_row; i++)
        (void)strcpy(virt_screen.win[window].body[i],(*list).body[i]);

    virt_screen.win[window].last_row = (*list).last_row;
    virt_screen.win[window].top_row = 0;
    virt_screen.win[window].curr_row = -1; /* no selected item */
    /*
     * copy virtual window onto curse screen and refresh it
     */
    browse_win_redraw(window);

    } /* write_list_to_window */


 
/*****************************************************************************/
void restore_browse()
    {
    w_number_type window;

    overwrite(browse_screen, stdscr);
    for (window = LEFT_W; window <= BOTTOM_W; window++)
        overwrite(curse_win[window], stdscr);
    /*
     * restore the cursor position in current window and current item
     */
    (void)wmove( curse_win[virt_screen.curr_window],
           virt_screen.win[virt_screen.curr_window].curr_row -
           virt_screen.win[virt_screen.curr_window].top_row, 0);

    (void)wrefresh(browse_screen);
    for (window = LEFT_W; window <= BOTTOM_W; window++)
        (void)wrefresh(curse_win[window]);
    refresh();

    } /* restore_browse */


 
/*****************************************************************************/
boolean ask_file_name(file_name)
    char file_name[];
    {
    line_type buffer;
    char ch ;
    int i;
    int last, low_limit, first_visible;
    int last_col;
    int ntoken;
    line_type token1, token2;
    /*
     * set the prompt and parameters for printing
     */
    string_copy(buffer,"New Candle filename: ",MAX_COL);

    low_limit = strlen(buffer) -1;   /* last char position for prompt msg */
    last_col = COLS - 1;             /* last visible column on the screen */
    last = low_limit;                /* last char i the buffer */
    first_visible = 0;               /* first visible char on screen */

    /*
     * get the erase char from curse
     */

    (void)wclear(menu_w);

    (void)wmove(menu_w, 0, 0);
    /*
     * print shifting if string too long to fit
     */
    for (i = 0; i<= low_limit; i++) {
        if (i < last_col) 
            (void)waddch(menu_w,buffer[i]);
        else {
            (void)wmove(menu_w,0,0);
            wdelch(menu_w);
            (void)wmove(menu_w,0,last_col-1);
            (void)waddch(menu_w,buffer[i]);
            first_visible++;
            (void)wrefresh(menu_w);
            }
        } /* for */
    (void)wrefresh(menu_w);
    /*
     * now ready to get input from keyboard
     */
    while ( (ch = wgetch(menu_w)) != CR_CH) {
        /*
         * if an erase char, backup
         */
        if  (ch == BACK_SP_CH) {
            if (last > low_limit) {   /* something to erase */
                (void)wmove(menu_w,0,last);
                wdelch(menu_w);
                if (last < last_col) {            /* no need for shifting */
                    (void)wmove(menu_w,0,last);
                    wdelch(menu_w);
                    }
                else {                            /* need shifting */
                    (void)wmove(menu_w, 0, last_col);
                    wdelch(menu_w);
                    (void)wmove(menu_w,0,0);
                    first_visible--;
                    winsch(menu_w, buffer[first_visible]);        
                    (void)wmove(menu_w, 0, last_col);
                    }
                last--;
                (void)wrefresh(menu_w);
                }
            else {
                 (void)printf("\007");
                 }
            }
        /*
         * if everything else, add to buffer
         */
        else if (last < (MAX_COL-1)) {
            last++;
            buffer[last] = ch;
            if ( last < last_col) {
                (void)waddch(menu_w,buffer[last]);
                (void)wrefresh(menu_w);
                }
            else {
                (void)wmove(menu_w,0,0);
                wdelch(menu_w);
                (void)wmove(menu_w,0,last_col-1);
                (void)waddch(menu_w,buffer[last]);
                (void)wrefresh(menu_w);
                first_visible++;
                }
            } /* if last < MAX_COL */

        } /* while */

    /*
     * now get the file name out of the buffer
     */
    buffer[last + 1] = '\0'; 
    string_copy(file_name, buffer + low_limit + 1 , MAX_COL);
    /*
     * check if the name is acceptable
     */
    if (strlen(file_name) == 0)
        return(FALSE);
    /*
     * eliminate leading and trailing blanks, check for blanks in between
     */
    ntoken = sscanf(file_name,"%s%s",token1, token2);
    if ( (ntoken < 1) || (ntoken > 1) )
        return(FALSE);
    /*
     * now restore the refined name
     */ 
    string_copy(file_name,token1, MAX_COL);

    return(TRUE);


    } /* ask_file_name */



/*****************************************************************************/
void send_file_message(error_message)
    char error_message[];
    {
    write_message(error_message);

    } /* send_file_message */


 
/*****************************************************************************/
void display_file_name(file_name)
    char file_name[];
    {
    line_type buffer;
    /*
     * set the string to print and parameters for printing
     */
    string_concat(buffer,"File: ", 
              file_name, MAX_COL);

    write_message(buffer);

    } /* display_file_name */

void display_verbose_info()
{
 line_type buffer;
 char revision_str[30], hostname[30];
 void get_revision_number();

 get_revision_number(revision_str);
 (void)gethostname(hostname,30);
 (void)sprintf(buffer,"%s, Version %s  Process %d on machine \"%s\"",
	this_program_name, revision_str, getpid(), hostname);
 write_message(buffer);
}


/*****************************************************************************/
void display_menu_line()
    {
    line_type buffer;
    /*
     * get from virtual screen the bottom line with the menu
     */
    string_copy(buffer, virt_screen.menu_line, COLS - 1);
    (void)wstandout(menu_w);  /* reverse video on */
    mvwaddstr(menu_w, 0, 0, buffer);   
    (void)wstandend(menu_w);   /* reverse video off */
    (void)wrefresh(menu_w);
    
    } /* display_menu_line */
 
/*****************************************************************************/
boolean ask_new_setup(col1,col2,line)
    int *col1;
    int *col2;
    int *line;
    {
    line_type buffer;
    char ch ;
    int i;
    int last, low_limit, first_visible;
    int last_col;
    int ntoken,itoken, ich;
    line_type numbers;
    line_type token[4];

    /*
     * set the default values for col1, col2 and line. If error occurs
     * they will be the new values returned
     */
    (*col1) = (COLS - 2)/3;
    (*col2) = ((COLS - 2)/3) * 2 + 1;
    (*line) = ( 7 * LINES ) / 18;

    /*
     * set the prompt and parameters for printing
     */
    string_copy(buffer,"idlbrowse Setup <col1> <col2> <line> <cr>: ",MAX_COL);

    low_limit = strlen(buffer) -1;   /* last char position for prompt msg */
    last_col = COLS - 1;             /* last visible column on the screen */
    last = low_limit;                /* last char i the buffer */
    first_visible = 0;               /* first visible char on screen */

    /*
     * get the erase char from curse
     */

    (void)wclear(menu_w);

    (void)wmove(menu_w, 0, 0);
    /*
     * print shifting if string too long to fit
     */
    for (i = 0; i<= low_limit; i++) {
        if (i < last_col) 
            (void)waddch(menu_w,buffer[i]);
        else {
            (void)wmove(menu_w,0,0);
            wdelch(menu_w);
            (void)wmove(menu_w,0,last_col-1);
            (void)waddch(menu_w,buffer[i]);
            first_visible++;
            (void)wrefresh(menu_w);
            }
        } /* for */
    (void)wrefresh(menu_w);
    /*
     * now ready to get input from keyboard
     */
    while ( (ch = wgetch(menu_w)) != CR_CH) {
        /*
         * if an erase char, backup
         */
        if  (ch == BACK_SP_CH) {
            if (last > low_limit) {   /* something to erase */
                (void)wmove(menu_w,0,last);
                wdelch(menu_w);
                if (last < last_col) {            /* no need for shifting */
                    (void)wmove(menu_w,0,last);
                    wdelch(menu_w);
                    }
                else {                            /* need shifting */
                    (void)wmove(menu_w, 0, last_col);
                    wdelch(menu_w);
                    (void)wmove(menu_w,0,0);
                    first_visible--;
                    winsch(menu_w, buffer[first_visible]);        
                    (void)wmove(menu_w, 0, last_col);
                    }
                last--;
                (void)wrefresh(menu_w);
                }
            else {
                 (void)printf("\007");
                 }
            }
        /*
         * if everything else, add to buffer
         */
        else if (last < (MAX_COL-1)) {
            last++;
            buffer[last] = ch;
            if ( last < last_col) {
                (void)waddch(menu_w,buffer[last]);
                (void)wrefresh(menu_w);
                }
            else {
                (void)wmove(menu_w,0,0);
                wdelch(menu_w);
                (void)wmove(menu_w,0,last_col-1);
                (void)waddch(menu_w,buffer[last]);
                (void)wrefresh(menu_w);
                first_visible++;
                }
            } /* if last < MAX_COL */

        } /* while */

    /*
     * now get the useful info out of the buffer
     */
    buffer[last + 1] = '\0'; 
    string_copy(numbers, buffer + low_limit + 1 , MAX_COL);
    /*
     * check if the name is acceptable
     */
    if (strlen(numbers) == 0)
        return(FALSE);
    /*
     * must extract 3 blocks of non blank characters
     */
    ntoken = sscanf(numbers,"%s%s%s%s",token[0],token[1],token[2],token[3]);
    if ( ntoken != 3 )
        return(FALSE);
    /*
     * for each of them must have digits only ('0'.....'9')
     */
    for (itoken = 0; itoken <ntoken; itoken++) 
        for (ich = 0; ich < strlen(token[itoken]); ich++)
            if ( (token[itoken][ich] < '0') || (token[itoken][ich] > '9') )
                return(FALSE);
    /*
     * now convert into integers
     */
    if ( sscanf(token[0],"%d",col1) != 1)
       return(FALSE);
    if ( sscanf(token[1],"%d",col2) != 1)
       return(FALSE);
    if ( sscanf(token[2],"%d",line) != 1)
       return(FALSE);
    /*
     * check the correct range
     */
    if ( (*col1 < 4) || ( (*col1) > (COLS - 1)))  
        return(FALSE);
    if ( (*col2 < 4) || ( (*col2) > (COLS - 1)))  
        return(FALSE);
    if ( (*col2) < ( (*col1) + 1))
        return(FALSE);
    if ( ((*line) < 2) || ( (*line) > (LINES -3)) )
        return(FALSE);

    (*col1)--;
    (*col2)--;
    (*line)--;

    return(TRUE);

    } /* ask_new_setup */

 
/*****************************************************************************/
void setup(col1,col2,line)
    int col1;
    int col2;
    int line;
    {
    w_number_type window;

    /*
     * update the window boundaries
     */
    virt_screen.col1 = col1;
    virt_screen.col2 = col2;
    virt_screen.line = line;
    /*
     * check if the new window length could affect the cursor. If so
     * calculate the new top window for a scroll
     */
    for ( window = LEFT_W; window <= BOTTOM_W; window++)
      if ( virt_screen.win[window].curr_row > 
              (virt_screen.win[window].top_row + last_visible_line(window)) ) {
          virt_screen.win[window].top_row = virt_screen.win[window].curr_row
                                     - last_visible_line(window);
          }

    /*
     * dispose the browse window and its subwindows. The help window
     * is not affected.
     */
    for (window = LEFT_W; window <= BOTTOM_W; window++) {
        (void)wclear(curse_win[window]);
        delwin(curse_win[window]);
        }
    (void)wclear(menu_w);
    (void)wclear(browse_screen);
    delwin(menu_w);
    delwin(browse_screen);
    /*
     * now initialize a new set of windows
     */
    browse_win_init();
    /*
     * now redraw the subwindows with their contents
     */
    for (window = LEFT_W; window <= BOTTOM_W; window++)
        browse_win_redraw(window);


    } /* setup */


 
/*****************************************************************************/
void refresh_screen()
    {
    (void)wrefresh(curscr);

    } /* refresh_screen */

 
/*****************************************************************************/
void quit()
    {
/*    move(LINES - 1, COLS - 1);
    refresh();
*/
    endwin();
    (void)printf("\n");

    } /* quit */


 
/*****************************************************************************/
void error_message()
    {
    (void)printf("\007");
    } /* error_message */


/***** PRIVATE ROUTINES  *****/


/*
 *  Routine: browse_win_init
 *
 * Description: initializes windows and draws boundaries between windows
 *              It doesn't affect the virtual windows.
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects: it initializes the screen 
 */
static void browse_win_init()
    {
    int i;   /* index to draw boundaries */
    int col1, col2, line ; /* local copies of global data, for sorthand */

    /*
     * copy windows boundaries into local copies
     */
    col1 = virt_screen.col1;
    col2 = virt_screen.col2;
    line = virt_screen.line;
    /*
     * create the screen window and the menu/command line.
     */
    browse_screen = newwin(LINES, COLS, 0,0);  
    menu_w = subwin(browse_screen, 1, COLS, LINES - 1, 0);
    /*
     * create the four working subwindows
     */
    curse_win[LEFT_W] = subwin(browse_screen, line, col1, 0,0);
    curse_win[CENTER_W] = subwin(browse_screen, line, col2 - col1 - 1, 0,
                                                                   col1 + 1);
    curse_win[RIGHT_W] = subwin(browse_screen, line, COLS - col2 - 1, 0, 
                                                                   col2 + 1);
    curse_win[BOTTOM_W] = subwin(browse_screen, LINES - 3 - line, COLS, 
                                                             line + 1, 0);
    /*
     * draw vertical boundaries
     */
    (void)wstandout(browse_screen);  /* reverse video on */
    for (i = 0; i < line; i++) {
        mvwaddch(browse_screen, i, col1, '|');
        mvwaddch(browse_screen, i, col2, '|');
        }
    /*
     * now draw horizontal boundaries
     */
    for (i = 0; i < COLS; i++) {
        mvwaddch(browse_screen, line, i, '-');
        mvwaddch(browse_screen, LINES - 2, i, '-');
        }
    (void)wstandend(browse_screen);   /* reverse video off */

    display_menu_line();
    /*
     * to test only, load with buffers
     */
/*@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
    for( window = LEFT_W; window <= BOTTOM_W; window++) {
        for(i = 0; i <= virt_screen.win[window].last_row; i++) {
            my_mvwaddstr(window,i,0,
              virt_screen.win[window].body[i+ virt_screen.win[window].top_row]);
            if ( virt_screen.win[window].curr_row == i)
                select_item(window, virt_screen.win[window].curr_row,
                                    virt_screen.win[window].top_row);
            }
        }
    (void)wmove(curse_win[window], virt_screen.win[window].curr_row -
                             virt_screen.win[window].top_row, 0); 
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@*/

    } /* browse_win_init */


/*
 *  Routine: browse_win_redraw
 *
 * Description: redraws the virtual windows onto the curse windows.
 *              It doesn't affect virtual windows.
 *
 *  Arguments: window -- (IN) window to consider
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void browse_win_redraw(window)
    w_number_type window;
    {
    int i, n_last;
    int n_first;
    /*
     * no action if wrong window
     */
    if ( (window < LEFT_W) || (window > BOTTOM_W) )
        return;
    /*
     * check if the list is empty
     */
    if (virt_screen.win[window].last_row < 0)
        return;
    /*
     * get the last line to draw onto the screen window
     */
    n_first = virt_screen.win[window].top_row;

    if ( (virt_screen.win[window].last_row - n_first) <= 
                      last_visible_line(window) )
        n_last = virt_screen.win[window].last_row - n_first;
    else
        n_last = last_visible_line(window);
    /*
     * copy onto the screen
     */
    (void)wclear(curse_win[window]);
    for (i = 0; i <= n_last; i++)
        my_mvwaddstr(window, i, 0, virt_screen.win[window].body[n_first + i]);
    /*
     * if some item is selected, draw in in reverse
     */
    if ( (virt_screen.win[window].curr_row >= n_first) &&
         (virt_screen.win[window].curr_row <= 
                            (n_first + last_visible_line(window)) ))
        select_item(window, virt_screen.win[window].curr_row,
                            virt_screen.win[window].top_row);
    /*
     * restore the cursor current position
     */
    (void)wmove(curse_win[virt_screen.curr_window],
          virt_screen.win[window].curr_row - virt_screen.win[window].top_row,
          0);

    (void)wrefresh(browse_screen);
    (void)wrefresh(curse_win[window]);
    (void)wrefresh(curse_win[virt_screen.curr_window]);
    refresh();


    } /* browse_win_redraw */

/*
 *  Routine: last_visible_line
 *
 *  Description: given a window it returns the index of the last
 *               visible line on it. E.g. if the window has 10 lines
 *               it returns 9 ( index of line starts from zero)
 *
 *  Arguments: window --(IN) window to consider
 *
 *  Return Value: number of the line
 *
 *  Side Effects: none
 */
static int last_visible_line(window)
    w_number_type window;
    {
    switch(window) {
        case LEFT_W:
        case CENTER_W:
        case RIGHT_W:
                return(virt_screen.line -1);
        case BOTTOM_W:
                return(LINES -3 - virt_screen.line -1);
        default:
                return(-1);
        }
    /*NOTREACHED*/

    } /* last_visible_line */

/*
 *  Routine: width
 *
 *  Description: it calculates the number of colums available in a
 *               given virtual window
 *
 *  Arguments: window -- (IN) window of the virtual screen
 *
 *  Return Value: number of columns if correct, -1 if error
 *
 *  Side Effects: none
 */
static int width(window)
    w_number_type window;
    {
    switch (window) {
        case LEFT_W:
            return(virt_screen.col1);
        case CENTER_W:
            return(virt_screen.col2 - virt_screen.col1 - 1);
        case RIGHT_W:
            return(COLS - virt_screen.col2 - 1);
        case BOTTOM_W:
            return(COLS);
        default:
            return(-1);
        } /* switch */
    /*NOTREACHED*/


    } /* widht */

/*
 *  Routine: select_item
 *
 *  Description: it rewrites in reverse video the given item
 *               of the given window.
 *
 *  Arguments: curr_window -- (IN) window to consider
 *             curr_row -- (IN) current row in the virtual window
 *             top_row -- (IN) top row in the virtual window 
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void select_item(curr_window,curr_row, top_row)
    w_number_type curr_window;
    int curr_row;
    int top_row;
    {
    (void)wstandout(curse_win[curr_window]);
    my_mvwaddstr(curr_window, curr_row - top_row, 0,
               virt_screen.win[curr_window].body[curr_row]);
    (void)wstandend(curse_win[curr_window]);
    (void)wmove(curse_win[curr_window], curr_row - top_row, 0);

    } /* select_item */


/*
 *  Routine: disselect_item
 *
 *  Description: it rewrites in normal video the given item
 *               of the given window.
 *
 *  Arguments: curr_window -- (IN) window to consider
 *             curr_row -- (IN) current row in the virtual window
 *             top_row -- (IN) top row in the virtual window 
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void disselect_item(curr_window,curr_row, top_row)
    w_number_type curr_window;
    int curr_row;
    int top_row;
    {
    my_mvwaddstr(curr_window, curr_row - top_row, 0,
               virt_screen.win[curr_window].body[curr_row]);
    (void)wmove(curse_win[curr_window], curr_row - top_row, 0);

    } /* disselect_item */

/*
 *  Routine: my_mvwaddstr
 *
 *  Description: it writes a string into the given virtual window. If
 *               the string is too long to fit in the row, it is truncated
 *
 *  Arguments: window -- (IN) window number where to write
 *             y, x -- (IN) curse coordinates in the given window
 *             str -- (IN) string to write in the given y,x coord.
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void my_mvwaddstr(window, y, x, str)
    w_number_type window;
    int y;
    int x;
    char *str;
    {
    int n_chars;  /* number of chars to write */
    WINDOW *win;  /* corresponding curse window */
    line_type buffer;

    if ( (window < LEFT_W) || (window > BOTTOM_W) )
        return;
    /*
     * calculate the number of chars to output
     */
    n_chars = width(window) - x;

    if (n_chars <= 0)
        return;
    else if (n_chars > MAX_COL)
        n_chars = MAX_COL;
    string_copy(buffer,str, n_chars);

    win = curse_win[window];
    (void)wmove(win, y, x);
    (void)wprintw(win,"%-*.*s",n_chars, n_chars, buffer);

    } /* my_mvwaddstr */



/*
 *  Routine: write_message
 *
 *  Description: it writes the given string to the bottom line of the
 *               screen. If the line is too long to fit, the leftmost
 *               characters are erased fro the screen and the line
 *               shifted
 *
 *  Arguments: message -- (IN) string to print
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void write_message(message)
    char message[];
    {
    int i;
    int max_len;
    int last_col;
    /*
     * set the string to print and parameters for printing
     */
    max_len = strlen(message);
    last_col = COLS - 1;

    (void)wclear(menu_w);
    (void)wstandout(menu_w);  /* reverse video on */

    (void)wmove(menu_w, 0, 0);
    /*
     * print shifting if string too long to fit
     */
    for (i = 0; i< max_len; i++) {
        if (i < last_col) 
            (void)waddch(menu_w,message[i]);
        else {
            (void)wmove(menu_w,0,0);
            wdelch(menu_w);
            (void)wmove(menu_w,0,last_col-1);
            (void)waddch(menu_w,message[i]);
            (void)wrefresh(menu_w);
            }
        }

    (void)wstandend(menu_w);   /* reverse video off */
    (void)wrefresh(menu_w);

    } /* write_message */
