/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlbrowse/help.c 
*				 					* 
*   Copyright (C) 1991 Marco Chierotti
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *
 *  Description: module to provide the on-line help. 
 *               For more details refer to the header of its 
 *               corresponding .h file.
 *
 */

/***************************************************************************/

/***** INCLUDE FILES *****/

#include <curses.h>
#include <string.h>
#include "global.h"
#include "help.h"

/***** FUNCTION PROTOTYPES *****/

static void help_win_init();
static void load_help_file();
static void enter_help();
static void page_up();
static void page_down();
static void write_page();

/***** PRIVATE DATA STRUCTURES *****/

/* #define HELP_FILE_NAME "idlbrowse.hlp" ASCII text file to display */
#define MAX_H_COL  81          /* max columns in the help screen */
#define MAX_H_LINE 100         /* max lines in the help file */
#define PAGE_UP_CH 'u'         /* key to scroll up one page */
#define PAGE_DOWN_CH 'd'       /* key to scroll down one page */

typedef char h_line_type[MAX_H_COL+1]; /* line on the screen */
/*
 * declaration of the buffer containing the help text
 */
typedef struct {
    int last;                 /* last row in the current buffer */
    int top;                  /* current first row displayed on screen */
    h_line_type body[MAX_H_LINE]; /* array of lines */
               } h_buffer_type;
/*
 * global variables private to this module
 */
static h_buffer_type h_buffer;   /* buffer where to load the help file */
static WINDOW  *help_window;      /* instance of a curse window for help */


/***** CODE *****/

/*****************************************************************************/
boolean init_help()
    {
    help_win_init();

    load_help_file(HELP_FILE_NAME);

    return (TRUE);
    } /* init_help */

/***************************************************************************/
void help()
    {
    char key; /* key pressed by the user */ 

    enter_help();
    /*
     * interpret the scroll commands, exit if any othe command is given
     */
    do {
        key = wgetch(help_window);
        if ( key == PAGE_UP_CH )
            page_up();
        else if ( key == PAGE_DOWN_CH) 
            page_down();
        }
        while (( key == PAGE_UP_CH) || ( key == PAGE_DOWN_CH) );


    } /* help */


/***** PRIVATE ROUTINES *****/

/*
 *  Routine: help_win_init
 *
 *  Description: it sets up the help curse window 
 *
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects:  none
 *
 */
static void help_win_init()
    {
    /*
     * create the new help window
     */
    help_window = newwin(LINES, COLS, 0, 0);

    } /* help_init */


/*
 *  Routine: load_help_file
 *
 *  Description: it loads the help file into the buffer. If file not
 *      available, it loads the buffer with an error message. If file
 *      too long, truncated.
 *
 *  Arguments: filename -- (IN) input file name
 *
 *  Return Value: none
 *
 *  Side Effects: it modifies the buffer if the file is read in correctly.
 */
static void load_help_file(filename)
    char filename[];
    {
    FILE *pt_file;
    char line[MAX_H_COL + 2]; /* buffer to read in */

    h_buffer.top = 0;

    if ((pt_file = fopen(filename,"r")) != NULL) {
        h_buffer.last = -1;         /* empty buffer */
        while ((h_buffer.last < (MAX_H_LINE -1)) &&
                           (fgets(line,MAX_H_COL + 1,pt_file) != NULL)) {
            line[strlen(line) - 1] = '\0';       /* eliminate <cr> */
            h_buffer.last++;
            (void)strcpy(h_buffer.body[h_buffer.last],line);
            } /* while */
        h_buffer.top = 0; /* position at the beginning of buffer */
        } 
        else { /* help not available */
            (void)strcpy(h_buffer.body[0],"");
            (void)strcpy(h_buffer.body[1],"");
            (void)strcpy(h_buffer.body[2],
                         "                        Help File Not Available");
            h_buffer.last = 2;
        } /* if */

    } /* load_help_file */


/*
 *  Routine: enter_help
 *
 *  Description: it swaps from browse to help window
 *
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects: it uses curses to swap window
 */
static void enter_help()
    {
    /*
     * position to first page, redraw to the  help screen
     */
    h_buffer.top = 0;
    write_page();
    (void)wmove(help_window, 0,0);
    /*
     * output the help screen
     */
    overwrite(help_window, stdscr);
    (void)wrefresh(help_window);
    refresh();

    } /* enter_help */


/*
 *  Routine: page_up
 *
 *  Description: it scrolls the help screen one page up. If scrolling is
 *      impossible, no action is taken
 *
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects: it modifies the current top line in the buffer and rewrites
 *      the curse window
 */
static void page_up()
    {
    /*
     * scroll only if one complete page is available above
     */
    if ( (h_buffer.top - (LINES -1)) >= 0) {
        h_buffer.top -= LINES -1;
        write_page();
        overwrite(help_window, stdscr);
        refresh();
        }
    else
        (void)printf("\007");

    } /* page_up */


/*
 *  Routine: page_down
 *
 *  Description: it scrolls the help screen one page up. If scrolling is
 *      impossible, no action is taken
 *
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects: it modifies the current top line in the buffer and rewrites
 *      the curse window
 */
static void page_down()
    {
    /*
     * scroll only if at least one more page (or part of it) available
     */
    if ((h_buffer.top + LINES -1) <= h_buffer.last ) {
        h_buffer.top += LINES - 1;                /* new top */
        write_page();
        overwrite(help_window, stdscr);    
        refresh();
        }
    else
        (void)printf("\007");

    } /* page_down */


/*
 *  Routine: write_page
 *
 *  Description: it gets the current page of the help buffer and puts
 *               it on the curse window. It clears the current screen
 *               and refreshes it.
 *
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void write_page()
    {
    int n_write; /* number of lines to write on screen */
    int i;
    line_type line;
 
    /*
     * eliminate last page on screen
     */
    (void)wclear(help_window);

    n_write = LINES - 1; /* default, save last line for menu */
    /*
     * check if the end of buffer will be reached. If so, calculate the
     * number of remaining lines to print 
     */
    if ( h_buffer.last < (h_buffer.top + LINES -1))
        n_write = h_buffer.last  - h_buffer.top + 1;

    for (i = 0; i < n_write; i++) {
        string_copy(line,h_buffer.body[i + h_buffer.top], COLS);
        mvwaddstr(help_window,i, 0, line);
        }
    /*
     * write the menu
     */
    string_copy(line,
                 "idlbrowse Help: Up Down (any other character exits help)\
                            ", COLS - 2);
    (void)wstandout(help_window);
    mvwaddstr(help_window, LINES -1, 0, line); 
    (void)wstandend(help_window);

    } /* write_page */




