/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlbrowse/command.c 
*				 					* 
*   Copyright (C) 1991 Marco Chierotti
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *
 *  Description: module to interpret commands. For more details refer
 *               to the header of its corresponding .h file.
 *
 */

/***** INCLUDE FILES *****/
#include "global.h"
#include "command.h"
#include "windows.h"
#include "candle.h"
#include "help.h"


void get_file_name();
void display_verbose_info();
boolean find_target_item_info();
boolean find_derived();

/***** FUNCTION PROTOTYPES *****/

boolean move_up();
boolean move_down();
boolean move_right();
boolean move_left();
static boolean scroll_up();
static boolean scroll_down();
void home();
static void update_next_window();
static void update_window_for_process();
static void update_window_for_structure();

/***** CODE *****/


/*****************************************************************************/
void command_interpreter()
    {
    list_type pr_str_list;
    command_type command;   
    line_type file_name;
    line_type error_message_line;
    reader_status_type reader_status;
    boolean must_restore_menu;
    int col1,col2,line;
   
    /*
     * get list of processes for the current candle strcuture
     */
    (void)find_proc_and_struct(&pr_str_list);
    if ( ! empty_list(&pr_str_list) ) {
        write_list_to_window(&pr_str_list,current_window());
        (void)select_top_element(current_window());
        }
    /*
     * loop to interpret commands
     */
    must_restore_menu = FALSE;
    while(( command = get_command()) != QUIT) {
        /*
         * restore the bottom menu line only if needed
         */
        if (must_restore_menu) {
            display_menu_line();
            must_restore_menu = FALSE;
            }
        switch(command){
            case UP:
                if (!move_up())
                    error_message();
                break;
            case DOWN:
               if (!move_down())
                    error_message();
                break;
            case SCR_UP:
                if (!scroll_up())
                    error_message();
                break;
            case SCR_DOWN:
                if (!scroll_down())
                    error_message();
                break;
            case RIGHT:
                if (!move_right())
                    error_message();
                break;
            case LEFT:
                if (!move_left())
                    error_message();
                break;
            case HELP:
                help();
                restore_browse();
                break;
            case C_FILE:
                must_restore_menu = TRUE;
                if(ask_file_name(file_name)) {
                    reader_status = load_candle_instance(file_name,
                                                   error_message_line); 
                    switch (reader_status) {
                        case NO_ERROR:
                            (void)find_proc_and_struct(&pr_str_list);
                            if ( ! empty_list(&pr_str_list) ) {
                                home();
                                write_list_to_window(&pr_str_list,
                                                        current_window());
                                (void)select_top_element(current_window());
                                 send_file_message("New file loaded");
                            }
                            break;
                        case OPEN_ERROR:
                            send_file_message(error_message_line);
                            break;
                        case IDL_ERROR:
                            refresh_screen();
                            send_file_message(error_message_line);
                            break;
                        } /* switch reader_status */
                    }
                else /* if ask_file_name fails */
                    send_file_message("Wrong file name");
                break;
            case F_NAME:
                get_file_name(file_name);
                display_file_name(file_name);
                must_restore_menu = TRUE;
                break;
            case SETUP:
                if ( ! ask_new_setup(&col1,&col2,&line))
                    error_message();
                setup(col1,col2,line);
                break;
            case ESCAPE:
                home();
                break;
	    case VERBOSE:
		display_verbose_info();
                must_restore_menu = TRUE;
		break;
            default:
                error_message();
                break;
            } /* switch (command) */

        } /* while */

    quit();

    } /* command_interpreter */

/************************ PRIVATE ROUTINES ********************/

/*
 *  Routine: move_up
 *
 *  Description: it moves up one element in the current window, if
 *               necessary it makes the window scroll
 *
 *  Arguments: none
 *
 *  Return Value: TRUE if movement allowed, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean move_up()
    {
    /*
     * move before changing next window
     */
    if ( select_prev_item() ) 
        return(TRUE); 
    else
        return(FALSE);

    } /* move_up */ 

/*
 *  Routine: move_down
 *
 *  Description: it moves down one element in the current window, if
 *               necessary it makes the window scroll
 *
 *  Arguments: none
 *
 *  Return Value: TRUE if movement allowed, otherwise FALSE
 *
 *  Side Effects: none
 */
boolean move_down()
    {
    /*
     * move before changing next window 
     */
    if (select_next_item() ) 
        return(TRUE);
    else
        return(FALSE);

    } /* move_down */ 

/*
 *  Routine: scroll_up
 *
 *  Description: it scrolls up half a window in the  current window.
 *
 *  Arguments: none
 *
 *  Return Value: TRUE if movement allowed, otherwise FALSE
 *
 *  Side Effects: none
 */ 
static boolean scroll_up()
    {
    /*
     * move before changing next window
     */
    if ( select_prev_scroll() ) 
        return(TRUE);
    else
        return(FALSE);

    } /* scroll_up */ 

/*
 *  Routine: scroll_down
 *
 *  Description: it scrolls down half a window in the  current window.
 *
 *  Arguments: none
 *
 *  Return Value: TRUE if movement allowed, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean scroll_down()
    {
    /*
     * move before changing next window
     */
    if ( select_next_scroll() ) 
        return(TRUE);
    else
        return(FALSE);

    } /* scroll_down */ 

/*
 *  Routine: move_right
 *
 *  Description: it moves to the top item of the next window, if it
 *               is not empty or it is not the bottom window.
 *
 *  Arguments: none
 *
 *  Return Value: TRUE if movement allowed, otherwise FALSE
 *
 *  Side Effects: none
 */
boolean move_right()
    {

    if (current_window() < BOTTOM_W) {
        update_next_window();
        if (move_to_next_window() ) {
            (void)select_top_element(current_window());
            return(TRUE);
            }
        }

    return(FALSE);

    } /* move_right */

/*
 *  Routine: move_left
 *
 *  Description: it moves to the selected item of the previous window, 
 *               disselecting the element of the current window. If in
 *               the left window no action is taken.
 *
 *  Arguments: none
 *
 *  Return Value: TRUE if movement allowed, otherwise FALSE
 *
 *  Side Effects:  by underlying functions
 */
boolean move_left()
    {
    if (current_window() <= LEFT_W)
        return(FALSE);    /* move not possible */

    clear_window(current_window());
    (void)move_to_prev_window();

    return(TRUE);

    } /* move_left */ 


/*
 *  Routine: home
 *
 *  Description: it moves the cursor to the first item of the
 *               first window, selecting it
 *  NOTE: only for level 2!!!!!!!!!!!
 *
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects: effects given by called routines: resets the current window
 *                and clears previous windows. 
 */
void home()
    {
    if(current_window ()< LEFT_W) 
        return; /* error condition */
    /*
     * backtrack up to the left window
     */
    while ( current_window() > LEFT_W) {
        if (current_window() < BOTTOM_W)
            clear_window(next_window(current_window()));
        (void)move_to_prev_window();
        }
    clear_window(next_window(current_window()));

    (void)select_top_element(current_window()); 

    } /* home */



/*
 *  Routine: update_next_window
 *
 *  Description: it checks the currently selected item and selects
 *               its related sublist of entries, displaying it onto
 *               the next window. It has no action if already in the
 *               bottom window. It behaves differently if a structure
 *               or a process is selected.
 *
 *  Arguments: none
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void update_next_window()
    {
    w_number_type curr_window;
    line_type item_name;

    curr_window = current_window();
    (void)give_selected_item(item_name, LEFT_W);

    if ( (curr_window < LEFT_W) || (curr_window > RIGHT_W) )
         return;  /* action not possible */

    if ( selected_item_is_process(item_name) )
        update_window_for_process(item_name+2);
    else if (selected_item_is_structure(item_name) )
        update_window_for_structure(item_name+2);
    else if (selected_item_is_defaultprocessinvariant(item_name) ) {
        update_window_for_structure(item_name);
    }
    else
        clear_window(next_window(current_window()));

    } /* update_next_window */

/*
 *  Routine: update_window_for_process
 *
 *  Description: it updates the next window in case a process is
 *               selected. It browses through the Candle instance.
 *
 *  Arguments: process_name -- (IN) process currently chosen
 *
 *  Return Value: none
 *
 *  Side Effects: none
 *
 */
static void update_window_for_process(process_name)
    line_type process_name; 
    {
    line_type process_property;
    list_type dummy_list;
    list_type item_list;
    line_type item_name;
    line_type prev_item_name;

    (void)give_selected_item(item_name, current_window());

    init_list(&dummy_list);

/*@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
    for (i = 0; i <= 1000; i++) {
        (void)sprintf(buffer,"Pl %2d w %2d frl %2d frw%2d xxxxxxxxxxxxxxxxxxxxxx",
                i, curr_window +1, n_row,
                curr_window );
        (void)append_line(buffer, &dummy_list);
        }
    write_list_to_window(&dummy_list, next_window(current_window()));
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@*/

    switch (current_window() ) {
        case LEFT_W:
            if ( find_process_properties(process_name, &item_list) ) 
                write_list_to_window(&item_list, next_window(current_window()));
            break;
        case CENTER_W:
            (void)give_selected_item(process_property, current_window());
            if (strcmp(process_property, "Invariant") == 0 ) {
               (void)find_invariant(process_name,&item_list);
               write_list_to_window(&item_list,next_window(current_window()));
               }
            else if ( strcmp(process_property,"Pre Ports") == 0 ) {
                (void)find_pre_ports(process_name, &item_list);
                write_list_to_window(&item_list,next_window(current_window()));
                }
            else if (strcmp(process_property, "Post Ports") == 0 ) {
                (void)find_post_ports(process_name, &item_list);
                write_list_to_window(&item_list,next_window(current_window()));
                }
            else if (strcmp(process_property,"Refined from") == 0 ) {
                (void)find_refined_proc(process_name,&item_list);
                write_list_to_window(&item_list,next_window(current_window()));
                }
            else if (strcmp(process_property,"Target") == 0 ) {
                (void)find_target_info(process_name,&item_list);
                write_list_to_window(&item_list,next_window(current_window()));
                }
            break;
        case RIGHT_W:
/*@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
            (void)append_line(item_name, &dummy_list); 
            write_list_to_window(&dummy_list,next_window(current_window()));
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@*/
            (void)give_selected_item(item_name, current_window());
            (void)give_selected_item(prev_item_name,prev_window(current_window()));
            if ( (strcmp(prev_item_name, "Pre Ports") == 0)
                || (strcmp(prev_item_name, "Post Ports") == 0) ) {
                (void)find_port_structure(process_name,item_name,&item_list);
                write_list_to_window(&item_list, next_window(current_window()));
                }
            else if ( strcmp(prev_item_name, "Target") == 0) {
                (void)find_target_item_info(process_name,item_name,&item_list);
                write_list_to_window(&item_list, next_window(current_window()));
                }
            break;
        } /* switch */


    } /* update_window_for_process */

/*
 *  Routine: update_window_for_structure
 *
 *  Description: it updates the next window in case a structure is 
 *               selected. It browses through the Candle instance.
 *
 *  Arguments: struct_name -- (IN) structure name to consider 
 *
 *  Return Value: none
 *
 *  Side Effects: none
 *
 */
static void update_window_for_structure(struct_name)
    line_type struct_name;
    {
    list_type item_list;
    line_type struct_property;
    line_type prev_item_name;
    list_type dummy_list;
    line_type item_name;

    (void)give_selected_item(item_name, current_window());

    init_list(&dummy_list);

    switch (current_window() ) {
        case LEFT_W:
            if ( find_struct_properties(struct_name, &item_list) ) 
                write_list_to_window(&item_list, next_window(current_window()));
            break;
        case CENTER_W:
            (void)give_selected_item(struct_property, current_window());
            if (strcmp(struct_property, "Derived from") == 0 ) {
               (void)find_derived(struct_name,&item_list);
               write_list_to_window(&item_list,next_window(current_window()));
               }
            if (strcmp(struct_property, "Refined from") == 0 ) {
               (void)find_refined_struct(struct_name,&item_list);
               write_list_to_window(&item_list,next_window(current_window()));
               }
            if (strcmp(struct_property, "Basic Types") == 0 ) {
               (void)find_basic_types(struct_name,&item_list);
               write_list_to_window(&item_list,next_window(current_window()));
               }
            if (strcmp(struct_property, "Private Types") == 0 ) {
               (void)find_priv_types(struct_name,&item_list);
               write_list_to_window(&item_list,next_window(current_window()));
               }
            else if ( strcmp(struct_property,"Classes") == 0 ) {
                (void)find_classes(struct_name, &item_list);
                write_list_to_window(&item_list,next_window(current_window()));
                }
            else if (strcmp(struct_property, "Nodes") == 0 ) {
                (void)find_nodes(struct_name, &item_list);
                write_list_to_window(&item_list,next_window(current_window()));
                }
            else if (strcmp(struct_property, "Root") == 0 ) {
                (void)find_root(struct_name, &item_list);
                write_list_to_window(&item_list,next_window(current_window()));
                }
            break;
        case RIGHT_W:
            (void)give_selected_item(item_name, current_window());
            (void)give_selected_item(prev_item_name,prev_window(current_window()));
            if (strcmp(prev_item_name, "Classes") == 0) {
                (void)find_class_info(struct_name,item_name,&item_list);
                write_list_to_window(&item_list, next_window(current_window()));
                }
            else if (strcmp(prev_item_name, "Nodes") == 0) {
                (void)find_node_info(struct_name,item_name,&item_list);
                write_list_to_window(&item_list, next_window(current_window()));
                }
            else if (strcmp(prev_item_name, "Root") ==  0) {
                if ( ! find_node_info(struct_name,item_name,&item_list) )
                    (void)find_class_info(struct_name, item_name, &item_list); 
                write_list_to_window(&item_list, next_window(current_window()));
                }
            else if (strcmp(prev_item_name, "Basic Types") ==  0) {
                (void)find_basic_type_occurrence(struct_name, item_name,&item_list); 
                write_list_to_window(&item_list, next_window(current_window()));
                }
            else if (strcmp(prev_item_name, "Private Types") ==  0) {
                (void)find_priv_type_info(struct_name, item_name,&item_list); 
                write_list_to_window(&item_list, next_window(current_window()));
                }
            break;
        } /* switch */



    } /* update_window_for_structure */ 

