/***********************************************************************\ 
*									* 
*   File: scorpion/src/idlbrowse/candle.c 
*				 					* 
*   Copyright (C) 1991 Marco Chierotti
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*
 *  Description: module to get info from the Candle structure. 
 *               For more details refer to the header of its 
 *               corresponding .h file.
 *
 */


/*****************************************************************************/


/***** INCLUDE FILES *****/

#include <stdio.h>
#include "global.h"   
#include "idlbrowse.h"   /* IDL generated header file */
#include "candle.h"

boolean selected_item_is_defaultprocessinvariant();

char current_file_name[1024];

/***** FUNCTION PROTOTYPES *****/

static StructureEntity find_structure();
static StructureEntity find_structure_from();
static ProcessEntity find_process();
static Class get_node();
static Class get_class();
static boolean is_basic_type();
static boolean is_private_type();
static boolean find_language();
static boolean find_language_version();
static boolean find_operating_system();
static boolean find_operating_system_version();
static boolean find_machine();
static boolean find_runtime_version();
static void catenate_tokens();

/***** PRIVATE DATA STRUCTURES *****/

/*
 * "handle" type to keep track of items
 */
typedef char *struct_handle_type;  /* handle to a structure */
typedef char *proc_handle_type;    /* handle to a process */
typedef char *node_handle_type;    /* handle to a node */
typedef char *class_handle_type;    /* handle to a class */

     /* other handle types can be declared if needed */

/*
 * private Candle instance loaded from file
 */
compilationUnit compUnit;
/*
 * file name of the current candle instance
 */

/***** CODE *****/

 
/*****************************************************************************/
void init_candle()
    {
    compUnit = NULL;         /* no compilation unit to start with */
    current_file_name[0] = '\0';  /* no name defined yet */

    } /* init_candle */


/*****************************************************************************/
reader_status_type load_candle_instance(file_name, error_message)
    line_type file_name;
    line_type error_message;
    {
    extern Boolean ReaderOK;    /* flag to signal error from the IDL reader */
    FILE *pt_file;              /* fiel where to read */
    compilationUnit tempCompUnit; /* temporary compilation unit */

    error_message[0] = '\0';
    /*
     * open the file and return if error occurs
     */
    if ( (pt_file = fopen(file_name, "r")) == NULL ) {
        string_concat(error_message,"Error: cannot open file ",file_name, 
                                                     MAX_COL);
        return(OPEN_ERROR);
        }
    /*
     * read the Candle instance from input port, and check for errors
     */
    if (!strcmp(current_file_name,file_name))
      return NO_ERROR;
      
    tempCompUnit = InCandle(pt_file);
    if ( (!ReaderOK) || (tempCompUnit == NULL) ) {
        string_concat(error_message,"IDL error in reading file ",
                                 file_name, MAX_COL);
        return(IDL_ERROR);
        }
    /*
     * if temporary compilation unit correct, update the compilation unit
     * and save the corresponding file name. If there is an old compilation
     * unit, dispose it first.
     */
    if (compUnit != NULL)
        DcompilationUnit(compUnit);
 
    compUnit = tempCompUnit;
    string_copy(current_file_name, file_name, MAX_COL);

    return(NO_ERROR);

    } /* load_candle_instance */


/*****************************************************************************/
void get_file_name(file_name)
    line_type file_name;
    {
    string_copy(file_name, current_file_name, MAX_COL);

    } /* get_file_name */

     
/*****************************************************************************/
boolean find_proc_and_struct(pr_str_list)
    list_type *pr_str_list;
    {
    SEQDeclaration  SDecl;          /* remaining SEQ of decl. to process */
    Declaration     currentDecl;    /* current declaration in syn_body */
    boolean         found;          /* to signal at least one element */
    line_type       lexical_name;   /* buffer for proc. or str. names */
    SEQStructureOrProcessRef  SStOrPrRef;
    StructureOrProcessRef  AStOrPrRef;
    /*
     * setup an empty list and usuccesfull flag
     */
    found = FALSE;
    init_list(pr_str_list);
    /*
     * process the declarations: consider structures or processes only
     */
    foreachinSEQDeclaration(compUnit->syn_body, SDecl, currentDecl) {
        switch (typeof(currentDecl)) {
	    case KImportDecl:
                found = TRUE;
		foreachinSEQStructureOrProcessRef(currentDecl.VImportDecl->syn_specs,SStOrPrRef,AStOrPrRef) {
		  switch (typeof(AStOrPrRef->sem_entity)) {
		  case KStructureEntity:
		    lexical_name[0] = 'S';
		    lexical_name[1] = ' ';
		    string_copy(lexical_name+2,
				AStOrPrRef->lex_name, MAX_COL -2);
		    if (! is_in_list(lexical_name,pr_str_list))
		      (void)append_line(lexical_name, pr_str_list);
		    mergefromstruct(AStOrPrRef->lex_name,pr_str_list);
		    break;
		  case KProcessEntity:
		    lexical_name[0] = 'I';
		    lexical_name[1] = ' ';
		    string_copy(lexical_name+2,
				AStOrPrRef->lex_name, MAX_COL -2);
		    if (! is_in_list(lexical_name,pr_str_list))
		      (void)append_line(lexical_name, pr_str_list);
		    lexical_name[0] = 'P';
		    lexical_name[1] = ' ';
		    string_copy(lexical_name+2,
				AStOrPrRef->lex_name, MAX_COL -2);
		    if (! is_in_list(lexical_name,pr_str_list))
		      (void)append_line(lexical_name, pr_str_list);

		    break;
		  }
		}
		break;
	      case KStructureEntity:
		found = TRUE;
		lexical_name[0] = 'S';
		lexical_name[1] = ' ';
		string_copy(lexical_name+2,
			    currentDecl.VStructureEntity->lex_name, MAX_COL -2);
		if (! is_in_list(lexical_name,pr_str_list))
		  (void)append_line(lexical_name, pr_str_list);
		mergefromstruct(currentDecl.VStructureEntity->lex_name,pr_str_list);
		break;
	      case KProcessEntity:
		found = TRUE;
		lexical_name[0] = 'I';
		lexical_name[1] = ' ';
		string_copy(lexical_name+2,
			    currentDecl.VProcessEntity->lex_name, MAX_COL -2);
		if (! is_in_list(lexical_name,pr_str_list))
		  (void)append_line(lexical_name, pr_str_list);
		lexical_name[0] = 'P';
		lexical_name[1] = ' ';
		string_copy(lexical_name+2,
			    currentDecl.VProcessEntity->lex_name, MAX_COL -2);
		if (! is_in_list(lexical_name,pr_str_list))
		  (void)append_line(lexical_name, pr_str_list);
		break;
	      } /* switch */
      } /* for */

    return(found);
    
} /* find_proc_and_struct */

mergefromstruct(struct_name, pr_str_list)
line_type struct_name;
list_type *pr_str_list;
{
  int i,j;
  list_type all_derived_list,tempderived_list;
  boolean find_derived();
  line_type lexical_name;
  
  init_list(&all_derived_list);
  (void)append_line(struct_name,&all_derived_list);
  for (i=0; i <= all_derived_list.last_row; i++) {
    if (find_derived(all_derived_list.body[i], &tempderived_list) == TRUE) {
      for (j=0; j <= tempderived_list.last_row; j++) {
	if (! is_in_list(tempderived_list.body[j],&all_derived_list))
	  (void)append_line(tempderived_list.body[j],&all_derived_list);
      }
    }
  }
  for (i = 0; i <= all_derived_list.last_row; i++) {
    lexical_name[0] = 'S';
    lexical_name[1] = ' ';
    string_copy(lexical_name+2,all_derived_list.body[i], MAX_COL -2);
    if (! is_in_list(lexical_name, pr_str_list))
      (void)append_line(lexical_name, pr_str_list);
  }
}

/*****************************************************************************/
boolean find_struct_properties(struct_name, struct_prop_list)
    line_type struct_name;
    list_type *struct_prop_list;
    {
    boolean found_derived,found_refined,found_classes,found_root,found_nodes;
    boolean found_set, found_seq, found_basic_type,found_private_types;
    StructureEntity AStr;  /* structure to refer to in the search */

    SEQStructureRef SStRef;
    StructureRef AStRef;
    TypeEntity currTypeEntity;
    SETTypeEntity tailSETTypeEntity;
    Class AClass;
    Attribute currAttribute;
    SEQAttribute SSEQAttribute;
    TypeRef ATypeRef;
    SetRef ASetRef;
    SeqRef ASeqRef;

    init_list(struct_prop_list);
    /*
     * find a structure with the given name, if any
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
	return(FALSE);
    /*
     * initialize the flags for the search
     */
    found_derived = FALSE;
    found_refined = FALSE;
    found_classes = FALSE;
    found_root    = FALSE;
    found_nodes   = FALSE;
    found_set     = FALSE;
    found_seq     = FALSE;
    found_basic_type = FALSE;
    found_private_types = FALSE;
    /*
     * check if the sequence of references to str. DERIVED FROM is not empty
     */
    foreachinSEQStructureRef(AStr->syn_from, SStRef, AStRef) {
	if (typeof(AStRef->sem_entity) == KStructureEntity) {
	    found_derived = TRUE;
	    break;
	    }
	} /* for */

    /*
     *  check if there is a structure the current structure has been REFINED
     *  FROM
     */
    if (typeof(AStr->syn_refines)==KStructureRef) {
	    AStRef = AStr->syn_refines.VStructureRef;
	    if (typeof(AStRef->sem_entity) == KStructureEntity) {
		found_refined = TRUE;
	    }
	}
    /*
     * look for NODES and CLASSES
     * AStr->sem_types: Set Of TypeEntity
     * TypeEntity ::= SetOrSeq | NamedType
     * NamedType ::= Atomic | Class
     * AClass->sem_subclasses : Set Of Class
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		AClass = currTypeEntity.VClass;
                foreachinSEQAttribute(AClass->sem_allattributes,
                                       SSEQAttribute, currAttribute) {
                    ATypeRef = currAttribute->syn_type;
                    switch (typeof (ATypeRef) ) {
                        case KNamedTypeRef:
                            if (is_basic_type(ATypeRef.VNamedTypeRef->lex_name))
                                found_basic_type = TRUE;
                            else if (is_private_type(AStr,
                                           ATypeRef.VNamedTypeRef->lex_name))
                                found_private_types = TRUE;
                            break;
                        /*
                         * look also for private types inside set or seq
                         */
                        case KSetRef:
                            found_set = TRUE;
                            ASetRef = ATypeRef.VSetRef;
                            if(is_private_type(AStr,
                                     ASetRef->syn_component->lex_name))
                                found_private_types = TRUE;
                            break;
                        case KSeqRef:
                            found_seq = TRUE;
                            ASeqRef = ATypeRef.VSeqRef;
                            if(is_private_type(AStr,
                                     ASeqRef->syn_component->lex_name))
                                found_private_types = TRUE;
                            break;
                        } /* switch */
                    } /* foreach */
		if (emptySETClass(AClass->sem_subclasses) )
		    found_nodes = TRUE;
		else
		    found_classes = TRUE;
	        }
	} /* foreachin */

    /*
     * check if there is a ROOT. AStr->sem_root: TypeEntityOrVoid
     *                           TypeEntityOrVoid ::= TypeEntity | Void
     */
    if ( typeof(AStr->sem_root) != KVoid)
	found_root = TRUE;

    /*
     * load the output list with attributes
     */
    if (found_derived)
	(void)append_line("Derived from",struct_prop_list);
    if (found_refined)
	(void)append_line("Refined from",struct_prop_list);
    if (found_basic_type || found_set || found_seq)
       (void)append_line("Basic Types", struct_prop_list);
    if (found_classes)
	(void)append_line("Classes",struct_prop_list);
    if (found_nodes)
	(void)append_line("Nodes",struct_prop_list);
    if (found_private_types)
       (void)append_line("Private Types",struct_prop_list);
    if (found_root)
	(void)append_line("Root",struct_prop_list);

    return(TRUE);

    } /* find_struct_prop */

 
/*****************************************************************************/
boolean find_derived(struct_name, struct_list)
    line_type struct_name;
    list_type *struct_list;
    {
    StructureEntity AStr;         /* structure to consider */
    SEQStructureRef SStRef;
    StructureRef AStRef;
    StructureEntity st;          /* current derived from structure */

    /*
     * find a structure with the given name, if any
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
	return(FALSE);

    init_list(struct_list);

    foreachinSEQStructureRef(AStr->syn_from, SStRef, AStRef) {
	if (typeof(AStRef->sem_entity) == KStructureEntity) {
	    st = AStRef->sem_entity.VStructureEntity;
	    (void)append_line(st->lex_name, struct_list);
	    }
	} /* for */

    return(TRUE);

    } /* find_derived */



/*****************************************************************************/
boolean find_refined_struct(struct_name, refined_list)
    line_type struct_name;
    list_type *refined_list;
    {
    StructureEntity AStr;         /* structure to consider */
    StructureRef AStRef;
    StructureEntity st;          /* refined from structure */
    /*
     * find a structure with the given name, if any
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
	return(FALSE);

    init_list(refined_list);
    /*
     * find the refined from structure
     */
    if (typeof(AStr->syn_refines)==KStructureRef) {
	    AStRef = AStr->syn_refines.VStructureRef;
	    if (typeof(AStRef->sem_entity) == KStructureEntity) {
	        st = AStRef->sem_entity.VStructureEntity;
	        (void)append_line(st->lex_name,refined_list);
	    }
	}

    return(TRUE);

    } /* find_refined_struct */





/*****************************************************************************/
boolean find_root(struct_name, root_list)
    line_type struct_name;
    list_type *root_list;
    {
    StructureEntity AStr;  /* structure to refer to in the search */

    init_list(root_list);
    /*
     * find a structure with the given name, if any
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
	return(FALSE);

    /*
     * check if there is a ROOT using semantics.
     * NOTE: in this way the root is found also if it has been inherited
     *       from some other structure
     */
    if (typeof(AStr->sem_root) != KVoid) {
        (void)append_line(AStr->sem_root.VTypeEntity.IDLclassCommon->sem_name,
                                 root_list);
        }


    return(TRUE);

    } /* find_root */



/*****************************************************************************/
boolean find_assert_struct(struct_name, assert_list )
    line_type struct_name;
    list_type *assert_list;
    {
    init_list(assert_list);

    } /* find_assert_struct */


/*****************************************************************************/
boolean find_classes(struct_name, class_list)
    line_type struct_name;
    list_type *class_list;
    {
    StructureEntity  AStr;
    SETTypeEntity tailSETTypeEntity;
    TypeEntity currTypeEntity;
    Class AClass;

    init_list(class_list);

    AStr = find_structure(struct_name);
    if ( AStr == NULL)
        return(FALSE);
    /*
     * if the structure exists, look inside the set of classes
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		AClass = currTypeEntity.VClass;
		if (! emptySETClass(AClass->sem_subclasses) )
                    (void)append_line(AClass->sem_name, class_list);
	        }
	} /* foreachin */

    return(FALSE);

    } /* find_classes */

 
/*****************************************************************************/
boolean find_nodes(struct_name, node_list)
    line_type struct_name;
    list_type *node_list;
    {
    StructureEntity  AStr;
    SETTypeEntity tailSETTypeEntity;
    TypeEntity currTypeEntity;
    Class ANode;

    init_list(node_list);

    AStr = find_structure(struct_name);
    if ( AStr == NULL)
        return(FALSE);
    /*
     * if the structure exists, look inside the set of nodes 
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		ANode = currTypeEntity.VClass;
		if (emptySETClass(ANode->sem_subclasses) )
                    (void)append_line(ANode->sem_name, node_list);
	        }
	} /* foreachin */

    return(FALSE);

    } /* find_nodes */

 
/*****************************************************************************/
boolean find_priv_types(struct_name, priv_types_list)
    line_type struct_name;
    list_type *priv_types_list;
    {
    StructureEntity AStr;  /* structure to refer to in the search */

    TypeEntity currTypeEntity;
    SETTypeEntity tailSETTypeEntity;
    NamedTypeRef ANamedTypeRef;
    Class AClass;
    Attribute currAttribute;
    SEQAttribute SSEQAttribute;
    TypeRef ATypeRef;
    SetRef ASetRef;
    SeqRef ASeqRef;
  
    init_list(priv_types_list); 
    /*
     * find the structure, if not present, return
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
        return(FALSE);
    /*
     * loop discarding all the basic types, sets and sequences
     * but looking inside sets and sequences for hidden private types
     * multiple occurrences are eliminated
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		AClass = currTypeEntity.VClass;
                foreachinSEQAttribute(AClass->sem_allattributes,
                                       SSEQAttribute, currAttribute) {
                    ATypeRef = currAttribute->syn_type;
                    switch (typeof (ATypeRef) ) {
                        case KNamedTypeRef:
                            ANamedTypeRef = ATypeRef.VNamedTypeRef;
                            if (is_private_type(AStr,ANamedTypeRef->lex_name))
                                if (! is_in_list(ANamedTypeRef->lex_name,
                                                             priv_types_list)) {
                                   (void)append_line(ANamedTypeRef->lex_name,
                                                  priv_types_list);
                                   }
                            break;
                        /*
                         * for sets and seqs, look inside
                         */
                        case KSetRef:
                            ASetRef = ATypeRef.VSetRef;
                            if (is_private_type(AStr,
                                   ASetRef->syn_component->lex_name))
                               if (!is_in_list(ASetRef->syn_component->lex_name,
                                                        priv_types_list)) {
                                   (void)append_line(ASetRef->syn_component->lex_name,
                                                        priv_types_list);
                                   }
                            break;
                        case KSeqRef:
                            ASeqRef = ATypeRef.VSeqRef;
                            if (is_private_type(AStr,
                                   ASeqRef->syn_component->lex_name))
                                if(!is_in_list(ASeqRef->syn_component->lex_name,
                                                        priv_types_list)) {
                                   (void)append_line(ASeqRef->syn_component->lex_name,
                                                        priv_types_list);
                                   }
                            break;
                        } /* switch */
                    } /* foreach */
	        }
	} /* foreachin */

    return(TRUE);

    } /* find_priv_types */

 
/*****************************************************************************/
boolean find_basic_types(struct_name, basic_types_list)
    line_type struct_name;
    list_type *basic_types_list;
    {

    StructureEntity AStr;  /* structure to refer to in the search */

    TypeEntity currTypeEntity;
    SETTypeEntity tailSETTypeEntity;
    NamedTypeRef ANamedTypeRef;
    Class AClass;
    Attribute currAttribute;
    SEQAttribute SSEQAttribute;
    TypeRef ATypeRef;
    SeqRef ASeqRef;
    SetRef ASetRef;
    boolean found_set, found_seq, found_integer, found_rational, found_string,
            found_boolean;
  
    init_list(basic_types_list); 
    /*
     * find the structure, if not present, return
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
        return(FALSE);

    found_boolean = FALSE;
    found_integer = FALSE;
    found_rational = FALSE;
    found_string = FALSE;
    found_seq = FALSE;
    found_set = FALSE;

    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		AClass = currTypeEntity.VClass;
                foreachinSEQAttribute(AClass->sem_allattributes,
                                       SSEQAttribute, currAttribute) {
                    ATypeRef = currAttribute->syn_type;
                    switch (typeof (ATypeRef) ) {
                        case KNamedTypeRef:
                            ANamedTypeRef = ATypeRef.VNamedTypeRef;
                            if (strcmp(ANamedTypeRef->lex_name,"Boolean") == 0)
                               found_boolean = TRUE;
                            else
                            if (strcmp(ANamedTypeRef->lex_name,"Integer") == 0)
                               found_integer = TRUE;
                            else
                            if (strcmp(ANamedTypeRef->lex_name,"Rational") == 0)
                               found_rational = TRUE;
                            else
                            if (strcmp(ANamedTypeRef->lex_name,"String") == 0)
                               found_string = TRUE;
                            break;
                        /*
                         * look also for basic types inside sets and sequences
                         */
                        case KSetRef:
                            found_set = TRUE;
                            ASetRef = ATypeRef.VSetRef;
                            if ( strcmp(ASetRef->syn_component->lex_name,
                                         "Boolean") == 0)
                                found_boolean = TRUE;
                            else if ( strcmp(ASetRef->syn_component->lex_name,
                                         "Integer") == 0)
                                found_integer = TRUE;
                            else if ( strcmp(ASetRef->syn_component->lex_name,
                                         "Rational") == 0)
                                found_rational = TRUE;
                            else if ( strcmp(ASetRef->syn_component->lex_name,
                                         "String") == 0)
                                found_string = TRUE;
                            break;
                        case KSeqRef:
                            found_seq = TRUE;
                            ASeqRef = ATypeRef.VSeqRef;
                            if ( strcmp(ASeqRef->syn_component->lex_name,
                                         "Boolean") == 0)
                                found_boolean = TRUE;
                            else if ( strcmp(ASeqRef->syn_component->lex_name,
                                         "Integer") == 0)
                                found_integer = TRUE;
                            else if ( strcmp(ASeqRef->syn_component->lex_name,
                                         "Rational") == 0)
                                found_rational = TRUE;
                            else if ( strcmp(ASeqRef->syn_component->lex_name,
                                         "String") == 0)
                                found_string = TRUE;
                            break;
                        } /* switch */
                    } /* foreach */
	        }
	} /* foreachin */
    /*
     * now load the list of items
     */
    if (found_boolean)
       (void)append_line("Boolean", basic_types_list);
    if (found_integer)
       (void)append_line("Integer", basic_types_list);
    if (found_rational)
       (void)append_line("Rational", basic_types_list);
    if (found_string)
       (void)append_line("String", basic_types_list);
    if (found_seq)
       (void)append_line("Seq Of", basic_types_list);
    if (found_set)
       (void)append_line("Set Of", basic_types_list); 


    return(TRUE);

    } /* find_basic_types */

 
/*****************************************************************************/
boolean find_struct_unnamed_assert(struct_name, struct_unnamed_assert_list)
    line_type struct_name;
    list_type *struct_unnamed_assert_list;
    {
    init_list(struct_unnamed_assert_list);

    } /* find_struct_unnamed_assert */

 
/*****************************************************************************/
boolean find_struct_assert_body(struct_name, assert_name, assertion_body_list)
    line_type struct_name;
    line_type assert_name;
    list_type *assertion_body_list;
    {
    init_list(assertion_body_list);

    } /* find_struct_assert_body */

 
/*****************************************************************************/
boolean find_class_info(struct_name, class_name, class_info_list)
    line_type struct_name;
    line_type class_name;
    list_type *class_info_list;
    {
    StructureEntity AStr;
    Class AClass;
    Class superClass, subClass;
    SETClass SSETClass;
    SEQAttribute SSEQAttribute;
    Attribute    currAttribute;
    TypeRef      ATypeRef;
    NamedTypeRef ANamedTypeRef;
    SetRef ASetRef;
    SeqRef ASeqRef;

    line_type buffer1, buffer2, buffer3, buffer4, buffer5;
    boolean first;
    int len_max, curr_len;
    int i;

    init_list(class_info_list);  
    /*
     * locate the structure and the class inside the compilation unit
     */
    if ( (AStr = find_structure(struct_name)) == NULL )
        return(FALSE);

    if ( ( AClass = get_class(AStr, class_name)) == NULL )
        return(FALSE);

    /*
     * block only for test purposes
     */
    if (FALSE) {
        for(i=0; i<=15; i++) {
            (void)sprintf(buffer1,"line %d",i);
            (void)append_line(buffer1,class_info_list);
            string_concat(buffer2,"     ", buffer1, MAX_COL);
            (void)append_line(buffer2,class_info_list);
            }
        return(TRUE);
        }

    /*
     * find the superclasses and list them
     */
    if ( emptySETClass(AClass->sem_ancestors) )
        (void)append_line("Superclasses: None", class_info_list);
    else {
        first = TRUE;
        foreachinSETClass(AClass->sem_ancestors,SSETClass, superClass) {
            if (first) {
                string_concat(buffer1,"Superclasses: ",
                                 superClass->sem_name, MAX_COL);
                (void)append_line(buffer1, class_info_list); 
                first = FALSE;
                }
            else {
                string_concat(buffer1,"              ",
                                 superClass->sem_name, MAX_COL);
                (void)append_line(buffer1, class_info_list); 
                 }
            } /* foreachin */
        }
    /*
     * find the subclasses and list them
     */
    if ( emptySETClass(AClass->sem_subclasses) )
        (void)append_line("Subclasses:   None", class_info_list);
    else {
        first = TRUE;
        foreachinSETClass(AClass->sem_subclasses,SSETClass, subClass) {
            if (first) {
                string_concat(buffer1,"Subclasses:   ",
                                    subClass->sem_name, MAX_COL);
                (void)append_line(buffer1, class_info_list); 
                first = FALSE;
                }
            else {
                string_concat(buffer1,"              ",
                                 subClass->sem_name, MAX_COL);
                (void)append_line(buffer1, class_info_list); 
                 }
            } /* foreachin */
        }

    /*
     * find all the attributes of the class
     */
    if ( emptySEQAttribute(AClass->sem_allattributes) )
       (void)append_line("Attributes: None",class_info_list);
    else {
        (void)append_line("Attributes:", class_info_list);
        /*
         * find the longest attribute string
         */
         len_max = -1;
         foreachinSEQAttribute(AClass->sem_allattributes,
                                SSEQAttribute, currAttribute) {
             if ((curr_len = strlen(currAttribute->lex_name)) > len_max)
                 len_max = curr_len;
             } /* foreachin */
        /*
         * print as a table
         */
        if (len_max > (MAX_COL - 1))
           len_max = MAX_COL - 1;
        foreachinSEQAttribute(AClass->sem_allattributes,
                                SSEQAttribute, currAttribute) {
            curr_len = strlen(currAttribute->lex_name); 
            for(i = 0; i <= len_max; i++) {
                if (i < curr_len)
                   buffer1[i] = currAttribute->lex_name[i];
                else if ( i == curr_len)
                   buffer1[i] = ':';
                else
                   buffer1[i] = ' ';
                } /* for */
            buffer1[len_max + 1 ] = '\0';
            string_concat(buffer2, "      ",buffer1,MAX_COL);
            string_concat(buffer3,buffer2,"   ", MAX_COL);
            ATypeRef = currAttribute->syn_type;
            switch (typeof (ATypeRef) ) {
                case KNamedTypeRef:
                    ANamedTypeRef = ATypeRef.VNamedTypeRef;
                    string_concat(buffer4,buffer3,
                             ANamedTypeRef->lex_name,MAX_COL);
                    (void)append_line(buffer4 ,class_info_list);  
                    break;
                case KSetRef:
                    ASetRef = ATypeRef.VSetRef;
                    string_concat(buffer4,buffer3,
                             "Set Of ",MAX_COL);
                    string_concat(buffer5,buffer4,
                              ASetRef->syn_component->lex_name,MAX_COL);
                    (void)append_line(buffer5 ,class_info_list);  
                    break;
                case KSeqRef:
                    ASeqRef = ATypeRef.VSeqRef;
                    string_concat(buffer4,buffer3,
                             "Seq Of ",MAX_COL);
                    string_concat(buffer5,buffer4,
                              ASeqRef->syn_component->lex_name,MAX_COL);
                    (void)append_line(buffer5 ,class_info_list);  
                    break;
                } /* switch */
            } /* foreachinSEQAttribute */
        } /* else */

    return(TRUE);

    } /* find_class_info */

 
/*****************************************************************************/
boolean find_node_info(struct_name, node_name, node_info_list)
    line_type struct_name;
    line_type node_name;
    list_type *node_info_list;
    {
    StructureEntity AStr;
    Class ANode;
    Class superClass;
    SETClass SSETClass;
    SEQAttribute SSEQAttribute;
    Attribute    currAttribute;
    TypeRef      ATypeRef;
    NamedTypeRef ANamedTypeRef;
    SetRef ASetRef;
    SeqRef ASeqRef;

    line_type buffer1, buffer2, buffer3, buffer4, buffer5;
    boolean first;
    int len_max, curr_len;
    int i;

    init_list(node_info_list);  
    /*
     * locate the structure and the node inside the compilation unit
     */
    if ( (AStr = find_structure(struct_name)) == NULL )
        return(FALSE);

    if ( ( ANode = get_node(AStr, node_name)) == NULL )
        return(FALSE);

    /*
     * block only for test purposes
     */
    if (FALSE) {
        for(i=0; i<=15; i++) {
            (void)sprintf(buffer1,"line %d",i);
            (void)append_line(buffer1,node_info_list);
            string_concat(buffer2,"     ", buffer1, MAX_COL);
            (void)append_line(buffer2,node_info_list);
            }
        return(TRUE);
        }

    /*
     * find the superclasses and list them
     */
    if ( emptySETClass(ANode->sem_ancestors) )
        (void)append_line("Superclasses: None",node_info_list);
    else {
        first = TRUE;
        foreachinSETClass(ANode->sem_ancestors,SSETClass, superClass) {
            if (first) {
                string_concat(buffer1,"Superclasses: ",
                                 superClass->sem_name, MAX_COL);
                (void)append_line(buffer1, node_info_list); 
                first = FALSE;
                }
            else {
                string_concat(buffer1,"              ",
                                 superClass->sem_name, MAX_COL);
                (void)append_line(buffer1, node_info_list); 
                 }
            } /* foreachin */
        }

    /*
     * find all the attributes of the node 
     */
    if ( emptySEQAttribute(ANode->sem_allattributes) )
       (void)append_line("Attributes: None",node_info_list);
    else {
        (void)append_line("Attributes:", node_info_list);
        /*
         * find the longest attribute string
         */
         len_max = -1;
         foreachinSEQAttribute(ANode->sem_allattributes,
                                SSEQAttribute, currAttribute) {
             if ((curr_len = strlen(currAttribute->lex_name)) > len_max)
                 len_max = curr_len;
             } /* foreachin */
        /*
         * print as a table
         */
        if (len_max > (MAX_COL - 1))
           len_max = MAX_COL - 1;
        foreachinSEQAttribute(ANode->sem_allattributes,
                                SSEQAttribute, currAttribute) {
            curr_len = strlen(currAttribute->lex_name); 
            for(i = 0; i <= len_max; i++) {
                if (i < curr_len)
                   buffer1[i] = currAttribute->lex_name[i];
                else if (i == curr_len)
                   buffer1[i] = ':';
                else
                   buffer1[i] = ' ';
                } /* for */
            buffer1[len_max + 1] = '\0';
            string_concat(buffer2, "      ",buffer1,MAX_COL);
            string_concat(buffer3,buffer2,"   ", MAX_COL);
            ATypeRef = currAttribute->syn_type;
            switch (typeof (ATypeRef) ) {
                case KNamedTypeRef:
                    ANamedTypeRef = ATypeRef.VNamedTypeRef;
                    string_concat(buffer4,buffer3,
                             ANamedTypeRef->lex_name,MAX_COL);
                    (void)append_line(buffer4 ,node_info_list);  
                    break;
                case KSetRef:
                    ASetRef = ATypeRef.VSetRef;
                    string_concat(buffer4,buffer3,
                             "Set Of ",MAX_COL);
                    string_concat(buffer5,buffer4,
                              ASetRef->syn_component->lex_name,MAX_COL);
                    (void)append_line(buffer5 ,node_info_list);  
                    break;
                case KSeqRef:
                    ASeqRef = ATypeRef.VSeqRef;
                    string_concat(buffer4,buffer3,
                             "Seq Of ",MAX_COL);
                    string_concat(buffer5,buffer4,
                              ASeqRef->syn_component->lex_name,MAX_COL);
                    (void)append_line(buffer5 ,node_info_list);  
                    break;
                } /* switch */
            } /* foreachinSEQAttribute */
        } /* else */

    return(TRUE);


    } /* find_node_info */

 
/*****************************************************************************/
boolean find_priv_type_info(struct_name, priv_type_name, priv_type_info_list)
    line_type struct_name;
    line_type priv_type_name;
    list_type *priv_type_info_list;
    {
    StructureEntity AStr;
    SETTypeEntity tailSETTypeEntity;
    TypeEntity currTypeEntity;
    Class AClass;
    SEQAttribute SSEQAttribute;
    Attribute currAttribute;
    TypeRef ATypeRef;
    NamedTypeRef ANamedTypeRef; 
    SetRef ASetRef;
    SeqRef ASeqRef;

    line_type buffer1, buffer2;

    init_list(priv_type_info_list); 
    /*
     * find the structure, if not present, return
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
        return(FALSE);
    /*
     * test the name to be a basic type name, if not, return
     */
    if ( is_basic_type(priv_type_name))
        return(FALSE);
    /*
     * search the occurrence for any class or node attribute
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		AClass = currTypeEntity.VClass;
                foreachinSEQAttribute(AClass->sem_allattributes,
                                       SSEQAttribute, currAttribute) {
                    ATypeRef = currAttribute->syn_type;
                    switch (typeof (ATypeRef) ) {
                        case KNamedTypeRef:
                            ANamedTypeRef = ATypeRef.VNamedTypeRef;
                            /*
                             * get the required attribute, if any and
                             * build the line to display
                             */
                            if (strcmp(ANamedTypeRef->lex_name,
                                                    priv_type_name) == 0) {
                                string_concat(buffer1,AClass->sem_name," => ",
                                                              MAX_COL);
                                string_concat(buffer2,buffer1,
                                              currAttribute->lex_name,MAX_COL);
                                string_concat(buffer1,buffer2,":   ",MAX_COL);
                                string_concat(buffer2,buffer1,
                                              ANamedTypeRef->lex_name,MAX_COL);
                                (void)append_line(buffer2, priv_type_info_list);
                                }
                            break;
                        /*
                         * look also for priv type occurrence inside 
                         * sets and seqs
                         */
                        case KSetRef:
                            ASetRef = ATypeRef.VSetRef;
                            /*
                             * build the line for set of elements
                             */
                            if (strcmp(priv_type_name,
                                   ASetRef->syn_component->lex_name) == 0) {
                                string_concat(buffer1,AClass->sem_name," => ",
                                                              MAX_COL);
                                string_concat(buffer2,buffer1,
                                                currAttribute->lex_name, 
							      MAX_COL);
                                string_concat(buffer1,buffer2,":   ",MAX_COL);
                                string_concat(buffer2,buffer1,"Set Of ",
                                                            MAX_COL);
                                string_concat(buffer1,buffer2,
                                      ASetRef->syn_component->lex_name,MAX_COL);
                                (void)append_line(buffer1 ,priv_type_info_list);  
                                }
                            break;
                        case KSeqRef:
                            ASeqRef = ATypeRef.VSeqRef;
                            /*
                             * build the line for set of elements
                             */
                            if (strcmp(priv_type_name,
                                   ASeqRef->syn_component->lex_name) == 0) {
                                string_concat(buffer1,AClass->sem_name," => ",
                                                              MAX_COL);
                                string_concat(buffer2,buffer1,
                                                currAttribute->lex_name, 
							      MAX_COL);
                                string_concat(buffer1,buffer2,":   ",MAX_COL);
                                string_concat(buffer2,buffer1,"Seq Of ",
                                                            MAX_COL);
                                string_concat(buffer1,buffer2,
                                      ASeqRef->syn_component->lex_name,MAX_COL);
                                (void)append_line(buffer1 ,priv_type_info_list);  
                                }
                            break;
                        } /* switch */
                    } /* foreach */
	        } /* if */
	} /* foreachin */

    return(TRUE);


    } /* find_priv_type_info */

 
/*****************************************************************************/
boolean find_basic_type_occurrence(struct_name, basic_type_name,
                                     basic_type_info_list)
    line_type struct_name;
    line_type basic_type_name;
    list_type *basic_type_info_list;
    {
    StructureEntity AStr;
    SETTypeEntity tailSETTypeEntity;
    TypeEntity currTypeEntity;
    Class AClass;
    SEQAttribute SSEQAttribute;
    Attribute currAttribute;
    TypeRef ATypeRef;
    NamedTypeRef ANamedTypeRef; 
    SetRef ASetRef;
    SeqRef ASeqRef;

    line_type buffer1, buffer2;

    init_list(basic_type_info_list); 
    /*
     * find the structure, if not present, return
     */
    if ( (AStr = find_structure(struct_name)) == NULL)
        return(FALSE);
    /*
     * test the name to be a basic type name, if not, return
     */
    if ( ! is_basic_type(basic_type_name))
        return(FALSE);
    /*
     * search the occurrence for any class or node attribute
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		AClass = currTypeEntity.VClass;
                foreachinSEQAttribute(AClass->sem_allattributes,
                                       SSEQAttribute, currAttribute) {
                    ATypeRef = currAttribute->syn_type;
                    switch (typeof (ATypeRef) ) {
                        case KNamedTypeRef:
                            ANamedTypeRef = ATypeRef.VNamedTypeRef;
                            /*
                             * get the required attribute, if any and
                             * build the line to display
                             */
                            if (strcmp(ANamedTypeRef->lex_name,
                                                    basic_type_name) == 0) {
                                string_concat(buffer1,AClass->sem_name," => ",
                                                              MAX_COL);
                                string_concat(buffer2,buffer1,
                                              currAttribute->lex_name,MAX_COL);
                                string_concat(buffer1,buffer2,":   ",MAX_COL);
                                string_concat(buffer2,buffer1,
                                              ANamedTypeRef->lex_name,MAX_COL);
                                (void)append_line(buffer2, basic_type_info_list);
                                }
                            break;
                        case KSetRef:
                            ASetRef = ATypeRef.VSetRef;
                            /*
                             * build the line for set of elements
                             */
                            if ( (strcmp(basic_type_name,"Set Of") == 0) ||
                                 (strcmp(basic_type_name,
                                     ASetRef->syn_component->lex_name) == 0)) {
                                string_concat(buffer1,AClass->sem_name," => ",
                                                              MAX_COL);
                                string_concat(buffer2,buffer1,
                                                currAttribute->lex_name,
							      MAX_COL);
                                string_concat(buffer1,buffer2,":   ",MAX_COL);
                                string_concat(buffer2,buffer1,"Set Of ",
                                                            MAX_COL);
                                string_concat(buffer1,buffer2,
                                      ASetRef->syn_component->lex_name,MAX_COL);
                                (void)append_line(buffer1 ,basic_type_info_list);  
                                }
                            break;
                        case KSeqRef:
                            ASeqRef = ATypeRef.VSeqRef;
                            /*
                             * build the line for set of elements
                             */
                            if ( (strcmp(basic_type_name,"Seq Of") == 0) ||
                                 (strcmp(basic_type_name,
                                     ASeqRef->syn_component->lex_name) == 0)) {
                                string_concat(buffer1,AClass->sem_name," => ",
                                                              MAX_COL);
                                string_concat(buffer2,buffer1,
                                                currAttribute->lex_name,
							      MAX_COL);
                                string_concat(buffer1,buffer2,":   ",MAX_COL);
                                string_concat(buffer2,buffer1,"Seq Of ",
                                                            MAX_COL);
                                string_concat(buffer1,buffer2,
                                      ASeqRef->syn_component->lex_name,MAX_COL);
                                (void)append_line(buffer1 ,basic_type_info_list);  
                                }
                            break;
                        } /* switch */
                    } /* foreach */
	        } /* if */
	} /* foreachin */

    return(TRUE);

    } /* find_basic_type_occurrence */

 
/*****************************************************************************/
boolean find_process_properties(process_name, proc_prop_list)
    line_type process_name;
    list_type *proc_prop_list;
    {
    ProcessRef APrRef;
    ProcessEntity APr;
    StructureRef AStRef;
    SETPort SPort;
    Port APort;
    processStatement curr_processStatement;
    SEQprocessStatement SSEQprocessStatement;

    boolean found_refined, found_invariant, found_pre_ports,
	    found_post_ports, found_target;


    init_list(proc_prop_list);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(process_name)) == NULL)
	return(FALSE);
    /*
     * initialize the flags for the search
     */
    found_refined      = FALSE;
    found_invariant    = FALSE;
    found_pre_ports    = FALSE;
    found_post_ports   = FALSE;
    found_target       = FALSE;
    /*
     *  check if there is a process the current process has been REFINED FROM
     */
    if (typeof(APr->syn_refines)==KProcessRef) {
	APrRef = APr->syn_refines.VProcessRef;
	if (typeof(APrRef->sem_entity) == KProcessEntity) {
	    found_refined = TRUE;
	    }
	}

    /*
     *  check if there is a structure the current process has been declared
     *  as an INVARIANT of
     */
    if (typeof(APr->syn_invariant) == KStructureRef) {
	AStRef = APr->syn_invariant.VStructureRef;
	if (typeof(AStRef->sem_entity) == KStructureEntity) {
	    found_invariant = TRUE;
	    }
	}
    /*
     * check for PRE and POST PORTS of the current process
     * APr--> sem_ports: Set Of Port
     * Port-->sem_portType: PortType
     * PortType ::= PrePort | PostPort
     */
     foreachinSETPort(APr->sem_ports, SPort, APort) {
         if (typeof(APort->syn_data->sem_entity) == KStructureEntity) {
		if ( typeof(APort->sem_portType) == KPrePort )
		    found_pre_ports = TRUE;
		else
		    found_post_ports = TRUE;
             }
         } /* for */
   /*
    * check if there are TARGET statements
    */
/*
    if ( typeof(APr->sem_target) == KTargetEntity )
	found_target = TRUE;
*/
    foreachinSEQprocessStatement(APr->syn_body,SSEQprocessStatement,
                                               curr_processStatement)
        if (typeof(curr_processStatement) == KtargetStmt)
            found_target = TRUE;

    /*
     * load the output list with attributes
     */
    if (found_invariant)
	(void)append_line("Invariant",proc_prop_list);
    if (found_pre_ports)
	(void)append_line("Pre Ports",proc_prop_list);
    if (found_post_ports)
	(void)append_line("Post Ports",proc_prop_list);
    if (found_refined)
	(void)append_line("Refined from",proc_prop_list);
    if (found_target)
	(void)append_line("Target",proc_prop_list);

    return(TRUE);

    } /* find_proc_properties */


 
/*****************************************************************************/
boolean find_refined_proc(proc_name, refined_name)
    line_type proc_name;
    list_type *refined_name;
    {
    ProcessRef APrRef;
    ProcessEntity APr;
    ProcessEntity pr;

    init_list(refined_name);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(proc_name)) == NULL)
	return(FALSE);
    /*
     *  get the process the current process has been REFINED FROM
     */
    if (typeof(APr->syn_refines)==KProcessRef) {
	APrRef = APr->syn_refines.VProcessRef;
	if (typeof(APrRef->sem_entity) == KProcessEntity) {
            pr = APrRef->sem_entity.VProcessEntity;
            (void)append_line(pr->lex_name, refined_name);
	    }
	}

    return(TRUE);

    } /* find_refined_proc */

 
/*****************************************************************************/
boolean find_assert_proc(proc_name, assert_list )
    line_type proc_name;
    list_type *assert_list;
    {
    init_list(assert_list);

    } /* find_assert_proc */


 
/*****************************************************************************/
boolean find_invariant(proc_name, invariant_name)
    line_type proc_name;
    list_type *invariant_name;
    {

    ProcessEntity APr;
    StructureRef AStRef;
    StructureEntity st;

    init_list(invariant_name);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(proc_name)) == NULL)
	return(FALSE);

    if (typeof(APr->syn_invariant) == KStructureRef) {
	AStRef = APr->syn_invariant.VStructureRef;
	if (typeof(AStRef->sem_entity) == KStructureEntity) {
            st = AStRef->sem_entity.VStructureEntity;
            (void)append_line(st->lex_name,invariant_name);
	    }
	}

    return(TRUE);

    } /* find_invariant */

/*****************************************************************************/
boolean find_pre_ports(proc_name, pre_ports_list )
    line_type proc_name;
    list_type *pre_ports_list;
    {
    ProcessEntity APr;
    SETPort SPort;
    Port APort;

    init_list(pre_ports_list);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(proc_name)) == NULL)
	return(FALSE);

    /*
     * APr--> sem_ports: Set Of Port
     * Port-->sem_portType: PortType
     * PortType ::= PrePort | PostPort
     */
	foreachinSETPort(APr->sem_ports, SPort, APort) {
	    if (typeof(APort->syn_data->sem_entity) == KStructureEntity) {
		if ( typeof(APort->sem_portType) == KPrePort ) {
                    (void)append_line(APort->lex_name, pre_ports_list);
                    } 
		}
	    } /* for */

    return(TRUE);

    } /* find_pre_ports */

 
/*****************************************************************************/
boolean find_post_ports(proc_name, post_ports_list )
    line_type proc_name;
    list_type *post_ports_list;
    {
    ProcessEntity APr;
    SETPort SPort;
    Port APort;

    init_list(post_ports_list);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(proc_name)) == NULL)
	return(FALSE);

    /*
     * APr--> sem_ports: Set Of Port
     * Port-->sem_portType: PortType
     * PortType ::= PrePort | PostPort
     */
	foreachinSETPort(APr->sem_ports, SPort, APort) {
	    if (typeof(APort->syn_data->sem_entity) == KStructureEntity) {
		if ( typeof(APort->sem_portType) == KPostPort ) {
                    (void)append_line(APort->lex_name, post_ports_list);
                    } 
		}
	    } /* for */

    return(TRUE);

    } /* find_post_ports */

 
/*****************************************************************************/
boolean find_proc_unnamed_assert(proc_name, proc_unnamed_assert_list)
    line_type proc_name;
    list_type *proc_unnamed_assert_list;
    {
    init_list(proc_unnamed_assert_list);

    } /* find_post_ports */

 
/*****************************************************************************/
boolean find_proc_assert_body(proc_name, assert_name, assertion_body_list)
    line_type proc_name;
    line_type assert_name;
    list_type *assertion_body_list;
    {
    init_list(assertion_body_list);

    } /* find_proc_assert_body */

 
/*****************************************************************************/
boolean find_port_structure(proc_name, port_name, struct_and_repr_stat_list)
    line_type proc_name;
    line_type port_name;
    list_type *struct_and_repr_stat_list;
    {

    ProcessEntity APr;
    SETPort SPort;
    Port APort;
    StructureEntity st;
    SEQRepRef SSEQRepRef;
    RepRef currRepRef;
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    nameToken curr_nameToken;
    line_type buffer1, buffer2;

    init_list(struct_and_repr_stat_list);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(proc_name)) == NULL)
	return(FALSE);

    /*
     * APr--> sem_ports: Set Of Port
     * Port-->sem_portType: PortType
     * PortType ::= PrePort | PostPort
     */
	foreachinSETPort(APr->sem_ports, SPort, APort) {
	    if (typeof(APort->syn_data->sem_entity) == KStructureEntity) {
                st = APort->syn_data->sem_entity.VStructureEntity;
                if ( strcmp(APort->lex_name, port_name) == 0 ) {
                    string_concat(buffer1,"Structure: ",st->lex_name,MAX_COL);

                    (void)append_line(buffer1,struct_and_repr_stat_list);
                    /*
                     * find representation for port
                     */
                    string_copy(buffer1,"  ",MAX_COL);
                    foreachinSEQRepRef(APort->sem_rep, SSEQRepRef, currRepRef) {
                        foreachinSEQparameter(currRepRef->syn_id, SSEQparameter,
                                                          curr_parameter) {
                            if (typeof(curr_parameter) == KnameToken) {
                                curr_nameToken = curr_parameter.VnameToken;
                                string_concat(buffer2,buffer1,
                                          curr_nameToken->lex_name, MAX_COL);
                                string_concat(buffer1, buffer2, " ", MAX_COL);
                                }
                            } /* foreachin */
                        (void)append_line(buffer1,struct_and_repr_stat_list);
                        } /* foreachin */
                    break;
                    } /* if strcmp */
		} /* if typeof */
	    } /* foreachinSETPort  */

    return(TRUE);

    } /* find_port_structure */



/*****************************************************************************/
boolean find_target_info(proc_name, target_info_list)
    line_type proc_name;
    list_type *target_info_list;
    {
    ProcessEntity APr;
    processStatement curr_processStatement;
    SEQprocessStatement SSEQprocessStatement;
    targetStmt curr_targetStmt;
    boolean found_language, found_language_version;
    boolean found_operating_system, found_operating_system_version;
    boolean found_machine, found_runtime_version;
    line_type dummy_line;

    init_list(target_info_list);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(proc_name)) == NULL)
	return(FALSE);

    /*
     * look inside the syn body to get statements related to Target
     */
    found_language = FALSE;
    found_language_version = FALSE;
    found_operating_system = FALSE;
    found_operating_system_version = FALSE;
    found_machine = FALSE;
    found_runtime_version = FALSE;

    foreachinSEQprocessStatement(APr->syn_body, SSEQprocessStatement,
                                                curr_processStatement) {
        if ( typeof(curr_processStatement) == KtargetStmt ) {
           curr_targetStmt = curr_processStatement.VtargetStmt;
           if (find_language_version(curr_targetStmt->syn_id, dummy_line)) {
                found_language_version = TRUE;
                }
           if (find_language(curr_targetStmt->syn_id, dummy_line)) {
                found_language = TRUE;
                }
           if (find_operating_system(curr_targetStmt->syn_id, dummy_line)) {
                found_operating_system = TRUE;
                }
           if (find_operating_system_version(curr_targetStmt->syn_id, 
                                                dummy_line)) {
                found_operating_system_version = TRUE;
                }
           if (find_machine(curr_targetStmt->syn_id, dummy_line)) {
                found_machine = TRUE;
                }
           if (find_runtime_version(curr_targetStmt->syn_id, dummy_line)) {
                found_runtime_version = TRUE;
                }
           } /* if typeof */
        } /* foreach inSEQprocessStatement */
    /*
     * now fill in the list of items
     */
    if (found_language)
        (void)append_line("Language", target_info_list);
    if (found_language_version)
        (void)append_line("Language Version",target_info_list);

    if (found_operating_system)
        (void)append_line("Operating System", target_info_list);
    if (found_operating_system_version)
        (void)append_line("Operating System Version", target_info_list);
    if (found_machine)
        (void)append_line("Machine", target_info_list); 
    if (found_runtime_version)
        (void)append_line("Runtime System Version", target_info_list);

    return(TRUE);

    } /* find_target_info */



/*****************************************************************************/
boolean find_target_item_info(proc_name, target_item, info_list)
    line_type proc_name;
    line_type target_item;
    list_type *info_list;
    {
    ProcessEntity APr;
    processStatement curr_processStatement;
    SEQprocessStatement SSEQprocessStatement;
    targetStmt curr_targetStmt;
    line_type info_line;

    init_list(info_list);
    /*
     * find a process with the given name, if any
     */
    if ( (APr = find_process(proc_name)) == NULL)
	return(FALSE);

    /*
     * look inside the syn body to get info related to the given item
     */

    foreachinSEQprocessStatement(APr->syn_body, SSEQprocessStatement,
                                                curr_processStatement) {
        if ( typeof(curr_processStatement) == KtargetStmt ) {
           curr_targetStmt = curr_processStatement.VtargetStmt;
           if (strcmp( target_item,"Language Version") == 0) {
               if (find_language_version(curr_targetStmt->syn_id, info_line)) {
                   (void)append_line(info_line,info_list);
                   }
               }
           else if (strcmp( target_item,"Language") == 0) {
               if (find_language(curr_targetStmt->syn_id, info_line)) {
                    (void)append_line(info_line,info_list);
                    }
               }
           else if (strcmp( target_item,"Operating System") == 0) {
               if (find_operating_system(curr_targetStmt->syn_id, info_line)) {
                   (void)append_line(info_line,info_list);
                   }
               }
           else if (strcmp( target_item,"Operating System Version") == 0) {
               if (find_operating_system_version(curr_targetStmt->syn_id, 
                                                info_line)) {
                   (void)append_line(info_line,info_list);
                   }
               }
           else if (strcmp( target_item,"Machine") == 0) {
               if (find_machine(curr_targetStmt->syn_id, info_line)) {
                   (void)append_line(info_line,info_list);
                   }
               }
           else if (strcmp( target_item,"Runtime System Version") == 0) {
               if (find_runtime_version(curr_targetStmt->syn_id, info_line)) {
                   (void)append_line(info_line,info_list);
                   }
               }
           } /* if typeof */
        } /* foreachin */

    return(TRUE);

    } /* find_target_item_info */


/**************************** PRIVATE ROUTINES ************************/



/*
 *  Routine: find_structure
 *
 *  Description:it looks for a structure given the name 
 *
 *  Arguments: struct_name -- (IN) name of the structure
 *
 *  Return Value: structure required, if found. Otherwise NULL
 *
 *  Side Effects: none
 */
static StructureEntity find_structure(struct_name)
line_type struct_name;
{
    SEQDeclaration  SDecl;          /* remaining SEQ of decl. to process */
    Declaration     currentDecl;    /* current declaration in syn_body */
    StructureEntity AStr;
    ProcessEntity APr;
    SEQStructureOrProcessRef  SStOrPrRef;
    StructureOrProcessRef  AStOrPrRef;

    if (selected_item_is_defaultprocessinvariant(struct_name)) {
      if (APr = find_process(struct_name+2)) {
	return APr->sem_invariant;
      }
      return NULL;
    }
      
    /*
     * process the declarations: consider structures only
     */
    foreachinSEQDeclaration(compUnit->syn_body, SDecl, currentDecl) {
      switch(typeof(currentDecl)) {
      case KStructureEntity:
	if (AStr = find_structure_from(currentDecl.VStructureEntity,struct_name))
	  return AStr;
	break;
      case KImportDecl:
	foreachinSEQStructureOrProcessRef(currentDecl.VImportDecl->syn_specs,SStOrPrRef,AStOrPrRef) {
	  if (typeof(AStOrPrRef->sem_entity) == KStructureEntity) {
	    if (AStr = find_structure_from(AStOrPrRef->sem_entity.VStructureEntity,struct_name))
	      return AStr;
	  }
	}
	break;
      }
    }
    
    return(NULL);
    
} /* find_structure */

static StructureEntity find_structure_from(StEn,struct_name)
StructureEntity StEn;
line_type struct_name;
{
  StructureRef AStRef;
  SEQStructureRef SStRef;
  StructureEntity AStr;

  if (strcmp(StEn->lex_name,struct_name) == 0)	      
    return(StEn);
  foreachinSEQStructureRef(StEn->syn_from,SStRef,AStRef) {
    switch(typeof(AStRef->sem_entity)) {
    case KStructureEntity:
      if (AStr = find_structure_from(AStRef->sem_entity.VStructureEntity,struct_name))
	return AStr;
      break;
    }
  }
  return NULL;
}


/*
 *  Routine: find_process
 *
 *  Description:it looks for a process given the name 
 *
 *  Arguments: struct_name -- (IN) name of the  process
 *
 *  Return Value: process required, if found. Otherwise NULL
 *
 *  Side Effects: none
 */
static ProcessEntity find_process(proc_name)
    line_type proc_name;
    {
    SEQDeclaration  SDecl;          /* remaining SEQ of decl. to process */
    Declaration     currentDecl;    /* current declaration in syn_body */

    /*
     * process the declarations: consider processes only
     */
    foreachinSEQDeclaration(compUnit->syn_body, SDecl, currentDecl) {
        if (typeof(currentDecl) == KProcessEntity ) {
	    if (strcmp(currentDecl.VProcessEntity -> lex_name,
                                               proc_name) == 0)
                return(currentDecl.VProcessEntity);
            } /* if */
	} /* for */

    return(NULL);

    } /* find_process */


/*
 *  Routine: get_node
 *
 *  Description: given a structure and a node name, it returns
 *               the corresponding node
 *
 *  Arguments: AStr -- (IN) structure to search
 *             node_name -- (IN) name of the wanted node 
 *
 *  Return Value: the requested node if found, otherwise NULL 
 *
 *  Side Effects: none
 */
static Class get_node(AStr, node_name)
    StructureEntity AStr;
    line_type node_name;
    {
    SETTypeEntity tailSETTypeEntity;
    TypeEntity currTypeEntity;
    Class ANode;
    /*
     * loop on the types of the structure comparing all the names
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		ANode = currTypeEntity.VClass;
		if ( emptySETClass(ANode->sem_subclasses) ) {
                    if (strcmp(ANode->sem_name, node_name) == 0)
                        return(ANode);
                        }
                }
        } /* foreachin */

    return(NULL);

    } /* get_node */



/*
 *  Routine: get_class
 *
 *  Description: given a structure and a class name, it returns
 *               the corresponding node
 *
 *  Arguments: AStr -- (IN) structure to search
 *             class_name -- (IN) name of the wanted class
 *
 *  Return Value: the requested class if found, otherwise NULL 
 *
 *  Side Effects: none
 */
static Class get_class(AStr, class_name)
    StructureEntity AStr;
    line_type class_name;
    {
    SETTypeEntity tailSETTypeEntity;
    TypeEntity currTypeEntity;
    Class AClass;
    /*
     * loop on the types of the structure comparing all the names
     */
    foreachinSETTypeEntity(AStr->sem_types, tailSETTypeEntity, currTypeEntity) {
	    if ( typeof(currTypeEntity) == KClass ) {
		AClass = currTypeEntity.VClass;
		if ( ! emptySETClass(AClass->sem_subclasses) ) {
                    if (strcmp(AClass->sem_name, class_name) == 0)
                        return(AClass);
                        }
                }
        } /* foreachin */

    return(NULL);

    } /* get_class */

/*
 *  Routine: is_basic_type
 *
 *  Description: it checks if an identifier name is a basic name identifier
 *
 *  Arguments: lex_name -- (IN) name to test
 *
 *  Return Value:  TRUE if a basic type identifier, FALSE otherwise.
 *
 *  Side Effects: none
 */
static boolean is_basic_type(lex_name)
    char *lex_name;
    {
    if ( strcmp(lex_name, "Integer") == 0)
        return(TRUE);
    else if ( strcmp(lex_name, "Rational") == 0)
        return(TRUE);
    else if ( strcmp(lex_name, "Boolean") == 0 )
        return(TRUE);
    else if ( strcmp(lex_name, "String") == 0 )
        return(TRUE);
    else if ( strcmp(lex_name, "Set Of") == 0 )
        return(TRUE);
    else if ( strcmp(lex_name, "Seq Of") == 0 )
        return(TRUE);
    else
        return(FALSE);

    } /* is_basic_type */




/*
 *  Routine: is_private_type
 *
 *  Description: it checks if a given identifier is a class or a node or
 *               a basic type: if not, it is a private type
 *
 *  Arguments: AStr -- (IN) structure to consider
 *             type_name -- (IN) identifier to check
 *
 *  Return Value: TRUE if private type, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean is_private_type(AStr, type_name)
    StructureEntity AStr;
    line_type type_name;
    {
    if (is_basic_type(type_name))
        return(FALSE);

    if ( (get_class(AStr,type_name) != NULL) || 
                          (get_node(AStr,type_name) != NULL) )
        return(FALSE);
    else
        return(TRUE);

    } /* is_private_type */



/*
 *  Routine: find_language
 *
 *  Description: it parses a sequence of parameters to find the target
 *               language
 *
 *  Arguments: ASEQparameter -- (IN) sequence of parameters to check
 *             language_info -- (OUT) language name
 *
 *  Return Value: TRUE if found the language, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean find_language(ASEQparameter, language_info)
    SEQparameter ASEQparameter;
    line_type language_info;
    {
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    nameToken curr_nameToken;
    int n_token;
    boolean found1, found2;
    /*
     * check for matching in the first tow tokens
     */
    n_token = 0;
    found1 = FALSE;
    found2 = FALSE;
    foreachinSEQparameter(ASEQparameter, SSEQparameter, curr_parameter) {
        n_token++;
        if (typeof(curr_parameter) == KnameToken) {
            curr_nameToken = curr_parameter.VnameToken;
            if ( (n_token == 1) && (strcmp(curr_nameToken->lex_name,
                                          "Language") == 0))
                found1 = TRUE;
            if ( (n_token == 2) && (strcmp(curr_nameToken->lex_name,
                                          "Version") != 0))
                found2 = TRUE;
            }
        if (n_token == 2) break;
        } /* foreachin */
     
    if ( ! (found1 && found2) ) {
        return(FALSE);
        }
    /*
     * if matching, then get the info from the tail of the sequence
     */
    catenate_tokens(ASEQparameter, n_token,  language_info);

    return(TRUE);

    } /* find_language */


/*
 *  Routine: find_language_version
 *
 *  Description: it parses a sequence of parameters to find the target
 *               language version
 *
 *  Arguments: ASEQparameter -- (IN) sequence of parameters to check
 *             language_version_info -- (OUT) language version name
 *
 *  Return Value: TRUE if found the language, version otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean find_language_version(ASEQparameter, language_version_info)
    SEQparameter ASEQparameter;
    line_type language_version_info;
    {
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    nameToken curr_nameToken;
    int n_token;
    boolean found1, found2;
    /*
     * check for matching in the first tow tokens
     */
    n_token = 0;
    found1 = FALSE;
    found2 = FALSE;
    foreachinSEQparameter(ASEQparameter, SSEQparameter, curr_parameter) {
        n_token++;
        if (typeof(curr_parameter) == KnameToken) {
            curr_nameToken = curr_parameter.VnameToken;
            if ( (n_token == 1) && (strcmp(curr_nameToken->lex_name,
                                          "Language") == 0))
                found1 = TRUE;
            if ( (n_token == 2) && (strcmp(curr_nameToken->lex_name,
                                          "Version") == 0))
                found2 = TRUE;
            }
        if (n_token == 2) break;
        } /* foreachin */
     
    if ( ! (found1 && found2) ) {
        return(FALSE);
        }
    /*
     * if matching, then get the info from the tail of the sequence
     */
     catenate_tokens(ASEQparameter, n_token + 1,  language_version_info);

     return(TRUE);

    } /* find_language_version */


/*
 *  Routine: find_operating_system 
 *
 *  Description: it parses a sequence of parameters to find the 
 *               operating system
 *
 *  Arguments: ASEQparameter -- (IN) sequence of parameters to check
 *             operating_system_info -- (OUT) operating system name
 *
 *  Return Value: TRUE if found the operating system, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean find_operating_system(ASEQparameter, operating_system_info)
    SEQparameter ASEQparameter;
    line_type operating_system_info;
    {
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    nameToken curr_nameToken;
    int n_token;
    boolean found1, found2, found3;
    /*
     * check for matching in the first tow tokens
     */
    n_token = 0;
    found1 = FALSE;
    found2 = FALSE;
    found3 = FALSE;
    foreachinSEQparameter(ASEQparameter, SSEQparameter, curr_parameter) {
        n_token++;
        if (typeof(curr_parameter) == KnameToken) {
            curr_nameToken = curr_parameter.VnameToken;
            if ( (n_token == 1) && (strcmp(curr_nameToken->lex_name,
                                          "Operating") == 0))
                found1 = TRUE;
            if ( (n_token == 2) && (strcmp(curr_nameToken->lex_name,
                                          "System") == 0))
                found2 = TRUE;
            if ( (n_token == 3) && (strcmp(curr_nameToken->lex_name,
                                          "Version") != 0))
                found3 = TRUE;
            }
        if (n_token == 3) break;
        } /* foreachin */
     
    if ( ! (found1 && found2 && found3) ) {
        return(FALSE);
        }
    /*
     * if matching, then get the info from the tail of the sequence
     */
     catenate_tokens(ASEQparameter, n_token,  operating_system_info);

     return(TRUE);

    } /* find_operating_system */



/*
 *  Routine: find_operating_system_version 
 *
 *  Description: it parses a sequence of parameters to find the 
 *               operating system version
 *
 *  Arguments: ASEQparameter -- (IN) sequence of parameters to check
 *             os_version_info -- (OUT) operating system version info 
 *
 *  Return Value: TRUE if found the operating system version, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean find_operating_system_version(ASEQparameter, os_version_info)
    SEQparameter ASEQparameter;
    line_type os_version_info;
    {
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    nameToken curr_nameToken;
    int n_token;
    boolean found1, found2, found3;
    /*
     * check for matching in the first tow tokens
     */
    n_token = 0;
    found1 = FALSE;
    found2 = FALSE;
    found3 = FALSE;
    foreachinSEQparameter(ASEQparameter, SSEQparameter, curr_parameter) {
        n_token++;
        if (typeof(curr_parameter) == KnameToken) {
            curr_nameToken = curr_parameter.VnameToken;
            if ( (n_token == 1) && (strcmp(curr_nameToken->lex_name,
                                          "Operating") == 0))
                found1 = TRUE;
            if ( (n_token == 2) && (strcmp(curr_nameToken->lex_name,
                                          "System") == 0))
                found2 = TRUE;
            if ( (n_token == 3) && (strcmp(curr_nameToken->lex_name,
                                          "Version") == 0))
                found3 = TRUE;
            }
        if (n_token == 3) break;
        } /* foreachin */
     
    if ( ! (found1 && found2 && found3) ) {
        return(FALSE);
        }
    /*
     * if matching, then get the info from the tail of the sequence
     */
     catenate_tokens(ASEQparameter, n_token + 1,  os_version_info);

     return(TRUE);

    } /* find_operating_system_version */



/*
 *  Routine: find_machine
 *
 *  Description: it parses a sequence of parameters to find the 
 *               machine
 *
 *  Arguments: ASEQparameter -- (IN) sequence of parameters to check
 *             machine_info  --(IN) machine name
 *
 *  Return Value: TRUE if found the machine, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean find_machine(ASEQparameter, machine_info)
    SEQparameter ASEQparameter;
    line_type machine_info;
    {
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    nameToken curr_nameToken;
    int n_token;
    boolean found1;
    /*
     * check for matching in the first token
     */
    n_token = 0;
    found1 = FALSE;
    foreachinSEQparameter(ASEQparameter, SSEQparameter, curr_parameter) {
        n_token++;
        if (typeof(curr_parameter) == KnameToken) {
            curr_nameToken = curr_parameter.VnameToken;
            if ( (n_token == 1) && (strcmp(curr_nameToken->lex_name,
                                          "Machine") == 0))
                found1 = TRUE;
            }
        if (n_token == 1) break;
        } /* foreachin */
     
    if ( ! found1 ) {
        return(FALSE);
        }
    /*
     * if matching, then get the info from the tail of the sequence
     */
    catenate_tokens(ASEQparameter, n_token + 1,  machine_info);

    return(TRUE);

    } /* find_machine */


/*
 *  Routine: find_runtime_version
 *
 *  Description: it parses a sequence of parameters to find the 
 *               runtime version
 *
 *  Arguments: ASEQparameter -- (IN) sequence of parameters to check
 *             runtime_version_info -- (OUT) runtime version info
 *
 *  Return Value: TRUE if found the runtime version, otherwise FALSE
 *
 *  Side Effects: none
 */
static boolean find_runtime_version(ASEQparameter, runtime_version_info)
    SEQparameter ASEQparameter;
    line_type runtime_version_info;
    {
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    nameToken curr_nameToken;
    int n_token;
    boolean found1, found2;
    /*
     * check for matching in the first two tokens
     */
    n_token = 0;
    found1 = FALSE;
    found2 = FALSE;
    foreachinSEQparameter(ASEQparameter, SSEQparameter, curr_parameter) {
        n_token++;
        if (typeof(curr_parameter) == KnameToken) {
            curr_nameToken = curr_parameter.VnameToken;
            if ( (n_token == 1) && (strcmp(curr_nameToken->lex_name,
                                          "Runtime") == 0))
                found1 = TRUE;
            if ( (n_token == 2) && (strcmp(curr_nameToken->lex_name,
                                          "Version") == 0))
                found2 = TRUE;
            }
        if (n_token == 2) break;
        } /* foreachin */
     
    if ( ! (found1 && found2) ) {
        return(FALSE);
        }
    /*
     * if matching, then get the info from the tail of the sequence
     */
     catenate_tokens(ASEQparameter, n_token + 1,  runtime_version_info);

     return(TRUE);

    } /* find_runtime_version */



/*
 *  Routine: catenate_tokens
 *
 *  Description: given a seq of parameters, it catenates their names into
 *               a string of chars. Names are separated by blanks
 *
 *  Arguments: ASEQparameter -- (IN) sequence to catenate
 *             first_token -- (IN) first token to consider in the sequence
 *             line -- (OUT) resulting string
 *
 *  Return Value: none
 *
 *  Side Effects: none
 */
static void catenate_tokens(ASEQparameter,first_token,line)
    SEQparameter ASEQparameter;
    int first_token;
    line_type line;
    {
    SEQparameter SSEQparameter;
    parameter curr_parameter;
    line_type buffer;
    int n_token;
    nameToken AnameToken;
    integerToken AintegerToken;
    rationalToken ArationalToken;
    stringToken AstringToken;

    line[0] = '\0';
    /*
     * parameter ::= nameToken | otherToken
     * nameToken => lex_name
     * otherToken ::= integerToken | rationalToken | stringToken
     * otherToken => lex_externalform
     */
    n_token = 0;
    foreachinSEQparameter(ASEQparameter,SSEQparameter,curr_parameter) {
        n_token++;
        if (n_token >= first_token) {
            switch(typeof(curr_parameter)) {
                case KnameToken:
                    AnameToken = curr_parameter.VnameToken;
                    string_concat(buffer, line, AnameToken->lex_name,MAX_COL);
                    break;
                case KintegerToken:
                    AintegerToken = curr_parameter.VintegerToken;
                    string_concat(buffer, line, 
                               AintegerToken->lex_externalform,MAX_COL);
                    break;
                case KrationalToken:
                    ArationalToken = curr_parameter.VrationalToken;
                    string_concat(buffer, line, 
                               ArationalToken->lex_externalform,MAX_COL);
                    break;
                case KstringToken:
                    AstringToken = curr_parameter.VstringToken;
                    string_concat(buffer, line, 
                               AstringToken->lex_externalform,MAX_COL);
                    break;
                    } /* switch */
            string_concat(line,buffer," ",MAX_COL);
            } /* if */
        } /* foreachinSEQparameter */
    /*
     * eliminate the last blank left to the end of the string
     */
     if ( (strlen(line) > 0) && (line[strlen(line) - 1] == ' ') )
         line[strlen(line) - 1] = '\0';

    } /* catenate_tokens */
