/***********************************************************************\ 
*									* 
*   File: scorpion/src/candlewalk/main.c 
*				 					* 
*   Copyright (C) 1991 C. Alexander Nelson, Karen Shannon, Richard Snodgrass
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#include "candlewalk.h"
#include <stdio.h>
#include <string.h>
#include <sys/file.h>

#define FOUND TRUE
#define NOTFOUND FALSE
#define DEFAULTPLAINTABSETTING 8
#define DEFAULTFORMATTEDTABSETTING DEFAULTPLAINTABSETTING
#define DEFAULTLATEXTABSETTING 8
extern Boolean ReaderOK;
Boolean debugswitch = FALSE;	/* print debugging information */
Boolean recursiveswitch = FALSE; /* recursively process included files */
Boolean verboseswitch = FALSE;	/* print out verbose messages */
int tabsetting = 0; /* Tab stops are set every tabsetting spaces */
int currentkillmode = 2;	/* kill mode used in Print***String */
/* killmode      = 0: insert nothing */
/*                 1: insert attribute/subclass tabbing kill line */
/*			after last \n, as well as initial tab */
/*                 2: insert normal tabbing kill line after */
/*			first \n */

/* output formats */
Boolean plainswitch = FALSE;	/* simply unparse into plain text */
Boolean formattedplainswitch = FALSE; /* plain text formatted with tabs */
Boolean LaTeXswitch = FALSE;	/* format output for LaTeX */
Boolean sourceposswitch = FALSE; /* add source position to Candle */

Boolean formattedcandleswitch = FALSE; /* format LexInfo attribute in Candle */

SEQString includes;		/* directories to look for files in */
SETcompilationUnit compUnits;	/* units that have been processed */
SETDeclaration existingdecls; /* structures and processes in main compUnit */
FILE *inputfp, *outputfp;
int currentfilenumber = 1;

Boolean FindFilePath();
void exit();
void usage(),revision_number();

main(argc, argv)
int argc;
char *argv[];
{
  compilationUnit compUnit, AcompUnit;
  SETcompilationUnit ScU;
  int i, count;
  String mainfilename = NULL;

  /* initialize */
  inputfp = stdin;
  outputfp = stdout;
  initializeSEQString(includes);

  /* process invocation arguments */
  if (argc == 1) {
    usage(argv[0]);
    exit(1);
  }

  for (i=1; i<argc; i++) {
/* first, miscellaneous switches */
    if (!strcmp(argv[i], "-d")) {
      debugswitch = TRUE;
    }
    else if (!strcmp(argv[i], "-V")) {
       revision_number(argv[0]);
       exit(0);
    }
    else if (!strcmp(argv[i], "-o")) {
      if (i+1 >= argc) {
	(void) fprintf(stderr, 
          "candlewalk: -o switch not followed with an file name\n");
         usage(argv[0]);
	 exit(1);
      }
      else {
	++i;
	if (argv[i][0] != '-') {
	  if ((outputfp = fopen(argv[i], "w")) == NULL){
	    (void) fprintf(stderr, 
              "candlewalk: can't open output file named %s.\n", argv[i]);
            usage(argv[0]);
	    exit(1);
	  }
	}			/* otherwise, stdout already open */
      }
    }
    else if (!strcmp(argv[i], "-r")) {
      recursiveswitch = TRUE;
    }
    else if (!strncmp(argv[i], "-t", 2)) {
      tabsetting = atoi(&argv[i][2]);
      if (tabsetting < 1) {
	(void) fprintf(stderr,
		"candlewalk: an illegal tab stop of %d was specified; ignored.\n",
		tabsetting);
	(void) fflush(stderr);
	tabsetting = 0;
      }
    }
    else if (!strcmp(argv[i], "-v")) {
      verboseswitch = TRUE;
    }
    else if (!strncmp(argv[i], "-I", 2)) {
      appendrearSEQString(includes, NewString(&(argv[i][2])));
    }

/* output option */
    else if (!strcmp(argv[i], "-FC")) {
      formattedcandleswitch = FALSE;
    }
    else if (!strcmp(argv[i], "-FP")) {
      formattedplainswitch = FALSE;
    }
    else if (!strcmp(argv[i], "-L")) {
      LaTeXswitch = TRUE;
    }
    else if (!strcmp(argv[i], "-P")) {
      plainswitch = TRUE;
    }
    else if (!strcmp(argv[i], "-S")) {
      sourceposswitch = TRUE;
    }

/* input file name */
    else if (argv[i][0] == '-') {
      if (strcmp(argv[i], "-")) {
	(void) fprintf(stderr, "candlewalk: illegal option (%s) ignored.\n", 
          argv[i]);
	(void) fflush(stderr);
      }
      else if (i == argc-1) {	/* this is the last option */
	mainfilename = NewString("stdin"); /* otherwise, stdin already open */
      }
      else {
	(void) fprintf(stderr,
		"candlewalk: illegal option (\"-\" not as last option) ignored.\n");
	(void) fflush(stderr);
      }
    }
    else if (i==argc-1) {
      mainfilename = argv[i];
      if ((inputfp=fopen(mainfilename,"r"))==NULL) {
	(void) fprintf(stderr, "candlewalk: can't open input file named %s.\n",
          mainfilename);
        usage(argv[0]);
	exit(1);
      }
    }
    else {
      (void) fprintf(stderr, 
        "candlewalk: too many file names specified; %s ignored.\n",
	      argv[i]);
      (void) fflush(stderr);
    }
  }

  if (debugswitch) verboseswitch = TRUE;
  count = 0;
  if (plainswitch) {
    count++;
    if (tabsetting==0)
      tabsetting = DEFAULTPLAINTABSETTING;
  }
  if (formattedplainswitch) {
    count++;
    if (tabsetting==0)
      tabsetting = DEFAULTFORMATTEDTABSETTING;
  }
  if (LaTeXswitch) {
    count++;
    if (tabsetting==0)
      tabsetting = DEFAULTLATEXTABSETTING;
  }
  if (sourceposswitch) count++;
  if (formattedcandleswitch) {
    count++;
    if (tabsetting==0)
      tabsetting = DEFAULTFORMATTEDTABSETTING;
    (void) fprintf(stderr, "candlewalk: -FC not yet supported.\n");
     exit(1);
  }
    

  /* check for valid arguments */
  if (count==0)
    plainswitch = TRUE;
  else if (count > 1) {
    (void) fprintf(stderr, 
      "candlewalk: should specify at most one output format.\n");
     usage(argv[0]);
     exit(1);
  }

  if (mainfilename == NULL) {
    (void) fprintf(stderr, "candlewalk: input file name not provided.\n");
    usage(argv[0]);
    exit(1);
  }

  if (recursiveswitch && LaTeXswitch) {
    (void) fprintf(stderr,
      "candlewalk: can't use both -L and -r simultaneously.\n");
    exit(1);
  }

  /* read in structure */
  if (verboseswitch) {
    (void) fprintf(stderr, "candlewalk: Reading input Candle file...\n");
    (void) fflush(stderr);
  }
    
  compUnit = input(inputfp);

  if (!ReaderOK) {
    (void) fprintf(stderr,"candlewalk: reader errors on input of %s.\n",
	    mainfilename);
    exit(-1);
  }
  compUnit->inv_filename = mainfilename;
	
  if (debugswitch) {
    (void) fprintf(stderr,
      "candlewalk: Successful read of input Candle file.\n");
    (void) fflush(stderr);
  }

  /* fix up Import clauses (bring in imported compilationUnits) */
  initializeSETDeclaration(existingdecls);
  if (recursiveswitch) {
    CollectDeclarations(compUnit); /* computes existingdecls */
    ProcessImportClauses(compUnit); /* computes compUnits */
  }
  else compUnit->inv_filenumber = 1;
    
  /* main treewalk */
  if (recursiveswitch)
    foreachinSETcompilationUnit(compUnits, ScU, AcompUnit) {
      ProcessCompUnit(AcompUnit);
    }
  else ProcessCompUnit(compUnit);	/* just do this one */

  if (sourceposswitch) {
    if (verboseswitch) {
      (void) fprintf(stderr, 
        "candlewalk: Updating source position in copied Attributes...\n");
      (void) fflush(stderr);
    }
    UpdateCopied(compUnit);

    if (verboseswitch) {
      (void) fprintf(stderr, "candlewalk: Writing CandlePos instance...\n");
      (void) fflush(stderr);
    }
    posout(outputfp, compUnit);	/* write out the compilationUnit with
				   source position */
  }
  else if (formattedcandleswitch) {
    if (verboseswitch) {
      (void) fprintf(stderr, "candlewalk: Writing Candle instance...\n");
      (void) fflush(stderr);
    }
    Candleout(outputfp, compUnit);
  }
  return 0;
}

CollectDeclarations(compUnit)
compilationUnit compUnit;
{
  SEQDeclaration SDecl;
  Declaration ADecl;

  foreachinSEQDeclaration(compUnit->syn_body, SDecl, ADecl) {
    if (typeof(ADecl)==KStructureEntity)
      CollectDeclsFromStructure(ADecl.VStructureEntity);
    else if (typeof(ADecl)==KProcessEntity)
      CollectDeclsFromProcess(ADecl.VProcessEntity);
  }
}

CollectDeclsFromStructure(aStructure)
StructureEntity aStructure;
{
  SEQStructureRef SSR;
  StructureRef ASR;
  Declaration Adecl;

  Adecl.VStructureEntity = aStructure;
  addSETDeclaration(existingdecls, Adecl);
  if (typeof(aStructure->syn_refines)==KStructureRef)
    CollectDeclsFromStructure(aStructure->syn_refines.VStructureRef->sem_entity.VStructureEntity);
  foreachinSEQStructureRef(aStructure->syn_from, SSR, ASR) {
    if (typeof(ASR->sem_entity)==KStructureEntity)
      CollectDeclsFromStructure(ASR->sem_entity.VStructureEntity);
  }
}

CollectDeclsFromProcess(aProcess)
ProcessEntity aProcess;
{
  Declaration Adecl;
  
  Adecl.VProcessEntity = aProcess;
  addSETDeclaration(existingdecls, Adecl);
  if (typeof(aProcess->syn_refines)==KProcessRef)
    CollectDeclsFromProcess(aProcess->syn_refines.VProcessRef->sem_entity.VProcessEntity);
}
	
ProcessImportClauses(compUnit)
compilationUnit compUnit;
{
  SEQDeclaration SDecl, SDecl2, newbody;
  Declaration ADecl, ADecl2;
  ImportDecl import;
  char *filename, *filenamebuf;
  Boolean foundDecl, foundfilepath;
  FILE *tmpfileptr;
  compilationUnit newcompUnit;

  if (compUnit->inv_filenumber!=0) return;
  compUnit->inv_filenumber = currentfilenumber++;
  addSETcompilationUnit(compUnits, compUnit);

  initializeSEQDeclaration(newbody);
  foreachinSEQDeclaration(compUnit->syn_body, SDecl, ADecl) {
    /* replace structures and processes in compUnit with existing ones */
    if ((typeof(ADecl)==KStructureEntity)
	|| (typeof(ADecl)==KProcessEntity)) {
	foundDecl = FALSE;
	foreachinSETDeclaration(existingdecls, SDecl2, ADecl2) {
	  if ((ADecl.VStructureOrProcess.IDLclassCommon->lex_name
	       ==ADecl2.VStructureOrProcess.IDLclassCommon->lex_name) &&
	      (typeof(ADecl)==typeof(ADecl2))) {
	      foundDecl = TRUE;
	      appendrearSEQDeclaration(newbody, ADecl2);
	    }
	};
	if (! foundDecl) 
	appendrearSEQDeclaration(newbody, ADecl);
      }
    else if (typeof(ADecl)==KImportDecl) {
      appendrearSEQDeclaration(newbody, ADecl);
      import = ADecl.VImportDecl;
      filename = import->syn_compUnit->lex_name;

      /* first check if file has already been read in */
      if (FindCompUnit(filename, &newcompUnit) != FOUND) {
	foundfilepath = FindFilePath(&filenamebuf, filename);
	if (foundfilepath) {
	  /* read in the new compilationUnit */
	  if ((tmpfileptr = fopen(filenamebuf, "r"))==NULL) {
	    (void) fprintf(stderr, 
              "candlewalk: error opening imported file: %s.\n", filenamebuf);
	    exit(1);
	  }
	  else {
	    newcompUnit = input(tmpfileptr);
	    if (!ReaderOK) {
	      (void)  fprintf(stderr, 
                "candlewalk: error reading imported file: %s.\n", filenamebuf);
	      exit(1);
	    }
	    newcompUnit->inv_filename = NewString(filenamebuf);
	  }
	}
	else {
	  (void) fprintf(stderr,"candlewalk: error finding Candle file: %s.\n",
		  filename);
	  exit(1);
	}
      }
      ProcessImportClauses(newcompUnit);
    }
  }
  compUnit->syn_body = newbody;
}

Boolean FindFilePath(finalfilenamebuf, filename)
char **finalfilenamebuf;
String filename;
{
  SEQString SStr;
  String AStr;
  char *filenamebuf;
  Boolean foundfilepath;
  String directory;
  int buflen;

  /* check for file in directories given */
  foundfilepath = FALSE;
  directory = NewString(".");
  buflen = strlen(filename)+strlen(directory)+6;
  filenamebuf = (char *)GetHeap(buflen);
  (void) sprintf(filenamebuf, "%s/%s.Cdl", directory, filename);
  if (access(filenamebuf, F_OK) != 0) {
    foreachinSEQString(includes, SStr, AStr) {
      directory = AStr;
      buflen = strlen(filename)+strlen(directory)+6;
      filenamebuf = (char *)GetHeap(buflen);
      (void) sprintf(filenamebuf, "%s/%s.Cdl", directory, filename);
      if (access(filenamebuf, F_OK) == 0) {
	foundfilepath = TRUE;
	*finalfilenamebuf = filenamebuf;
	break;
      }
    }
  }
  else {
    foundfilepath = TRUE;
    *finalfilenamebuf = filenamebuf;
  }
  return(foundfilepath);
}

int FindCompUnit(name, foundcunit)
String name;
compilationUnit *foundcunit;
{
  SETcompilationUnit Scu;
  compilationUnit Acu;
  char filename[300];
  int len;
  int found=NOTFOUND;
  char *cp;
    
  foreachinSETcompilationUnit(compUnits, Scu, Acu) {
    len = strlen(Acu->inv_filename);
    /* strip off preceding "./(etc)" */
    for (cp = Acu->inv_filename+len; cp !=Acu->inv_filename && *cp !='/';
	 cp--) ;
    if (*cp == '/')
      (void) strcpy(filename, cp+1);
    else (void) strcpy(filename, Acu->inv_filename);

    /* strip off suffix */
    len = strlen(filename);
    if (!strcmp(&(filename[len-4]), ".Cdl"))
      filename[len-4]='\0';

    if (!strcmp(filename, name)){
      found = FOUND;
      *foundcunit = Acu;
      break;
    }
    if (found == FOUND)
      break;
  }
  return(found);
}

/* SyntacticEntity */
Assertion IAssertion(this)
Assertion this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

Attribute IAttribute(this)
Attribute this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

CaseName ICaseName(this)
CaseName this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

Control IControl(this)
Control this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

Formal IFormal(this)
Formal this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

cyclicdef Icyclicdef(this)
cyclicdef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

noncyclicdef Inoncyclicdef(this)
noncyclicdef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

PrivateDefInstance IPrivateDefInstance(this)
PrivateDefInstance this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

Port IPort(this)
Port this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

StructureEntity IStructureEntity(this)
StructureEntity this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

ProcessEntity IProcessEntity(this)
ProcessEntity this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

/* EntityReference */
AssertRef IAssertRef(this)
AssertRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

AttributeRef IAttributeRef(this)
AttributeRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

AttributeRefs IAttributeRefs(this)
AttributeRefs this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

compUnitRef IcompUnitRef(this)
compUnitRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

DefineRef IDefineRef(this)
DefineRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

DefInstanceOrDefineRef IDefInstanceOrDefineRef(this)
DefInstanceOrDefineRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

NamedTypeRef INamedTypeRef(this)
NamedTypeRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

OperRef IOperRef(this)
OperRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

PortRef IPortRef(this)
PortRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

ProcessRef IProcessRef(this)
ProcessRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

StructureRef IStructureRef(this)
StructureRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

StructureOrProcessRef IStructureOrProcessRef(this)
StructureOrProcessRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

ExpNameRef IExpNameRef(this)
ExpNameRef this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

nameToken InameToken(this)
nameToken this;
{
  SourcePosition thispos;

  thispos = NSourcePosition;
  this->lex_srcpos = thispos;
  return(this);
}

UpdateCopied(compUnit)
compilationUnit compUnit;
{
  SEQDeclaration SD;
  Declaration ADecl;
  SEQStructureOrProcessRef SS;
  StructureOrProcessRef SPspec;

  foreachinSEQDeclaration(compUnit->syn_body, SD, ADecl) {
    switch (typeof(ADecl)) {
    case KImportDecl:
      foreachinSEQStructureOrProcessRef(ADecl.VImportDecl->syn_specs,
					SS, SPspec) {
	switch (typeof(SPspec->sem_entity)) {
	case KStructureEntity:
	  UpdateCopiedStructure(SPspec->sem_entity.VStructureEntity);
	  break;
	case KProcessEntity:
	  UpdateCopiedProcess(SPspec->sem_entity.VProcessEntity);
	  break;
	}
      };
      break;
    case KStructureEntity:
      UpdateCopiedStructure(ADecl.VStructureEntity);
      break;
    case KProcessEntity:
      UpdateCopiedProcess(ADecl.VProcessEntity);
      break;
    }
  }
}

/* returns ultimate ancestor */
Attribute UpdateCopiedAttribute(AnAtt)
Attribute AnAtt;
{
  Attribute Att2;

  if (typeof(AnAtt->sem_copiedfrom)==KAttribute) {
    Att2 = UpdateCopiedAttribute(AnAtt->sem_copiedfrom.VAttribute);
    AnAtt->lex_srcpos = Att2->lex_srcpos;
    switch (typeof(AnAtt->syn_type)) {
      case KSetRef: AnAtt->syn_type.VSetRef->syn_component->lex_srcpos
	= Att2->syn_type.VSetRef->syn_component->lex_srcpos;
	break;
      case KSeqRef: AnAtt->syn_type.VSeqRef->syn_component->lex_srcpos
	= Att2->syn_type.VSeqRef->syn_component->lex_srcpos;
	break;
      case KNamedTypeRef: AnAtt->syn_type.VNamedTypeRef->lex_srcpos
	= Att2->syn_type.VNamedTypeRef->lex_srcpos;
	break;
      }
    return(Att2);
  }
  else return(AnAtt);
}

/*SourcePosition UpdateCopiedTypeEntity(AType)
TypeEntity AType;
{
  if (typeof(AType.IDLclassCommon->sem_copiedfrom)!=KTVoid)
    AType.IDLclassCommon->lex_srcpos
      = UpdateCopiedTypeEntity(AType.IDLclassCommon->sem_copiedfrom.VTypeEntity);
  return(AType.IDLclassCommon->lex_srcpos);
}
*/

UpdateCopiedStructure(this)
StructureEntity this;
{
  SEQStructureRef SSR;
  StructureRef ASRef;
  SETTypeEntity STE;
  TypeEntity ATypeEntity;
  SEQAttribute SA;
  Attribute AnAtt;

  if (typeof(this->syn_refines)==KStructureRef)
    UpdateCopiedStructure(this->syn_refines.VStructureRef->sem_entity.VStructureEntity);
  foreachinSEQStructureRef(this->syn_from, SSR, ASRef) {
    if (typeof(ASRef->sem_entity)==KStructureEntity)
      UpdateCopiedStructure(ASRef->sem_entity.VStructureEntity);
  }
  foreachinSETTypeEntity(this->sem_types, STE, ATypeEntity) {
/*    UpdateCopiedTypeEntity(ATypeEntity);
*/
    if (typeof(ATypeEntity)==KClass) {
      foreachinSEQAttribute(ATypeEntity.VClass->sem_allattributes, SA, AnAtt) {
	(void) UpdateCopiedAttribute(AnAtt);
      }
    }
  }
}

UpdateCopiedProcess(this)
ProcessEntity this;
{
  if (typeof(this->syn_refines)==KProcessRef) {
    if (typeof(this->syn_refines.VProcessRef->sem_entity)==KProcessEntity)
      UpdateCopiedProcess(this->syn_refines.VProcessRef->sem_entity.VProcessEntity);
  }
  UpdateCopiedStructure(this->sem_invariant);
}


void usage(name)
char *name;
{
 (void)fprintf(stderr,"usage: %s [-V] [-d] [-r] [-t<number>] [-o <file>] [-v] [-I<dir>]... [-P | -FC | -FP | -L | -S] <.Cdl file>\n",name);
}


void revision_number(name)
char *name;
{
 char revision_str[20];

 (void)sscanf("$Revision: 5.0 $","%*s %s",revision_str);
 (void)printf("%s: IDL specification walker Version %s\n",name,revision_str);
 usage(name);
}
