/***********************************************************************\ 
*									* 
*   File: scorpion/src/candlewalk/ProcessCompUnit.c 
*				 					* 
*   Copyright (C) 1991 Richard Snodgrass
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 

*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/**************************************************************\
 * Filename: ProcessCompUnit.c                                  *
 * Function: ProcessCompUnit                                    *
 * Authors:  Alex Nelson, Karen Shannon, and Rick Snodgrass     *
 * Used by:  main                                               *
 * Algorithm:  Treewalk the compUnit of a Candle file given as  *
 *              input.  Special formatting commands for node    *
 *              and class productions as well as general        *
 *              formatting is applied, if LaTeXswitch is set.   *
 *		Otherwise, perform other tasks on each node.    *
 \**************************************************************/

#include <stdio.h>
#include <string.h>
#include "candlewalk.h"

extern Boolean debugswitch;
extern Boolean verboseswitch;

/* output formats */
extern Boolean LaTeXswitch;
extern Boolean sourceposswitch;
extern Boolean plainswitch;
extern Boolean formattedplainswitch;
extern Boolean formattedcandleswitch;

char *rm1char();
char *rmlastchar();
extern char *FormatString();
extern void PrintTexString();
void UpdateCounts();
void exit();

char lhs[200], rhs[200];
LexInfo restlexinfo, newlexinfo;
FILE *outputfp;
int filenumber = 0,
  pagenumber = 0,
  linenumber = 0,
  charnumber = 0,
  startlineoffset = 0;

/* Process String appearing in original specification with formatting  */
#define PStr(str, tab, kill) {if (plainswitch) (void) fprintf(outputfp, str);\
else if (formattedplainswitch) (void) fprintf(outputfp,\
  FormatString(str, tab, kill));\
else if (LaTeXswitch) PrintTexString(outputfp, str, tab, kill);\
else if (sourceposswitch) UpdateCounts(str); \
else if (formattedcandleswitch) \
  appendrearSEQString(newlexinfo, FormatString(str, tab, kill));}

/* Process additional strings with formatting */
#define ProcessStr(str) {if (plainswitch) (void) fprintf(outputfp, str); \
else if (formattedplainswitch) (void) fprintf(outputfp,\
  FormatString(str, TRUE, 0)); \
else if (LaTeXswitch) PrintTexString(outputfp, str, TRUE, 0); \
else if (sourceposswitch) /* nothing */ ; \
else if (formattedcandleswitch) \
  appendrearSEQString(newlexinfo, FormatString(str, TRUE, 0));}

/* Just output an additional string, without processing */
#define PrintStr(str) {if (plainswitch) /* nothing */ ; \
else if (formattedplainswitch) /* nothing */ ; \
else if (LaTeXswitch) (void) fprintf(outputfp, str); \
else if (sourceposswitch) /* nothing */ ; \
else if (formattedcandleswitch) /* nothing */;}

#define PName(named) {if(plainswitch) (void)fprintf(outputfp,named->lex_name); \
else if (formattedplainswitch) PStr(named->lex_name, TRUE, 0) \
else if (LaTeXswitch) PStr(named->lex_name, TRUE, 0) \
else if (sourceposswitch) MakeSourcePosition(&(named->lex_srcpos)); \
else if (formattedcandleswitch) /* nothing */;}

/* process the next LexInfo (LaTeX formatted)  and update the sequence */
#define PNextLexInfo \
       {String currentlexinfo;\
	retrievefirstSEQString(restlexinfo, currentlexinfo);\
	PStr(currentlexinfo, TRUE, 0); \
        removefirstSEQString(restlexinfo);\
       }\

ProcessCompUnit(compUnit)
compilationUnit compUnit;
{
  SEQDeclaration SDecl;
  Declaration ADecl;

  if (verboseswitch) {
    (void) fprintf(stderr, 
            "candlewalk: Processing file %s (file number %d)...\n",
	    compUnit->inv_filename, compUnit->inv_filenumber);
    (void) fflush(stderr);
  }
  restlexinfo = copySEQString(compUnit->lex_information);
  initializeSEQString(newlexinfo);
  filenumber = compUnit->inv_filenumber;
  pagenumber = 1;
  linenumber = 1;
  charnumber = 1;
  startlineoffset = charnumber;
  (void) strcpy(lhs, "  ");
  (void) strcpy(rhs, "  ");

  if (LaTeXswitch)
    PrintStr("\\ \\ \\ \\ \\=\\ \\ \\ \\ \\=\\ \\ \\ \\ \\=\\ \\ \\ \\=\\ \\ \\ \\ \\=\\ \\ \\ \\ \\=\\ \\ \\ \\ \\=\\ \\ \\ \\ \\=\\kill\n\\SPA{1}{1}");
  PNextLexInfo;			/* before 1st decl */

  foreachinSEQDeclaration(compUnit->syn_body, SDecl, ADecl){
    switch (typeof(ADecl)) {
    case KStructureEntity:
      ProcessStructure(ADecl.VStructureEntity);
      break;
    case KProcessEntity:
      ProcessProcess(ADecl.VProcessEntity);
      break;
    case KImportDecl:
      ProcessImport(ADecl.VImportDecl);
      break;
    }
    PNextLexInfo;		/* after previous decl */
  }
  if (LaTeXswitch)
    PrintStr("\\kill\n");	/* get rid of trailing \\SPA */
  compUnit->lex_information = newlexinfo;
}

ProcessImport(import)
ImportDecl import;
{
  LexInfo copyrestlexinfo, copynewlexinfo;
  SEQStructureOrProcessRef SS;
  StructureOrProcessRef SPspec;

  copyrestlexinfo = restlexinfo;
  copynewlexinfo = newlexinfo;
  restlexinfo = copySEQString(import->lex_information);
  initializeSEQString(newlexinfo);

  PNextLexInfo;			/* import... */
  foreachinSEQStructureOrProcessRef(import->syn_specs, SS, SPspec) {
    PName(SPspec);
    PNextLexInfo;		/* from... or , */
  }
  PName(import->syn_compUnit);
  PNextLexInfo;			/* ..end */

  import->lex_information = newlexinfo;
  newlexinfo = copynewlexinfo;
  restlexinfo = copyrestlexinfo;
}


ProcessStructure(st)
StructureEntity st;
{
  SEQstructureStatement Sstate;
  structureStatement Astate;
  LexInfo copyrestlexinfo, copynewlexinfo;
  SEQStructureRef SStRef;
  StructureRef AStRef;
  int i;
  String str;

  copyrestlexinfo = restlexinfo;
  copynewlexinfo = newlexinfo;
  restlexinfo = copySEQString(st->lex_information);
  initializeSEQString(newlexinfo);

  /* get a copy of the longest node/class name and longest attr. name */
  TexIdlTabs(st);

  if (strlen(lhs)==0) (void) strcpy(lhs,"A");
  if (strlen(lhs)==0) (void) strcpy(lhs,"B");
	
  /* Really only care about number of chars, this will remove all special chars */
  for (i=0; i<strlen(lhs); i++) lhs[i]='A';
  for (i=0; i<strlen(rhs); i++) rhs[i]='B';

  PNextLexInfo;			/* structure ... */
  PName(st);			/* name of the struct. */

  if (typeof(st->syn_root) != KVoid){
    PNextLexInfo;		/* text between structure name and root */
    PNextLexInfo;		/* Root ...     */ 
    ProcessType(st->syn_root.VTypeRef, FALSE); /* name of root */
  }

  if (typeof(st->syn_refines) == KStructureRef){
    PNextLexInfo;		/* Refines */
    PName(st->syn_refines.VStructureRef);
  }
  foreachinSEQStructureRef(st->syn_from, SStRef, AStRef){
    PNextLexInfo;		/* From ... */
    PName(AStRef);		/* name of the from structures */
  }

  foreachinSEQstructureStatement(st->syn_body, Sstate, Astate) {

    /* First, process the lexinfo coming before the next statement */
    /* Is ... or end of previous statement*/
    if (lengthSEQString(restlexinfo)>1) {
      if ((typeof(Astate)==KattributeProduction)
	  || (typeof(Astate)==KsubclassProduction)) {
	  /* Need to switch to special mode */
	  retrievefirstSEQString(restlexinfo, str);
	  PStr(str, FALSE, 1);
	  removefirstSEQString(restlexinfo);
	}
      else {			/* we want regular mode */
	retrievefirstSEQString(restlexinfo, str);
	PStr(str, TRUE, 2);
	removefirstSEQString(restlexinfo);
      }
    }
    else {			/* we want regular mode */
      retrievefirstSEQString(restlexinfo, str);
      PStr(str, TRUE, 2);
      removefirstSEQString(restlexinfo);
    };

    if (debugswitch) {(void) fflush(stderr);}

    /* now process the statement */
    ProcessStatement(Astate);
  }

  retrievefirstSEQString(restlexinfo, str);
  PStr(str, TRUE, 2);		/* End... */
  removefirstSEQString(restlexinfo);

  restlexinfo = copyrestlexinfo;
  st->lex_information = newlexinfo;
  newlexinfo = copynewlexinfo;
}


ProcessProcess(pr)
ProcessEntity pr;
{
  LexInfo copyrestlexinfo, copynewlexinfo;
  SEQprocessStatement Spst;
  processStatement Apst;

  copyrestlexinfo = restlexinfo;
  restlexinfo = copySEQString(pr->lex_information);
  copynewlexinfo = newlexinfo;
  initializeSEQString(newlexinfo);

  PNextLexInfo;			/* Process.. */
  PName(pr);

  if (typeof(pr->syn_refines)==KProcessRef){
    PNextLexInfo;		/* Refines.. */
    PName(pr->syn_refines.VProcessRef);
  }

  if (typeof(pr->syn_invariant) == KStructureRef){
    PNextLexInfo;		/* Inv.. */
    PName(pr->syn_invariant.VStructureRef);
  }

  PNextLexInfo;			/* Is */
  foreachinSEQprocessStatement(pr->syn_body, Spst, Apst){
    ProcessStatement(Apst);
    PNextLexInfo;
  }

  /* End is processed with last statement */
  restlexinfo = copyrestlexinfo;
  pr->lex_information = newlexinfo;
  newlexinfo = copynewlexinfo;
}

ProcessStatement(st)
statement st;
{
  String str;

  if (debugswitch) {
    (void) fprintf(stderr,"Beginning ProcessStatement.\n");
    (void) fflush(stderr);
  }

  switch (typeof(st)) {

  case KsubclassProduction: {
    subclassProduction sp;
    SEQNamedTypeRef Snt;
    NamedTypeRef Ant;
    int NumClasses;
      
    if (debugswitch) {
      (void) fprintf(stderr,"Beginning subclass production.\n");
      (void) fflush(stderr);
    }

    sp = st.VsubclassProduction;

    /* Try to match one of three  options.                         */
    /* Option 1: No <cr> for the subclass production.  Obey input. */
    /* Option 2: <cr>s before the vertical bars.                   */
    /* Option 3: <cr>s after the vertical bars.                    */

    NumClasses = lengthSEQNamedTypeRef(sp->syn_subclasses);
    if (debugswitch) {
      (void) fprintf(stderr,"The number of subclasses is %d\n",NumClasses);
    }

    PName(sp->syn_class);
    if (NumClasses==0)
      break;			/* ::= will be printed before next statement */
    else {
      if (plainswitch)
	PNextLexInfo
      else {
	removefirstSEQString(restlexinfo); /* throw away ::= */
	ProcessStr("\t::=\t");
      }
      foreachinSEQNamedTypeRef(sp->syn_subclasses, Snt, Ant) {
	PName(Ant);
	NumClasses--;
	if (NumClasses > 0) {	/* not at last subclass */
	  if (plainswitch)
	    PNextLexInfo
	  else {
	    retrievefirstSEQString(restlexinfo,str);
	    removefirstSEQString(restlexinfo);
	    if ((firsdex("\n",str)) > (firsdex("|",str))) {
	      /* Class ::= sub1 <comm1 | \n> sub2 <comm2 \n ;>  */
	      ProcessStr(" | ");
	      str = rm1char('|',str);
	      PStr(str, FALSE, 0); /* <comm1> \n  */
	      ProcessStr("\t\t"); /* initial tab already processed */
	    }
	    else if ((firsdex("\n", str) >= 0)
		     && ((firsdex("\n",str)) < (firsdex("|",str)))) {
	      /* Class ::= sub1 <comm1 \n |> sub2 <comm2 \n ;>  */
	      str = rm1char('|',str);
	      PStr(str, FALSE, 0);	/* <comm1> \n  */
	      ProcessStr("\t\t| "); /* initial tab already processed */
	    }
	    else {		/* no \n */
	      PStr(str, FALSE, 0);
	    }
	  }
	};			/* <comm2 \n ;>  processed at end of stmt */
      }				/* foreachIn */
    }				/* NumClasses > 0 */
  }
    break;
  case KattributeProduction: {
    attributeProduction ap;
    SEQAttribute Sa;
    Attribute Aatt;
    String str;
    int NumAttributes;
    int first;

    if (debugswitch) {
      (void) fprintf(stderr,"Begining attribute production.\n");
      (void) fflush(stderr);
    }

    ap = st.VattributeProduction;

    NumAttributes = lengthSEQAttribute(ap->syn_attributes);

    /* Process the    NODE => ATT : TYPE , */ 
    PName(ap->syn_class);
    if (NumAttributes == 0)
      break;
    if (plainswitch)
      PNextLexInfo
    else {
      removefirstSEQString(restlexinfo);	/* throw away => */
      ProcessStr("\t=>\t");
    }
    first = TRUE;
    foreachinSEQAttribute(ap->syn_attributes, Sa, Aatt) {
      if (first) first = FALSE;
      else if (! plainswitch)
	ProcessStr("\t\t");	/* ??? removed one \t */
      PName(Aatt);
      NumAttributes--;
      if (plainswitch)
	PNextLexInfo
      else {
	removefirstSEQString(restlexinfo); /* Rem ":" from  lexinfo  */
	PStr(":\t", TRUE, 0);
      }
      ProcessType(Aatt->syn_type, (! plainswitch));
      if (NumAttributes > 0) {	/* not last attribute */
	if (plainswitch)
	  PNextLexInfo
	else {
	  retrievefirstSEQString(restlexinfo,str);
	  removefirstSEQString(restlexinfo);
	  str = rm1char(',',str); str = rmlastchar('\n', str);
	  ProcessStr(",");
	  PStr(str, TRUE, 0);
	  PStr("\n", TRUE, 0);
	}
      }
    }
  }
    break;			  
  case KatomicDecl: {
    atomicDecl ad;
      
    ad = st.VatomicDecl;
    /* "Type" is processed with previous statement or "Is" */
    PName(ad->syn_atomic);
  }
    break;
  case KwithoutClause: {
    withoutClause wc;
    SEQwithoutItem Swi;
    withoutItem wi;
    ClassOrAllClasses CorAll;
    AttributeOrAllAttributes AorAll;
    char firsttime;
      
    wc = st.VwithoutClause;
    /* "Without" is processed with previous statement or "Is" */
    firsttime = TRUE;
    foreachinSEQwithoutItem(wc->syn_list, Swi, wi){
      if (! firsttime) {
	PNextLexInfo;	/* between without items */
      }
      firsttime = FALSE;
      switch (typeof(wi)){
      case KwithoutAssert:
	/* "Assert" is processed with "Without" */
	PName(wi.VwithoutAssert->syn_item);
	break;
      case KwithoutDefine:
	/* "Define" is processed with "Without" */
	PName(wi.VwithoutDefine->syn_item);
	break;
      case KwithoutType:
	/* "Type" is processed with "Without" */
	ProcessType(wi.VwithoutType->syn_item, FALSE);
	break;
      case KwithoutSubclass:
	CorAll = wi.VwithoutSubclass->syn_lefthandside;
	if (typeof(CorAll)==KNamedTypeRef){
	  PName(CorAll.VNamedTypeRef);
	}
	/* else "*" is processed with "Without" */
	  
	CorAll = wi.VwithoutSubclass->syn_righthandside;
	if (typeof(CorAll)==KNamedTypeRef){
	  PNextLexInfo;		/* ::=  or , */
	  PName(CorAll.VNamedTypeRef);
	};
	/* else * is processed with ::= */
	break;
      case KwithoutAttribute:
	CorAll = wi.VwithoutAttribute->syn_lefthandside;
	if (typeof(CorAll)==KNamedTypeRef){
	  PName( CorAll.VNamedTypeRef);
	}
	/* else "*" will be processed with "Without" */
	  
	AorAll = wi.VwithoutAttribute->syn_righthandside;
	if (typeof(AorAll)==KAttributeRefs){
	  PNextLexInfo;		/* => or , */
	  PName(AorAll.VAttributeRefs);
	}
	/* else "*" will be processed with => */
	break;
      }
    }
  }
    break;
  case KattributeRep: {
    attributeRep ar;

    ar = st.VattributeRep;
    /* "For" is processed with last statement or "Is" */
    PName(ar->syn_class);
    PNextLexInfo;
    PName(ar->syn_attribute);
    ProcessParams(ar->syn_rep->syn_id);
  }
    break;
  case KtypeRepDecl: {
    typeRepDecl tr;
      
    tr = st.VtypeRepDecl;
    /* "For" is processed with last statement or "Is" */
    ProcessType(tr->syn_type, FALSE);
    ProcessParams(tr->syn_spec->syn_id);
  }
    break;
  case KportAssociation: {
    portAssociation pa;
      
    pa = st.VportAssociation;
    /* "For" is processed with last statement or "Is" */
    PName(pa->syn_port);
    ProcessParams(pa->syn_rep->syn_id);
  }
    break;
  case KportDefinition: {
    portDefinition pd;
    SEQPort Sp;
    Port aport;
    int cnt, numports;
      
    /* "Pre", "Post", etc is processed with last statement or "Is"*/
    pd = st.VportDefinition;
    numports = lengthSEQPort(pd->syn_ports);
    cnt = 0;
    foreachinSEQPort(pd->syn_ports, Sp, aport){
      PName(aport);
      PNextLexInfo;		/* : */
      PName(aport->syn_data);
      ++cnt;
      if (cnt < numports)
	PNextLexInfo;
    }
  }
    break;
  case KtargetStmt: {
    targetStmt ts;
    SEQparameter Sparm;
    parameter Aparm;

    int numparms, cnt;
    /* replace with ProcessParams, it cnt not needed */
    ts = st.VtargetStmt;
    /* "Target" is processed with last statement or "Is" */
    numparms = lengthSEQparameter(ts->syn_id);
    cnt = 0;
    foreachinSEQparameter(ts->syn_id, Sparm, Aparm){
      if (typeof(Aparm)== KnameToken) {
	PStr(Aparm.VnameToken->lex_name, TRUE, 0);
      }
      else if (typeof(Aparm)==KstringToken) {
	PStr("\"", TRUE, 0);
	PStr(Aparm.VstringToken->lex_externalform, TRUE, 0);
	PStr("\"", TRUE, 0);
	}
      else {
	PStr(Aparm.VotherToken.IDLclassCommon->lex_externalform, TRUE, 0);
      }
      ++cnt;
      if (cnt < numparms)
	PNextLexInfo;
    }
  }
    break;
  case KtypeRestriction: 
  case KattributeRestriction: {
    TypeOrAllTypes TorAll;
    AttributeOrAllAttributes AorAll;
    SEQOperRef opseq;
    SEQOperRef Sop;
    OperRef Aop;
    int numops, cnt;
     
    TorAll = st.Vrestriction.IDLclassCommon->syn_type;
    opseq = st.Vrestriction.IDLclassCommon->syn_operations;
     
    /* "Restrict" is processed with last statement */
    if (typeof(TorAll) != KAllTypes){
      ProcessType(TorAll.VTypeRef, FALSE);
      PNextLexInfo;		/* "To" or "."*/
    }
    /* else " * To " or "* ." is processed with "Restrict" */
    if (typeof(st)==KattributeRestriction){
      AorAll = st.VattributeRestriction->syn_attribute;
      if (typeof(AorAll)==KAttributeRefs){
	PName(AorAll.VAttributeRefs);
	PNextLexInfo;		/* "To" */
      }
    }
    /* else "* To" is processed with "." */
     
    numops = lengthSEQOperRef(opseq);
    cnt = 0;
    foreachinSEQOperRef(opseq, Sop, Aop){
      PName(Aop);
      ++cnt;
      if (cnt < numops)
	PNextLexInfo;		/* "," */
    }
  }
    break;
  case KAssertion: {
    Assertion as;

    as = st.VAssertion;
    if (strlen(as->lex_name) > 0){
      PName(as);
      PNextLexInfo;		/* "Assert" */
    }
    /* else "Assert" is processed with last statement or "Is" */

    ProcessExpression(as->syn_body);
  }
    break;
  case Kcyclicdef:
  case Knoncyclicdef:
  case KPrivateDefInstance: {
    DefineRef defref;
    SEQFormal formseq, Sf;
    Formal f;
     
    defref = st.VDefInstance.IDLclassCommon->syn_def;
    formseq = st.VDefInstance.IDLclassCommon->syn_list;
     
    /* "Define" and "Cyclic Define" processed with last statement */
    PName(defref);
    foreachinSEQFormal(formseq, Sf, f){
      PNextLexInfo;		/* ( or , */
      PName(f);
      PNextLexInfo;		/* : */
      ProcessType(f->syn_type, FALSE);
    }
    PNextLexInfo;		/* ), =, or "Returns" */
    if (typeof(st)==Kcyclicdef){ /* previous LexInfo was ") Returns" */
      ProcessType(st.Vcyclicdef->syn_returnType, FALSE);
      PNextLexInfo;		/* "=" */
      ProcessExpression(st.Vcyclicdef->syn_body);
    }
    else if (typeof(st)==Knoncyclicdef){ /* previous LexInfo was ") =" */
      ProcessExpression(st.Vnoncyclicdef->syn_body);
    }
    else {		/* private def, previous LexInfo was ") Returns" */
      ProcessType(st.VPrivateDefInstance->syn_returnType, FALSE);
    }
  }
    break;
  default: {
    (void) fprintf(stderr,"candlewalk: the statement is not a known type.\n");
    exit(1);
  }

  }				/* End of Switch on stmt. type */
}

/*VARARGS*/
ProcessType(type, SeqSetProcess)
TypeRef type;
int SeqSetProcess;		/* If True, then fabricate Seq Of and Set Of */
{
  switch (typeof(type)){
  case KNamedTypeRef:
    if (debugswitch) {
      (void) fprintf(stderr,"The type is >>%s<<",type.VNamedTypeRef->lex_name);
    }
    PName(type.VNamedTypeRef);
    break;
  case KSetRef:
    if (SeqSetProcess) PStr("Set Of ", TRUE, 0);
    PName(type.VSetOrSeqRef.IDLclassCommon->syn_component);
    break;
  case KSeqRef:
    if (SeqSetProcess) PStr("Seq Of ", TRUE, 0);
    PName(type.VSetOrSeqRef.IDLclassCommon->syn_component);
    break;
  }
}

/* general algorithm is to process lexical information after
 * each recorded token is processed
 */
ProcessExpression(exp)
expression exp;
{
  switch (typeof(exp)) {
  case Kconditional: {
    conditional c;
    SEQexpressionPair Sep;
    expressionPair ep;

    c = exp.Vconditional;
    /* "If" is processed with previous keyword or white space */
    ProcessExpression(c->syn_test);
    PNextLexInfo;		/* "Then" */
    ProcessExpression(c->syn_then);
    foreachinSEQexpressionPair(c->syn_orif, Sep, ep){
      PNextLexInfo;		/* "OrIf" */
      ProcessExpression(ep->syn_test);
      PNextLexInfo;		/* "Then" */
      ProcessExpression(ep->syn_then);
    }
    PNextLexInfo;		/* "Else" */
    ProcessExpression(c->syn_else);
  }
    break;
  case Kforallq:
  case Kexistsq: {
    quantifier q;

    q = exp.Vquantifier;
    /* "ForAll" and "Exists" are processed previously */
    PName(q.IDLclassCommon->syn_control);
    PNextLexInfo;		/* "In..." */
    ProcessExpression(q.IDLclassCommon->syn_set);
    PNextLexInfo;		/* "Do" */
    ProcessExpression(q.IDLclassCommon->syn_body);
  }
    break;
  case Kbinary: {
    binary b;

    b = exp.Vbinary;
    ProcessExpression(b->syn_left);
    PNextLexInfo;		/* operator */
    ProcessExpression(b->syn_right);
  }
    break;
  case Kunary: {
    /* operator was processed previously */
    ProcessExpression(exp.Vunary->syn_body);
  }
    break;
  case Kapplication: {
    application a;
    SEQexpression Sexp;
    expression Anexp;

    a = exp.Vapplication;
    PName(a->syn_instance);
    foreachinSEQexpression(a->syn_arguments, Sexp, Anexp){
      PNextLexInfo;		/* (..  or separator */
      ProcessExpression(Anexp);
    }
  }
    break;
  case Kdotted: {
    dotted d;

    d = exp.Vdotted;
    ProcessExpression(d->syn_left);
    /*	    PNextLexInfo;	/* before the dot */
    PNextLexInfo;		/* . */
    PName(d->syn_right);
  }
    break;
  case KportExpression: {
    portExpression pe;

    pe = exp.VportExpression;
    PName(pe->syn_portName);
    PNextLexInfo;		/* : */
    ProcessType(pe->syn_type);
  }
    break;
  case KExpSetRef:
  case KExpSeqRef: {
    ExpSetSeqRef er;

    er = exp.VExpSetSeqRef;
    /* "Set, Seq Of" are processed previously */
    PName(er.IDLclassCommon->syn_component);
  }
    break;
  case KExpNameRef: {
    PName(exp.VExpNameRef);
  }
    break;
  case KintegerToken:
  case KrationalToken: {
    PStr(exp.VotherToken.IDLclassCommon->lex_externalform, TRUE, 0);
  }
    break;
  case KstringToken: {
    PStr("\"", TRUE, 0);
    PStr(exp.VotherToken.IDLclassCommon->lex_externalform, TRUE, 0);
    PStr("\"", TRUE, 0);
  }
    break;
  case KrootExp:{
    rootExp re;

    re = exp.VrootExp;
    if (typeof(re->syn_portName)==KPortRef){
      PName(re->syn_portName.VPortRef);
      PNextLexInfo;		/* ":Root" */
    };
    /* else Root will be processed eventually */
  }
    break;
  case KemptyExp:
  case KtrueExp:
  case KfalseExp:
    /* Empty, True, False will be processed eventually */
    break;
  case KcaseExp: {
    caseExp c;
    SEQcase_select scase_select;
    case_select thecase_select;

    c = exp.VcaseExp;
    /* "Case" was processed previously */
    PName(c->syn_casename);
    PNextLexInfo;		/* "Is" */
    ProcessExpression(c->syn_exp);
    foreachinSEQcase_select(c->syn_select, scase_select, thecase_select) {
      PNextLexInfo;		/* before case select */
      ProcessType(thecase_select->syn_type, FALSE);
      PNextLexInfo;		/* "Do" */
      ProcessExpression(thecase_select->syn_exp);
    }
    if (typeof(c->syn_otherwise) != KVoid) {
      PNextLexInfo;		/* "Otherwise Do" */
      ProcessExpression(c->syn_otherwise.Vexpression);
    }
    PNextLexInfo;		/* "Od End" */
  }
    break;
    default: {
      (void) fprintf(stderr,"candlewalk: Illegal expression type.\n");
      exit(-1);
    }
  }

}

ProcessParams(Sp)
SEQparameter Sp;
{
  SEQparameter Sparm;
  parameter Aparm;
  foreachinSEQparameter(Sp, Sparm, Aparm) {
    PNextLexInfo;
    if (typeof(Aparm)== KnameToken)
	PName((Aparm.VnameToken))
    else if (typeof(Aparm)==KstringToken) {
      PStr("\"", TRUE, 0);
      PStr(Aparm.VstringToken->lex_externalform, TRUE, 0);
      PStr("\"", TRUE, 0);
    }
    else {
      PStr(Aparm.VotherToken.IDLclassCommon->lex_externalform, TRUE, 0);
    }
  }
}

void UpdateCounts(thisstr)
String thisstr;
{
  int i;

  for (i=0; i<strlen(thisstr); i++) {
    charnumber++;
    switch (thisstr[i]) {
    case '\n': linenumber++;
      startlineoffset = charnumber;
      break;
    case '\f': pagenumber++;
      linenumber = 1;
      startlineoffset = charnumber;
      break;
    }
  }
}

MakeSourcePosition(thispointer)
SourcePosition *thispointer;
{
  if (*thispointer==NULL) (*thispointer) = NSourcePosition;
  (*thispointer)->lex_file = filenumber;
  (*thispointer)->lex_page = pagenumber;
  (*thispointer)->lex_line = linenumber;
  (*thispointer)->lex_column = charnumber - startlineoffset + 1;
}
