/***********************************************************************\ 
*									* 
*   File: scorpion/src/IDLlib/libci/stabsearch.c 
*				 					* 
*   Copyright (C) 1991 Steven Shafer
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*  stabsearch --  search for best match within string table
 *
 *  int stabsearch (arg,table,quiet);
 *  char *arg;
 *  char **table;
 *  int quiet;
 *
 *  Just like stablk(3), but a match score is determined for each
 *  string in the table, and the best match is used.  If quiet=0,
 *  the user will be asked if he really meant the best matching
 *  string; if he says "no", a list of several other good matches
 *  will be printed.
 *  If there is exactly one perfect match, then its index will be
 *  returned, and the user will not be asked anything.  If there are
 *  several perfect matches, then up to 50 will be listed for the
 *  user to review and among which he can select one.
 *
 * HISTORY
 * 06-Feb-81  Steven Shafer (sas) at Carnegie-Mellon University
 *	Fixed bug in previous bug fixes.
 *
 * 27-Jan-81  Steven Shafer (sas) at Carnegie-Mellon University
 *	Added SCROLLSIZE, KEEPPERFECT, and associated code for better
 *	handling of long string tables.
 *
 * 16-Apr-80  Steven Shafer (sas) at Carnegie-Mellon University
 *	Changed option-printing code to use "prstab"; this will use
 *	multiple columns where appropriate.
 *
 * 12-Mar-80  Steven Shafer (sas) at Carnegie-Mellon University
 *	Added check for unique perfect match; will now return appropriate index
 *	without asking "Did you mean X?".
 *
 * 23-Jan-80  Steven Shafer (sas) at Carnegie-Mellon University
 *	Created.  Almost identical to Dave McKeown's matching routine from
 *	the PDP-11, but the scores are normalized to lie in the range 0 to 100.
 *
 */

#include <stdio.h>
#define KEEPBEST 6		/* # of matches to keep around */
#define KEEPPERFECT 50		/* # of perfect matches to keep */
#define PERFECT 100		/* perfect match score */
#define THRESHOLD 35		/* minimum acceptable score */
#define NOMATCH -1		/* return value if no match */
#define MAXLENGTH 400		/* max length of arg and table entries */
#define SCROLLSIZE 20		/* size of each screenful for scrolling */

int srchscore();		/* score function */
void prstab();
char *folddown();

int stabsearch (arg,table,quiet)
char *arg,**table;
int quiet;
{
	int bestentry[KEEPPERFECT], bestscore[KEEPPERFECT];
	int maxscore;		/* best possible score */
	register int i,j,k;	/* temps */
	int nperfect;		/* # of perfect matches */
	int nentries;		/* # of entries in table */
	char line[MAXLENGTH];
	char a[MAXLENGTH],e[MAXLENGTH];

	if (strcmp (arg,"?") != 0) {

		for (i=0; i<KEEPPERFECT; i++) {
			bestentry[i] = -1;
			bestscore[i] = -1;
		}
		maxscore = strlen (arg);
		maxscore *= maxscore;
		(void)folddown (a,arg);

		nperfect = 0;
		for (i=0; table[i]; i++) {
			(void)folddown (e,table[i]);
			j = (srchscore (e,a) * PERFECT) / maxscore;
			if (nperfect == 0 && j != PERFECT) {
				if (j >= bestscore[KEEPBEST-1]) {
					k = KEEPBEST - 1;
					while ((k > 0) && (j > bestscore[k-1])) {
						bestscore[k] = bestscore[k-1];
						bestentry[k] = bestentry[k-1];
						--k;
					}
					bestscore[k] = j;
					bestentry[k] = i;
				}
			}
			else if (j == PERFECT && nperfect < KEEPPERFECT) {
				bestscore[0] = PERFECT;
				bestentry[nperfect] = i;
				nperfect++;
			}
		}
		nentries = i;

		if (bestscore[0] <= THRESHOLD) {
			if (!quiet) {
				(void)printf ("Sorry, nothing matches \"%s\" reasonably.\n",arg);
				if (getbool("Do you want a list?",(nentries<=SCROLLSIZE))) goto makelist;
			}
			return (NOMATCH);
		}

		if (quiet)  return (bestentry[0]);

		if (nperfect > 1) {	/* multiple max matches */
			(void)printf ("There are %d perfect matches for \"%s\":\n",nperfect,arg);
			for (j=0; j<nperfect && ((j%SCROLLSIZE!=SCROLLSIZE-1) || getbool("Continue?",1)); j++) {
				(void)printf ("(%d)\t%s\n",j+1,table[bestentry[j]]);
			}
			j = getint ("Which one should be used?  (0 if none ok)",0,nperfect,1);
			return ((j > 0) ? bestentry[j-1] : NOMATCH);
		}
		else if (nperfect == 1) {	/* unique perfect match */
			return (bestentry[0]);
		}

		(void)sprintf (line,"Did you mean \"%s\" [%d] ?",
			table[bestentry[0]],bestscore[0]);
		if (getbool(line,1))  return (bestentry[0]);

		(void)printf ("May I suggest the following?\n");
		for (i=0; i<KEEPBEST && bestscore[i] >= THRESHOLD; i++) {
			(void)printf ("(%d)\t%-15s \t[%d]\n",i+1,table[bestentry[i]],bestscore[i]);
		}
		j = getint ("Which one should be used?  (0 if none ok)",0,i,1);
		if (j>0)  return (bestentry[j-1]);
		if (getbool("Do you want a list of all possibilities?",(nentries<=SCROLLSIZE)))  goto makelist;

	}
	else {
makelist:

		(void)printf ("The choices are as follows:\n");
		(void)prstab (table);
	}

	return (NOMATCH);
}
