/* ******************************************************************* *\
*									*
*   File: scorpion/src/IDLlib/libci/pdate.y
*									*
*   Copyright (C) 1991 Steven Shafer
*									*
*   The Scorpion System is free software in the public domain; you can  *
*   redistribute it and/or modify it as you wish. We ask that you 	*
*   retain credits referencing the University of Arizona and that you	*
*   identify any changes you make.					*
*									*
*   Report problems to scorpion-project@cs.arizona.edu			*
*   Direct all inquiries to:	The Scorpion Project			*
*				Department of Computer Science		*
*				Gould-Simpson Building			*
*				University of Arizona			*
*				Tucson, AZ 85721			*
*				U.S.A.					*
*									*
*   Revision Log:							*
*	$Log:$
*									*
*   Edit Log:								*
*									*
\* ******************************************************************* */

/*
 *  pdate - parse date specification
 *
 **********************************************************************
 * HISTORY
 * 21-Feb-80  Mike Accetta (mja) at Carnegie-Mellon University
 *	Fixed bug in grammar which failed to parse dates followed by
 *	times when no year was specified in the date;  changed
 *	date string limit from 25 to 50 characters.
 *
 * 03-Jan-80  Mike Accetta (mja) at Carnegie-Mellon University
 *	Created.
 *
 **********************************************************************
 */

%{
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <setjmp.h>
#include	<sys/time.h>
#include "c.h"

static struct tm *tm;			/* parsed time structure */
static char *strp;			/* current position in date string */
static int delim;			/* previous field delimiter */
static jmp_buf errbuf;			/* jump location for parse errors */
extern char *nxtarg();
char *foldup();

/*
 *  Month and week day names (upper case only)
 */
static char *strings[] =
{
    "JANUARY",				/* months (0-11) */
    "FEBRUARY",
    "MARCH",
    "APRIL",
    "MAY",
    "JUNE",
    "JULY",
    "AUGUST",
    "SEPTEMBER",
    "OCTOBER",
    "NOVEMBER",
    "DECEMBER",
    "MONDAY",				/* days of the week (12-18) */
    "TUESDAY",
    "WEDNESDAY",
    "THURSDAY",
    "FRIDAY",
    "SATURDAY",
    "SUNDAY",
    0
};
%}

/*
 *  Grammar definitions for legal date specifications
 */

%start date
%token NUMBER WEEKDAY MONTH

%%

date	:	pdate
	|	WEEKDAY pdate
	;

pdate	:	time
	|	yday
	|	yday time
	|	yday time year
	|	pday
	|	pday time
	|	pday time year
	;


yday 	:	pday year
	|	pday '-' year
	|	pday '/' year
	;

pday	:	month '/' mday
	|	month_name mday
	|	mday '-' month_name
	|	mday month_name
	;

time	:	hm_time
	|	hm_time ':' sec
	;

hm_time	:	hour ':' min
	;

month_name :	MONTH
			{ tm->tm_mon = $1; }
	;

year	:	NUMBER
			{ tm->tm_year = $1 + (($1>=1900)?-1900:0); }
	;

month	:	NUMBER
			{ tm->tm_mon = $1; }
	;

mday	:	NUMBER
			{ tm->tm_mday = $1; }
	;

hour	:	NUMBER
			{ tm->tm_hour = $1; }
	;

min	:	NUMBER
			{ tm->tm_min = $1; }
	;

sec	:	NUMBER
			{ tm->tm_sec = $1; }
	;
%%

/*
 *  pdate - parse date specification into time structure
 *
 *  str = date specification string
 *  tmp = pointer to time structure to receive parsed fields
 *
 *      Parses the supplied date string and sets the appropriate
 *  fields into the time structure.  Fields which are not specified
 *  in the string are not changed in the structure (I think).
 *  The date string is copied into a temporary buffer before
 *  being parsed (maximum of 50 characters).
 *
 *     Returns 0 on a successful parse, -1 on error.
 */

pdate(str, tmp)
char *str;
struct tm *tmp;
{

    char tstr[51];

    /*  initialize lexical analyzer  */
    (void)strncpy (strp = tstr, str, 50);
    tstr[51] = 0;
    tm = tmp;
    delim = 0;

    if (setjmp(errbuf) == 0)
    {
	yyparse();
	return(0);			/* return here on successful parse */
    }
    else
	return(CERROR);			/* return here on error */

}

/*
 *  yyerror - error routine (called by yyparse)
 *
 *     Performs a jump to the error return location established
 *  by pdate().
 */

/*ARGUSED*/
static yyerror(s)
char *s;
{

    longjmp(errbuf,1);

}



/*
 *  yylex - return next token in date string
 *
 *     Uses nxtarg() to parse the next field of the date string.
 *  If a non-space, tab, comma or newline delimiter terminated the
 *  previous field it is returned before the next field is parsed.
 *
 *     Returns either one of the delimiter characters " -:/", NUMBER
 *  for a numeric field with its valus in yylval, MONTH for an ascii
 *  field which matches one of the twelve month names with its month
 *  index (0-11) in yylval, WEEKDAY for an ascii field which matches
 *  one of the seven day-of-week names, or -1 for any other ascii
 *  field.
 */

static int yylex()
{

    register int ret;			/* temp for previous delimiter */
    register char *fp;			/* current field */
    extern char _argbreak;		/* current delimiter */

    if (ret=delim)
    {
	delim = 0;
	if (ret != ' ' && ret != '\t' && ret != ',' && ret != '\n')
	    return(ret);
    }
    fp = nxtarg(&strp, " \t,-:/\n");
    delim = _argbreak;
    if (*fp == 0 && delim == 0) return (0);
    (void)foldup(fp, fp);
    if (*fp < '0' || *fp > '9')
    {
	if ((yylval=stablk(fp, strings, TRUE)) >= 0)
		return(yylval < 12?MONTH:WEEKDAY);
	return(-1);
    }
    yylval = atoi(fp);
    return(NUMBER);

}
