/***********************************************************************\ 
*									* 
*   File: scorpion/src/IDLlib/libci/glob.c 
*				 					* 
*   Copyright (C) 1991 Steven Shafer
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************/ 

#ifndef lint 
static char rcsid[] = "$Header:$"; 
#endif 

/*  glob  --  expand filename specification
 *
 *  Usage:  buffer = glob (spec);
 *	char *spec,**buffer;
 *
 *  Glob takes a file specification, and expands it into
 *  filenames by resolving the characters '*', '?', and '['
 *  (also ']') in the same manner as the shell.
 *  Expand will compute the corresponding filenames, and
 *  will return buffer, a vector of char *'s pointing to the
 *  matching file names.  The last pointer will be followed
 *  by the value NULL to indicate the end of the vector.
 *  Expand returns NULL if no files matched the pattern or if
 *  an error occured during the expansion.
 *
 *  HISTORY
 *  3-Mar-84  Mike Shaddock (shaddock) at University of North Carolina at
 *	Chapel Hill.  Changed to work under 4.2bsd.
 *
 * 14-Dec-81  David Nichols (nichols) at Carnegie-Mellon University
 *	Changed the way '.' is handled for the Nth time.  Here's how it
 *	works now: the files "." and ".." never match a wildcard.  A file
 *	whose name begins with '.' may only be matched by a pattern whose
 *	first character is '.' (i.e. it must be explicitly matched).
 *
 * 05-Dec-81  David Nichols (nichols) at Carnegie-Mellon University
 *	Changed to allocate vector for string pointers and return that
 *	instead of requiring user to guess how many files will match.
 *	Changed to never return a name that stat(2) doesn't think is there.
 *	Changed way '.' is handled.  The only thing special now is that
 *	the files "." and ".." never match any pattern.
 *
 * 23-Jan-80  Steven Shafer (sas) at Carnegie-Mellon University
 *	Added check for DOT in xmatch.
 *
 * 05-Dec-79  Steven Shafer (sas) at Carnegie-Mellon University
 *	Created.  The idea comes from the shell, but the implementation
 *	is somewhat different.
 *
 */

#include <stdio.h>

#ifdef hpux
#define BSD42
#endif

#ifndef BSD42
#include "sys/types.h"
#include "sys/stat.h"
#include "sys/dir.h"
#endif
#ifdef BSD42
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#define	CMU 1
#endif

#include	<string.h>

#ifndef BSD42
#define DNAMESIZE 14		/* size of one level of filename */
#endif
#ifdef BSD42
#define	DNAMESIZE MAXNAMLEN	/* from sys/dir.h */
#endif

#define META "[*?"		/* meta-characters to expand */
#define SLASH '/'		/* directory-name separator */

char *malloc(), *realloc(), *skipto();
char *IDLstrchr(), *IDLstrrchr();

static int xexpand();
static int xmatch();

static int nres;		/* # of results found so far */
static int smallsize, bigsize;	/* control Fibonacci storage allocator */
static char **results;		/* where to put them */

char **glob (fname)
char *fname;
{
	nres = 0;
	smallsize = 2;
	bigsize = 3;
	results = (char **) malloc ((unsigned)(bigsize * sizeof(char *)));
	if (results == NULL)
		return NULL;
	if (xexpand(fname) == -1)  return NULL;
	results[nres] = NULL;	/* mark end */
	return results;
}

static int xexpand (fname)
char *fname;
{
	register char *dirpart,*metapart,*endpart;
	/* ptrs to parts of fname */
	char *meta;		/* ptr to first meta-char */
	char whichmeta;		/* char at *meta */
	char *lastslash;	/* last / before meta */
	char *newname;		/* buffer for generated name */
	int isadir;		/* is "dirpart" a directory? */
	struct stat statb;	/* buffer for "stat" call */
	char dname[DNAMESIZE+1];	/* for name from directory */
#ifndef BSD42
	struct direct dirb;	/* buffer for directory entry */
	int dirfile;		/* file desc. for directory */
#endif
#ifdef BSD42
	struct direct *dirb;
	DIR *dirfile;
#endif

	meta = skipto (fname,META);
	/* find first meta-character */

	if (*meta == '\0') {	/* no meta-chars */
#ifndef SAS
		if (stat (fname, &statb) == -1)	/* no file */
			return (0);
#endif
		if (nres + 1 >= bigsize) {	/* too many names */
			int newbig = bigsize + smallsize;
			smallsize = bigsize;
			bigsize = newbig;
			results = (char **) realloc ((char *)results,
				(unsigned)(bigsize * sizeof(char *)));
			if (results == NULL)
				return (-1);
		}
		results[nres] = malloc ((unsigned)(1 + strlen(fname)));
		if (results[nres] == 0)  return (-1);
		(void)strcpy (results[nres],fname);
		nres++;
		return (0);
	}

	whichmeta = *meta;	/* remember which character */
	*meta = '\0';		/* terminate string for IDLstrrchr() */
	lastslash = IDLstrrchr (fname,SLASH);	/* find last '/' */
	if (lastslash == 0) {	/* no slashes before meta */
		dirpart = ".";
		metapart = fname;
	}
	else {			/* chop off directory part */
		dirpart = ((lastslash == fname) ? "/" : fname);
		*lastslash = '\0';
		metapart = lastslash + 1;
	}
	*meta = whichmeta;	/* replace meta-character */
	endpart = IDLstrchr (metapart,SLASH);	/* find / after meta */
	if (endpart)	*endpart++ = '\0';	/* chop it off */

#ifndef BSD42
	isadir = ((stat(dirpart,&statb) >= 0)
	    && ((statb.st_mode & S_IFMT) == S_IFDIR)
	    && ((dirfile = open (dirpart,0)) >= 0));
#endif
#ifdef BSD42
	isadir = ((stat(dirpart,&statb) >= 0)
	    && ((statb.st_mode & S_IFMT) == S_IFDIR)
	    && ((dirfile = opendir (dirpart)) != NULL));
#endif

	if (!isadir)	return (0);	/* dirpart not directory */

	/* To be considered for a match the directory entry must meet the
	 * following conditions:
	 *	1) inode is not zero (i.e. directory entry is valid)
	 *	2) name is not "." or ".." (NEVER match these)
	 *	3) A leading '.' is matched by an explicit '.' in the pattern
	 */
#ifndef BSD42
	while (read(dirfile,(char *)&dirb,(sizeof dirb)) == (sizeof dirb)) {
		if (dirb.d_ino != 0 && strcmp(dirb.d_name, ".") != 0
				&& strcmp(dirb.d_name, "..") != 0
				&& !(dirb.d_name[0] == '.' && metapart[0] != '.')) {
			(void)strncpy (dname,dirb.d_name,(int)DIRSIZ((&dirb)));
			dname[DNAMESIZE] = '\0';
			if (xmatch (dname,metapart)) {	/* match */
				newname = malloc ((unsigned)strlen(dirpart) + strlen(dname) + (endpart ? strlen(endpart) : 0) + 3);
				if (newname == 0) {
					(void)close (dirfile);
					if (lastslash)  *lastslash = SLASH;
					return (-1);
				}
				(void)sprintf (newname,"%s%s%s%s%s",
				((lastslash && (lastslash != fname)) ? dirpart : ""),
				(lastslash ? "/" : ""),
				dname,
				(endpart ? "/" : ""),
				(endpart ? endpart : ""));
				if (xexpand(newname) == -1) {
					(void)close (dirfile);
					if (lastslash)  *lastslash = SLASH;
					return (-1);
				}
				free (newname);
			}
		}
	}

	(void)close (dirfile);
#endif
#ifdef BSD42
	while ((dirb = readdir(dirfile)) != NULL) {
		if (dirb->d_ino != 0 && strcmp(dirb->d_name, ".") != 0
				&& strcmp(dirb->d_name, "..") != 0
				&& !(dirb->d_name[0] == '.' && metapart[0] != '.')) {
			(void)strncpy (dname,dirb->d_name,DNAMESIZE);
			dname[DNAMESIZE] = '\0';
			if (xmatch (dname,metapart)) {	/* match */
				newname = malloc ((unsigned)strlen(dirpart) + strlen(dname) + (endpart ? strlen(endpart) : 0) + 3);
				if (newname == 0) {
					(void)closedir (dirfile);
					if (lastslash)  *lastslash = SLASH;
					return (-1);
				}
				(void)sprintf (newname,"%s%s%s%s%s",
				((lastslash && (lastslash != fname)) ? dirpart : ""),
				(lastslash ? "/" : ""),
				dname,
				(endpart ? "/" : ""),
				(endpart ? endpart : ""));
				if (xexpand(newname) == -1) {
					(void)closedir (dirfile);
					if (lastslash)  *lastslash = SLASH;
					return (-1);
				}
				free (newname);
			}
		}
	}

	(void)closedir (dirfile);
#endif
	if (lastslash)  *lastslash = SLASH;
	return (0);
}

static int xmatch (name,pattern)
char *name,*pattern;
{
	register char *n,*p;	/* pointers into strings */
	register char nc,pc;	/* chars from strings */
	char cur;		/* for [a-z] specs */
	int ok;			/* for [...] specs */

	/* in this routine, one char (or spec) from pattern is
	 * parsed and compared with the appropriate number of chars
	 * from name.  If a match is made, the comparison is continued.
	 * If not, 0 is returned.  The comparison is continued by
	 * recursively calling xmatch with incremented arguuments;
	 * this is necessary for the easy evaluation of "*", which
	 * is probably the most common meta-character.
	 */

	n = name;
	p = pattern;
	nc = *n++;
	pc = *p++;

	switch (pc) {

	case '?':
		if (nc == 0)  return (0);
		return (xmatch (n,p));

	case '[':
		ok = 0;
		cur = 0;
		while (pc = *p++) {
			if (pc == ']') {
				if (!ok)  return (0);
				return (xmatch (n,p));
			}
			else if (pc == '-') {
				if (cur <= nc && nc <= *p++)  ok = 1;
			}
			else {
				if (pc == nc)  ok = 1;
				cur = pc;
			}
		}
		return (0);

	case '*':
		if (*p == 0)  return (1);	/* * at end matches anything */
		--n;
		while (*n) {
			if (xmatch (n,p)) return (1);
			n++;
		}
		return (0);

	case NULL:
		return (nc == '\0');

	default:
		if (nc != pc)  return (0);
		return (xmatch (n,p));
	}
}
