{***********************************************************************\ 
*									* 
*   File: scorpion/include/Pascal/reader.h 
*				 					* 
*   Copyright (C) 1991 Michael Shapiro and Richard Snodgrass
*									* 
*   The Scorpion System is free software in the public domain; you can  * 
*   redistribute it and/or modify it as you wish. We ask that you 	* 
*   retain credits referencing the University of Arizona and that you	* 
*   identify any changes you make.					* 
*									* 
*   Report problems to scorpion-project@cs.arizona.edu			* 
*   Direct all inquiries to:	The Scorpion Project			* 
*				Department of Computer Science		* 
*				Gould-Simpson Building			* 
*				University of Arizona			* 
*				Tucson, AZ 85721			* 
*				U.S.A.					* 
*									* 
*	This file is included in the reader routines			*
*	produced by the IDL processor.					*
*
*   Revision Log:							* 
*	$Log:$ 
*									* 
*   Edit Log:								* 
*									* 
\***********************************************************************} 

procedure Reader (var Input:text; var OK: boolean; var Root:nodeType);
{	Read a structure from an input file.
	Parameters:
	Input:	file from which to read the structure.  It must
		already be open.
	OK: 	Result parameter.  If TRUE, no problems were
		encountered during input.  If FALSE, the internal
		data structure may not be valid.
	Root:	Result parameter through which the root of the data
		structure is returned.
}
const
LTABSIZ=1024;

type

SeqType = ^CellType;
RepType = ^ RecRepType;
PosType = record
	line	:integer
	end;
AttrDesc = integer;

LexType = (
LexCloseAngleBracket, LexCloseBrace, LexCloseSquareBracket, LexEOF, LexFalse,
LexTrue, LexInteger, LexLabelDef, LexLabelRef, LexName, LexOpenAngleBracket,
LexOpenBrace, LexOpenSquareBracket, LexRational, LexSemiColon, LexString );

LabelKind = ( SeqLabel, NodeLabel );
LabelRefType = ^LabelRefRecord;
LabelType = ^LabelRecord;
LabelRefRecord = record
	next	:LabelRefType;
	RefList	:LabelType;
	Unresolved	:integer;
	Generated	:boolean;
	Name	:String;
	Val	:RepType
	end;
LabelRecord = record
	next	:LabelType;
	parent	:LabelRefType;
	Pos	:PosType;
	case Kind:LabelKind of
	SeqLabel:	(SLabel:SeqType);
	NodeLabel:	(NodeField:nodeType;Desc:AttrDesc)
	end;

Lexeme = record
	Kind	:LexType;
	Pos	:PosType;
	Token	:String;
	case boolean of
	true: (R: real);
	false: (I: integer)
	end;

RepKind = ( RepLabelRef, RepNode, AtrInteger, AtrRational, AtrString,
	AtrBoolean, AtrSequence );
RecRepType = record
	Pos		:PosType;
	StringField	:String;
	Labelled	:boolean;
	case Kind:RepKind of
		AtrSequence	:(seqVal:SeqType);
		AtrBoolean	:(boolVal:boolean);
		AtrInteger	:(intVal:integer);
		AtrRational	:(ratVal:real);
		AtrString	:();
		RepLabelRef	:(LabelField:LabelRefType);
		RepNode		:(NodeField:nodeType)
	end;

CellType = record
	next	:SeqType;
	value	:RepType
	end;

ErrType = ( ErrAngleExpected, ErrBraceExpected, ErrEOF, ErrSemiColonExpected,
	ErrSkipStopped, ErrValueExpected, ErrBadNodeType, ErrNodeExpected,
	ErrEOFExpected, ErrBadRootType, ErrNoRoot, ErrBadLex, 
	ErrBadEscapedChar );
LErrType = (ErrUnresolvedLabel, ErrUnresolvedKids, ErrBadLabelType );

var CallOK : boolean; TempRoot : RepType; Lex,TempLex : Lexeme;
	C:char;
	Line:integer;
	LabelTable:array[1..LTABSIZ] of LabelRefType;

(* Forward Declarations *)
procedure FindAttribute(var GoodPair:boolean; K:integer; AttrName:Lexeme;
			var Attribute:AttrDesc); forward;

procedure FindNodeType(var OK:boolean; NodeName:String; var K:integer);
	forward;

function NodeAlloc(K:integer):nodeType; forward;

procedure AttrStore(Ref:nodeType; Desc:AttrDesc; Val:RepType); forward;

function LabelRef(Lex:Lexeme):RepType; forward;

procedure DelRep(R:RepType); forward;

function AllocString: String; forward;

(* Utility routines *)
(*function StrCopy(S:String):String;
var Ret:String;
i:integer;
begin
	new(Ret);
	Ret^.length := S^.length;
	for i := 1 to S^.length do
		Ret^.value[i] := S^.value[i];
	StrCopy := Ret
end;
*)

function Gboolean(Val:RepType): boolean;
begin
	if Val=nil
		then Gboolean := false
		else	begin
				case Val^.Kind of
				AtrBoolean:
					Gboolean := Val^.boolVal
				end;
			DelRep(Val)
			end
end;

function Ginteger(Val:RepType): integer;
begin
	if Val=nil
		then Ginteger := 0
		else	begin
				case Val^.Kind of
				AtrInteger:
					Ginteger := Val^.intVal
				end;
			DelRep(Val)
			end
end;

function Greal(Val:RepType): real;
begin
	if Val=nil
		then Greal := 0
		else	begin
				case Val^.Kind of
				AtrRational:
					Greal := Val^.ratVal
				end;
			DelRep(Val)
			end
end;

function GString(Val:RepType): String;
begin
	if Val=nil
		then GString := NewString(0,'  ')
		else	begin
			GString := NewString(Val^.StringField^.length,
						Val^.StringField^.value);
			DelRep(Val)
			end
end;

(* error functions *)

procedure Error(What:ErrType; Where:PosType);
var lineNum:integer;
begin
	lineNum := Where.line;
	case What of
		ErrAngleExpected:
			message(
			'IDL reader error: Expected an angle bracket on line ',
			lineNum);
		ErrBraceExpected:
			message(
			'IDL reader error: Expected a square bracket on line ',
			lineNum);
		ErrEOF:
			message(
			'IDL reader error: Unexpected end of file on line ',
			lineNum);
		ErrSemiColonExpected:
			message(
			'IDL reader error: Expected a semicolon on line ',
			lineNum);
		ErrSkipStopped:
			message(
			'IDL reader error: Error recovery failed on line ',
			lineNum);
		ErrValueExpected:
			message(
			'IDL reader error: Expected a value on line ',
			lineNum);
		ErrBadNodeType:
			message(
			'IDL reader error: Bad node type on line ',
			lineNum);
		ErrNodeExpected:
			message(
			'IDL reader error: Expected a node on line ',
			lineNum);
		ErrEOFExpected:
			message(
			'IDL reader error: Expected end of file on line ',
			lineNum);
		ErrBadRootType:
			message(
			'IDL reader error: Root type invalid for this port');
		ErrNoRoot:
			message('IDL reader error: No root!!!');
		ErrBadLex:
			message('IDL reader error: Bad character on line ',
			lineNum);
		ErrBadEscapedChar:
			message('IDL reader error: Illegal escaped character on line ',
			lineNum);
	end;
end;

procedure ErrInit;
begin
end;

procedure ErrFinish;
	begin
	end;

(* sequence functions *)

function NewSeq:SeqType;
begin
	NewSeq := nil
end;

procedure AddSeq(var L:SeqType; V:RepType);
begin
	if L = nil then
		begin
		new(L);
		L^.next := nil;
		L^.value := V
		end
	else	AddSeq(L^.next,V)
end;

(* functions that return RepTypes *)

function MakeVal(Attr:RepKind; Lex:Lexeme):RepType;
var R:RepType;
begin
	new(R);
	R^.Pos := Lex.Pos;
	R^.StringField := Lex.Token;
	R^.Kind := Attr;
	R^.Labelled := false;
	MakeVal := R
end;


function MakeRat(Rat:real; P:PosType):RepType;
var R:RepType;
begin
	new(R);
	R^.Pos := P;
	R^.Kind := AtrRational;
	R^.ratVal := Rat;
	R^.StringField := nil;
	R^.Labelled := false;
	MakeRat := R
end;



function MakeInt(Int:integer; P:PosType):RepType;
var R:RepType;
begin
	new(R);
	R^.Pos := P;
	R^.Kind := AtrInteger;
	R^.intVal := Int;
	R^.StringField := nil;
	R^.Labelled := false;
	MakeInt := R
end;


function MakeBool(TorF:boolean; P:PosType):RepType;
var R:RepType;
begin
	new(R);
	R^.Pos := P;
	R^.Kind := AtrBoolean;
	R^.boolVal := TorF;
	R^.StringField := nil;
	R^.Labelled := false;
	MakeBool := R
end;

function MakeNode(N:nodeType; P:PosType):RepType;
var R:RepType;
begin
	new(R);
	R^.Pos := P;
	R^.StringField := nil;
	R^.Kind := RepNode;
	R^.NodeField := N;
	R^.Labelled := false;
	MakeNode := R
end;

function MakeSeq(S:SeqType; P:PosType):RepType;
var R:RepType;
begin
	new(R);
	R^.Pos := P;
	R^.StringField := nil;
	R^.Kind := AtrSequence;
	R^.seqVal := S;
	R^.Labelled := false;
	MakeSeq := R
end;

(*
procedure DelSequence(S:SeqType);
var N:SeqType;
	begin
	while S<>nil do
		begin
		N := S^.next;
		dispose(S);
		S := N
		end
	end;
*)

procedure DelRep(* R:RepType *);
begin
	if not R^.Labelled then
	begin	if R^.StringField <> nil then dispose(R^.StringField);
		(* if R^.Kind=RepNode then dispose(R^.NodeField)
		 * else if R^.Kind=AtrSequence then DelSequence(R^.seqVal);
		 *)
		dispose(R)
	end
end;

(* string functions *)

function AllocString;
var S:String;
	i: integer;
begin
	new(S);
	S^.length := 0;
	for i := 1 to 512 do S^.value[i] := ' ';
	AllocString := S
end;


function CompStrings   (S1: String; S2: packedArrayType; length: integer ):boolean;
var	match:boolean;
	i: integer;
begin	
	(* writeln('S1=''',S1^.value,''',S2=''',S2, '''');  *)
	match := length = S1^.length;
	i := 1;
	while (match and (i<=length)) do 
	begin
	    match := S1^.value[i] = S2[i];
	    i := i + 1;
	end;
	CompStrings := match
end;

procedure AddString(var S:String; c:char);
begin
	S^.length := S^.length + 1;
	S^.value[S^.length] := c
end;

procedure DelString(S:String);
begin
	if S <> nil then dispose(S)
end;

(************************************************************************)
(*	Lexical Analysis						*)
(*									*)
(*	Author: Tim Maroney, Spring, 1984				*)
(*									*)
(*	Function: returns the next Lexeme.				*)
(*	Method: Implements an FSA, with transitions for each char.	*)
(*									*)
(************************************************************************)

procedure LexInit(var MyFile:text);
begin
	Line := 1;
	read(MyFile,C)
end;

procedure LexRead(var F:text; var c:char);
begin
	if eoln(F) then Line := Line + 1;
	read(F,c)
end;

(* check for correct escaped characters in a string *)
procedure escapecheck(var MyFile:text; Ret:Lexeme);
var
	nc : char;

begin
	LexRead(MyFile, nc);
	if (nc >= '@') and (nc <= '_') then
	begin
	    nc := chr(ord(nc) - ord('@'));
	    AddString(Ret.Token,nc)
	end
	else if nc = '~' then
		 AddString(Ret.Token,nc)
	     else if nc = '{' then
		      AddString(Ret.Token,chr(127))
		  else
		      Error(ErrBadEscapedChar, Ret.Pos);
end;

procedure GetLex(var MyFile:text; var Ret:Lexeme);
label	0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21;
var i, intpart, fracpart, fraccount, exp, base, ordzero, ordA: integer;
	first: real;
	seendivide, ispositive, expispositive: boolean;
begin
	Ret.Token := AllocString;
	Ret.Pos.line := Line;
	seendivide := false;
	ordzero := ord('0');
	ordA := ord('A');

	19:	intpart := 0;
		fracpart := 0;
		fraccount := 0;
		exp := 0;
		base := 10;
		ispositive := true;
		expispositive := true;

	0:	if eof(MyFile) then
			begin
			Ret.Kind := LexEOF;
			goto 1
			end

		else if (C='0') or ((C >= '1') and (C <= '9')) then
			begin
			Ret.Kind := LexInteger;
			AddString(Ret.Token,C);
			intpart := ord(C) - ordzero;
			LexRead(MyFile,C);
			goto 3
			end

		else if ((C >= 'a') and (C <= 'z'))
			or ((C >= 'A') and (C <= 'Z')) then
			begin
			Ret.Kind := LexName;
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 13
			end

		else if C = '"' then
			begin
			Ret.Kind := LexString;
			LexRead(MyFile,C);
			goto 17
			end

		else case C of

		(* white space *)

		' ':	begin
			LexRead(MyFile,C);
			Ret.Pos.line := Line;
			goto 0
			end;
		'	':begin
			LexRead(MyFile,C);
			goto 0
			end;

		(* punctuation *)

		'{':	begin
			Ret.Kind :=  LexOpenBrace;
			LexRead(MyFile,C);
			goto 1
			end;
		'}':	begin
			Ret.Kind :=  LexCloseBrace;
			LexRead(MyFile,C);
			goto 1
			end;
		'<':	begin
			Ret.Kind :=  LexOpenAngleBracket;
			LexRead(MyFile,C);
			goto 1
			end;
		'>':	begin
			Ret.Kind :=  LexCloseAngleBracket;
			LexRead(MyFile,C);
			goto 1
			end;
		'[':	begin
			Ret.Kind :=  LexOpenSquareBracket;
			LexRead(MyFile,C);
			goto 1
			end;
		']':	begin
			Ret.Kind :=  LexCloseSquareBracket;
			LexRead(MyFile,C);
			goto 1
			end;
		';':	begin
			Ret.Kind :=  LexSemiColon;
			LexRead(MyFile,C);
			goto 1
			end;
		'+':	begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 2
			end;
		'-':	begin
			LexRead(MyFile,C);
			if (C = '-') then 	(* comment *)
			begin
			    while not eoln(MyFile) do
				LexRead(MyFile, C);
			    LexRead(MyFile, C);
			    goto 0
			end
			else begin
			    AddString(Ret.Token,'-');
			    ispositive := false;
			    goto 2
			end
			end;
		'#':	begin		(* soft EOF *)
			Ret.Kind := LexEOF;
			goto 1
			end
		end;

	2:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			Ret.Kind := LexInteger;
			AddString(Ret.Token,C);
			intpart := ord(C) - ordzero;
			LexRead(MyFile,C);
			goto 3
			end
		else	Error(ErrBadLex,Ret.Pos);

	3:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			AddString(Ret.Token,C);
			intpart := intpart * 10 + ord(C) - ordzero;
			LexRead(MyFile,C);
			goto 3
			end
		else if C = '#' then
			begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 4
			end
		else if C = 'E' then
			begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 5
			end
		else if C = '.' then
			begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 6
			end
		else if C = '/'
		then	goto 21
		else	goto 20;

	4:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			AddString(Ret.Token,C);
			base := intpart;
			intpart := ord(C) - ordzero;
			LexRead(MyFile,C);
			goto 7
			end
		else if ((C >= 'A') and (C <= 'F')) then
			begin
			AddString(Ret.Token,C);
			base := intpart;
			intpart := ord(C) - ordA + 10;
			LexRead(MyFile,C);
			goto 7
			end
		else	Error(ErrBadLex,Ret.Pos);

	5:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			Ret.Kind := LexRational;
			AddString(Ret.Token,C);
			exp := ord(C) - ordzero;
			LexRead(MyFile,C);
			goto 11
			end
		else if (C = '+') then
			begin
			expispositive := true;
			LexRead(MyFile,C);
			goto 5
			end
		else if (C = '-') then
			begin
			expispositive := false;
			LexRead(MyFile,C);
			goto 5
			end
		else	Error(ErrBadLex,Ret.Pos);

	6:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			Ret.Kind := LexRational;
			AddString(Ret.Token,C);
			fracpart := ord(C) - ordzero;
			fraccount := 1;
			LexRead(MyFile,C);
			goto 10
			end
		else	Error(ErrBadLex,Ret.Pos);

	7:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			AddString(Ret.Token,C);
			intpart := intpart * base + ord(C) - ordzero;
			LexRead(MyFile,C);
			goto 7
			end
		else if ((C >= 'A') and (C <= 'F')) then
			begin
			AddString(Ret.Token,C);
			intpart := intpart * base + ord(C) - ordA + 10;
			LexRead(MyFile,C);
			goto 7
			end
		else if C = '#' then
			begin
			Ret.Kind := LexRational;
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 12
			end
		else if C = '.' then
			begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 8
			end
		else	Error(ErrBadLex,Ret.Pos);

	8:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			AddString(Ret.Token,C);
			fracpart := ord(C) - ordzero;
			fraccount := 1;
			LexRead(MyFile,C);
			goto 9
			end
		else if ((C >= 'A') and (C <= 'F')) then
			begin
			AddString(Ret.Token,C);
			fracpart := ord(C) - ordA + 10;
			fraccount := 1;
			LexRead(MyFile,C);
			goto 9
			end
		else	Error(ErrBadLex,Ret.Pos);

	9:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			AddString(Ret.Token,C);
			fracpart := fracpart * base + ord(C) - ordzero;
			fraccount := fraccount + 1;
			LexRead(MyFile,C);
			goto 9
			end
		else if ((C >= 'A') and (C <= 'F')) then
			begin
			AddString(Ret.Token,C);
			fracpart := fracpart * base + ord(C) - ordA + 10;
			fraccount := fraccount + 1;
			LexRead(MyFile,C);
			goto 9
			end
		else if C = '#' then
			begin
			Ret.Kind := LexRational;
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 12
			end
		else	Error(ErrBadLex,Ret.Pos);

	10:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			AddString(Ret.Token,C);
			fracpart := fracpart * 10 + ord(C) - ordzero;
			fraccount := fraccount + 1;
			LexRead(MyFile,C);
			goto 10
			end
		else if C = 'E' then
			begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 5
			end
		else if C = '/'
		then	goto 21
		else	goto 20;

	11:	if (C = '0') or ((C >= '1') and (C <= '9')) then
			begin
			AddString(Ret.Token,C);
			exp := exp * 10 + ord(C) - ordzero;
			LexRead(MyFile,C);
			goto 11
			end
		else if C = '/'
		then	goto 21
		else	goto 20;

	12:	if C = 'E' then
			begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 5
			end
		else	goto 20;

	13:	if ((C >= 'a') and (C <= 'z')) or ((C >= 'A') and (C <= 'Z'))
			or (C = '0') or ((C >= '1') and (C <= '9'))
			or (C = '_') then
			begin
			AddString(Ret.Token,C);
			LexRead(MyFile,C);
			goto 13
			end
		else if C = '^' then
			begin
			LexRead(MyFile,C);
			goto 15
			end
		else if C = ':' then
			begin
			LexRead(MyFile,C);
			goto 16
			end
		else if C = ' ' then
			begin
			LexRead(MyFile,C);
			goto 14
			end
		else if C = '	' then
			begin
			LexRead(MyFile,C);
			goto 14
			end
		else	goto 18;

	14:	if C = '^' then
			begin
			LexRead(MyFile,C);
			goto 15
			end
		else if C = ':' then
			begin
			LexRead(MyFile,C);
			goto 16
			end
		else if C = ' ' then
			begin
			LexRead(MyFile,C);
			goto 14
			end
		else if C = '	' then
			begin
			LexRead(MyFile,C);
			goto 14
			end
		else	goto 18;	(* if no case matches *)

	15:	Ret.Kind := LexLabelRef;
		goto 1;

	16:	Ret.Kind := LexLabelDef;
		goto 1;

	17:	if C <> '"' then
			begin
			if (C = '~')
			    then escapecheck(MyFile, Ret)
			else AddString(Ret.Token,C);
			LexRead(MyFile, C);
			goto 17
			end
		else	begin
			LexRead(MyFile, C);
			if (C <> '"')
			    then goto 1
			else begin
			    AddString(Ret.Token, C);
			    LexRead(MyFile, C);
			    goto 17
			    end
			end;

	18:	if (Ret.Token^.length = 4) then
			begin
			if (Ret.Token^.value[1] = 'T') and
				(Ret.Token^.value[2] = 'R') and
				(Ret.Token^.value[3] = 'U') and
				(Ret.Token^.value[4] = 'E') then
				Ret.Kind := LexTrue
			end
		else if (Ret.Token^.length = 5) then
			if (Ret.Token^.value[1] = 'F') and
				(Ret.Token^.value[2] = 'A') and
				(Ret.Token^.value[3] = 'L') and
				(Ret.Token^.value[4] = 'S') and
				(Ret.Token^.value[5] = 'E') then
				Ret.Kind := LexFalse;
		goto 1;

	20:	if not seendivide and (Ret.Kind = LexInteger)
		then begin
			if ispositive
			then Ret.I := intpart
			else Ret.I := -intpart
			end
		else with Ret do
		begin
			R := fracpart;
			for i := 1 to fraccount do R := R / base;
			R := R + intpart;
			if (expispositive) then
			    for i := 1 to exp do R := R * base
			else for i := 1 to exp do R := R / base;
			if not ispositive then R := - R;
			if seendivide then R := first / R;
			Kind := LexRational
			end;
		goto 1;

	21:	seendivide := true;
		AddString(Ret.Token, C);
		first := fracpart;
		for i := 1 to fraccount do first := first / base;
		first := first + intpart;
		for i := 1 to exp do first := first * base;
		if not ispositive then first := - first;
		LexRead(MyFile, C);
		goto 19;

	1:	null (* finished *)
end; (* GetLex *)

procedure MsgString(S:String);
	var i:integer;
	begin
	for i := 1 to S^.length do write(S^.value[i])
	end;

function StringComp(S1,S2:String):integer; 
var	stat:integer;
	i:integer;
	done:boolean;
begin	i := 1;
	stat := 0;
	done := false;
	while not done do
	begin	if ord(S1^.value[i]) < ord(S2^.value[i]) then
		begin	stat := -1;
			done := true
		end
		else if ord(S1^.value[i]) > ord(S2^.value[i]) then
		begin	stat := 1;
			done := true
		end
		else
		begin	i := i+1;
			if i > S1^.length then
				if i > S2^.length then
					done := true
				else
				begin	stat := -1;
					done := true
				end
			else if i > S2^.length then
			begin	done := true;
				stat := 1
			end
		end
	end;
	StringComp := stat
end;
function hash(S:String):integer;
var i:integer;
    h:integer;
begin
	h := 0;
	for i := 1 to S^.length do
		h := h + (ord(S^.value[i]) * ord(S^.value[i]));
	hash := h mod LTABSIZ
end;

(* label handling functions *)

procedure LError(E:LErrType;S:String);
begin	case (E) of
	ErrUnresolvedLabel:
		begin
		MsgString(S);
		message(': Label Error: Unresolved label')
		end;
	ErrUnresolvedKids:
		begin
		MsgString(S);
		message(
		': Label Error: generated label with unresolved elements')
		end;
	ErrBadLabelType:
		message(
		'Label Error: bad label type in forward reference')
	end
end;

(* Label handling *)

procedure FixUpRefs(MyLabel:LabelRefType; P:PosType);
(* Fix up each forward reference with the given value.
 * Decrement the Unresolved member of the parent of each forward reference;
 * if it's 0, then call FixUpRefs on the parent, since it can now be safely
 * generated.  Delete each forward reference.
 *)
var	tmp:LabelType;
	L:LabelType;
	Value:RepType;
begin
	L := MyLabel^.RefList;
	MyLabel^.RefList := nil;
	Value := MyLabel^.Val;
	while (L <> nil) do
	begin	if (L^.Kind=SeqLabel) then
			L^.SLabel^.value := Value
		else if (L^.Kind=NodeLabel) then
			AttrStore(L^.NodeField,L^.Desc,Value)
		else	LError(ErrBadLabelType,nil);
		if (L^.parent^.Unresolved<>0) then
		begin	if (L^.parent^.Unresolved = 1) then
			begin	L^.parent^.Unresolved := 0;
				FixUpRefs(L^.parent,P);
				if (L^.parent^.Name = nil) then
					dispose(L^.parent)
			end
			else
				L^.parent^.Unresolved:=L^.parent^.Unresolved-1
		end;
		tmp := L^.next;
		dispose(L);
		L := tmp
	end;
end;

function FindLabel(Lex:Lexeme):LabelRefType;
var	R,P:LabelRefType;
	cmp,h:integer;
	found,under:boolean;
begin
	found := false;
	h := hash(Lex.Token);
	R := LabelTable[h];
	if (R = nil) then
	begin	new(R);
		LabelTable[h] := R;
		P := nil
	end
	else
	begin	cmp := StringComp(R^.Name,Lex.Token);
		if (cmp=0) then
			found := true
		else if (cmp > 0) then
		begin	P := R;
			new(R);
			LabelTable[h] := R
		end
		else
		begin	P := R;
			R := R^.next;
			cmp := 1;
			under := true;
			while (R <> nil) and under do
			begin	cmp := StringComp(R^.Name,Lex.Token);
				if cmp < 0 then
				begin	P := R;
					R := R^.next
				end
				else under := false
			end;
			if (cmp<>0) then	(* didn't find it *)
			begin	new(P^.next);
				P^.next^.next := R;
				R := P^.next;
				P := R^.next
			end
			else
				found := true
		end
	end;
	if found then
		dispose(Lex.Token)
	else
	begin	R^.RefList := nil;
		R^.Unresolved := 0;
		R^.Generated := false;
		R^.Val := nil;
		R^.next := P;
		R^.Name := Lex.Token;
	end;
	FindLabel := R
end;

procedure ReplaceLabel(Old,New:LabelRefType);
var	R,P:LabelRefType;
	h:integer;
begin
	h := hash(Old^.Name);
	P := nil;
	R := LabelTable[h];
	while (R <> Old) do
	begin	P := R;
		R := R^.next
	end;
	if (P = nil) then	(* Old was first *)
		LabelTable[h] := New
	else	(* Old appeared later in chain *)
		P^.next := New;
	New^.next := R^.next;
	New^.Name := R^.Name;
	New^.RefList := R^.RefList;
	dispose(R)
end;

procedure SaveLabel(Lex:Lexeme;Value:RepType);
(* Find the label in the table.  Set the Val member to the Value arg,
 * and resolve every forward reference in the RefList member.
 *)
var	R:LabelRefType;

begin
	R := FindLabel(Lex);
	(* If we are passed a label reference as the value, it is a generated
	 * label.  All labels for which the fake label is the parent must now
	 * refer to the real label.  This is accomplished by replacing the
	 * real label that FindLabel returns with the generated label.  The
	 * forward references cannot be resolved yet, since the generated label
	 * has at least one unresolved entry.  It will be resolved when the
	 * last unresolved entry is resolved.
	 *)
	if (Value^.Kind = RepLabelRef) then
	begin	ReplaceLabel(R,Value^.LabelField);
		R := Value^.LabelField;
	end
	else
		R^.Val := Value;
	R^.Val^.Labelled := true;
	FixUpRefs(R,Lex.Pos);
end;

function LabelRef{Lex:Lexeme):RepType};
(* Find the label in the label table.  If the label is defined and completely
 * resolved, return the representation of the labelled value.  If the label
 * is not defined, return the label reference associated with the label.
 *)
var	R:LabelRefType;
	V:RepType;

begin
	R := FindLabel(Lex);
	if ((R^.Val<>nil) and (R^.Unresolved = 0)) then
		LabelRef := R^.Val
	else
	begin	new(V);
		V^.Pos := Lex.Pos;
		V^.StringField := nil;
		V^.Kind := RepLabelRef;
		V^.LabelField := R;
		LabelRef := V
	end;
end;

function FakeLabel(ForVal:RepType):RepType;
(* Fake labels are what Lamb calls 'generated' labels.  They are created
 * for nodes, sequences, and sets which contain unresolved labels.  The
 * effect is to treat these values as if they were themselves unresolved
 * label references, to be resolved when all their internal labelled values
 * are resolved.
 *
 * Create a new RepType with variant RepLabelRef.  Create a new
 * LabelRefRecord, with no name, Generated=true, and Val=ForVal.
 * Set the new RepType's LabelField to point to the new LabelRefRecord,
 * and return it.
 *)
var	V:RepType;
	R:LabelRefType;

begin	new(R);
	new(V);
	R^.RefList := nil;
	R^.next := nil;
	R^.Unresolved := 0;
	R^.Name := nil;
	R^.Generated := true;
	R^.Val := ForVal;
	V^.Pos := ForVal^.Pos;
	V^.StringField := nil;
	V^.Kind := RepLabelRef;
	V^.LabelField := R;
	FakeLabel := V
end;

procedure DelaySeqRef(List:SeqType; MyLabel,Fake:LabelRefType; Where:PosType);
(* Find the end of the list.  Add a pointer to the value field of the end
 * of the list to the label table entry for MyLabel.  Its parent is set
 * to Fake, and its Pos to Where.
 *)
var	L:LabelType;
begin	while (List^.next<>nil) do List := List^.next;
	new(L);
	L^.next := MyLabel^.RefList;
	MyLabel^.RefList := L;
	L^.parent := Fake;
	L^.Pos := Where;
	L^.Kind := SeqLabel;
	L^.SLabel := List
end;

procedure DelayedRef(NType: nodeType; Desc: AttrDesc;
			MyLabel, Fake:LabelRefType; Where: PosType);
(* Add an entry with NType and the attribute number to the list of
 * forward references.  The parent of the MyLabel entry should be Fake,
 * and its Pos should be Where.
 *)
var	L:LabelType;
begin	new(L);
	L^.next := MyLabel^.RefList;
	MyLabel^.RefList := L;
	L^.parent := Fake;
	L^.Pos := Where;
	L^.Kind := NodeLabel;
	L^.NodeField := NType;
	L^.Desc := Desc
end;

procedure LabelInit;
var i:integer;
begin
	for i := 1 to LTABSIZ do LabelTable[i] := nil
end;

function LabelFinish:boolean;
(* For every entry of the label table, check for unresolved labels and
 * cause an error (and set OK = 0) if so.  Regardless, delete it and set
 * its pointer in the table to nil.
 *)
var	i:integer;
	R,tmp:LabelRefType;
	LOK:boolean;

begin	LOK := true;
	for i := 1 to LTABSIZ do
	begin	R := LabelTable[i];
		while (R <> nil) do
		begin	(* MsgString(R^.Name);
			message(' seen in label table.'); *)
			if (R^.Val = nil) then
			begin	LOK := false;
				LError(ErrUnresolvedLabel,R^.Name);
			end
			else
			begin	R^.Val^.Labelled := false;
				DelRep(R^.Val);
				if (R^.Unresolved<>0) then
				begin	LOK := false;
					LError(ErrUnresolvedKids,R^.Name);
				end
			end;
			tmp := R^.next;
			dispose(R^.Name);
			dispose(R);
			R := tmp;
		end;
		LabelTable[i] := nil;
	end;
	LabelFinish := LOK;
end;

(************************************************************************)
(*	Syntactic Analysis						*)
(*									*)
(*	Author: Tim Maroney, Spring, 1984				*)
(*									*)
(*	Method: Implements a top-down, recursive descent parser		*)
(*									*)
(************************************************************************)

procedure RelLex(Lex:Lexeme);
{	Release the dynamically-allocated portions of a
		lexeme.
	Parameters:
		Lex: Lexeme to be released
}
begin
		if Lex.Token <> nil then begin
			DelString(Lex.Token); Lex.Token := nil
			end
end;

procedure NextLex;
{ Cover function for calls on the lexer. }
begin GetLex(Input,Lex); end;

function IsLex(Expected:LexType) : boolean;
begin
{	Test whether the current lexeme is a particular type,
	and advance to the next lexeme if so.  Used in common
	case of recognizing a simple lexeme (i.e., one for
	which no token need be saved) }

	if Lex.Kind=Expected
	then begin RelLex(Lex); NextLex; IsLex := true end
	else IsLex := false;
end; { IsLex }

procedure SkipLex(LexWanted,CloseLex,OpenLex : LexType);
{	Skip over lexemes until a particular one is found.
	LexWanted:	The lexeme to look for.
	CloseLex:	Lexeme that closes the environment
			in which LexWanted may be found.
	OpenLex:	Open lexeme corresponding to CloseLex.
}
var CloseCount:integer;
begin
	CloseCount := 1;
	while CloseCount > 0 do
		begin
		RelLex(Lex); NextLex;
		if Lex.Kind=CloseLex then CloseCount:=CloseCount-1
		else if Lex.Kind=OpenLex then CloseCount:=CloseCount+1
		else if (Lex.Kind = LexWanted) and (CloseCount=1) then
			CloseCount := 0
		else if Lex.Kind = LexEOF then
			begin
			Error(ErrEOF,Lex.Pos);
			CloseCount:=0
			end
		end;
	Error(ErrSkipStopped,Lex.Pos);
	if Lex.Kind = LexWanted then
	begin RelLex(Lex); NextLex end;
end; { SkipLex }

procedure PValue(var OK:boolean; var Val:RepType); forward;
{	Parse potentially labelled values.
	OK:	Result parameter, true if no parse errors.
	Val:	Result parameter, representation of value.
}

procedure ValueList(var Ret : RepType);
{	Parse a list of potentially labelled values.
	Ret: Result parameter, representation of value list.
}
var ThisVal : RepType; TempOK : boolean;
	FakeRef : RepType; Unresolved : integer;
begin
	Unresolved := 0; TempOK := true;
	Ret := MakeSeq(nil,Lex.Pos);
	while TempOK do
	begin
		PValue(TempOK,ThisVal);
		if TempOK then begin
			AddSeq(Ret^.seqVal,ThisVal);
			if ThisVal^.Kind = RepLabelRef then begin
				if Unresolved=0 then
				FakeRef := FakeLabel(Ret);
				Unresolved := Unresolved+1;
				DelaySeqRef(Ret^.seqVal,
					ThisVal^.LabelField,
					FakeRef^.LabelField,
					ThisVal^.Pos)
				end;
			end; { if TempOK }
		end; { while }
	if Unresolved <> 0 then
	begin	FakeRef^.LabelField^.Unresolved := Unresolved;
		Ret := FakeRef
	end
end; { ValueList }

procedure Node(var OK:boolean; var Ref:RepType); forward;

procedure Value(var OK:boolean; var Val: RepType);
{	Parse a value.  In this routine, tokens must not be
	released by RelLex, since they are used to form
	string values.
	OK:	Result parameter, true iff there were no
		parse errors.
	Val:	Result parameter, representation of the
		parsed value.
}
begin
	OK := true;
	if Lex.Kind = LexInteger then
		begin Val := MakeInt(Lex.I,Lex.Pos); NextLex end
	else if Lex.Kind = LexRational then
		begin Val := MakeRat(Lex.R,Lex.Pos); NextLex end
	else if Lex.Kind = LexString then
		begin Val := MakeVal(AtrString,Lex); NextLex end
	else if IsLex(LexTrue) then Val := MakeBool(true,Lex.Pos)
	else if IsLex(LexFalse) then Val := MakeBool(false,Lex.Pos)
	else if IsLex(LexOpenBrace) then begin
		ValueList(Val);
		if not IsLex(LexCloseBrace) then begin
			OK := false;
			Error(ErrBraceExpected,Lex.Pos)
			end
		end
	else if IsLex(LexOpenAngleBracket) then begin
		ValueList(Val);
		if not IsLex(LexCloseAngleBracket) then begin
			OK := false;
			Error(ErrAngleExpected,Lex.Pos)
			end
		end
	else Node(OK,Val);
end; { value }

procedure PValue (*var OK:boolean, var Val : RepType*);
{	Parse potentially labelled values.	}
var Labelled : boolean; Name : Lexeme;
begin
	if Lex.Kind = LexLabelRef
	then begin
		Val := LabelRef(Lex); OK := true;
		NextLex
		end
	else begin
		if Lex.Kind = LexLabelDef
		then begin
			Labelled := true;
			Name := Lex;
			NextLex
			end
		else Labelled := false;
		Value(OK,Val);
		if OK then if Labelled then
		SaveLabel(Name,Val);
		end;
end; { PValue }

procedure Pair(var OK:boolean; var Name:Lexeme; var Val:RepType);
{	Parse an attribute/value pair.
	OK:	Result parameter, true iff no parsing errors.
	Name:	Result parameter, attribute name.
	Val:	Result parameter, repn. of the value.
}
begin
	if Lex.Kind = LexName
	then begin
		Name := Lex;
		NextLex;
		PValue(OK,Val);
		if not OK then
			Error(ErrValueExpected,Name.Pos)
		end
	else OK := false;
end; { Pair }

procedure ParseAttributes(var GoodPair:boolean; A:integer;
	var Ref:RepType);
{	Parse attributes for a node.
	GoodPair:	result parameter, TRUE if
		no errors occurred.
	A:	Node kind of node read
	Ref:	Representation of node into which to store
		attributes.  This may be replaced by a
		generated label reference if the repn. of
		the node contains references to unresolved
		labels.
}
var ThisPos:PosType; AttrName:Lexeme; Val,FakeRef:RepType;
	Desc:AttrDesc; Unresolved:integer;
begin
	Unresolved := 0;
	FakeRef := nil;
	while not IsLex(LexCloseSquareBracket) do
	begin
		ThisPos := Lex.Pos;
		Pair(GoodPair,AttrName,Val);
		if GoodPair then begin { each attr/value pair }
			FindAttribute(GoodPair,A,AttrName,Desc);
			if GoodPair then
			begin
				if Val^.Kind = RepLabelRef then
				begin
					if Unresolved=0 then
						FakeRef :=
						     FakeLabel(Ref);
					Unresolved := Unresolved+1;
					DelayedRef(
					     Ref^.NodeField,
					     Desc,
					     Val^.LabelField,
					     FakeRef^.LabelField,
					     ThisPos);
					end
				else AttrStore(Ref^.NodeField,Desc,Val);
				end
			end; { each attr/value pair }
		if not IsLex(LexSemiColon) then
		begin
			if Lex.Kind<>LexCloseSquareBracket then begin
				Error(ErrSemiColonExpected,Lex.Pos);
				SkipLex(LexSemiColon,
					LexCloseSquareBracket,
					LexOpenSquareBracket);
				end;
			end;
		end; { while }
	if Unresolved <> 0 then begin
		Ref := FakeRef;
		Ref^.LabelField^.Unresolved := Unresolved;
		end;
end; { ParseAttributes }

procedure Node(* var OK:boolean, var Ref:RepType *);
{	Parse a node.
	OK:	result parameter, TRUE iff a node was found.
	Ref:	Result parameter.  If OK is true, the
		representation of the node.  It will be
		either variant RepNode or variant RepLabelRef.
}
var A:integer; TempRef : nodeType;
begin
	OK := true;
	if Lex.Kind <> LexName
	then OK := false
	else begin { parse the node }
		FindNodeType(OK,Lex.Token,A);
		if OK
		then TempRef := NodeAlloc(A)
		else begin
			TempRef := nil;
			Error(ErrBadNodeType,Lex.Pos);
			end;
		Ref := MakeNode(TempRef,Lex.Pos);
		RelLex(Lex);
		NextLex;
		if IsLex(LexOpenSquareBracket) then
		begin { find attributes }
			if OK
				then ParseAttributes(OK,A,Ref)
				else SkipLex(
					LexCloseSquareBracket,
					LexEOF, LexEOF)
			end { find attributes }
		end { parse the node }
end; { Node }

function LValueList:boolean;
{	Parse a list of (potentially) labelled values
	Returns TRUE if there were no problems during parsing.
}
var Name: Lexeme; TempOK : boolean; TempVal : RepType;
begin
	LValueList := true;
	while Lex.Kind = LexLabelDef do
	begin
		Name := Lex;
		NextLex;
		Value(TempOK,TempVal);
		if TempOK
			then SaveLabel(Name,TempVal)
			else LValueList := false;
		end;
end; { LValueList }

procedure NodeRef(var OK:boolean; var Ref:RepType);
{	Parse a node reference.
	OK:	Result parameter, TRUE if a node reference
		is found.
	Ref:	Result parameter.  If OK is true, a data
		structure represnting the node reference.
}
var	Name:Lexeme;
begin
	if Lex.Kind=LexLabelRef then begin
		Ref := LabelRef(Lex);
		NextLex;
		OK := true
		end
	else if Lex.Kind=LexLabelDef then begin
		Name := Lex;
		NextLex;
		Node(OK,Ref);
		if OK then begin
			SaveLabel(Name,Ref);
			if Ref^.Kind = RepLabelRef then
				Ref := Ref^.LabelField^.Val;
		end
	end
	else if Lex.Kind=LexEOF then begin
		Error(ErrNodeExpected,Lex.Pos);
		OK := false
		end
	else Node(OK,Ref);
end; { NodeRef }
