(* file: functionlang/frontend/Pascal/lexical.i *)

const maxidentifierlength = 128;

type tokenType = (endoffileToken,
	(* binary operators *)
		PlusToken, MinusToken, TimesToken,  DivideToken,
	(* unary operators *)
		UnaryMinusToken, UnaryPlusToken,
	(* punctuation *)
		LParenToken, RParenToken, SemiToken, EqualToken,
		ColonToken, CommaToken,
	(* keywords *)
		functionKeyword, integerKeyword, realKeyword,
	(* tokens with values *)
		IntegerToken, IdentifierToken, RealToken);

var 	token: tokenType; (* current token, set by lexer *)
	lasttoken: tokenType;
	yylval: record case tokenType of
		IntegerToken: (Vintegerconstant: integerconstant);
		IdentifierToken: (Videntifier: identifier);
		RealToken: (Vrealconstant: realconstant);
		end;
	linecount, charoffset, linestart: integer; (* source position *)
	thischar: char; (* computed by GetChar *)

procedure GetChar;
begin
	charoffset := charoffset + 1;
	if eof(input)
	then thischar := chr(0)
	else if eoln(input)
	then begin
		thischar := chr(12);
		readln;
		linecount := linecount + 1;
		linestart := charoffset;
		end
	else read(thischar)
end;
procedure Lexer;
var 
	fracdigits,				(* number of digits seen to the right *)
					(* of a decimal point *)
		identifierlength,
		i,			(* temporary *)
		thisint: integer;					(* constants accumulated *)
	thisreal: real;
	thisidentifier: packed array [1..maxidentifierlength] of char;
	n: identifier;
	thispat: packedArrayType;

procedure MakeInteger;
var i: integerconstant;
begin
	i := Nintegerconstant;
	i^.lexvalue := thisint;
	yylval.Vintegerconstant := i;
	token := IntegerToken;
end;

procedure MakeReal;
var r: realconstant;
	i: integer;
begin
	r := Nrealconstant;
	for i := 1 to fracdigits do thisreal := thisreal / 10;
	r^.lexvalue := thisint + thisreal;
	yylval.Vrealconstant := r;
	token := RealToken;
end;

begin (* Lexer *)
	lasttoken := token;
	for i := 1 to maxidentifierlength do
		thisidentifier[i] := ' ';
	while (thischar = ' ') or (thischar = chr(9))
		or (thischar = chr(11)) or (thischar = chr(12))
	do GetChar; (* skip white space *)
	if thischar = chr(0)
	then token := endoffileToken
	else if thischar = '+'
	then begin
		if lasttoken in [RParenToken, IntegerToken, IdentifierToken, RealToken]
		then token := PlusToken
		else token := UnaryPlusToken;
		GetChar end
	else if thischar = '-'
	then begin
		if lasttoken in [RParenToken, IntegerToken, IdentifierToken, RealToken]
		then token := MinusToken
		else token := UnaryMinusToken;
		GetChar end
	else if thischar = '*'
	then begin token := TimesToken; GetChar end
	else if thischar = '/'
	then begin token := DivideToken; GetChar end
	else if thischar = '('
	then begin token := LParenToken; GetChar end
	else if thischar = ')'
	then begin token := RParenToken; GetChar end
	else if thischar = ';'
	then begin token := SemiToken; GetChar end
	else if thischar = '='
	then begin token := EqualToken; GetChar end
	else if thischar = ':'
	then begin token := ColonToken; GetChar end
	else if thischar = ','
	then begin token := CommaToken; GetChar end
	else if thischar in ['0'..'9']
	then begin (* saw a number *)
		thisint := ord(thischar) - ord('0');
		GetChar;
		while thischar in ['0'..'9']
		do begin
			thisint := thisint * 10 + ord(thischar) - ord('0');
			GetChar;
			end;
		if thischar = '.'
		then begin
			fracdigits := 0;
			thisreal := 0;
			GetChar;
			while thischar in ['0'..'9']
			do begin
				fracdigits := fracdigits + 1;
				thisreal := thisreal * 10 + ord(thischar) - ord('0');
				GetChar;
				end;
			MakeReal;
			end
		else MakeInteger;
		end
	else if thischar in ['a'..'z', 'A'..'Z']
	then begin (* identifier *)
		identifierlength := 1;
		thisidentifier[identifierlength] := thischar;
		GetChar;
		while thischar in ['a'..'z', 'A'..'Z', '0'..'9', '_']
		do begin
			if identifierlength < maxidentifierlength
			then identifierlength := identifierlength + 1;
			thisidentifier[identifierlength] := thischar;
			GetChar;
			end;
		if thisidentifier = 'function'
		then token := functionKeyword
		else if thisidentifier = 'integer'
		then token := integerKeyword
		else if thisidentifier = 'real'
		then token := realKeyword
		else begin
			n := Nidentifier;
			with n^ do begin
				thispat := '  ';
				for i := 1 to identifierlength do
					thispat[i] := thisidentifier[i];
				lextoken := NewString(identifierlength, thispat);
				lexpos := Npos;
				lexpos^.charoffset := charoffset;
				lexpos^.lineoffset := charoffset - linestart;
				lexpos^.linenumber := linecount;
				end;
			yylval.Videntifier := n;
			token := IdentifierToken
			end
		end
	else begin
		writeln(output, 'frontend: illegal character encountered, with',
				' ASCII value ', ord(thischar), '.');
		halt;
		end
end (* Lexer *);
		
procedure InitLexer;
begin
	linecount := 1;
	charoffset := 0;
	linestart := 0;
	token := endoffileToken;
	GetChar;
end;
