/*
 * (c) Universal Access Inc., 1995
 *
 * Basic sql database admin form generator and parser.
 *
 * We require a value for the CGI variable 'db_name', and
 * table operations require the 'db_table_name' CGI variable to be set
 * as well. These can be set either in the QUERY_STRING, the command
 * line, or in a POST method input.
 *
 * This program will look for a CGI variable called "db_opcode" , and if
 * there is one it executes the specified operation, one of:
 *
 * "New Record"
 * "Insert"
 * "Delete"
 * "Update"
 * "Select"
 * "List All"
 *
 * Thus, this same script can be used to generate and execute the
 * forms for modifying the database.
 *    
 */
 
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "list.h"
#include "msql.h"
#include "forms.h"
#include "bprintf.h"

#define caseless_equal(str1,str2)   (!strcasecmp(str1, str2))

/****************************************************************/

char *websql_version = "websql 0.91";

char *db_hostname;
char *db_name;
char *db_table_name;
m_result *db_fields;
char *db_admin_url;
char *db_screen_num;
char *db_opcode;

#define DB_RECORDS_PER_SCREEN 10

POSTED_ITEM **cgi_items;
#define DB_FIND_ALL 1
#define NEW_CMD "New Record"
#define INSERT_CMD "Insert"
#define DELETE_CMD "Delete"
#define UPDATE_CMD "Update"
#define SELECT_CMD "Select"
#define LIST_CMD "List All"

#define DB_FORM_HELP_STRING "You can do a simple SELECT of a record
from this table by doing the following:
<ol>
<li> Click in the checkbox of any fields you
want to match on, and setting a value in that field. 
<li> Click on the <b><tt>Select</tt></b> button.
</ol>
A null value for a checked field indicates you want records which 
match on <i>any</i> value for that field.
<p>
"

int min (int a, int b)
{
  if (a < b) return a;
  else return b;
}

int max (int a, int b)
{
  if (a > b) return a;
  else return b;
}

/*
   Given a table, with its field names, we need:
   
   Insert form
   Delete form
   Update form
   Select form

*/

/* Make a string of n spaces */
char *spaces (int n)
{
  int i;
  char *str;

  str = malloc (n+1);
  for (i = 0; i < n; i++)
    {
      str[i] = ' ';
    }

  str[i] = 0;
  return str;

}

/* Make string safe to use in URL */
char *
sanitize (char *s) {
  BPRINTF_BUFFER *out;
  out = bprintf_create_buffer ();

  for (; *s; s++)
    if (*s == ' ') bprintf (out, "+");
    else if (isalnum(*s)) bprintf (out, "%c", *s);
    else {
      bprintf(out, "%%%2X", *s);
    }
  return out->buffer;
}

/* Get a the value of a single valued variable */
char *
get_form_var (char *name)
{
  char **vals;

  vals = forms_find_tag (cgi_items, name);

  if (vals == NULL) 
    return NULL;
  else
    return (vals[0]);
}

/* Returns a string of comma separated items from field names. */
char *
sql_column_names (m_result *result)
{
  BPRINTF_BUFFER *out;
  int multi = 0;
  int i = 0;
  char *name;

  out = bprintf_create_buffer ();

  msqlFieldSeek (result, 0);

  for (i = 0; i < msqlNumFields(result); i++)
    {
      m_field *field;
      field = msqlFetchField (result);

      if (multi == 0)
	{
	  multi = 1;
	  bprintf (out, "%s", field->name);
	}
      else
	{
	  bprintf (out, ",%s", field->name);
	}
    }

  return out->buffer;
}

/* Search terms is a list of m_field structures. We can extract
   the field name and type from them. */
char *
make_sql_search_string (LIST *search_terms)
{
  BPRINTF_BUFFER *bp;
  NODE *node = NULL;
  m_field *field;
  int multi=0;

  bp = bprintf_create_buffer ();

  field = (m_field *) list_next (search_terms, &node);

  while (field != NULL)
    {
      char *val;
      char *string_term = "%s='%s'";
      char *numeric_term = "%s=%s";
      char *format;

      val = get_form_var (field->name);

      /* We need to decide if we put '''s around the values or not. 
       * Numeric type fields don't get them, but char types do.
       */
      if ((field->type == REAL_TYPE) || 
	  (field->type == INT_TYPE))
	{
	  format = numeric_term;
	}
      else 
	{
	  format = string_term;
	}

      if (val && (strlen (val) > 0))
	{

	  if (multi == 0)
	    {
	      multi = 1;
	    }
	  else
	    {
	      bprintf (bp, " AND ");
	    }

	  bprintf (bp, format, field->name, val);
	}

      field = (m_field *) list_next (search_terms, &node);
    }

  return bp->buffer;
}

/* Do a SELECT query. If FIND_ALL is nonzero, then
 * do a select over all primary keys.
 */

m_result *
get_database_record (int sock, 
		     char *table_name, 
		     m_result *fields,
		     LIST *search_terms)

{
  BPRINTF_BUFFER *query;
  m_result *result;

  query = bprintf_create_buffer ();

  bprintf (query, "SELECT %s FROM %s",
	   sql_column_names (db_fields),
	   table_name);

  if (list_length (search_terms) > 0)
    {
      char *sql_string;

      sql_string = make_sql_search_string (search_terms);

      if (sql_string && !strlen (sql_string) == 0)
	bprintf (query, " WHERE %s", sql_string);
    }

  printf ("<pre>Search query: %s\n</pre>\n", query->buffer);

  if (msqlQuery (sock, query->buffer) < 0)
    {
      fprintf (stdout, "Query failed (%s)\n",msqlErrMsg);
      exit(0);
    }

  result = msqlStoreResult ();
      
  return result;
}



/* Returns the row value of the field named NAME */
char *
get_field_named (m_result *result, m_row row, char *name)
{
  int i;
  m_field *field;

  msqlFieldSeek (result, 0);

  for (i = 0; i < msqlNumFields(result); i++)
	{
	  field = msqlFetchField(result);

	  if (caseless_equal (name, field->name))
	    return (row[i]);
	}

  return (char *) NULL;
}

char *	      
field_type_string (int type)
{
  if (type == INT_TYPE) 
    {
      return "int";
    }
  else if (type == CHAR_TYPE)  
    {
      return "char";      
    }
  else if (type == REAL_TYPE)  
    {
      return "real";      
    }
  else if (type == IDENT_TYPE)  
    {
      return "ident";      
    }
  else if (type == NULL_TYPE)  
    {
      return "null";      
    }
  else
    {
      return "unknown";
    }
}

/* Format the table column names as input forms. If the column name
   has a form-variable set, print its value as the default.
*/
void
format_record_fields (m_result *result, m_row row)
{
  
  int i;
  char *val;

  msqlFieldSeek (db_fields, 0);

  for (i = 0; i < msqlNumFields(db_fields); i++)
    {
      m_field *field;
      field = msqlFetchField(db_fields);

      printf ("<tr><td align=center><input type=checkbox name=\"db_checkbox_%s\">", field->name);

      printf ("<td>%s</td><td align=right>%d</td>",
	      field_type_string (field->type),
	      field->length);

      printf ("<th align=right>%s:</td>", field->name);

      printf ("<td><input name=\"%s\" type=text size=32 maxsize=1024", 	      
	      field->name);


      if (row)
	{
	  val = row[i];
	}
      else
	{
	  val = get_form_var (field->name);
	}

      if (val)
	{
	  printf (" value=\"%s\"", val);	  
	}
      
      printf ("></td></tr>\n");
    }
}


/* Take a result of a query, or NULL. If result is non-null, 
   format one form per entry, else format a blank entry. 
   
   If there are more than DB_RECORDS_PER_SCREEN records selected, show
   only the first N starting at db_record_start.

   */
void
make_db_admin_form (m_result *result)
{
  int i;
  int nrows;
  int screen_num;
  m_row row = NULL;

  /* Where do we start displaying records from? */
  screen_num = atoi (db_screen_num);

  printf ("<h3>Database: <a href=\"%s?db_name=%s\"><tt>%s</tt></a> Table: <tt>%s</tt></h3>\n", 	      
	  db_admin_url,
	  sanitize(db_name),
	  db_name, db_table_name);

  printf (DB_FORM_HELP_STRING);

  /****************************************************************/
  /* Allow button to see previous screen of records, if any */

  if (screen_num > 0)
    {
      printf ("<form method=post action=\"%s?db_name=%s&db_table_name=%s\">", 
	      db_admin_url,
	      sanitize(db_name),
	      sanitize(db_table_name));

      printf ("<input type=hidden name=\"db_screen_num\" value=\"%d\">\n",
	      max (0, (screen_num - 1)));
	
      printf ("<input type=hidden value=\"%s\" name=\"db_opcode\">", db_opcode);
      printf ("<input type=submit value=\"View Previous %d Records\">", DB_RECORDS_PER_SCREEN);

      printf ("</form><hr>");

    }
  /****************************************************************/



  if (result == NULL) 
    {
      nrows = 1;
    }
  else
    {
      nrows = msqlNumRows(result);
    }

  if (result) msqlDataSeek(result, 0);

  for (i = 0; i < screen_num * DB_RECORDS_PER_SCREEN; i++)
    {
      if (result)
	row = msqlFetchRow(result);
    }

  for (i = (screen_num * DB_RECORDS_PER_SCREEN); 
       i < min ( ((screen_num+1) * DB_RECORDS_PER_SCREEN), nrows); 
       i++)
    {
      if (result)
	row = msqlFetchRow(result);

      printf ("<form method=post action=\"%s?db_name=%s&db_table_name=%s\">", 
	      db_admin_url,
	      sanitize(db_name),
	      sanitize(db_table_name));


      if (db_hostname)
	{
	  printf ("<input type=hidden name=\"db_hostname\" value=\"%s\">\n",
		db_hostname);
	}

      printf ("<input type=submit value=\"New Record\" name=\"db_opcode\">
<input type=submit value=\"Select\" name=\"db_opcode\">
<input type=submit value=\"Insert\" name=\"db_opcode\">
<input type=submit value=\"Update\" name=\"db_opcode\">
<input type=submit value=\"Delete\" name=\"db_opcode\">
<input type=submit value=\"List All\" name=\"db_opcode\">");


      printf ("<table border>\n");
      printf ("<tr><th>Use</th><th>Type</th><th>Size</th><th>Name</th><th>Value</th></tr>\n");
      format_record_fields (db_fields, row); 
      printf ("</table>\n");

      printf ("</form><hr>");
    }

  /****************************************************************/
  /* Allow button to see next screen of records, if any */

  if ((screen_num + DB_RECORDS_PER_SCREEN) < nrows)
    {
      printf ("<form method=post action=\"%s?db_name=%s&db_table_name=%s\">", 
	      db_admin_url,
	      sanitize(db_name),
	      sanitize(db_table_name));

      printf ("<input type=hidden name=\"db_screen_num\" value=\"%d\">\n",
	      (screen_num + 1));
	
      printf ("<input type=hidden value=\"%s\" name=\"db_opcode\">", db_opcode);
      printf ("<input type=submit value=\"View Next %d Records\">", DB_RECORDS_PER_SCREEN);

      printf ("</form><hr>");
    }
  /****************************************************************/

}

m_field *
copy_field (m_field *f)
{
  m_field *new;

  new = (m_field *) xmalloc (sizeof (m_field));
  memcpy (new, f, (sizeof (m_field)));
  return (new);
}


/* Returns a list of * m_field which want to be searched on */
LIST *
get_search_terms ()
{
  LIST *terms;
  char buf[1024];
  int i;

  terms = list_make (NULL, NULL);

  msqlFieldSeek (db_fields, 0);

  for (i = 0; i < msqlNumFields(db_fields); i++)
    {
      m_field *field;
      char *val;

      field = msqlFetchField(db_fields);
      
      /* Look for any checkboxes which indicate activated field name
         terms. */

      sprintf (buf, "db_checkbox_%s", field->name);
      val = get_form_var (buf);

      if (val)
	{
	  list_add (terms, (char *) copy_field (field));
	}
    }

  return terms;

}

/* Do a SELECT query with the current form values, using primary key 
 * only, for now.
 *
 *
 */
m_result *
db_do_select (int sock, int find_all)
{
  m_result *result;
  int i, j;
  char *cname;
  char *val;
  m_row row;
  LIST *search_terms;

  /* A list of field names to be used in the search. */
  search_terms = get_search_terms ();

  if (!find_all && (list_length (search_terms) == 0))
    {
      printf ("<h3>You must check at least one field from the table to be active when doing a SELECT operation.</h3>\n");
      return (NULL);
    }
  result = get_database_record (sock,
				db_table_name, 
				db_fields,
				search_terms);

  return result;
}

/* Update a record with the fields which are in the form. */
void
db_do_update (int sock)
{
  BPRINTF_BUFFER *query;
  int i;
  int multi = 0;
  LIST *search_terms;

  /* A list of field names to be used in the update. */
  search_terms = get_search_terms ();

  if (list_length(search_terms) == 0) {

      printf("<h3>You must check at least one field from the table to be active when doing a UPDATE operation.</h3>\n");
      return;
  }

  query = bprintf_create_buffer();

  bprintf(query, "UPDATE %s SET ", db_table_name);

  msqlFieldSeek(db_fields, 0);

  for (i = 0; i < msqlNumFields(db_fields); i++) {
      char *val;
      m_field *field;

      field = msqlFetchField(db_fields);

      val = get_form_var(field->name);

      if (multi == 0) {
	  multi = 1;
      } else {
	  bprintf (query, ", ");
      }

      if (val) {
	      if (field->type == CHAR_TYPE)
		      bprintf (query, "%s='%s'", field->name, val);
	      else
		      bprintf (query, "%s=%s", field->name, val);
      }
  }

  if (list_length(search_terms) > 0)
    {
      char *sql_string;

      sql_string = make_sql_search_string(search_terms);

      if (sql_string && strlen(sql_string) > 0)
	bprintf (query, " WHERE %s", sql_string);
    }

  printf ("<pre>Update statement: %s\n</pre>\n", query->buffer);

  if (msqlQuery (sock, query->buffer) < 0) {
      fprintf (stdout, "Query failed (%s)\n",msqlErrMsg);
      exit(0);
  }
}

/* Inset a record with the fields which are in the form. */

db_do_insert (int sock)
{
  BPRINTF_BUFFER *query;
  int i;
  int multi = 0;

  query = bprintf_create_buffer ();

  bprintf (query, "INSERT INTO %s (", db_table_name);

  /* Print column names */

  msqlFieldSeek (db_fields, 0);
  for (i = 0; i < msqlNumFields(db_fields); i++)
    {
      char *val;
      m_field *field;

      field = msqlFetchField(db_fields);

      if (i != 0)
	{
	  bprintf (query, ", ");
	}

      bprintf (query, "%s", field->name);
    }

  bprintf (query, ") VALUES (");

  msqlFieldSeek (db_fields, 0);

  for (i = 0; i < msqlNumFields(db_fields); i++)
    {
      char *val;
      m_field *field;

      field = msqlFetchField(db_fields);

      val = get_form_var (field->name);

      if (i != 0)
	{
	  bprintf (query, ", ");
	}

      if (val && strlen (val) > 0)
	{
	  if (field->type == CHAR_TYPE)
	    {
	      bprintf (query, "'%s'", val);
	    }
	  else 
	    {
	      /* Numeric types don't get quotes around values. */
	      bprintf (query, "%s", val);
	    }

	}
      else
	{
	  bprintf (query, "NULL");
	}
    }


  bprintf (query, ")");

  printf ("<pre>Search query: %s\n</pre>\n", query->buffer);

  if (msqlQuery (sock, query->buffer) < 0)
    {
      fprintf (stdout, "Query failed (%s)\n",msqlErrMsg);
      exit(0);
    }
}





/* Delete a record with the given matching terms. */
void
db_do_delete (int sock)
{
  BPRINTF_BUFFER *query;
  int i;
  int multi = 0;
  LIST *search_terms;

  query = bprintf_create_buffer ();

  bprintf (query, "DELETE FROM %s", db_table_name);

  /* A list of field names to be used in the search. */
  search_terms = get_search_terms ();

  if (list_length (search_terms) > 0)
    {
      char *sql_string;

      sql_string = make_sql_search_string (search_terms);

      if (sql_string && !strlen (sql_string) == 0)
	{
	  bprintf (query, " WHERE %s", sql_string);
	}
      else
	{
      printf ("<h3>Error: You MUST check at least one field and
 supply a value in order to do a DELETE, otherwise you would delete
 every record in the table.</h3>");
      return;
	}
    }
  else
    {
      printf ("<h3>Error: You MUST check at least one field and
 supply a value in order to do a DELETE, otherwise you would delete
 every record in the table.</h3>");
      return;
    }

  printf ("<pre>Search query: %s\n</pre>\n", query->buffer);

  if (msqlQuery (sock, query->buffer) < 0)
    {
      fprintf (stdout, "Query failed (%s)\n",msqlErrMsg);
      exit(0);
    }
}


/* List the tables in the database, and their fields. */
void
db_describe_db (int sock)
{
  m_result *result, *fresult;
  int i, j;
  char *cname;
  char *val;
  m_row row;


  printf ("<h3>Database <i>%s</i> Tables</h3>\n", db_name);
  printf ("<form method=post action=\"%s\">", db_admin_url);
  if (db_hostname)
    {
      printf ("<input type=hidden name=\"db_hostname\" value=\"%s\">\n",
	      db_hostname);
    }


  result = msqlListTables (sock);

  /* The table names will appear one per row in row[0]. */
  for (j = 0; j < msqlNumRows(result); j++)
    {	  
      char *tableName;
      row = msqlFetchRow(result);
      if (row == NULL) return;

      tableName = row[0];
      printf ("<table border>\n");
      printf ("<tr><th colspan=5>");
      printf ("<a href=\"%s?db_name=%s&db_table_name=%s\">Table Name: <font size=+2><tt>%s</tt></font></a>\n", db_admin_url, sanitize(db_name), sanitize(tableName), tableName);
      printf ("<tr><th>Field<th>Type<th>Size<th>Key<th>Non Null</tr>\n");

      /* Get a list of field names for this table */
      fresult = msqlListFields (sock,tableName);

      /* Print the field names */
      for (i = 0; i < msqlNumFields (fresult); i++)
	{
	  m_field *field;
	  field = msqlFetchField (fresult);

	  if (field == NULL) continue;

	  printf ("<tr>\n");

	  printf ("<td><tt>%s</tt>", field->name);

	  if (field->type == REAL_TYPE)
	    {
	      printf ("<td align=left>real<td>&nbsp");
	    }
	  else if (field->type == INT_TYPE)
	    {
	      printf ("<td align=left>int<td>&nbsp");
	    }
	  else 
	    {
	      printf ("<td>string<td>%d", field->length);
	    }

	  if (IS_PRI_KEY(field->flags))
	    {
	      printf ("<td>PRIMARY_KEY");	      
	    }
	  else
	    printf ("<td>&nbsp");

	  if (IS_NOT_NULL(field->flags))
	    {
	      printf ("<td>NOT_NULL");	      
	    }
	  else
	    printf ("<td>&nbsp");

	  printf ("</tr>\n");
	}
      printf ("</table><p>\n");
    }

  printf ("</form>");

}

void
do_db_operation (char *opcode)      
{
  int sock;
  m_result *result;

  if (!opcode) opcode = "NONE";

  db_hostname = get_form_var ("db_hostname");

  if (db_hostname && strlen (db_hostname) == 0)
    db_hostname = NULL;

  if ((sock = msqlConnect (db_hostname)) < 0)
    {
      fprintf (stdout,"Couldn't connect to engine!\n%s\n\n",
	       msqlErrMsg);
      exit(0);
    }

  if (msqlSelectDB (sock, db_name) < 0)
    {
      fprintf (stdout, "Couldn't select database named %s! %s\n", 
	       db_name, msqlErrMsg);
      exit(0);
    }


  /* Get the field names and primary key for the desired table. */
  if (db_table_name != NULL)
    db_fields = msqlListFields (sock, db_table_name);

  if (db_table_name != NULL)
    {
      if (caseless_equal (opcode, INSERT_CMD))
	{
	  db_do_insert (sock);
	  make_db_admin_form (NULL);
	}
      else if (caseless_equal (opcode, LIST_CMD))
	{
	  result = db_do_select (sock, DB_FIND_ALL); 
	  make_db_admin_form (result);
	}
      else if (caseless_equal (opcode, DELETE_CMD))
	{
	  db_do_delete (sock); 
	  make_db_admin_form (NULL);
	}
      else if (caseless_equal (opcode, UPDATE_CMD))
	{
	  db_do_update (sock);
	  make_db_admin_form (NULL);
	}
      else if (caseless_equal (opcode, SELECT_CMD))
	{
	  result = db_do_select (sock, 0);
	  make_db_admin_form (result);
	}
      else if (caseless_equal (opcode, NEW_CMD))
	{
	  cgi_items = NULL;
	  make_db_admin_form (NULL);
	}
      else
	{
	  make_db_admin_form (NULL);
	}
    }
  else
    {
      /* Default is to list the tables in the database. */
      db_describe_db (sock);
    }
  
  msqlClose(sock);

}

void
main (int argc,char **argv)
{
  db_name = NULL;
  db_table_name = NULL;

  cgi_items = forms_input_data (argc, argv);
  db_table_name = get_form_var ("db_table_name");
  db_name = get_form_var ("db_name");

  db_admin_url = getenv ("SCRIPT_NAME");

  db_screen_num = get_form_var ("db_screen_num");
  if (!db_screen_num || strlen (db_screen_num) == 0)
    {
      db_screen_num = "0";
    }

  /* Look for the database operation */
  db_opcode = get_form_var ("db_opcode");

  if (!db_name || strlen (db_name) == 0)
    {
      printf ("Content-type: text/html\n\n");
      printf ("<html><head><title>MSQL Browser</title></head><body>\n");
      printf ("<form method=post action=\"%s\">", db_admin_url);
      printf ("<h3>MSQL Database Browser</h3>");
      printf ("Enter a database name:<br> <input type=text size=32 name=\"db_name\"><p>\n");
      printf ("Enter hostname of MSQL server:<br> <input type=text size=32 name=\"db_hostname\"><p>\n");
      printf ("<input type=submit> <input type=reset>");
      printf ("</form></body></html>");
      exit (0);
    }

  { 
    FILE *foo;
    foo = fopen ("/tmp/foobar", "w");
    fprintf (foo, "db_name = %s\n", db_name);
    fprintf (foo, "db_hostname = %s\n", db_hostname);
    fclose (foo);
  }
    


  printf ("Content-type: text/html\n\n");
  printf ("<html><head><title>MSQL Operations For %s %s</title></head><body>\n", 
	  db_name, db_table_name);


  do_db_operation (db_opcode);

  printf ("<hr>
<address><a href=\"http://www.ua.com\">&copy; Universal Access Inc.</a>, 1995,
<a href=\"mailto:hqm@ua.com\">Henry Minsky (hqm@ua.com)</a></address>
</body></html>");

  exit(0);
}


