/*
 * Filename: test.c
 * Project:  Sequence Generation System
 * Module:   Examples
 *
 * Function: This program uses many API functions.  It serves as a quality
 *           assurance tool as well as a benchmarking tool for comparing
 *           the speed of different hardware.
 *
 * Author:   Pascal Forget (pascal@wsc.com)
 * Date:     October 1995
 *
 * Copyright (C)1995 Pascal Forget
 *
 * You may freely distribute, copy or modify this program.
 */

#include "../../api/sgs.h"

#define DEFAULT_NB_SEQUENCES 5

int
createTestSequences(int nbSequences)
{
    double diff;
    struct timezone tzp;
    struct timeval start, end;
    char sequenceName[4];
    int i;
    
    fprintf(stdout, "Test: about to create %d named sequences.\n",
	    nbSequences);

    gettimeofday(&start, &tzp);
    
    for (i=0; i<nbSequences; i++) {
	sprintf(sequenceName, "S%i",i);
	fprintf(stdout, "About to create sequence: %s\n", sequenceName);
	sgsCreateSequence(sequenceName);
    }

    gettimeofday(&end, &tzp);
    diff=(end.tv_sec - start.tv_sec)+(end.tv_usec - start.tv_usec)/1000000.0;
    
    fprintf(stdout, "%d sequences created in %3.2f seconds.\n\n",
	    nbSequences, diff);
    
    return 1; 
}

int
readSequenceValues(int nbSequences, int showMessages)
{
    double diff;
    struct timezone tzp;
    struct timeval start, end;
    SGAnswer answer;
    char sequenceName[4];    
    int i, j;
    
    fprintf(stdout, "About to get 25 values for each of the %d sequences.\n",
	    nbSequences);
    
    gettimeofday(&start, &tzp);
    
    for(i=0; i<nbSequences; i++) {
	sprintf(sequenceName, "S%i",i);
	
	for (j=0; j<25; j++) {
	    answer = sgsNextValue(sequenceName);
	    
	    if (answer.returnStatus == 0) {
		fprintf(stdout, "Error getting the sequence's next value.\n");
		break;
	    } else {
		if (showMessages) {
		    fprintf(stderr, "Value for sequence %s: %d.\n",
			    sequenceName, answer.value);
		}
	    }
	}
    }

    gettimeofday(&end, &tzp);
    diff=(end.tv_sec - start.tv_sec)+(end.tv_usec - start.tv_usec)/1000000.0;

    fprintf(stdout, "%d sequence values read in %2.2f seconds. "
	    "(%2.1f values/sec)\n\n",
	    25 * nbSequences, diff, 25 * nbSequences / diff);

    return 1;
}

int
setSequenceValues(int nbSequences)
{
    double diff;
    struct timezone tzp;
    struct timeval start, end;    
    char sequenceName[4];    
    int i;
    
    fprintf(stdout, "Setting the cache and increment for the sequences.\n");

    gettimeofday(&start, &tzp);

    for (i=0; i<nbSequences; i++) {
	sprintf(sequenceName, "S%i",i);
	sgsSetCache(sequenceName, 10);
	sgsSetIncrement(sequenceName, 2);
    }

    gettimeofday(&end, &tzp);
    diff=(end.tv_sec - start.tv_sec)+(end.tv_usec - start.tv_usec)/1000000.0;

    fprintf(stdout, "%d sequences were modified in %3.2f seconds.\n\n",
	    nbSequences, diff);

    return 1;
}

int
isNumber(char *string)
{
    int i, len;
    char *ptr;

    ptr = string;
    len = strlen(string);

    for (i=0; i<len; i++) {
	if (!isdigit(ptr[i])) {
	    return 0;
	}
    }
    return 1;
}

int
main(int argc, char *argv[], char *env[])
{
    int nbSequences;
    SGAnswer cache, increment;

    
    fprintf(stdout, "**  This is the SGS test program. It is aimed at "
	    "verifying the behavior of many \n**  API functions on your "
	    "system.\n");

    if ((argc == 2) && (isNumber(argv[1]))) {
	nbSequences = atoi(argv[1]);
    } else {
	nbSequences = DEFAULT_NB_SEQUENCES;
    }
    
    createTestSequences(nbSequences);

    readSequenceValues(nbSequences, 1);

    /*
     * We repeat the step, but this time without printing to stdout
     * in order to see the "real" speed of the sgs daemon
     */
    readSequenceValues(nbSequences, 0);

    setSequenceValues(nbSequences);

    readSequenceValues(nbSequences, 1);    

    cache = sgsCache("S3");
    increment = sgsIncrement("S3");
    
    fprintf(stdout, "Sequence 3 (S3) cache:%i increment:%i\n\n ** THE END.\n",
	    cache.value, increment.value);

    return 0;
}


