/*
 * Filename:	main.c
 * Project:	Sequence Generation System
 * Module:	Daemon
 * Function:	Please refer to the header file for a full description
 *
 * Author:	Pascal Forget
 * Date:	October 1995
 *
 * Copyright (C) 1995 Pascal Forget
 * All Rights reserved.
 *
 * This software falls under the GNU General Public Licence, as published
 * by FSF, the Free Software Foundation.  You should have received a copy 
 * of the GNU General Public License along with this program; if not,
 * write to the Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
 * MA 02139, USA.
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include "main.h"

#ifndef DEBUG
#define DEBUG 1
#endif

extern int errno;

static int sock_fd;

int intSize;

#define DEFS_FILENAME "/data/defs"
#define VALUES_FILENAME "/data/values"

#if DEBUG
#define MAX_ENTRIES_FOR_SEQUENTIAL_SEARCH 55
#else
#define MAX_ENTRIES_FOR_SEQUENTIAL_SEARCH 100
#endif

#define VERSION "v1.0.0"
#define STREAM_VERSION 0

Definition *definitions;
DefsHeader header;
char *defsFilename;
char *valuesFilename;

int
sgsInit(void)
{
    struct sockaddr_in sin;

    intSize = sizeof(int);

    /*
     * Get a socket to work with.  This socket will be in
     *  the Internet domain, and will be a stream socket.
     */
    if ((sock_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	perror("Sequence generator daemon: socket");
	return 0;
    }

    /* Create the address that we will be binding to.
     * We use port DM_PORT but put it into network
     * byte order.  Also, we use bcopy to copy the
     * network number.
     */
    bzero((char *) &sin, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = htonl(INADDR_ANY);
    sin.sin_port = htons(sgsGetPortNumber());

    /*
     * Try to bind the address to the socket.
     */
    if (bind(sock_fd, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
	perror("Sequence generator daemon: bind");
	return 0;
    }

    return 1;
}

/*
 * _sgsGetSetting returns either the number of entries to be cached
 * for the specified sequence, or the increment by which the value
 * grows after each call to sgsGetValue()
 */

SGAnswer
sgsGetSetting(SGRequest req)
{
    SGAnswer answer;
    Location loc;

    assert(req.sequenceName);

    loc = sgsLocate(req.sequenceName);

    if (!loc.exactMatch) {
        answer.returnStatus = 0; /* Failure */
    } else {
        answer.returnStatus = 1;

	if (req.rtype == SG_cache) {
	    answer.value = (unsigned int)definitions[loc.index].cache;
	} else {
	    answer.value = (unsigned int)definitions[loc.index].increment;
	}
    }
    return answer;
}

/*
 *  _sgsLoadNextValuesFromFile updates the values file with the value
 *  stored in the sequence definition cache.
 */

int
sgsLoadNextValuesFromFile(int index)
{
    int fd;
    int newValue = definitions[index].value;
    
    if ((fd = open(valuesFilename, O_WRONLY, 0666)) < 0) {
	perror(valuesFilename);
	sgsQuit(1);
    }
    
    lseek(fd, (long)(index*intSize), SEEK_SET);
    write(fd, &newValue, intSize);
    close(fd);

    return 1;
}

/*
 * _sgsGetValue is called whenever the client requests the next
 * sequence number
 */
SGAnswer
sgsGetValue(SGRequest req)
{
    Location loc;
    SGAnswer answer;

    assert(req.sequenceName);

    loc = sgsLocate(req.sequenceName);

    if (!loc.exactMatch) {
        answer.value = 0;
	answer.returnStatus = 0;
    } else {
        answer.value = definitions[loc.index].value;
	answer.returnStatus = 1;

	if (req.rtype == SG_nextValue) {
	    definitions[loc.index].value += definitions[loc.index].increment;

	    if (definitions[loc.index].leftInCache) {
	        definitions[loc.index].leftInCache--;
	    } else {
	        definitions[loc.index].leftInCache =
				definitions[loc.index].cache;
	        sgsLoadNextValuesFromFile(loc.index);
	    }
        }
    }
    return answer;
}

SGAnswer
sgsSetValue(SGRequest req)
{
    SGAnswer answer;
    Location loc;

    assert(req.sequenceName);

    loc = sgsLocate(req.sequenceName);

    if (!loc.exactMatch) {
        answer.value = 0;
	answer.returnStatus = 0;
    } else {
        definitions[loc.index].value = req.arg;
	definitions[loc.index].leftInCache = definitions[loc.index].cache;
        sgsLoadNextValuesFromFile(loc.index);
        answer.value = req.arg;
	answer.returnStatus = 1;
    }
    return answer;
}

SGAnswer
sgsSetCache(SGRequest req)
{
    SGAnswer answer;
    Location loc;

    assert(req.sequenceName);

    loc = sgsLocate(req.sequenceName);

    if (!loc.exactMatch) {
        answer.value = 0;
        answer.returnStatus = 0;
    } else {
        definitions[loc.index].cache = (unsigned char)req.arg;
        answer.value = 0;
	answer.returnStatus = 1;
    }
    return answer;
}

SGAnswer
sgsSetIncrement(SGRequest req)
{
    SGAnswer answer;
    Location loc;

    assert(req.sequenceName);

    loc = sgsLocate(req.sequenceName);

    if (!loc.exactMatch) {
        answer.value = 0;
        answer.returnStatus = 0;
    } else {
        definitions[loc.index].increment = req.arg;
        answer.value = 0;
	answer.returnStatus = 1;
    }
    return answer;
}

SGAnswer
sgsProcessRequest(int stream)
{
    SGAnswer answer;
    SGASCIIRequest asciiReq;
    SGRequest req;
    
    read(stream, &asciiReq, sizeof(SGASCIIRequest));
    req = sgsConvertRequestFromAscii(asciiReq);
    
    switch(req.rtype) {
      case SG_createSequence: return sgsCreateSequence(req);
      case SG_dropSequence: return sgsDropSequence(req);
      case SG_currentValue: return sgsGetValue(req);
      case SG_nextValue: return sgsGetValue(req);
      case SG_cache: return sgsGetSetting(req);
      case SG_increment: return sgsGetSetting(req);
      case SG_setCache: return sgsSetCache(req);
      case SG_setIncrement: return sgsSetIncrement(req);
      case SG_setValue: return sgsSetValue(req);
      default: sgsQuit(1); /* Error */
    }
    
    answer.returnStatus = 0;
    return answer;
}

SGAnswer
sgsCreateSequence(SGRequest req)
{
    SGAnswer answer;
    Definition newDef;

    Location loc;

    loc = sgsLocate(req.sequenceName);

    if (loc.exactMatch) {
        fprintf(stderr, "Sequence generator: cannot create sequence %s.  "
	        "Name is already in use.\n", req.sequenceName);
	answer.value = 0;
	answer.returnStatus = 0; /* Failure */
        return answer;
    } else {
      strncpy(newDef.name, req.sequenceName, SEQ_NAME_LEN);
      newDef.name[SEQ_NAME_LEN-1] = '\0';
      newDef.value = 1;
      newDef.cache = 0;
      newDef.increment = 1;
      newDef.leftInCache = 0;

      if (loc.index == -1) {
	  loc.index = 0;
      }
      
      sgsInsertDefinition(newDef, loc);
      sgsWriteValues();
    }
    answer.value = 0;
    answer.returnStatus = 1; /* Success */
    return answer;
}

SGAnswer
sgsDropSequence(SGRequest req)
{
    SGAnswer answer;
    Location loc;
    int i;

    loc = sgsLocate(req.sequenceName);

    if (!loc.exactMatch) {
        fprintf(stderr, "Sequence generator: Cannot drop sequence %s.  "
		"Invalid sequence name.\n", req.sequenceName);
	answer.returnStatus = 0; 
	return answer;
    } else {  /* Note that we do not realloc to decrease the array size */
      for (i=loc.index; i<header.numberOfEntries-1; i++) {
	  definitions[i] = definitions[i+1];
      }
      header.numberOfEntries--;
      sgsWriteValues();
    }
    answer.returnStatus = 1; /* Success */
    return answer;
}

Location
sgsLocate(const char *name)
{
    return sgsLocateInSubset(0, header.numberOfEntries-1, name);
}

Location
sgsLocateSequentially(int min, int max, const char *name)
{
    Location loc;
    int i;
    int comparisonResult;

    for (i=min; i<=max; i++) {
        comparisonResult = strcmp(definitions[i].name, name);

	if (comparisonResult == 0) {
	    loc.index = i;
	    loc.exactMatch = 1; /* YES */
	    return loc;
	} else if (comparisonResult > 0) {
	    loc.index = i;
	    loc.exactMatch = 0; /* NO */
	}
    }

    /* Not Found */

    loc.index = max;
    loc.exactMatch = 0;
    return loc;
}

Location
sgsLocateInSubset(int min, int max, const char *name)
{
    Location loc;
    int centerPosition = (max - min) / 2;
    int comparisonResult;

    if (min > max) {
        loc.index = max;
        loc.exactMatch = 0;
        return loc;
    } else if ((max-min) < MAX_ENTRIES_FOR_SEQUENTIAL_SEARCH) {
        return sgsLocateSequentially(min, max, name);
    }

    comparisonResult = strcmp(definitions[centerPosition].name, name);

    if (comparisonResult == 0) {
        loc.index = centerPosition;
        loc.exactMatch = 1;
        return loc;
    } else if (comparisonResult > 0) {
        return sgsLocateInSubset(min, centerPosition, name);
    } else {
        return sgsLocateInSubset(centerPosition, max, name);
    }
}

int
sgsInsertDefinition(Definition newDef, Location loc)
{
    int i, size;

    header.numberOfEntries++;

    size = header.numberOfEntries * sizeof(Definition);
    
    if (!definitions) {
	definitions = malloc(size);
    } else {
	definitions = realloc(definitions, size);
    }

    for (i=header.numberOfEntries-1; i>loc.index; i--){
	definitions[i] = definitions[i-1];
    }

    definitions[loc.index] = newDef;

    sgsWriteDefinitions();
    sgsWriteValues();
    return 1;
}

int
sgsListenOnTheSocket(void)
{
    SGASCIIAnswer answer;
    int ns, len, size;
    struct sockaddr_in client;

    size = sizeof(client);
    
    /*
     * Listen on the socket.
     */
    if (listen(sock_fd, 5) < 0) {
	perror("Sequence generator daemon: listen");
	return 0;
    }

    for ( ; ; ) {
	/*
	 * Accept connections.
	 */

	len = size; 
	
	if ((ns = accept(sock_fd, (struct sockaddr *) &client, &len)) < 0) {
	    perror("Sequence generator daemon (accept)");
	    return 0;
	}
    
	answer = sgsConvertAnswerToAscii(sgsProcessRequest(ns));

	write(ns, &answer, sizeof(SGASCIIAnswer));

	close(ns);
    }
}

int
sgsReadSequenceDefinitions(void)
{
    int fd;
    
    /* Init the header */
    header.valuesStreamVersion = 0;
    header.numberOfEntries = 0;

    /* Open the file if it exists */
    fd = open(defsFilename, O_RDONLY);
	
    /* Read in the header */

    if (fd == -1) {
	header.valuesStreamVersion = STREAM_VERSION;
	header.numberOfEntries = 0;
    } else {
	read(fd, &header, sizeof(DefsHeader));

	if (header.numberOfEntries > 0) {
	    switch(header.valuesStreamVersion) {
	      case 0:
		sgsReadDefinitionsVersion0(fd);
		break;
	      default:
		fprintf(stderr, "Sequence generator daemon: invalid stream"
			" version, exiting.\n");
		sgsQuit(1);
	    }
	}
    }
    close(fd);
    return 1;
}
   
int
sgsReadDefinitionsVersion0(int filedesc)
{
    int bufSize;
    int i;
    
    bufSize = header.numberOfEntries * sizeof(Definition);
    definitions = malloc(bufSize);

    for (i=0; i<header.numberOfEntries; i++) {
	read(filedesc, &definitions[i], sizeof(Definition));
    }

    return sgsReadValues();
}

int
sgsReadValues(void)
{
    int fd;
    int i;

    if ((fd = open(valuesFilename, O_RDONLY|O_CREAT, 0666)) < 0) {
	perror(valuesFilename);
	sgsQuit(1);
    }

    for (i=0; i<header.numberOfEntries; i++) {
	read(fd, &definitions[i].value, intSize);
	definitions[i].leftInCache = definitions[i].cache;
    }

    close(fd);
    
    return 1; /* success */
}

void
main(int argc, char *argv[])
{
  int fd;

  intSize = sizeof(int);
  
  if ((argc == 1) || (argc == 2)) {
      switch(fork()) {
	case -1: /* error */
	  fprintf(stderr, "The sequence generator daemon could not fork!\n");
	  exit(-1);
	case 0:  /* child process */
	  /*
	   * Close the standard file descriptors (stdin, stdout, stderr)
	   */
	  (void) close(0);
	  (void) close(1);
	  (void) close(2);
	  
	  /*
	   *	open() /dev/tty (could be /dev/console)
	   */
	  fd = open("/dev/console", O_RDWR);
	  
	  /*
	   *	dup() it twice...so that fd 0, 1, and 2
	   *	are all the control terminal.
	   */
	  (void) dup(dup(fd));
	  
	  if (!sgsInit()) {
	      sgsQuit(1);
	  }
	  
	  if (argc == 2) {
	      sgsCreateFilenames(argv[1]);
	  } else {
              sgsCreateFilenames(SGS_DEFAULT_PATH);
	  }
	  
	  sgsReadSequenceDefinitions();
	  fprintf(stderr, "Sequence daemon %s started.\n", VERSION);	  
	  sgsListenOnTheSocket();
	  
	default: /* parent process */
	  fprintf(stderr, "SGS startup process completed.\n");
	  sgsQuit(0);
      }
  } else {
      if (!sgsInit()) {
	  sgsQuit(1);
      }
      
      sgsCreateFilenames(SGS_DEFAULT_PATH);
      sgsReadSequenceDefinitions();
      fprintf(stderr, "SGS process running in shell (debug mode).\n");
      sgsListenOnTheSocket();	
  }
  sgsQuit(0);
}

int
sgsCreateFilenames(const char * path)
{
  
  defsFilename = malloc(strlen(path) + strlen(DEFS_FILENAME) + 1);
  strcpy(defsFilename, path);
  strcat(defsFilename, DEFS_FILENAME);

  valuesFilename = malloc(strlen(path) + strlen(VALUES_FILENAME) + 1);
  strcpy(valuesFilename, path);  
  strcat(valuesFilename, VALUES_FILENAME);

  return 1; /* Success */
}

void
sgsQuit(int status)
{
    exit(status);
}

int
sgsWriteDefinitions(void)
{
    int fd;

    header.valuesStreamVersion = STREAM_VERSION;

    fd = creat(defsFilename, 0666);

    lseek(fd, 0L, 0);
    
    /* Write the header */
    write(fd, &header, sizeof(DefsHeader));

    /* Write all definitions */
    write(fd, &definitions[0], header.numberOfEntries * sizeof(Definition));
    
    close(fd);
    return 1;
}    

int
sgsWriteValues(void)
{
    int i, fd;
    
    if ((fd = open(valuesFilename, O_RDWR|O_CREAT, 0666)) < 0) {
	perror(valuesFilename);
	sgsQuit(1);
    }

    /* Write the values  */
    for (i=0; i<header.numberOfEntries; i++) {
	write(fd, &definitions[i].value, intSize);
    }
    
    close(fd);
    return 1; /* Success */
}
