/*
 * Filename: sgs.c
 * Project:  Sequence Generation System
 * Module:   API
 *
 * Function: Please refer to the header file for a description
 *
 * Author:   Pascal Forget
 * Date:     October 1995
 *
 * Copyright (C) 1995 Pascal Forget <pascal@wsc.com>
 *
 * This software falls under the GNU General Public License.
 * Please read the COPYING file for all the details.
 */

#include "sgs.h"
#include "sgsPrivate.h"

char *_sgsHostname;

/* Private functions */

int
sgsConnectToDaemon(const char *hostname)
{
    int sockfd;
    struct sockaddr_in serv_addr;
    char host[64];
    struct hostent *hp = NULL;
    int anyhost = 0;

    if (!hostname) {
	gethostname(host, sizeof(host));
    } else if (!(anyhost = (!strcmp(hostname, "*")))) {
	sprintf(host, "%s", hostname);
    }
    
    /* look up the network address of our host */
    if ((!anyhost) && ((hp = gethostbyname(host)) == NULL)) {
	fprintf(stderr, "%s: unknown host.\n", hostname);
    }
    
    bzero((char *) &serv_addr, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_port = htons(sgsGetPortNumber());
    
    if (!anyhost) {
	bcopy(hp->h_addr, &serv_addr.sin_addr, hp->h_length);
    }
    
    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	fprintf(stderr, "Sequence generator error: could not find the "
		"sequence daemon.\n");
    }
    
    if (connect(sockfd, (struct sockaddr *)&serv_addr,
		sizeof(serv_addr)) < 0) {
	fprintf(stderr, "Sequence generator error: could not "
		"connect to the sequence daemon.\n");
	return -1;
    }
    return sockfd;
}

int
sgsStartServerLocally(void)
{
    system(DEFAULT_DM_FILENAME);
    return 1;
}

SGAnswer
sgsSendRequest(SGRequest req)
{
    int  socketfd;
    SGASCIIAnswer answer;
    SGAnswer returnVal;
    SGASCIIRequest ascReq;
    
    if ((socketfd = sgsConnectToDaemon(_sgsHostname)) < 0) {
	returnVal.returnStatus = 0;
	return returnVal;
    }

    ascReq = sgsConvertRequestToAscii(req);
    
    send(socketfd,  (char *)&ascReq, sizeof(SGASCIIRequest), 0);
    read(socketfd, &answer, sizeof(SGASCIIAnswer));
    
    close(socketfd);

    returnVal = sgsConvertAnswerFromAscii(answer);
    return returnVal;
}

int
sgsCreateSequence(const char *name)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: invalid argument "
		"to createSequence() (NULL).\n");
	return 0;
    } else {    
	req.rtype = SG_createSequence;
	strcpy(req.sequenceName, name);
    }

    answer = sgsSendRequest(req);

    return (int)answer.returnStatus;
}

int
sgsDropSequence(const char *name)
{
    SGRequest req;
    SGAnswer answer;

    if (!name) {
	fprintf(stderr, "Sequence generator error: invalid argument "
		"to dropSequence() (NULL).\n");
	return 0;
    } else {    
	req.rtype = SG_dropSequence;
	strcpy(req.sequenceName, name);
    }

    answer = sgsSendRequest(req);

    return (int)answer.returnStatus;
}

SGAnswer
sgsCurrentValue(const char *name)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: invalid argument "
		"to currentValue() (NULL).\n");
	answer.returnStatus = 0;
	return answer;
    } else {    
	req.rtype = SG_currentValue;
	strcpy(req.sequenceName, name);
    }

    return sgsSendRequest(req);
}

SGAnswer
sgsNextValue(const char *name)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: invalid argument "
		"to nextValue() (NULL).\n");
	answer.returnStatus = 0;
	return answer;
    } else { 
	req.rtype = SG_nextValue;
	strcpy(req.sequenceName, name);
    }

    return sgsSendRequest(req);
}

SGAnswer
sgsCache(const char *name)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: invalid argument "
		"to cache() (NULL).\n");
	answer.returnStatus = 0;
	return answer;
    } else { 
	req.rtype = SG_cache;
	strcpy(req.sequenceName, name);
    }

    return sgsSendRequest(req);
}

SGAnswer
sgsIncrement(const char *name)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: invalid argument "
		"to increment() (NULL).\n");
	answer.returnStatus = 0;
	return answer;
    } else { 
	req.rtype = SG_increment;
	strcpy(req.sequenceName, name);
    }

    return sgsSendRequest(req);
}

int
sgsSetCache(const char *name, unsigned char newCache)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: sequence name (NULL).\n ");
	return 0;
    } else { 
	req.rtype = SG_setCache;
	strcpy(req.sequenceName, name);
	req.arg = newCache;
    }

    answer = sgsSendRequest(req);
    return answer.returnStatus;
}

int
sgsSetHostname(const char *name)
{
    if ((_sgsHostname != NULL) &&
	(name != NULL) &&
	(!strcmp(_sgsHostname,name)))
    {
	return 1; /* Nothing to do */
    }

    
    if (_sgsHostname != (char *)NULL) {
	free(_sgsHostname);
    }
    
    if (name != (char *)NULL) {
	_sgsHostname = (char *)malloc(strlen(name) + 1);
	strcpy(_sgsHostname, name);
    } else {
	_sgsHostname = (char *)malloc(2);
	strcpy(_sgsHostname, "*");
    }
    return 1;
}

int
sgsSetIncrement(const char *name, unsigned int newIncrement)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: sequence name (NULL).\n ");
	return 0;
    } else { 
	req.rtype = SG_setIncrement;
	strcpy(req.sequenceName, name);
	req.arg = newIncrement;
    }

    answer = sgsSendRequest(req);
    return answer.returnStatus;
}

int
sgsSetValue(const char *name, unsigned int newValue)
{
    SGRequest req;
    SGAnswer answer;
    
    if (!name) {
	fprintf(stderr, "Sequence generator error: sequence name (NULL).\n ");
	return 0;
    } else { 
	req.rtype = SG_setValue;
	strcpy(req.sequenceName, name);
	req.arg = newValue;
    }

    answer = sgsSendRequest(req);
    return answer.returnStatus;
}
