/******************************************************************************
	  CCCC	    A	  BBBBB	  L	EEEEE  N     N	EEEEE	TTTTTTT
	 C    C    A A	  B    B  L	E      NN    N	E	   T
	C	  A   A	  B    B  L	E      N N   N	E	   T
	C	 AAAAAAA  BBBBB	  L	EEEEE  N  N  N	EEEEE	   T
	C        A     A  B    B  L	E      N   N N	E	   T
	 C    C  A     A  B    B  L	E      N    NN  E 	   T
	  CCCC	 A     A  BBBBB	  LLLL	EEEEE  N     N	EEEEE	   T
*******************************************************************************

CableNet Source Module:
	Copyright 1994-1995 (C) CableNet Limited. All Rights Reserved.

    Module Name:		$RCSfile: DfltsPaths.c,v $
    Module Description:	Defaults Handling library routines

Description:

    Defaults are stored in sets (directories) and groups (files).
    A default is stored in a group (of defaults) file in the format
    <default identifier>=<value>

    To retreive a default value you call a function passing ;
        - the set name   (or use the default/current set)
	- a group name   (or use the default/current group)
	- an identifier   which identifies the default value
	- a default return value    - returned if nothing is found

    The value returned is either a string or int depending on the
    function call used to access the default

    There are three levels of defaults in ascending order of precedence;
        - global (i.e. system defaults)
	- local  (i.e. local to the running program)
	- user   (specific to the current user)

    Therefore user defaults override all others and are ordinarily
    found in a tree below $(HOME)/defaults

    Local defaults are normally found in  /usr/local/defaults

    global defaults are normally found in /var/spool/defaults

Edit History:

	$Log: DfltsPaths.c,v $
 * Revision 1.5  1995/12/21  18:04:51  V
 * latest revision
 *
 * Revision 1.4  1995/08/29  09:58:52  V
 * many changes
 *
 * Revision 1.3  1995/02/27  17:40:04  V
 * cleaned up unused warnings
 *
 * Revision 1.2  1995/02/09  19:01:56  V
 * don't make the sets when setting the current set and group
 *
 * Revision 1.1  1995/02/03  18:27:53  V
 * Initial revision
 *


*/

/* RCS identification string (for "what" program) */
static char moduleRCSid[] = "@(#) $Id: DfltsPaths.c,v 1.5 1995/12/21 18:04:51 V Exp $";

/* must come first header files */
#include "V.h"			/* virtual header file */
#include "Vport.h"		/* port header file */


/*
 *   system header files
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <memory.h>
#include <unistd.h>

#ifdef Vsunos40
#include "Vansi.h"
#endif

/*
 * third party headers ie, X, informix, ctree
 */

/*
 * project header files
 */
#include "Vlib.h"

/*
 *  local application/library header files
 */

#include "dfltsP.h"

/* the global default base for all sets */
static char *defaults_base = "/var/spool/defaults";
static char *local_defaults_base = "/usr/local/defaults";

/*
 *  global or normal set/group paths
 global_base_set is an absolute path ie '/var/spool/defaults'
 global_set is an absolute path  ie '${base_set}/smallads/recover'
 global_group is a path relative to current set ie '${curr_set}/config'
 global_path is an absolute value to the current group
 ie /var/spool/defaults/smallads/recover/config
 */
static char *global_set = NULL;
static char *global_group = NULL;
static char *global_path = NULL;
static char *global_base_set = NULL;

/*
 * local set/group paths
 these can be used to provide an alternative set which can
 overide the values in the global defaults database
 typically you would first of all look for a value in the local
 set, and if you couldn't find it then look in the global set
 */
static char *local_set = NULL;
static char *local_group = NULL;
static char *local_base_set = NULL;

/*
 *   user set/group paths
 */
static char *user_set = NULL;
static char *user_group = NULL;
static char *user_path = NULL;
static char *user_base_set = NULL;




/*++ ***************  function  DfltMakeSet *************** */
/*
   Purpose:        ensure that a set (directory) exists, if it
   doesn't then make it (and recursively make parents)

   Globals changed: none

   Parameters:
   char     *path      == path to directory to ensure exists

   Return Values: int
   0    success
   < 0  error

   Dependencies:
   EndOfString(), CopyString()

 */
/************************************************************************ ++**/

#ifdef VANSI_1
int
DfltMakeSet (char *setpath)
#else
int
DfltMakeSet (setpath)
     char *setpath;
#endif
{
  char *set;
  char sysbuf[MAXPATHLEN + 128];
  char *t;
  int sysval = 0;
  struct stat st;

/*-- verify arguments */
  if (!setpath)
    return -1;

/*-- make a working copy */
  set = CopyString (setpath);

/*-- find the end of the path and remove any trailing '/' */
  t = EndOfString (set);

  /* remove trailing slashes */
  while (*t == '/')
    *t-- = '\0';

/*-- try to open this set */
  if (stat (set, &st) < 0)
    {
/*-- no joy */
      if ((t = strrchr (set, '/')) != NULL)
	{

	  *t = '\0';

/*-- recursivley make previous sets */
	  if (DfltMakeSet (set) < 0)
	    {
	      fprintf (stderr, "%s couldn't make set %s", ProgramName (), set);
	      Free (set);
	      return -1;
	    }
	  *t = '/';
/*-- now make this set */
	  sprintf (sysbuf, "mkdir %s", set);
	  sysval = system (sysbuf);
	  if ((sysval >> 8) != 0)
	    {
	      fprintf (stderr, "%s system error with %s\n", ProgramName (),
		       sysbuf);
	      Free (set);
	      return -1;
	    }
	}
    }
  else
/*-- we opened this set ok, now check that it is a directory */
  if ((st.st_mode & S_IFMT) != S_IFDIR)
    {
/*-- not a directory, return error (-1) */
      Free (set);
      return -1;
    }
/*-- everything ok, return success (0) */
  Free (set);

  return 0;
}


/*++ **************** function DfltsInit() **********************

   Purpose:     Initialise the defaults system

   Globals changed:    base_set, local_base_set, user_base_set

   **** ++ */
#ifdef VANSI_1
int
DfltsInit (void)
#else
int
DfltsInit (void)
#endif
{
  char buf[MAXPATHLEN];
  char *base, *home;

/*-- get the standard defaults base */
  if ((base = GlobalDefaults ()) == NULL)
    global_base_set = CopyString (defaults_base);
  else
    global_base_set = CopyString (base);

/*-- get the local defaults base */
  if ((base = LocalDefaults ()) == NULL)
    local_base_set = CopyString (local_defaults_base);
  else
    local_base_set = CopyString (base);

/*-- get the standard defaults base */
  if ((base = UserDefaults ()) == NULL)
    {
      home = getenv ("HOME");

      if (home == NULL)
	  return -1;

      sprintf (buf, "%s/defaults", home);

      user_base_set = CopyString (buf);
    }
  else
    user_base_set = CopyString (base);

  return 0;
}

/*++ ****************  function  DfltSetBaseSet ******************** */
/*
   Purpose:     set the base set to a new path and ensure it exists

   Globals changed:  base_set or local_base_set or user_base_set

   Parameters:
   char     *path      ==  path to use as new base set
   int      dflt_set   ==  indicates which set to use

   Return Values: int
   0    success
   < 0  error

   Dependencies:
   CopyString(), EndOfString()

 */
/************************************************************************ ++**/
#ifdef VANSI_1
int
DfltSetBaseSet (DfltsInfo * di)
#else
int
DfltSetBaseSet (di)
     DfltsInfo *di;
#endif
{
  char buf[MAXPATHLEN];
  char *base;

  if (!di)
    return -1;

  switch (di->base)
    {
    case GLOBAL_DFLT_SET:
      if ((base = GlobalDefaults ()) == NULL)
	base = defaults_base;

/*-- free any previous value of base set */
      if (global_base_set)
	Free (global_base_set);

/*-- generate new base set absolute path */
      if (di->set)
	sprintf (buf, "%s/%s", base, di->set);
      else
	sprintf (buf, "%s", base);

      global_base_set = CopyString (buf);
      break;
    case LOCAL_DFLT_SET:
      if ((base = LocalDefaults ()) == NULL)
	base = local_defaults_base;

/*-- free any previous value of base set */
      if (local_base_set)
	Free (local_base_set);

/*-- generate new base set absolute path */
      if (di->set)
	sprintf (buf, "%s/%s", base, di->set);
      else
	sprintf (buf, "%s", base);

      local_base_set = CopyString (buf);
      break;
    case USER_DFLT_SET:
      if ((base = UserDefaults ()) == NULL)
	base = getenv ("HOME");

      if (base == NULL)
	  return -1;

/*-- free any previous value of base set */
      if (user_base_set)
	Free (user_base_set);

/*-- generate new base set absolute path */
      if (di->set)
	sprintf (buf, "%s/defaults/%s", base, di->set);
      else
	sprintf (buf, "%s/defaults", base);

      user_base_set = CopyString (buf);
      break;
    case ANY_DFLT_SET:
    default:
      return -1;
      break;
    }

  return 0;
}

/*++ ****************  function DfltGetBaseSet ******************** */
/*
   Purpose:        determine the current base_set

   Parameters:
   int      dflt_set   == which set to use

   Return Values:
   char        *base       == current base set
 */
/*********************************************************************** ++*/

#ifdef VANSI_1
char *
DfltGetBaseSet (int dflt_set)
#else
char *
DfltGetBaseSet (dflt_set)
     int dflt_set;
#endif
{
  switch (dflt_set)
    {
    case GLOBAL_DFLT_SET:
      return global_base_set;
      break;
    case LOCAL_DFLT_SET:
      return local_base_set;
      break;
    case USER_DFLT_SET:
      return user_base_set;
      break;
    case ANY_DFLT_SET:
    default:
      return NULL;
      break;
    }

  return NULL;
}

/*++ ****************  function DfltGetCurrentPath ******************** */
/*
   Purpose:     determine the current absolute set/group path

   Parameters:
   int      dflt_set    == which set to use

   Return Values:
   char        *path       ==  current base path
 */
/*********************************************************************** ++*/

#ifdef VANSI_1
char *
DfltGetCurrentPath (int dflt_set)
#else
char *
DfltGetCurrentPath (dflt_set)
     int dflt_set;
#endif
{
  static char buf[MAXPATHLEN];
  char *base, *set, *group;

  switch (dflt_set)
    {
    case GLOBAL_DFLT_SET:
    case LOCAL_DFLT_SET:
    case USER_DFLT_SET:
      base = DfltGetBaseSet (dflt_set);
      set = DfltGetCurrentSet (dflt_set);
      group = DfltGetCurrentGroup (dflt_set);

      if (set && group)
	sprintf (buf, "%s/%s/%s", base, set, group);
      else if (set)
	sprintf (buf, "%s/%s", base, set);
      else
	sprintf (buf, "%s/%s", base, group);

      break;

    case ANY_DFLT_SET:
    default:
      return NULL;
      break;
    }

  return buf;

}


/*++ ***************  function DfltSetCurrentPath *************** */
/*
   Purpose:        set the current path to an absolute value

   Globals changed: current_path

   Parameters:
   char     *path      ==   absolute path to set

   Return Values: void

   Dependencies:
   CopyString()

 */
/************************************************************************ ++**/
#ifdef VANSI_1
void
DfltSetCurrentPath (DfltsInfo * di)
#else
void
DfltSetCurrentPath (di)
     DfltsInfo *di;
#endif
{
  /* set an absolute path */
  char buf[MAXPATHLEN];

/*-- verify arguments */
  if (!di || !di->set)
    return;

/*-- make a temporary copy */
  if (di->group)
    sprintf (buf, "%s/%s/%s", DfltGetBaseSet (di->base),
	     di->set, di->group);
  else
    sprintf (buf, "%s/%s", DfltGetBaseSet (di->base),
	     di->set);

  switch (di->base)
    {
    case GLOBAL_DFLT_SET:
/*-- free any previous value */
      if (global_path)
	Free (global_path);

/*-- set the new value */
      global_path = CopyString (buf);

      break;
    case LOCAL_DFLT_SET:
/*-- free any previous value */
      if (global_path)
	Free (global_path);

/*-- set the new value */
      global_path = CopyString (buf);
      break;
    case USER_DFLT_SET:
/*-- free any previous value */
      if (user_path)
	Free (user_path);

/*-- set the new value */
      user_path = CopyString (buf);
      break;
    case ANY_DFLT_SET:
    default:
      return;
    }

}

#ifdef VANSI_1
void
DfltResetPaths (DfltsInfo * di)
#else
void
DfltResetPaths (di)
     DfltsInfo *di;
#endif
{
  char buf[MAXPATHLEN];

  if (di->base != GLOBAL_DFLT_SET &&
      di->base != LOCAL_DFLT_SET &&
      di->base != USER_DFLT_SET &&
      di->base != ANY_DFLT_SET)
    return;

  sprintf (buf, "%s/%s", DfltGetCurrentSet (di->base),
	   DfltGetCurrentGroup (di->base));

  switch (di->base)
    {
    case GLOBAL_DFLT_SET:
/*-- free any previous value */
      if (global_path)
	Free (global_path);

/*-- set the new value */
      global_path = CopyString (buf);

      break;
    case LOCAL_DFLT_SET:
/*-- free any previous value */
      if (global_path)
	Free (global_path);

/*-- set the new value */
      global_path = CopyString (buf);
      break;
    case USER_DFLT_SET:
/*-- free any previous value */
      if (user_path)
	Free (user_path);

/*-- set the new value */
      user_path = CopyString (buf);
      break;
    case ANY_DFLT_SET:
      di->base = GLOBAL_DFLT_SET;
      DfltResetPaths (di);
      di->base = LOCAL_DFLT_SET;
      DfltResetPaths (di);
      di->base = USER_DFLT_SET;
      DfltResetPaths (di);
      di->base = ANY_DFLT_SET;
      break;
    default:
      return;
    }

}

/*++ ****************  function  DfltGetCurrentSet ******************** */
/*
   Purpose:    determine the current set

   Parameters: void

   Return Values:  char *
   group       == current set
 */
/*********************************************************************** ++*/

char *
DfltGetCurrentSet (int dflt_set)
{
  switch (dflt_set)
    {
    case GLOBAL_DFLT_SET:
      return global_set;
      break;
    case LOCAL_DFLT_SET:
      return local_set;
      break;
    case USER_DFLT_SET:
      return user_set;
      break;
    case ANY_DFLT_SET:
    default:
      return NULL;
      break;
    }

  return NULL;
}

/*++ ***************  function DfltSetCurrentSet *************** */
/*
   Purpose:   set the current set to an absolute value

   Globals changed:  curr_set

   Parameters:
   char     *path      == new current set from base set

   Return Values: int
   0       success
   < 0     error

   Dependencies:
   CopyString(), DfltGetBaseSet()

 */
/************************************************************************ ++**/

#ifdef VANSI_1
int
DfltSetCurrentSet (DfltsInfo * di)
#else
int
DfltSetCurrentSet (di)
     DfltsInfo *di;
#endif
{

  /*-- no need to verify arguments as calling this with NULL resets the
      current set to it's default value */
  if (!di && !di->set)
    return -1;

  switch (di->base)
    {
    case GLOBAL_DFLT_SET:
/*-- free any previous value */
      if (global_set)
	Free (global_set);

      global_set = CopyString (di->set);
      break;
    case LOCAL_DFLT_SET:
      if (local_set)
	Free (local_set);

      local_set = CopyString (di->set);
      break;
    case USER_DFLT_SET:
      if (user_set)
	Free (user_set);

      user_set = CopyString (di->set);
      break;
    case ANY_DFLT_SET:
      if (di->set)
	{
	  di->base = GLOBAL_DFLT_SET;
	  DfltSetCurrentSet (di);
	  di->base = LOCAL_DFLT_SET;
	  DfltSetCurrentSet (di);
	  di->base = USER_DFLT_SET;
	  DfltSetCurrentSet (di);
	  di->base = ANY_DFLT_SET;
	}
      else
	{
	  if (global_set)
	    Free (global_set);

	  if (local_set)
	    Free (local_set);

	  if (user_set)
	    Free (user_set);

	  DfltsInit ();
	}
      break;
    default:
      return -1;
      break;
    }


  return 0;
}

/*++ ***************  function DfltSetCurrentGroup *************** */
/*
   Purpose:        set the current group to some value, also
   update the current path to reflect the change

   Globals changed: curr_group

   Parameters:
   char     *group      == new group
   int      set        == which set to use

   Return Values:int
   0       success
   < 0     error

   Dependencies:
   DfltSetCurrentPath(), DfltGetBaseSet(), CopyString()

 */
/************************************************************************ ++**/

#ifdef VANSI_1
int
DfltSetCurrentGroup (DfltsInfo * di)
#else
int
DfltSetCurrentGroup (di)
     DfltsInfo *di;
#endif
{

/*-- verify arguments */
  if (!di || !di->group)
    return -1;

  switch (di->base)
    {
    case GLOBAL_DFLT_SET:
/*-- free previous value */
      if (global_group)
	Free (global_group);

/*-- copy the new group */
      global_group = CopyString (di->group);

      break;
    case LOCAL_DFLT_SET:
/*-- free previous value */
      if (local_group)
	Free (local_group);

/*-- copy the new group */
      local_group = CopyString (di->group);

      break;
    case USER_DFLT_SET:
/*-- free previous value */
      if (user_group)
	Free (user_group);

/*-- copy the new group */
      user_group = CopyString (di->group);

      break;
    case ANY_DFLT_SET:
      di->base = GLOBAL_DFLT_SET;
      DfltSetCurrentGroup (di);
      di->base = LOCAL_DFLT_SET;
      DfltSetCurrentGroup (di);
      di->base = USER_DFLT_SET;
      DfltSetCurrentGroup (di);
      di->base = ANY_DFLT_SET;
      break;
    default:
      return -1;
      break;
    }


  return 0;
}

/*++ ****************  function  DfltGetCurrentGroup ******************** */
/*
   Purpose:    determine the current group

   Parameters: int set

   Return Values:  char *
   group       == current group
 */
/*********************************************************************** ++*/

char *
DfltGetCurrentGroup (int dflt_set)
{
  switch (dflt_set)
    {
    case GLOBAL_DFLT_SET:
      return global_group;
      break;
    case LOCAL_DFLT_SET:
      return local_group;
      break;
    case USER_DFLT_SET:
      return user_group;
      break;
    case ANY_DFLT_SET:
    default:
      return NULL;
      break;
    }

  return NULL;
}


/* in this type either set or group is non NULL, but not both */
typedef struct _SetGroup
{
  char *set;			/* if !NULL then list is a set list */
  char *group;			/* if !NULL then list is a list of TagDefs */
  LLITEMPTR list;		/* list of SetGroup or TagDefs */
}

SetGroup;


#ifdef VANSI_1
SetGroup *
CreateSetGroup (char *set, char *group)
#else
SetGroup *
CreateSetGroup (group, set)
     char *set;
     char *group;
#endif
{
  SetGroup *sg;

  sg = (SetGroup *) Malloc (sizeof (SetGroup));

  Memset (sg, 0, sizeof (SetGroup));

  if (set)
    sg->set = CopyString (set);
  else
    sg->group = CopyString (set);

  sg->list = LLCreate ();

  return sg;
}


#ifdef VANSI_1
int
DfltAddGroupDefs (LLITEMPTR list, char *group)
#else
int
DfltAddGroupDefs (list, group)
     LLITEMPTR list;
     char *group;
#endif
{
  TagDef *tdef;
  char *ftag, *fval;
  char buf[4096];
  char *str, *t;
  FILE *fp;

  ftag = fval = NULL;

  if ((t = strrchr (group, '/')) != NULL)
    if ((strcmp (t, ".") != 0) || (strcmp (t, "..") != 0))
      return -1;

  if ((fp = fopen (group, "r")) == NULL)
    {
      printf ("%s couldn't open group %s\n", ProgramName (), group);
      return -1;
    }

  while ((str = fgets (buf, 4096, fp)) != NULL)
    {

      if (SeperateTagAndValue (str, &ftag, &fval) < 0)
	continue;

      StripLTBlanks (ftag);
      StripLTBlanks (fval);
      StripLTBlanks (ftag);
      StripLTBlanks (fval);

      tdef = CreateTagDef (ftag, fval);
      LLAddEnd (list, (void *) tdef);
    }

  return 0;
}


#ifdef VANSI_1
int
DfltAddSetDefs (LLITEMPTR list, char *set)
#else
int
DfltAddSetDefs (list, set)
     LLITEMPTR list;
     char *set;
#endif
{
  DIR *dir;
  struct dirent *dent;
  char buf[MAXPATHLEN];
  SetGroup *sg = NULL;

  /* open up this set */
  if ((dir = opendir (set)) == NULL)
    {
      printf ("%s couldn't open data set %s", ProgramName (), set);
      return -1;
    }
  else
    {
      struct stat stbuf;

      while ((dent = readdir (dir)) != NULL)
	{
	  if (strcmp (dent->d_name, ".") == 0)
	    continue;
	  if (strcmp (dent->d_name, "..") == 0)
	    continue;

	  if (stat (dent->d_name, &stbuf) < 0)
	    continue;

	  sprintf (buf, "%s/%s", set, dent->d_name);

	  if ((stbuf.st_mode & S_IFMT) == S_IFDIR)
	    {
	      /* this is a set so process it */
	      sg = CreateSetGroup (dent->d_name, NULL);

	      LTRACE (VLT_DFLTSET, "%s processing set %s\n",
		      ProgramName (), buf);

	      if (DfltAddSetDefs (sg->list, buf) < 0)
		return -1;

	    }
	  else if ((stbuf.st_mode & S_IFMT) == S_IFREG)
	    {

	      sg = CreateSetGroup (NULL, dent->d_name);

	      if (DfltAddGroupDefs (sg->list, buf) < 0)
		return -1;
	    }
	  else
	    fprintf (stderr, "%s invalid directory entry %s/%s",
		     ProgramName (), set, dent->d_name);

	  /* now add it to this sets list */
	  LLAddEnd (list, (void *) sg);

	}
      closedir (dir);
    }
  return 0;
}


/*++ ****************  function DfltDeleteGroup ******************** */
/*
   Purpose:         delete an entire group by renaming the group file
   to ${group}.bak

   Parameters:
   char     *group      == group to delete

   Return Values: int
   0        success
   < 0      error

 */
/*********************************************************************** ++*/

#ifdef VANSI_1
int
DfltDeleteGroup (char *path)
#else
int
DfltDeleteGroup (path)
     char *group;
#endif
{
  char backup[MAXPATHLEN];

/*-- generate group backup name */
  sprintf (backup, "%s.bak", path);

/*-- use system rename function to rename the file */
  rename (path, backup);

  return 0;
}



#ifdef VANSI_1
int
DestroySet (DfltsInfo * di)
#else
int
DestroySet (di)
     DfltsInfo *di;
#endif
{
  char *setstr;
  char newset[MAXPATHLEN];
  int retval = 0;
  struct stat stbuf;
  char buf[MAXPATHLEN];

  sprintf (buf, "%s/%s", DfltGetBaseSet (di->base),
	   di->set);

  if (stat (buf, &stbuf) < 0)
    return 0;

  DfltSetCurrentSet (di);

  setstr = DfltGetSet (di);

  if (setstr)
    {
      char *t, *next;

      t = setstr;

      if (t)
	do
	  {
	    next = TerminateAtNextChar (t, ",");
	    DfltSetCurrentSet (di);

	    sprintf (buf, "%s/%s/%s", DfltGetBaseSet (di->base),
		     di->set, t);

	    if (stat (buf, &stbuf) < 0)
	      {
		fprintf (stderr, "can't access set %s\n", buf);
		return -1;
	      }

	    if ((stbuf.st_mode & S_IFMT) == S_IFREG)
	      {
		unlink (buf);
	      }
	    else
	      {
		sprintf (newset, "%s/%s", di->set, t);

		di->set = newset;
		if (DestroySet (di) < 0)
		  {
		    fprintf (stderr, "%s error deleting set %s\n",
			     ProgramName (), t);
		    retval = -1;
		  }
	      }

	  }
	while ((t = next) != NULL);

      Free (setstr);
    }
  else
    return -1;

  return retval;

}

#ifdef VANSI_1
int
DfltDeleteSet (DfltsInfo * di)
#else
int
DfltDeleteSet (di)
     DfltsInfo *di;
#endif
{
  char backup[MAXPATHLEN];
  char newbak[MAXPATHLEN];

  sprintf (backup, "%s.bak", DfltGetCurrentSet (di->base));
  sprintf (newbak, "%s/%s", DfltGetBaseSet (di->base),
	   DfltGetCurrentSet (di->base));

  /* ensure the backup set does not exist */
  di->set = backup;
  DestroySet (di);

  sprintf (backup, "%s/%s.bak", DfltGetBaseSet (di->base),
	   DfltGetCurrentSet (di->base));

  rename (newbak, backup);

  return 0;
}
