/******************************************************************************
	  CCCC	    A	  BBBBB	  L	EEEEE  N     N	EEEEE	TTTTTTT
	 C    C    A A	  B    B  L	E      NN    N	E	   T
	C	  A   A	  B    B  L	E      N N   N	E	   T
	C	 AAAAAAA  BBBBB	  L	EEEEE  N  N  N	EEEEE	   T
	C        A     A  B    B  L	E      N   N N	E	   T
	 C    C  A     A  B    B  L	E      N    NN  E 	   T
	  CCCC	 A     A  BBBBB	  LLLL	EEEEE  N     N	EEEEE	   T
*******************************************************************************

CableNet Source Module:
	Copyright 1994-1995 (C) CableNet Limited. All Rights Reserved.

    Module Name:		$RCSfile: AppInit.c,v $
    Module Description:	Generic command line argument processing functions


Description:


Edit History:

	$Log: AppInit.c,v $
 * Revision 2.3  1996/04/04  11:15:53  damian
 * include errno.h regardless of OS
 *
 * Revision 2.2  1996/03/07  00:01:16  damian
 * add ability to send LTRACE outout to syslog
 *
 * Revision 2.1  1996/01/04  12:33:52  V
 * Version 2
 * change char arg to int in FindArg..etc...
 *
 * Revision 1.4  1995/08/29  10:01:57  damian
 * changes for sunos
 *
 * Revision 1.3  1995/02/03  11:11:52  V
 * added defaults intialisation and access
 *
 * Revision 1.2  1995/01/20  18:32:44  V
 * add ProgramName and setup recover mode
 *
 * Revision 1.1  1995/01/09  18:21:23  V
 * Initial revision
 *


*/

/* RCS identification string (for "what" program) */
static char moduleRCSid[] = "@(#) $Id: AppInit.c,v 2.3 1996/04/04 11:15:53 damian Exp $";

/* must come first header files */
#include "V.h"			/* virtual header file */
#include "Vport.h"		/* port header file */


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/fcntl.h>
#include <errno.h>

#ifdef Vsunos40
#include "Vansi.h"
#endif

#ifdef VANSI_1
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "Vlib.h"


static int BuildArgList ();
static int ValidateArgList ();

static char *CLO_ProgramName = NULL;
static char *CLO_GlobalDefaults = NULL;
static char *CLO_LocalDefaults = NULL;
static char *CLO_UserDefaults = NULL;

/* linked list of argument info items */
static LLlist arglist;


extern int via_syslog;

/*
 * this variable determines whether to process generic (common) arguments ie.
 * -L   <mode>       - ltrace mode
 * -D   <mode>       - debug mode
 * -H                - do help
 * -F                - turn on ftrace
 * -defaults_dir  <dir>  - set new defaults base directory
*/

static BOOLEAN process_generic_args = FALSE;

/*
 * used to test the command line processing suite
 * by printing the arg info list
 */
void
PrintArgList ()
{
  LLnode *node;
  VargInfoPtr va;

  node = arglist.head;

  printf ("addr    num id name\tvalue\toptarg\tnext\tprev\trange\tdesc\n");
  while (node)
    {

      va = (VargInfoPtr) node->data;
      printf ("0x%p = %d  %c  %s\t%s\t0x%p\t0x%p\t0x%p\t%s\t%s\n",
	      va,
	      va->argindex,
	      va->shortname,
	      va->longname,
	      va->value,
	      va->optarg,
	      va->next,
	      va->prev,
	      va->range,
	      va->desc);

      node = node->next;

    }

}

/*++ ****************  function ProgramName() *********************/
/*
    Purpose:   determine the name of the program

    Parameters: none

    Return Values:
    char     *name       ==   name of program (argv[0])

*/
/*********************************************************************** ++*/

char *
ProgramName ()
{
  return CLO_ProgramName;
}

/*++ ****************  function  SetProgramName *********************/
/*
    Purpose:  set up the program name object

    Parameters:
    char        *name       == new program name string

    Return Values: void

*/
/*********************************************************************** ++*/

void
SetProgramName (name)
     char *name;
{
  char *p;

  if (!name)
    return;

  if (CLO_ProgramName)
    Free (CLO_ProgramName);

  while (name[strlen (name) - 1] == '/')
    name[strlen (name) - 1] = '\0';

  if ((p = strrchr (name, '/')) == NULL)
    p = name;
  else
    p++;

  CLO_ProgramName = CopyString (p);

}

/*++ ****************  function GlobalDefaults() *********************/
/*
    Purpose:   determine the base defaults directory

    Parameters: none

    Return Values:
    char     *dir       ==   base directory

*/
/*********************************************************************** ++*/

char *
GlobalDefaults ()
{
  return CLO_GlobalDefaults;
}

/*++ ****************  function LocalDefaults() *********************/
/*
    Purpose:   determine the base defaults directory

    Parameters: none

    Return Values:
    char     *dir       ==   base directory

*/
/*********************************************************************** ++*/

char *
LocalDefaults ()
{
  return CLO_LocalDefaults;
}

/*++ ****************  function UserDefaults() *********************/
/*
    Purpose:   determine the base defaults directory

    Parameters: none

    Return Values:
    char     *dir       ==   base directory

*/
/*********************************************************************** ++*/

char *
UserDefaults ()
{
  return CLO_UserDefaults;
}

/*++ **********************  function  VappGenericArgs ***************

	Purpose:  signal that this program will accept the common set of
	command line arguments

	Return Values: void

******************************************************************* ++*/


#ifdef VANSI_1
void
VappGenericArgs (void)
#else
void
VappGenericArgs ()
#endif
{
  process_generic_args = TRUE;

}


#ifdef VANSI_1
static void
DoHelp (char *progpath)
#else
static void
DoHelp (progpath)
     char *progpath;

#endif
{
  char *p, helpfile[512], sysbuf[1024];
  char *pager, *vhome;

  if ((p = strrchr (progpath, '/')) == NULL)
    p = progpath;

  pager = getenv ("PAGER");
  vhome = getenv ("VHOME");

  if (vhome)
    sprintf (helpfile, "%s/help/%s", vhome, p);
  else
    sprintf (helpfile, "~V/help/%s", p);

  if (access (helpfile, R_OK) != 0)
    {
      fprintf (stderr, "%s can't access helpfile %s\n",
	       progpath, helpfile);
      return;
    }
  if (pager)
    sprintf (sysbuf, "%s %s", pager, helpfile);
  else
    sprintf (sysbuf, "more %s", helpfile);

  system (sysbuf);

}

#ifdef VANSI_1
static void
OpenLogfile (char *fname, char *progname)
#else
static void
OpenLogfile (fname, progname)
     char *fname;
     char *progname;

#endif
{
  char logfilename[256];
  int fd;
  time_t t;
  char *p;

  if ((p = strrchr (progname, '/')) == NULL)
    p = progname;

  /*-- generate the log file name and open the file */
  /*-- if the log filename is null or == "TRUE" use program name + pid */
  if ((fname != NULL) && strncmp (fname, "TRUE", 4) != 0)
    sprintf (logfilename, "%s", fname);
  else
    sprintf (logfilename, "%s.%ld", p, (long) getpid ());

  if ((fd = open (logfilename, O_WRONLY | O_CREAT, 0666)) < 0)
    {
      fprintf (stderr, "%s: error opening logfile %s\n",
	       progname, logfilename);
      exit (1);
    }
  /*-- close stderr and duplicate the log file descriptor so that
      stderr goes to the log file */
  close (2);
  dup (fd);

  t = time ((time_t *) NULL);
  /*-- print a banner in the log file */
  fprintf (stderr, "%s: pid:%ld log file started at %s\n",
	   progname, (long) getpid (), ctime (&t));
  fprintf (stderr,
	   "---------------------------------------------------------\n");

}

/*++ **********************  function  VappArgList ***************

	Purpose:  get a pointer to the arglist

	Return Values: LLIST_HDR *
	hdr    == address of argument list header
******************************************************************* ++*/
#ifdef VANSI_1
LLlist *
VappArgList (void)
#else
LLlist *
VappArgList ()
#endif
{
  return &arglist;
}

/*++ ***************  function  VappInit ****************/
/*
    Purpose:  process command line arguments

    Prototype:
    char *VappInit( int *argc,        // address of argc
                     char **argv,      // argv untouched
		     char *ver,        // version string ie 2.11
		     *(func)(),        // ptr to usage function
		     flags );          // set of default actions to perform


    Globals changed: none

    Parameter Description:
    argc        == pointer to an integer - number of arguments
    argv        == pointer to list of command line arguments
    ver         == version string, may be displayed on startup
    func        == pointer to a replacement usage function instead of generic
    flags       == control flags determine startup actions

    Return Values: char *
    progname    == program name as per argv[0]
    NULL        if argv is NULL argument

    Dependencies:
    VappArgCHAR, ValidateArgList, BuildArgList

    Notes:
    flags is a BitMask, the values of which are;

        0x01   if set don't print program name & version string
	0x02   if set create a log file called argv[0].getpid() and redirect
	       stderr to it.  This can also be achived with the generic
	       argument -logfile with no arguments or with an argument to
	       explicitly name the logfile
        0x04   if set don't print usage when exiting, regardless of any
	       individual argument setups

    Example Use:

    progname = VappInit( &argc,        // address of argc
               argv,         // argv untouched
               ver,          // char * version string ie 2.11
               usage,        // ptr to usage function (may be null)
               flags );      // set of default actions to perform


*/
/************************************************************************ ++**/

#ifdef VANSI_1
char *
VappInit (int *argc, char *argv[], char *ver, int (*usage) (), BitMask flags)
#else
char *
VappInit (argc, argv, ver, usage, flags)
     int *argc;
     char *argv[];
     char *ver;
     int (*usage) ();
     BitMask flags;

#endif
{
  int num;
  char *logfname;

  /*-- initialise the arglist */
  Memset (&arglist, 0, sizeof (LLlist));

  /*-- by default print version string unless told not to */
  if (!(flags & VARG_NOVERSION) && ver)
    {
      fprintf (stderr, "%s: version %s\n", argv[0], ver);
    }
  /*-- build argument info list */
  BuildArgList (*argc, argv);

  /*-- validate the arguments */
  if (ValidateArgList (&arglist) < 0)
    {
      fprintf (stderr, "%s: argument validation failed\n", argv[0]);
      if (usage != NULL && !(flags & VARG_USAGE))
	usage ();
      exit (1);
    }
  /*-- open a logfile if the program explicitly demands it or the user
      asked for one on the command line */
  logfname = VappArgCHAR ("logfile");

  if ((flags & VARG_LOGFILE) || logfname)
    {
      OpenLogfile (logfname, argv[0]);
    }
  /*-- if the process generic args flag has been set then
      process the generic args */
  if (process_generic_args != 0)
    {
      char *val;
      BitMask level;

      if ((val = VappArgCHAR ("L")) != NULL)
	{
	  level = (BitMask) strtol (val, (char **) NULL, 0);
	  AddLtraceLevel (level);
	  /*	    LTRACEON(level);		*/

	  /* the user can ask for output via syslog */
	  if ((val = VappArgCHAR ("syslog")) != NULL)
	  {
	      via_syslog = TRUE;
	  }
	}

      if ((val = VappArgCHAR ("F")) != NULL)
	FTRACEON ();

      if ((val = VappArgCHAR ("D")) != NULL)
	{
	  level = (BitMask) strtol (val, (char **) NULL, 0);
	  AddDebugMode (level);
	}

      if ((val = VappArgCHAR ("H")) != NULL)
	{
	  DoHelp (argv[0]);
	  exit (0);
	}
      if ((val = VappArgCHAR ("recover")) != NULL)
	{
	  level = (BitMask) strtol (val, (char **) NULL, 0);
	  SetRecoveryMode (level);
	}

      if ((val = VappArgCHAR ("global_defaults")) != NULL)
	{
	  CLO_GlobalDefaults = CopyString (val);
	}
      if ((val = VappArgCHAR ("local_defaults")) != NULL)
	{
	  CLO_LocalDefaults = CopyString (val);
	}
      if ((val = VappArgCHAR ("user_defaults")) != NULL)
	{
	  CLO_UserDefaults = CopyString (val);
	}

      if ((val = VappArgCHAR ("Strace")) != NULL)
	{
	  SetStracePattern (val);
	}
      errno = 0;
      num = VappArgINT ("Stracewin");
      if (errno == 0)
	{
	  SetStraceWindow (num);
	}
      if ((val = VappArgCHAR ("Stracecase")) != NULL)
	{
	  SetStraceWindow (1);
	}

    }

  SetProgramName (argv[0]);

  return argv[0];
}



/*++ ***************  function BuildArgList ****************/
/*
    Purpose:  build the list of argument info structures

    Prototype:  static int BuildArgList( int argc, char **argv )

    Globals changed: arglist

    Parameter Description:
    argc     == intger number of arguments
    argv     == argument vector pointer

    Return Values:int
    0       == success
    <0      == error

    Dependencies:
    CopyString, _AddNewArgInfo, _FindArgInfoShort, _FindArgInfoLong

    Notes:

*/
/************************************************************************ ++**/
#ifdef VANSI_1
static int
BuildArgList (int argc, char **argv)
#else
static int
BuildArgList (argc, argv)
     int argc;
     char **argv;

#endif
{
  int i = 1, append_new, duplicate_arg;
  char *id, *arg;
  VargInfoPtr nextarg, prevarg = NULL;
  VargInfoPtr longarg, shortarg;

  /*-- for each arg in the vector */
  while (i < argc)
    {
      shortarg = longarg = NULL;
      duplicate_arg = FALSE;

      if ((id = arg = argv[i]) == NULL)	/* something dicky with argc */
	return -1;

      append_new = FALSE;

      /*-- get a handle on the identifier proper */
      while (*id == '-')
	id++;

      if (*id)
	{
	  /*-- determine if we have seen the same initial letter */
	  shortarg = _FindArgInfoShort (*id);

	  /*-- retrieve any previously setup arg info item */
	  if (strlen (id) > 1)
	    longarg = _FindArgInfoLong (id);

	  if (longarg != NULL && longarg->argindex > 0)
	    append_new = TRUE;
	  else if (longarg == NULL && shortarg == NULL)
	    append_new = TRUE;	/* no prior knowledge  */
	  else if (strlen (id) > 1 && longarg == NULL)
	    append_new = TRUE;
	  else if (strlen (id) == 1 && shortarg == NULL)
	    append_new = TRUE;
	  else if (longarg == NULL && shortarg->argindex > 0)
	    append_new = TRUE;

	  /*-- if we haven't seen this arg before add a new item */
	  if (append_new == TRUE)
	    {
	      nextarg = _AddNewArgInfo (NULL, (char) '\0');
	    }
	  else if (longarg != NULL)
	    nextarg = longarg;
	  else
	    nextarg = shortarg;

	  nextarg->argindex = i;

	  /*-- if this argument is a member of a 'same argument' list
	      ie. -Wall -Wnothing then add it to the list for the id */
	  if (append_new == TRUE && (longarg || shortarg))
	    {
	      VargInfoPtr lastva;

	      if (longarg)
		lastva = longarg;
	      else
		lastva = shortarg;

	      /*-- seek to the end of the list and add the arg info item */
	      while (lastva->next != NULL)
		{
		  lastva = lastva->next;
		}
	      lastva->next = nextarg;
	    }
	  if (*arg != '-')
	    {
	      /*-- if this is not an option  */
	      /*-- set the value of the previous arg info item's optarg
		 pointer to point to this arg */
	      if (prevarg != NULL)
		prevarg->optarg = nextarg;

	      nextarg->value = CopyString (id);

	    }
	  else
	    {
	      /*-- else this is an option so set the shortname and */
	      if (nextarg->shortname == '\0')
		nextarg->shortname = *id;

	      if (strlen (id) > 1)
		{
		  /*-- set longname if the id is longer than a single char */
		  if (nextarg->longname == NULL)
		    nextarg->longname = CopyString (id);

		  nextarg->value = CopyString (id + 1);
		}
	    }

	}
      prevarg = nextarg;
      i++;
    }

  return 0;
}



/*++ ***************  function  ValidateArgList ****************/
/*
    Purpose: validate the argument list

    Prototype: static int ValidateArgList( LLlist *arglist )

    Globals changed: none

    Parameter Description:
    arglist     == list of arg info items

    Return Values: int
    0       == success  all args passed validation
    <0      == error 1 or more args failed validation

    Dependencies:
    llist_delete

    Notes:

*/
/************************************************************************ ++**/

#ifdef VANSI_1
static int
ValidateArgList (LLlist * list)
#else
static int
ValidateArgList (list)
     LLlist *list;

#endif
{
  LLnode *node, *next;
  VargInfoPtr va = NULL, frstva;
  BOOLEAN val;
  int start, end, value;
  int mutexId, nextMutexId;

  node = list->head;

  /*-- for each node on the arg info list */
  while (node != NULL)
    {

      next = node->next;

      if ((va = (VargInfoPtr) node->data) == NULL)
	LLdelete (list, node);

      /*-- if this node's argument was setup by the program but wasn't
	  given on the command line then delete it's info item */
      if (va->argindex == 0)
	{

	  if (va->longname)
	    Free (va->longname);

	  if (va->value)
	    Free (va->value);

	  if (va->range)
	    Free (va->range);

	  if (va->desc)
	    Free (va->desc);

	  LLdelete (list, node);

	  node = next;

	  continue;
	}
      /*-- if a validation function or a range was given for this arg  */
      if (va->func || va->range)
	{
	  /*-- evaluate the arg */
	  if (va->func)
	    {
	      /*-- evaluate the function */
	      if (va->optarg && va->optarg->value)
		val = va->func (va->optarg->value);
	      else if (va->value)
		val = va->func (va->value);
	      else
		val = FALSE;
	    }
	  else
	    {
	      /*-- or evaluate the range */
	      char *rst, *rcomma, *rend;
	      char buf[64];

	      strcpy (buf, va->range);

	      /*-- get start & end of range */
	      rst = buf;
	      rcomma = strchr (buf, ',');
	      rend = rcomma + 1;

	      if (va->optarg && va->optarg->value)
		value = atoi (va->optarg->value);
	      else if (va->value)
		value = atoi (va->value);
	      else
		rcomma = NULL;

	      /*-- determine if value falls within range */
	      if (rcomma && (*rcomma == ','))
		{
		  *rcomma = '\0';
		  start = atoi (rst);
		  end = atoi (rend);
		  if (value >= start && value <= end)
		    val = TRUE;
		  else
		    val = FALSE;
		}
	      else
		val = FALSE;
	    }

	  /*-- if the evaluation of the argument */
	  /*-- (either true or false) means failure */
	  if ((val && (va->flags & VARG_FAILTRUE)) ||
	      (!val && !(va->flags & VARG_FAILTRUE)))
	    {

	      /*-- if required print usage for this arg */
	      if ((va->flags & VARG_PRNUSAGE) &&
		  (va->longname || va->shortname) &&
		  va->desc)
		{
		  if (va->longname)
		    fprintf (stderr, "evaluation failed for -%s (%s) ",
			     va->longname, va->desc);
		  else
		    fprintf (stderr, "evaluation failed for -%c (%s) ",
			     va->shortname, va->desc);

		  if (va->range)
		    fprintf (stderr, "range: %s\n", va->range);
		  else
		    fprintf (stderr, "\n");
		}
	      /*-- then exit if failure means exit */
	      if (va->flags & VARG_EXITFAIL)
		{
		  fprintf (stderr, "exiting....\n");
		  exit (va->exval);
		}
	    }
	}
      node = next;
    }

  /*-- now check for mutually exclusive arguments */
  /*-- for each unique mutual exclusion ID */
  mutexId = 0;
  do
    {
      int found_one = FALSE;

      mutexId++;
      nextMutexId = mutexId;

      node = list->head;

      /*-- for each node on the arg info list */
      while (node != NULL)
	{

	  next = node->next;

	  if ((va = (VargInfoPtr) node->data) == NULL)
	    continue;

	  /*-- if this arg has the current mutual exclusion Id */
	  if (va->mutexid == mutexId)
	    {
	      nextMutexId = mutexId + 1;

	      /*-- if we have already seen this id then */
	      if (found_one == TRUE)
		{
		  /*-- two mutually exclusive args present */
		  /*-- print an error message and return error */
		  fprintf (stderr, "mutually exclusive arguments:  ");

		  if (frstva->longname)
		    fprintf (stderr, "-%s ", frstva->longname);
		  else
		    fprintf (stderr, "-%c ", frstva->shortname);

		  if (frstva->desc)
		    fprintf (stderr, "(%s) ", frstva->desc);

		  fprintf (stderr, "and ");

		  if (va->longname)
		    fprintf (stderr, "-%s ", va->longname);
		  else
		    fprintf (stderr, "-%c ", va->shortname);

		  if (va->desc)
		    fprintf (stderr, "(%s) ", va->desc);

		  fprintf (stderr, "\n");
		  return -1;
		}
	      found_one = TRUE;
	      frstva = va;
	    }
	  node = next;
	}
    }
  while (nextMutexId != mutexId);

  return 0;
}

/*++ **********************  function _AddNewArgInfo ***************

    Purpose: create a new arg info item and add it to the list

    Return Values: VargInfoPtr
    pointer to the newly created item

******************************************************************* ++*/

#ifdef VANSI_1
VargInfoPtr
_AddNewArgInfo (char *longname, int shortname)
#else
VargInfoPtr
_AddNewArgInfo (longname, shortname)
     char *longname;
     int shortname;

#endif
{
  VargInfoPtr va;

  va = (VargInfoPtr) Malloc (sizeof (VargInfo));

  /*-- initialise it to all 0's */
  Memset (va, 0, sizeof (VargInfo));

  /*-- set the arg info item's identifier and return the handle */
  if (longname != NULL)
    va->longname = CopyString (longname);

  if (shortname != '\0')
    va->shortname = shortname & 0xff;

  /*-- append the space for the arg item to the list */
  LLappend (&arglist, va);

  return va;
}

/*++ **********************  function _FindArgInfoLong ***************

    Purpose: find an arg info item for a long name

    Return Values: VargInfoPtr
    pointer to the found item or NULL

******************************************************************* ++*/

#ifdef VANSI_1
VargInfoPtr
_FindArgInfoLong (char *name)
#else
VargInfoPtr
_FindArgInfoLong (name)
     char *name;

#endif
{
  LLnode *node;
  VargInfoPtr va = NULL;

  node = arglist.head;

  /*-- for each item on the arg info list */
  while (node != NULL)
    {

      if ((va = (VargInfoPtr) node->data) == NULL)
	LLdelete (&arglist, node);
      /*-- if the longname compares with the name we ar looking for then
	  stop searching and return a pointer to the item */
      if (va->longname != NULL)
	{
	  if (strcmp (va->longname, name) == 0)
	    break;
	}
      node = node->next;
    }
  /*-- otherwise return NULL */
  if (node != NULL)
    return va;
  else
    return NULL;

}

/*++ **********************  function _FindArgInfoShort ***************

	Purpose: find an arg info item for a short (char) name

	Return Values: VargInfoPtr
	pointer to the found arg info item or NULL

******************************************************************* ++*/

#ifdef VANSI_1
VargInfoPtr
_FindArgInfoShort (int name)
#else
VargInfoPtr
_FindArgInfoShort (name)
     int name;

#endif
{
  LLnode *node;
  VargInfoPtr va = NULL;

  node = arglist.head;

  /*-- for each arg info item on the list */
  while (node != NULL)
    {

      if ((va = (VargInfoPtr) node->data) == NULL)
	LLdelete (&arglist, node);

      /*-- if the shortname == the name we are looking for then
	  stop searching and return a pointer to the arg info item */
      if (va->shortname == name & 0xff)
	break;

      node = node->next;
    }
  /*-- otherwise return NULL */
  if (node != NULL)
    return va;
  else
    return NULL;

}
