/* 
 * This is an interface to the miniSQL database for Python.
 *   Written by Anthony Baxter, anthony@aaii.oz.au.
 *
 *   Copyright (C) 1994 Anthony Baxter.
 *
 *   Permission is hereby granted, free of charge, to any person obtaining
 *   a copy of this source file to use, copy, modify, merge, or publish it
 *   subject to the following conditions:
 *
 *   The above copyright notice and this permission notice shall be included
 *   in all copies or in any new file that contains a substantial portion of
 *   this file.
 *
 *   THE  AUTHOR  MAKES  NO  REPRESENTATIONS ABOUT  THE  SUITABILITY  OF
 *   THE  SOFTWARE FOR  ANY  PURPOSE.  IT IS  PROVIDED  "AS IS"  WITHOUT
 *   EXPRESS OR  IMPLIED WARRANTY.  THE AUTHOR DISCLAIMS  ALL WARRANTIES
 *   WITH  REGARD TO  THIS  SOFTWARE, INCLUDING  ALL IMPLIED  WARRANTIES
 *   OF   MERCHANTABILITY,  FITNESS   FOR  A   PARTICULAR  PURPOSE   AND
 *   NON-INFRINGEMENT  OF THIRD  PARTY  RIGHTS. IN  NO  EVENT SHALL  THE
 *   AUTHOR  BE LIABLE  TO  YOU  OR ANY  OTHER  PARTY  FOR ANY  SPECIAL,
 *   INDIRECT,  OR  CONSEQUENTIAL  DAMAGES  OR  ANY  DAMAGES  WHATSOEVER
 *   WHETHER IN AN  ACTION OF CONTRACT, NEGLIGENCE,  STRICT LIABILITY OR
 *   ANY OTHER  ACTION ARISING OUT OF  OR IN CONNECTION WITH  THE USE OR
 *   PERFORMANCE OF THIS SOFTWARE.
 *
 * $Id: msqlmodule.c,v 1.3 1994/09/08 10:23:05 anthony Exp anthony $
 */

/*#define DEBUG_MSQLMOD*/

#include "allobjects.h"
#include "modsupport.h"
#include <sys/types.h>
#include <sys/stat.h>

#include "msql.h"

char   PROGNAME[] = "python";
static object *pythonify_res();
static object *pythonify_lf_res();

typedef struct {
	OB_HEAD
	int handle;
	int valid;
} msqlobject;

staticforward typeobject MsqlType;

#define is_msqlobject(v) ((v)->ob_type == &MsqlType)

static object * 
msqlmod_connect(self,args)
object *self,*args;
{
    char *dbname;
    msqlobject *n;
    if(!getargs(args,"s",&dbname)) {
	if(!getnoarg(args)) {
	    err_setstr(TypeError,"connect has one optional arg, a database host name");
	    return NULL;
    	} else 
	    dbname=NULL;
    }
    n=NEWOBJ(msqlobject,&MsqlType);
    n->valid=0;
    if(!n)
    	return NULL;
    if((n->handle=msqlConnect(dbname))==-1) {
    	err_setstr(ValueError,msqlErrMsg); 
    	DECREF(n);
    	return NULL;
    } else {
    	n->valid=1;
    	return((object *)n);
    }
}
static object *
msqlobj_selectdb(self,args)
msqlobject *self;
object *args;
{
    char *dbname;
    if(!getargs(args,"s",&dbname)) {
        err_setstr(TypeError,"selectdb has one arg, a database name"); 
        return NULL;
    }
    if(msqlSelectDB(self->handle,dbname)==-1) {
        err_setstr(ValueError,msqlErrMsg);
        return NULL;
    } 
    INCREF(None);
    return(None);
}   

static object * 
msqlobj_listdbs(self,args)
msqlobject *self;
object *args;
{
    m_result *res;
    object *resobj;

    if(!getnoarg(args)) {
    	err_setstr(TypeError,"listdbs takes no args");
    	return NULL;
    }
    if((res=msqlListDBs(self->handle))==NULL) {
        INCREF(None);
        return(None);
    }
    resobj=pythonify_res(res);
    return(resobj);
}

static object * 
msqlobj_listtables(self,args)
msqlobject *self;
object *args;
{
    m_result *res;
    object *resobj;

    if(!getnoarg(args)) {
    	err_setstr(TypeError,"listtables takes no args");
    	return NULL;
    }
    if((res=msqlListTables(self->handle))==NULL) {
        INCREF(None);
        return(None);
    }
    resobj=pythonify_res(res);
    return(resobj);
}

static object *
msqlobj_listfields(self,args)
msqlobject *self;
object *args;
{
    char *tname;
    m_result *res;
    object *resobj;

    if(!getargs(args,"s",&tname)) {
        err_setstr(TypeError,"listfields takes one arg, a table name");
        return NULL;
    }
    if((res=msqlListFields(self->handle,tname))==NULL) {
        INCREF(None);
        return(None);
    }
    resobj=pythonify_lf_res(res);
    return(resobj);
}

static object * 
msqlobj_query(self,args)
msqlobject *self;
object *args;
{
    char *query;
    m_result *res;
    object *resobj;

    if(!getargs(args,"s",&query)) {
    	err_setstr(TypeError,"query has one arg, a query string");
    	return NULL;
    }
    if(msqlQuery(self->handle,query)==-1) {
    	err_setstr(ValueError,msqlErrMsg); 
    	return NULL;
    }
    res=msqlStoreResult();
    if(!res) {
    	INCREF(None);
    	return(None);
    }
    resobj=pythonify_res(res);
    return(resobj);
}

/*
 * Take an mSQL m_result, turn it into a list of tuples.
 */
static object *
pythonify_res(res)
m_result *res;
{
    object *reslist,*rowtuple, *str;
    m_row thisrow;
    int i,n;

#ifdef DEBUG_MSQLMOD
    printf("data ready, %d rows of %d fields\n",msqlNumRows(res),msqlNumFields(res));
#endif
    reslist=newlistobject(0);
    n=msqlNumFields(res);
    while(thisrow=msqlFetchRow(res)) {
	rowtuple=newtupleobject(n);
	for (i=0;i<n;i++) {
    	    str=newstringobject(thisrow[i]);
	    settupleitem(rowtuple,i,str);
    	}
	addlistitem(reslist,rowtuple);
	DECREF(rowtuple);
    }
    return(reslist);
}

/*
 * Take an mSQL m_result, return a list of tuples of the FetchField data.
 */
static object *
pythonify_lf_res(res)
m_result *res;
{
    object *reslist, *thistuple, *str;
    int i,n;
    char *type,flags[14];
    m_field *tf;
    
#ifdef DEBUG_MSQLMOD
    printf("data ready, %d fields\n", msqlNumFields(res));
#endif
    reslist=newlistobject(0);
    n=msqlNumFields(res);
    for(i=0;i<n;i++) {
    	tf=msqlFetchField(res);
    	switch(tf->type) {
    	    case INT_TYPE: type="int"; break;
    	    case CHAR_TYPE: type="char"; break;
    	    case REAL_TYPE: type="real"; break;
    	    default: type="????"; break;
    	}
    	if (IS_PRI_KEY(tf->flags)) 
    	    strcpy(flags,"pri");
    	else 
    	    flags[0]=0;
    	if (IS_NOT_NULL(tf->flags)) 
    	    if(flags[0])
		strcat(flags," notnull");
    	    else
    	    	strcpy(flags,"notnull");
    	else 
    	    flags[0]=0;

    	thistuple=mkvalue("(sssis)",tf->name,tf->table,type,tf->length,
    	    	    	    	    	    	    	    	flags);
    	addlistitem(reslist,thistuple);
    	DECREF(thistuple);
    }
    return(reslist);
}

static struct methodlist msqlobj_methods[] = {
    {"SelectDB",(method)msqlobj_selectdb},
    {"selectdb",(method)msqlobj_selectdb},
    {"Query",(method)msqlobj_query},
    {"query",(method)msqlobj_query},
    {"ListDBs",(method)msqlobj_listdbs},
    {"listdbs",(method)msqlobj_listdbs},
    {"ListTables",(method)msqlobj_listtables},
    {"listtables",(method)msqlobj_listtables},
    {"ListFields",(method)msqlobj_listfields},
    {"listfields",(method)msqlobj_listfields},
    {NULL, NULL}           /* sentinel */
};

static object *
msqlobj_getattr(ms, name)
        msqlobject *ms;
        char *name;
{
        return findmethod(msqlobj_methods, (object *)ms, name);
}
static void     
msqlobj_dealloc(m)
        register msqlobject *m;
{
        if ( m->valid )
          msqlClose(m->handle);
        DEL(m);
}       

int
msqlobj_print(sp, fp, flags)
msqlobject *sp;
FILE *fp;
int flags;
{
    fprintf(fp,"MSQL handle");
    return 0;
}

static int
msqlobj_len(self,subs)
object *self, *subs;
{
    err_setstr(TypeError,"cant take length of an msql handle");
    return 0;
}
static object *
msqlobj_subscript(self,subs)
object *self, *subs;
{
    return msqlobj_query(self,subs);
}
static int
msqlobj_ass_sub(self,subs,val)
object *self, *subs,*val;
{
    err_setstr(TypeError,"cant assign to an msql handle (use query insert)");
    return 0;
}

static mapping_methods msql_as_mapping = {
        (inquiry)msqlobj_len,            /*length*/
        (binaryfunc)msqlobj_subscript,      /*subscript*/
        (objobjargproc)msqlobj_ass_sub,     /*assign subscript*/
};              


static typeobject MsqlType = {
        OB_HEAD_INIT(&Typetype)
        0,
        "msqlobject",
        sizeof(msqlobject),
        0,
        (destructor)msqlobj_dealloc,  /*tp_dealloc*/
        0,                            /*tp_print*/
        (getattrfunc)msqlobj_getattr, /*tp_getattr*/
        0,                            /*tp_setattr*/
        0,                            /*tp_compare*/
        0,                            /*tp_repr*/
        0,                            /*tp_as_number*/
        0,                            /*tp_as_sequence*/
        &msql_as_mapping,             /*tp_as_mapping*/
};      


static struct methodlist msql_methods[] = {
    {"Connect",msqlmod_connect},
    {"connect",msqlmod_connect},
/*  To Be Implemented 
    {"FetchData",msqlmod_fetchdata},
    {"fetchdata",msqlmod_fetchdata},
    {"ListDBs",msqlmod_listdbs},
    {"listdbs",msqlmod_listdbs},
    {"ListTables",msqlmod_listtables},
    {"listtables",msqlmod_listtables},
    {"ListFields",msqlmod_listfields},
    {"listfields",msqlmod_listfields},
    {"Close",msqlmod_close},
    {"close",msqlmod_close},
*/
    {NULL,NULL}
};


void
initmsql()
{
    object *module, *dict;
    char *err;

    module = initmodule("msql", msql_methods);
    dict = getmoduledict(module);
}
