/* Routines to read and write xBase-files (.dbf)

   By Maarten Boekhold, 29th of oktober 1995
*/
#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>

#include "dbf.h"
#include "endian.h"

/* open a dbf-file, get it's field-info and store this information */

dbhead *dbf_open(u_char *file, int flags) {
	int			FILE;
	dbhead			*dbh;
	f_descr			*fields;
	dbf_header		*head;
	dbf_field		*fieldc;
	int			t;

	if ((dbh = (dbhead *)malloc(sizeof(dbhead))) == NULL) {
		return (dbhead *)DBF_ERROR;
	}

	if ((head = (dbf_header *)malloc(sizeof(dbf_header))) == NULL) {
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

	if ((fieldc = (dbf_field *)malloc(sizeof(dbf_field))) == NULL) {
		free(head);
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

	if ((FILE = open(file, flags)) == -1) {
		free(fieldc);
		free(head);
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

/* read in the disk-header */

	if (read(FILE, head, sizeof(dbf_header)) == -1) {
		close(FILE);
		free(fieldc);
		free(head);
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

	if (!(head->dbh_dbt & DBH_NORMAL)) {
                close(FILE);
                free(fieldc);
                free(head);
                free(dbh);
                return (dbhead *)DBF_ERROR;
	}

	dbh->db_fd = FILE;
	if (head->dbh_dbt & DBH_MEMO) {
		dbh->db_memo = 1;
	} else {
		dbh->db_memo = 0;
	}
	dbh->db_year = head->dbh_year;
	dbh->db_month = head->dbh_month;
	dbh->db_day = head->dbh_day;
	dbh->db_records = get_long((u_char *)&head->dbh_records);
	dbh->db_currec = 0;
	dbh->db_rlen = get_short((u_char *)&head->dbh_rlen);
	dbh->db_nfields = (get_short((u_char *)&head->dbh_hlen) - 
				sizeof(dbf_header)) / sizeof(dbf_field);

	if ((fields = (f_descr *)calloc(dbh->db_nfields, sizeof(f_descr)))
		== NULL) {
                close(FILE);
                free(fieldc);
                free(head);
                free(dbh);
                return (dbhead *)DBF_ERROR;
	}

	for (t = 0; t < dbh->db_nfields; t++) {
		read(FILE, fieldc, sizeof(dbf_field));
		strncpy(fields[t].db_name, fieldc->dbf_name,
			strlen(fieldc->dbf_name));
		fields[t].db_type = fieldc->dbf_type;	
		fields[t].db_flen = fieldc->dbf_flen;
		fields[t].db_dec  = fieldc->dbf_dec;
	}

	dbh->db_offset = get_short((u_char *)&head->dbh_hlen);
	dbh->db_fields = fields;

        free(fieldc);
        free(head);
	return dbh;
}

void dbf_close(dbhead *dbh) {
	int t;

	close(dbh->db_fd);

	for (t = 0; t < dbh->db_nfields; t++) {
		free(&dbh->db_fields[t]);
	}

	free(dbh);
}
	
int dbf_get_record(dbhead *dbh, field *fields,  u_long rec) {
	u_char  *data;
	int     t, i, offset;
	u_char  *dbffield, *end;

/* calculate at which offset we have to read. *DON'T* forget the
   0x0D which seperates field-descriptions from records!
*/
	offset = sizeof(dbf_header) +
			(dbh->db_nfields * sizeof(dbf_header)) + 1;
	offset += (rec * dbh->db_rlen);

	if (lseek(dbh->db_fd, offset, SEEK_SET) == -1) {
		lseek(dbh->db_fd, 0, SEEK_SET);
		dbh->db_offset = 0;
		return DBF_ERROR;
	}

	dbh->db_offset 	= offset;
	dbh->db_currec	= rec;

    if ((data = (u_char *)malloc(dbh->db_rlen)) == NULL) {
            return DBF_ERROR;
    }

    read(dbh->db_fd, data, dbh->db_rlen);

    if (data[0] == DBF_DELETED) {
            free(data);
            return DBF_DELETED;
    }

    dbffield = &data[1];
    for (t = 0; t < dbh->db_nfields; t++) {
            fields[t].db_name = dbh->db_fields[t].db_name;
            fields[t].db_type = dbh->db_fields[t].db_type;
            fields[t].db_flen = dbh->db_fields[t].db_flen;
            fields[t].db_dec  = dbh->db_fields[t].db_dec;

            if (fields[t].db_type == 'C') {
				end = &dbffield[fields[t].db_flen - 1 ];
				i = fields[t].db_flen;
                while (( i > 0) && ((*end < 0x21) || (*end > 0x7E))) {
       	          	end--;
					i--;
                }
				strncpy(fields[t].db_contents, dbffield, i);
                fields[t].db_contents[i] = '\0';
            } else {
				end = dbffield;
				i = fields[t].db_flen;
                while (( i > 0) && ((*end < 0x21) || (*end > 0x7E))) {
					end++;
					i--;
				}
				strncpy(fields[t].db_contents, end, i);
				fields[t].db_contents[i] = '\0';
            }

            dbffield += fields[t].db_flen;
	}

    dbh->db_offset += dbh->db_rlen;

	return DBF_VALID;
}

field *dbf_build_record(dbhead *dbh) {
	int t;
	field	*fields;

	if (!(fields = (field *)calloc(dbh->db_nfields, sizeof(field)))) {
		return (field *)DBF_ERROR;
	}
	
	for ( t = 0; t < dbh->db_nfields; t++) {
		if (!(fields[t].db_contents =
			(u_char *)malloc(dbh->db_fields[t].db_flen + 1))) {
			for (t = 0; t < dbh->db_nfields; t++) {
				if (fields[t].db_contents != 0) {
					free(fields[t].db_contents);
					free(fields);
				}
				return (field *)DBF_ERROR;
			}
		}
	}

	return fields;
}

void dbf_free_record(dbhead *dbh, field *rec) {
	int t;

	for ( t = 0; t < dbh->db_nfields; t++) {
		free(rec[t].db_contents);
	}

	free(rec);
}
