/***[lex.c]*******************************************************[TAB=4]****\
*                                                                            *
* PHP/FI                                                                     *
*                                                                            *
* Copyright 1995,1996 Rasmus Lerdorf                                         *
*                                                                            *
*  This program is free software; you can redistribute it and/or modify      *
*  it under the terms of the GNU General Public License as published by      *
*  the Free Software Foundation; either version 2 of the License, or         *
*  (at your option) any later version.                                       *
*                                                                            *
*  This program is distributed in the hope that it will be useful,           *
*  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*  GNU General Public License for more details.                              *
*                                                                            *
*  You should have received a copy of the GNU General Public License         *
*  along with this program; if not, write to the Free Software               *
*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
*                                                                            *
\****************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <php.h>
#if HAVE_MMAP
#include <sys/mman.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <parse.h>
#if NSAPI
#include <sys/stat.h>
#endif

#if HAVE_MMAP
static caddr_t pa=NULL;
static long pa_pos=0L;
#else
static FILE *fp = NULL;
#endif
static int gfd = 0;
static long gsize = 0L;
static int state = 0;
static int lstate = 0;
static unsigned char inbuf[LINEBUFSIZE];	/* input line buffer  */
static int inpos=0; /* current position in inbuf */
static int inlength=0; /* current length of inbuf */
static int inmarker;  /* marked position in inbuf      */
static int tokenmarker; /* marked position in inbuf      */
static long SeekPos=0L;

static FileStack *top = NULL;

YYSTYPE yylval;

extern int PrintHeader;
extern int HeaderPrinted;

int yylex_linenumber = 0;

#if NSAPI
static filebuf *groupbuf=NULL;
extern Session *gsn;
extern Request *grq;
extern pblock  *gpb;
#endif

typedef struct _cmd_table_t {
	char *cmd;
	unsigned int token;
} cmd_table_t;

static cmd_table_t cmd_table[22][30] = {
	{ { NULL,0 } },        /* 0 */

	{ { NULL,0 } },        /* 1 */

	{ { "if", IF },        /* 2 */
	  { NULL,0 } }, 

	{ { "set", SET },      /* 3 */
	  { "max", MAX },
	  { "min", MIN },
	  { NULL,0 } }, 

	{ { "echo",ECHO },     /* 4 */
	  { "else",ELSE },
	  { "case",CASE },
	  { "msql",MSQL },
	  { "exit",EXIT },
	  { "exec",EXEC },
	  { "time",TIME },
	  { "date",DATE },
	  { "sort",SORT },
	  { "rand",RAND },
	  { NULL,0 } }, 

	{ { "endif",ENDIF },   /* 5 */
	  { "while",WHILE },
	  { "break",BREAK },
	  { "isset",ISSET },
	  { "count",COUNT },
	  { "crypt",CRYPT },
	  { "srand",SRAND },
	  { "sleep",SLEEP },
	  { "fopen",PHPFOPEN },
	  { "fgets",PHPFGETS },
	  { "fputs",PHPFPUTS },
	  { "fseek",PHPFSEEK },
	  { "ftell",PHPFTELL },
	  { "chdir",CHDIR },
	  { "chmod",CHMOD },
	  { "chown",CHOWN },
	  { "chgrp",CHGRP },
	  { "mkdir",MKDIR },
	  { NULL,0 } }, 

	{ { "elseif",ELSEIF }, /* 6 */
	  { "switch",SWITCH }, 
	  { "strlen",STRLEN }, 
	  { "strval",STRVAL }, 
	  { "intval",INTVAL }, 
	  { "strtok",STRTOK }, 
	  { "strstr",STRSTR }, 
	  { "strchr",STRCHR }, 
	  { "substr",SUBSTR }, 
	  { "system",SYSTEM }, 
	  { "header",HEADER },
	  { "gmdate",GMDATE }, 
	  { "dblist",DBLIST }, 
	  { "unlink",UNLINK }, 
	  { "rename",RENAME }, 
	  { "putenv",PUTENV }, 
	  { "mktime",MKTIME }, 
	  { "fclose",PHPFCLOSE }, 
	  { "rewind",PHPREWIND }, 
	  { "bindec",BINDEC }, 
	  { "decbin",DECBIN }, 
	  { "hexdec",HEXDEC }, 
	  { "dechex",DECHEX }, 
	  { "octdec",OCTDEC }, 
	  { "decoct",DECOCT }, 
	  { "usleep",USLEEP },
	  { NULL,0 } }, 

	{ { "default", DEFAULT }, /* 7 */
	  { "include", INCLUDE },
	  { "dbmopen", DBMOPEN },
	  { "strrchr", STRRCHR },
	  { "opendir", OPENDIR },
	  { "readdir", READDIR },
	  { "tempnam", TEMPNAM }, 
	  { "settype", SETTYPE }, 
	  { "gettype", GETTYPE }, 
	  { NULL,0 } },

	{ { "endwhile",ENDWHILE }, /* 8 */
	  { "dbmclose",DBMCLOSE },
	  { "dbmfetch",DBMFETCH },
	  { "gettotal",GETTOTAL },
	  { "gettoday",GETTODAY },
	  { "closedir",CLOSEDIR },
	  { "filesize",FILESIZE },
	  { "getmyuid",GETMYUID },
	  { "getmypid",GETMYPID },
	  { "imagegif",IMAGEGIF },
	  { "imagearc",IMAGEARC },
	  { NULL,0 } },

	{ { "endswitch", ENDSWITCH }, /* 9 */
	  { "reg_match", REG_MATCH },
	  { "dbminsert", DBMINSERT },
	  { "dbmexists", DBMEXISTS },
	  { "dbmdelete", DBMDELETE },
	  { "rewinddir", REWINDDIR },
	  { "fileperms", FILEPERMS },
	  { "fileinode", FILEINODE },
	  { "fileowner", FILEOWNER },
	  { "filegroup", FILEGROUP },
	  { "fileatime", FILEATIME },
	  { "filemtime", FILEMTIME },
	  { "filectime", FILECTIME },
	  { "getlogdir", GETLOGDIR },
	  { "getaccdir", GETACCDIR },
	  { "imageline", IMAGELINE },
	  { "imagefill", IMAGEFILL },
	  { "imagechar", IMAGECHAR },
	  { "doubleval", DOUBLEVAL }, 
	  { "securevar", SECUREVAR }, 
	  { NULL,0 } },        

	{ { "strtoupper", STRTOUPPER }, /* 10 */
	  { "strtolower", STRTOLOWER },
	  { "reg_search", REG_SEARCH },
	  { "dbmreplace", DBMREPLACE },
	  { "dbmnextkey", DBMNEXTKEY },
	  { "getlogfile", GETLOGFILE },
	  { "getlastref", GETLASTREF },
	  { "getlastmod", GETLASTMOD },
	  { "getmyinode", GETMYINODE },
	  { "getrandmax", GETRANDMAX },
	  { "setlogging", SETLOGGING },
	  { NULL,0 } },

	{ { "msql_result", MSQL_RESULT }, /* 11 */
	  { "reg_replace", REG_REPLACE },
	  { "dbmfirstkey", DBMFIRSTKEY },
	  { "getlasthost", GETLASTHOST },
	  { "imagecreate", IMAGECREATE },
	  { "imagecharup", IMAGECHARUP },
	  { "imagestring", IMAGESTRING },
	  { "setshowinfo", SETSHOWINFO },
	  { NULL,0 } },

	{ { "getlastemail", GETLASTEMAIL }, /* 12 */
	  { "msql_connect", MSQL_CONNECT },
	  { "msql_numrows", MSQL_NUMROWS },
	  { "msql_regcase", MSQL_REGCASE },
	  { "imagedestroy", IMAGEDESTROY },
	  { "imagepolygon", IMAGEPOLYGON },
	  { NULL,0 } }, 

	{ { "gethostbyaddr", GETHOSTBYADDR }, /* 13 */
	  { "gethostbyname", GETHOSTBYNAME },
	  { "getlastaccess", GETLASTACCESS },
	  { "msql_fieldlen", MSQL_FIELDLEN },
	  { "imagesetpixel", IMAGESETPIXEL },
	  { "imagestringup", IMAGESTRINGUP },
	  { NULL,0 } },

	{ { "getlastbrowser", GETLASTBROWSER }, /* 14 */
	  { "msql_fieldname", MSQL_FIELDNAME },
	  { "msql_fieldtype", MSQL_FIELDTYPE },
	  { "msql_numfields", MSQL_NUMFIELDS },
	  { "imagerectangle", IMAGERECTANGLE },
	  { "imageinterlace", IMAGEINTERLACE },
	  { NULL,0 } },

	{ { "msql_freeresult", MSQL_FREERESULT }, /* 15 */
	  { "msql_fieldflags", MSQL_FIELDFLAGS },
	  { NULL,0 } },

	{ { NULL,0 } }, /* 16 */

	{ { "imagefilltoborder", IMAGEFILLTOBORDER }, /* 17 */
	  { NULL,0 } }, 

	{ { "imagecolorallocate", IMAGECOLORALLOCATE }, /* 18 */
	  { "imagefilledpolygon", IMAGEFILLEDPOLYGON },
	  { NULL,0 } },

	{ { NULL,0 } }, /* 19 */

	{ { "imagefilledrectangle", IMAGEFILLEDRECTANGLE }, /* 20 */
	  { NULL,0 } }, 

	{ { "imagecolortransparent", IMAGECOLORTRANSPARENT }, /* 21 */
	  { NULL,0 } } 

};

void FilePush(char *fn, long file_size, int fd) {
	FileStack *new;

#if DEBUG
	Debug("FilePush called\n");
#endif
	new = emalloc(sizeof(FileStack));
#if HAVE_MMAP
	new->pa = pa;
#else
	new->fp = fp;
#endif
	new->size = file_size;
	new->fd = fd;
	new->pos = SeekPos + inpos;
	new->state = state;
	new->next = top;
	new->lineno = yylex_linenumber-1;
#if DEBUG
	Debug("Filename pushed onto file stack: [%s]\n",fn);
#endif
	new->filename = estrdup(fn);
	top = new;
}

#if HAVE_MMAP
caddr_t FilePop(void) {
#else
FILE *FilePop(void) {
#endif
	FileStack *s;

#if DEBUG
	Debug("FilePop() called\n");
#endif
#if HAVE_MMAP
	if(pa) {
#if DEBUG
		Debug("munmap'ing %ld bytes\n",gsize);
#endif
		munmap(pa,gsize);
		pa=NULL;
		close(gfd);
	}
#else
	if(fp) {
		fclose(fp);
		fp=NULL;
	}
#endif
#if DEBUG
	Debug("FilePop()\n");
#endif
	if(top) {
#if DEBUG
		Debug("Top is non-null, top->filename is [%s]\n",top->filename);
#endif
#if HAVE_MMAP
		pa = top->pa;
#else
		fp = top->fp;
#endif
		gfd = top->fd;
		gsize = top->size;
		state = top->state;
		yylex_linenumber = top->lineno;
		SetCurrentFilename(top->filename);
		SetCurrentFileSize(top->size);
#if HAVE_MMAP
		pa_pos = top->pos;
#else
		if(fp) fseek(fp,top->pos,SEEK_SET);
#endif
		s = top;
		top = top->next;
		free(s->filename);
		efree(s);
		inpos = -1;
		tokenmarker = 0;
#if HAVE_MMAP
		return(pa);
#else
		return(fp);
#endif
	} else return(NULL);
}

void Include(void) {
	Stack *s;
	int fd;
	char *ofn=NULL;
#ifndef NSAPI
	long file_size=0L;
#endif
	long ofile_size=0L;

	s = Pop();
	if(!s) {
		Error("Stack error in include");
		return;
	}
	if(s->strval) {
		ofn = estrdup(GetCurrentFilename());
		ofile_size = GetCurrentFileSize();
#if NSAPI
		fd = OpenFile(s->strval,0);
#else
		fd = OpenFile(s->strval,0,&file_size);
#endif
		if(fd>-1) {
			FilePush(ofn,ofile_size,gfd);
			gfd = fd;
#if NSAPI
			ParserInit(fd);
#else
			ParserInit(fd,file_size);
#endif
		} else {
			Error("Include error: %d %s",errno,strerror(errno));
		}
		if(ofn) efree(ofn);
	}
}

void outputchar(unsigned char ch) {
	if(PrintHeader && !HeaderPrinted) {
		fputs("Content-type: text/html\n\n",stdout);
		HeaderPrinted = 1;
	}	
	if(GetCurrentState(NULL)) {
		if(fputc(ch,stdout)==EOF) {
			/* browser has probably gone away */
			bailout();
		}
	}
}

/*
 * This reads a line into the input buffer if inpos is -1 and returns the 
 * first char.  If inpos is not -1, it returns the char at inpos
 */
unsigned char getnextchar(void) {
	static int length=0;
	unsigned char ch;
#if HAVE_MMAP
	int i=0;
	int cont=1;
#endif

try_again:
	if(inpos==-1 || inpos>=length) {
		/* Read the next line with something on it into the buffer */
		length=0;
		inlength=0;
		while(length==0) {
#if HAVE_MMAP
			SeekPos = pa_pos;
			if(SeekPos > gsize) {
#if DEBUG
				Debug("End of File\n");
#endif
				pa = FilePop();
				if(!pa) return(0);
				goto try_again;
			}
#else
			SeekPos = ftell(fp);
			if(SeekPos<0) {
#if DEBUG
				Debug("End of File\n");
#endif
				fp = FilePop();
				if(!fp) return(0);
				goto try_again;
			}
#endif
			yylex_linenumber++;
#if HAVE_MMAP
			cont=1;
			i=0;
			while(cont) {
				ch = *(pa+pa_pos+i);
				if(ch==10 || ch==13 || ch==0 || i==LINEBUFSIZE-1) cont=0;
				if(SeekPos+i >= gsize) {
#if DEBUG
					Debug("End of File\n");
#endif
					pa = FilePop();
#if DEBUG
					Debug("Back from FilePop\n");
#endif
					if(!pa) return(0);
#if DEBUG
					Debug("Trying again\n");
#endif
					goto try_again;
				}
				inbuf[i++] = ch;
			}
			length=i;
			inlength=i;
			pa_pos = SeekPos;
			pa_pos+=i;
#else
			if(!fgets(inbuf,LINEBUFSIZE-1,fp)) {
				if(feof(fp)) {
#if DEBUG
					Debug("End of File\n");
#endif
					fp = FilePop();
					if(!fp) return(0);
					goto try_again;
				}
			}
			length = strlen(inbuf);
			inlength=length;
#endif
			if(!length) outputchar('\n');
		}
		inpos=0;
	}	
	ch = inbuf[inpos++];
	return(ch);
}

void putback(unsigned char ch) {
	if(inpos>0) inpos--;
}

void output_from_marker(void) {
	int i;
	
	i=inmarker;
	while(i<inpos) {
		outputchar(inbuf[i++]);
	}
}

unsigned char *MakeToken(char *string, int len) {
	unsigned char *s;

	s = (unsigned char *)malloc(sizeof(unsigned char)*(len+1));
	memcpy(s,string,len);	
	*(s+len) = '\0';	
	return(s);
}

int CommandLookup(int cmdlen) {
	int i=0;

	while(cmd_table[cmdlen][i].cmd) {
		if(!strncasecmp(&inbuf[tokenmarker],cmd_table[cmdlen][i].cmd,cmdlen)) {
			yylval = (YYSTYPE) MakeToken(&inbuf[tokenmarker],cmdlen);
			return(cmd_table[cmdlen][i].token);
		}	
		i++;
	}
	return(0);
}
	
int yylex(void) {
	register unsigned char c;
	static int inIf=-1;
	static int inWhile=-1;
	int tokenlen=0;
	unsigned char *s;
	char temp[8];
	static int NewExpr=1;
	int bs=0;

	if(lstate==99) return(0);
	if(lstate==20 && inIf==-1 && inWhile==-1) {
		ClearStack();
		ErrorPrintState(1);
	}

	while(1) switch(state) {
		case 0:  /* start of token '<' gets us to state 1 */
			lstate=0;
			c = getnextchar();
			if(!c) { 
				state=99; break; 
			}
			if(c!='<') {
				outputchar(c);	
				break;
			}
			inmarker=inpos-1;
			/* fall-through */

		case 1:	/* '?' gets us to state 2 */
			lstate=1;
			c = getnextchar();
			if(!c) { state=99; break; }
			if(c!='?') {
				putback(c);	
				output_from_marker();
				state=0;
				break;
			}	
			NewExpr=1;
			/* fall-through */
			state=2;

		case 2: /* Start of  a command - [a-z][A-Z] gets us to state 3 */
				/* Start of a number  - [0-9] takes us to state 10 */
			lstate=2;
			c = getnextchar();
			if(!c) { state=99; break; }
			if(c==VAR_INIT_CHAR) { 
				state=40; 
				tokenlen=0;
				tokenmarker=inpos;
				break; 
			}
			if(c=='>') { state=20; break; }
			if(c=='<') { state=21; break; }
			if(c==' ' || c=='\t' || c=='\n') break;
			if(c=='\'') { 
				state = 9;
				tokenlen=0;
				tokenmarker=inpos;
				break;
			}
			if(c=='\"') {
				state=30;
				tokenlen=0;
				tokenmarker=inpos;
				break;
			}	
			if(isdigit(c)) {
				state=10;
				tokenlen=1;
				tokenmarker=inpos-1;
				break;
			}
			if(c=='.') {
				state=11;
				tokenlen=1;
				tokenmarker=inpos-1;
				break;
			}
			if(c=='@') { return(c); }
			if(c==';') { 
				NewExpr=1;
				return(c); 
			}
			if(c=='!') { state=12; break; }
			if(c=='&') { state=13; break; }
			if(c=='|') { state=14; break; }
			if(c=='=') { state=15; break; }	
			if(c=='+') { state=16; break; }
			if(c=='-') { state=17; break; }
			if(c=='/') { state=18; break; }
			if(c=='%') { state=8; break; }
			if(c=='*') { state=19; break; }
			if(c==',') { 
				NewExpr=1;
				return(','); 
			}
			if(c=='(') {
				NewExpr=1;
				if(inIf>-1) inIf++;	
				if(inWhile>-1) inWhile++;	
				return(c);
			}
			if(c==')') {
				if(inIf>-1) inIf--;
				if(inIf==0) inIf=-1;
				if(inWhile>-1) inWhile--;
				if(inWhile==0) inWhile=-1;
				NewExpr=0;
				return(c);
			}	
			if(c==']') return(c);
			if(!isalpha(c) && c != '_') {
				putback(c);	
				output_from_marker();
				state=0;
				break;
			}
			tokenlen=1;
			tokenmarker=inpos-1;
			/* fall-through */

		case 3: /* continue command - non [a-z][A-Z] gets us to state 4 */
			lstate=3;
			while(isalpha((c=getnextchar())) || c=='_') tokenlen++;
			if(!c) { state=99; break; }
			putback(c);
			
		case 4: /* command finished */
			lstate=4;
			if(tokenlen > MAX_CMD_LEN) {
				/* unrecognized command */
				output_from_marker();
				state=0;
				break;
			}
			state=2;
			if(tokenlen==2 && !strncasecmp(&inbuf[tokenmarker],"if",2)) {
				inIf++;
			} else if(tokenlen==6 && !strncasecmp(&inbuf[tokenmarker],"elseif",6)) {
				inIf++;		
			} else if(tokenlen==5 && !strncasecmp(&inbuf[tokenmarker],"while",5)) {
				inWhile++;
				if(GetCurrentState(NULL)) {
					WhilePush(SeekPos,tokenmarker,yylex_linenumber-1);
				}
			}
			NewExpr=1;
			return(CommandLookup(tokenlen));
			break;

		case 8: /* % */
			lstate=8;
			state=2;
			NewExpr=1;
			return('%');
			break;

		case 9: /* 'c' */
			lstate=9;
			NewExpr=0;
			while(1) {
				c=getnextchar();
				if(c=='\\') {
					bs=bs?0:1;
					tokenlen++;
					continue;
				}
				if(bs) {
					tokenlen++;
					bs=0;
					continue;
				}	
				if(c=='\'') break;
				tokenlen++;
			}
/*
			while((c=getnextchar())!='\'' 
				|| (c=='\'' && inbuf[inpos-2]=='\\')) tokenlen++;
*/
			if(!c) { state=99; break; }
			s = (unsigned char *) MakeToken(&inbuf[tokenmarker],tokenlen);
			if(s && *s) {
				sprintf(temp,"%d",(int)*s);
			} else strcpy(temp,"0");
			yylval = (YYSTYPE) MakeToken(temp,strlen(temp));
			state = 2;
			return(LNUMBER);
			
		case 10: /* LNUMBER */
			lstate=10;
			NewExpr=0;
			while(isdigit((c=getnextchar()))) tokenlen++;
			if(!c) { state=99; break; }
			if(c=='.') { tokenlen++; state=11; break; }
			putback(c);
			yylval = (YYSTYPE) MakeToken(&inbuf[tokenmarker],tokenlen);
			state = 2;
			return(LNUMBER);

		case 11: /* Double */
			lstate=11;
			NewExpr=0;
			while(isdigit((c=getnextchar()))) tokenlen++;
			if(!c) { state=99; break; }
			putback(c);
			yylval = (YYSTYPE) MakeToken(&inbuf[tokenmarker],tokenlen);
			state = 2;
			return(DNUMBER);

		case 12: /* ! */
			lstate=12;
			NewExpr=1;
			c = getnextchar();
			state=2;
			if(c=='=') return(COND_NE);
			else {
				putback(c);
				return(NOT);
			}
			break;

		case 13: /* & */
			lstate=13;
			NewExpr=1;
			c = getnextchar();
			state=2;
			if(c=='&') return(LOG_AND);
			else if(c=='=') return(ANDEQ);
			else {
				putback(c);
				return('&');
			}
			break;

		case 14: /* | */
			lstate=14;
			NewExpr=1;
			c = getnextchar();
			state=2;
			if(c=='|') return(LOG_OR);
			else if(c=='=') return(OREQ);
			else {
				putback(c);
				return('|');
			}
			break;
			
		case 15: /* = */
			lstate=15;
			NewExpr=1;
			c = getnextchar();
			state=2;
			if(c=='=') return(COND_EQ);
			else {
				putback(c);
				return('=');
			}
			break;

		case 16: /* + */
			lstate=16;
			c = getnextchar();
			state=2;
			if(c=='+') return(INC);
			else if(c=='=') return(PEQ);
			else {
				putback(c);
				NewExpr=1;
				return('+');
			}
			break;

		case 17: /* - */
			lstate=17;
			c = getnextchar();
			state=2;
			if(c=='-') return(DEC);
			else if(c=='=') return(MEQ);
			else {
				putback(c);
				if(NewExpr) {
					NewExpr=0;
					return(NEG);
				} else {
					NewExpr=1;
					return('-');
				}
			}
			break;

		case 18: /* / */
			lstate=18;
			c = getnextchar();
			state=2;
			if(c=='*') {
				while(1) {
					c=getnextchar();
					if(c=='/' && inbuf[inpos-2]=='*') break;
					tokenlen++;
				}
				return(COMMENT);
			} else {
				putback(c);
				NewExpr=1;
				return('/');
			}
			break;

		case 19: /* * */
			lstate=19;
			state=2;
			NewExpr=1;
			return('*');
			break;
	
		case 20: /* > */
			lstate=20;
			NewExpr=1;
			if(inIf>-1 || inWhile>-1) {
				state=2;
				c=getnextchar();
				if(c=='=') return(COND_GE);
				putback(c);
				return(COND_GT);
			}
			state=0;
			return(END_TAG);

		case 21: /* < */
			lstate=21;
			NewExpr=1;
			if(inIf>-1 || inWhile>-1) {
				state=2;
				c=getnextchar();
				if(c=='=') return(COND_LE);
				putback(c);
				return(COND_LT);
			}
			return('<');

		case 30: /* string */
			lstate=30;
			NewExpr=0;
			while(1) {
				c=getnextchar();
				if(c=='\\') {
					bs=bs?0:1;
					tokenlen++;
					continue;
				}
				if(bs) {
					tokenlen++;
					bs=0;
					continue;
				}	
				if(c=='\"') break;
				tokenlen++;
			}
/*
			while((c=getnextchar())!='\"'
				|| (c=='\"' && inbuf[inpos-2]=='\\')) {
#if DEBUG
				Debug("Fetching string %c[%c%c%c]\n",c,inbuf[inpos-2],inbuf[inpos-1],inbuf[inpos]);
#endif
				tokenlen++;
			}
*/
			if(!c) { state=99; break; }
			yylval = (YYSTYPE) MakeToken(&inbuf[tokenmarker],tokenlen);
			state = 2;
			return(STRING);

		case 40: /* Variable */
			lstate=40;
			NewExpr=0;
			while((c=getnextchar()) && ((isalnum(c) || c=='_') || (c==VAR_INIT_CHAR && tokenlen==0))) tokenlen++;
			if(!c) { state=99; break; }
			yylval = (YYSTYPE) MakeToken(&inbuf[tokenmarker],tokenlen);
			state = 2;
			while(c==' ' || c=='\t' || c=='\n') c=getnextchar();
			if(c=='[') return(ARRAY);
			putback(c);
			return(VAR);

		case 99: /* EOF reached */
			lstate=99;
			state=0;
			return(END_OF_FILE);		
	}
	return(0);
} /* yylex */

void yyerror(char *string) {
#if DEBUG
	Debug("Printing error [%s] %d\n",string,state);
#endif
	if(string) Error("%s", string);
	else Error("Unknown error");
}

#if NSAPI
int ParserInit(SYS_FILE fd) {
	struct stat finfo;
#else
void ParserInit(int fd, long file_size) {
#endif
#if NSAPI
    /* Use server native buffered I/O on the file */
	groupbuf = filebuf_open_nostat(fd, FILE_BUFFERSIZE, &finfo);
	if(!groupbuf) {
		protocol_status(gsn, grq, PROTOCOL_FORBIDDEN, NULL);
		log_error(LOG_WARN,"send-file",gsn,grq,"error opening buffer (%s)",system_errmsg(fd));
		system_fclose(fd);
		return REQ_ABORTED;
	}
	protocol_status(gsn, grq, PROTOCOL_OK, NULL);
#else
#if HAVE_MMAP
#if DEBUG
	Debug("mmap'ing %ld bytes\n",file_size);
#endif
	pa = mmap(NULL, file_size, PROT_READ, MAP_PRIVATE, fd, 0);
	pa_pos = 0L;
#else
	fp = fdopen(fd,"r");
#endif
#endif
#if NSAPI
	gsize = finfo.st_size;
#else
	gsize = file_size;
#endif
	gfd = fd;
	state = 0;
	inpos = -1;
	tokenmarker=0;
	yylex_linenumber = 0;
#if NSAPI
	return(0);
#endif
}	

void WhileAgain(long seekpos, int offset, int lineno) {
#if HAVE_MMAP
	pa_pos = seekpos+offset;
#else
	fseek(fp,seekpos+offset,SEEK_SET);
#endif
	
	yylex_linenumber = lineno;
	tokenmarker = 0;
	state=2;
}

void Exit(void) {
#if DEBUG
	Debug("Exit called\n");
#endif
#if HAVE_MMAP
	if(pa) {
#if DEBUG
		Debug("munmap'ing %ld bytes\n",gsize);
#endif
		munmap(pa,gsize);
		pa=NULL;
		close(gfd);
	}
#else
	if(fp) fclose(fp);
#endif
	dbmCloseAll();
	MsqlClose();
	if(getshowinfo() && getlogging()) ShowPageInfo();
#if DEBUG
	CloseDebug();
#endif
	exit(0);
}

char *GetCurrentLexLine(int *pos, int *length) {
	*pos = inpos;
	*length = inlength;
	return(inbuf);
}
