/* st_util.c - generic symbol table utility routines */

/*  Copyright 1993 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_st_util_c_sccsid[] = "@(#)st_util.c	1.5 04 Jun 1995 (UKC)";

#include <mtrprog/ifdefs.h>

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#ifdef __STDC__
#include <unistd.h>
#endif

#include <local/ukcprog.h>
#include <local/arg.h>
#include <mtrprog/genmergesort.h>
#include <mtrprog/strcache.h>
#include <mtrprog/alloc.h>

#include "ups.h"
#include "symtab.h"
#include "ci.h"
#include "target.h"
#include "st.h"
#include "obj_bpt.h"
#include "srcbuf.h"
#include "srcpath.h"

/*  Element in the linked list of global variable names and 
 *  addresses for a symbol table.
 */
struct addrlist_s {
	const char *al_name;
	taddr_t al_addr;
	var_t *al_var;
	struct addrlist_s *al_next;
};

static block_t *make_rootblock PROTO((void));

void
adjust_addrlist_addr_offset(addrlist, delta)
addrlist_t *addrlist;
long delta;
{
	addrlist_t *al;

	for (al = addrlist; al != NULL; al = al->al_next)
		al->al_addr += delta;
}

/*  Insert a new entry at the front of the globals address map.
 */
void
insert_global_addr(ap, p_addrlist, name, addr)
alloc_pool_t *ap;
addrlist_t **p_addrlist;
const char *name;
taddr_t addr;
{
	addrlist_t *al;

	al = (addrlist_t *)alloc(ap, sizeof(addrlist_t));
	al->al_name = name;
	al->al_addr = addr;
	al->al_var = NULL;
	al->al_next = *p_addrlist;
	*p_addrlist = al;
}

/*  Look up the address of global variable name.  Return the
 *  address, or 0 if name is not found.
 */
taddr_t
find_addr_in_addrlist(addrlist, name)
addrlist_t *addrlist;
const char *name;
{
	addrlist_t *al;

	for (al = addrlist; al != NULL; al = al->al_next)
		if (strcmp(al->al_name, name) == 0)
			return al->al_addr;
	return 0;
}

void
scan_addrlist(st, matchf, pat, addf, addf_arg)
symtab_t *st;
matchfunc_t matchf;
const char *pat;
void (*addf)PROTO((char *, var_t *));
char *addf_arg;
{
	addrlist_t *al;

	for (al = st->st_addrlist; al != NULL; al = al->al_next) {
		if (!(*matchf)(al->al_name, pat))
			continue;

		if (al->al_var == NULL) {
			var_t *v;

			v = ci_make_var(st->st_apool, al->al_name, CL_EXT,
					ci_code_to_type(TY_INT_ASSUMED),
					al->al_addr);
			v->va_language = LANG_UNKNOWN;
			al->al_var = v;
		}
		
		(*addf)(addf_arg, al->al_var);
	}
}

/*   Find the address of a global.  Return 0 if not found (this will happen
 *   for an extern with no definition).
 */
taddr_t
lookup_global_addr(st, name)
symtab_t *st;
const char *name;
{
	return find_addr_in_addrlist(st->st_addrlist, name);
}

symtab_t *
make_symtab(ap, path, sfiles, funclist, ops, data)
alloc_pool_t *ap;
const char *path;
fil_t *sfiles;
func_t *funclist;
sym_ops_t *ops;
char *data;
{
	symtab_t *st;
	func_t *f;
	fil_t *fil;
	
	st = (symtab_t *)alloc(ap, sizeof(symtab_t));
	
	st->st_apool = ap;
	st->st_path = alloc_strdup(ap, path);
	st->st_sfiles = sfiles;
	st->st_cblist = NULL;
	st->st_modules = NULL;
	st->st_funclist = funclist;
	st->st_addrlist = NULL;
	st->st_data = data;
	st->st_ops = ops;

	for (f = funclist; f != NULL; f = f->fu_next)
		f->fu_symtab = st;

	for (fil = sfiles; fil != NULL; fil = fil->fi_next)
		fil->fi_symtab = st;

	return st;
}

/*  Close down a symbol table.  This is called if we rerun a
 *  target and find that it uses a different shared library
 *  from the last run (rare, but could happen if the
 *  environment variable LD_LIBRARY_PATH is changed).
 *  If we extended ups to allow multiple binaries in the
 *  same run this routine would also get called.
 *
 *  Free all the resources and memory associated with this
 *  symbol table (the memory part is made easy by alloc()).
 */
void
close_symtab(st)
symtab_t *st;
{
	fil_t *fil;

	/*  Delete any breakpoints from this symbol table
	 */
	remove_symtab_breakpoints(st);

	/*  Close any source files we have open.
	 */
	for (fil = st->st_sfiles; fil != NULL; fil = fil->fi_next) {
		if (fil->fi_srcbuf != NULL)
			srcbuf_destroy(fil->fi_srcbuf);
	}
	
	free_cblist_info(st->st_cblist);

	st_close_symtab_data(st);

	/*  Free all storage associated with this symtab.
	 *  Vars, functions, common blocks, the lot.
	 */
	alloc_free_pool(st->st_apool);
}

block_t *
get_rootblock()
{
	static block_t *block = NULL;

	if (block == NULL)
		block = make_rootblock();
	
	return block;
}

static block_t *
make_rootblock()
{
	static struct {
		const char *regname;
		int regno;
	} regtab[] = {
		{ "$pc",	UPSREG_PC	},
		{ "$fp",	UPSREG_FP	},
		{ "$sp",	UPSREG_SP	},
	};
#ifdef PURIFY
	static alloc_pool_t *save_apool;
#endif
	alloc_pool_t *ap;
	type_t *type;
	block_t *block;
	int i;

	ap = alloc_create_pool();
#ifdef PURIFY
	save_apool = ap;
#endif

	block = ci_make_block(ap, (block_t *)NULL);

	type = ci_code_to_type(TY_INT);

	for (i = 0; i < sizeof regtab / sizeof *regtab; ++i) {
		var_t *v;

		v = ci_make_var(ap, regtab[i].regname, CL_REG, type,
						     (taddr_t)regtab[i].regno);

		/*  This is pretty bogus.  The problem is that the register
		 *  values for these registers are stored in a data
		 *  structure in our address space, and we don't have an
		 *  address in the target to hand.  Thus we pass back
		 *  (in get_reg_addr) the address in our address space,
		 *  and set the flag below to make it look there.
		 */
		v->va_flags |= VA_IS_CI_VAR;

		v->va_next = block->bl_vars;
		block->bl_vars = v;
	}

	return block;
}

void
iterate_over_vars_of_block(block, func, args)
block_t *block;
void (*func)PROTO((var_t *v, char *c_args));
char *args;
{
	block_t *bl;
	var_t *v;

	if (block != NULL) {
		for (bl = block->bl_blocks; bl != NULL; bl = bl->bl_next)
			iterate_over_vars_of_block(bl, func, args);
		for (v = block->bl_vars; v != NULL; v = v->va_next)
			(*func)(v, args);
	}
}

/*  Return an lno pointer given a function and an address.
 *  Return NULL if no line number information is available.
 *
 *  This function searches for the first lno that has an address larger
 *  than text_addr, and returns the one before that.
 */
bool
addr_to_lno(f, text_addr, p_ln)
func_t *f;
taddr_t text_addr;
lno_t **p_ln;
{
	lno_t *ln, *last;
	taddr_t min_addr;

	if ((f->fu_flags & FU_BAD) != 0)
		return FALSE;

	if (get_min_bpt_addr(f, &min_addr) != 0 || text_addr < min_addr)
		return FALSE;

	last = NULL;
	for (ln = FU_LNOS(f); ln != NULL; ln = ln->ln_next) {
		if (ln->ln_addr > text_addr)
			break;
		last = ln;
	}

	if (last == NULL)
		return FALSE;

	*p_ln = last;
	return TRUE;
}

void
addr_to_fil_and_lnum(f, addr, p_fil, p_lnum)
func_t *f;
taddr_t addr;
fil_t **p_fil;
int *p_lnum;
{
	lno_t *lno;
	
	if (addr_to_lno(f, addr, &lno)) {
		*p_fil = lno->ln_fil;
		*p_lnum = lno->ln_num;
	}
	else {
		*p_fil = NULL;
		*p_lnum = 0;
	}
}

/*  Return the lnum that matches address text_addr.
 *
 *  This function is used to matching RBRAC addresses to line numbers.
 *  Some compilers emit several lnos for a for loop, all with the same
 *  source line number.  For the end of a block we want a source line
 *  number near the end of the loop, hence the odd way this routine works.
 */
int
rbrac_addr_to_lnum(f, text_addr)
func_t *f;
taddr_t text_addr;
{
	int max_lnum;
	lno_t *ln;
	taddr_t min_addr;

	if (get_min_bpt_addr(f, &min_addr) != 0 || text_addr < min_addr)
		return 0;

	max_lnum = 0;
	for (ln = FU_LNOS(f); ln != NULL; ln = ln->ln_next) {
		if (ln->ln_addr > text_addr)
			break;
		if (ln->ln_num > max_lnum)
			max_lnum = ln->ln_num;
	}
	return max_lnum;
}

/*  Map line number lnum to an address, given that the line is within
 *  function f.
 *
 *  Return 0 if there is no line number information, 1 if we can't find
 *  the line number.
 */
taddr_t
lnum_and_fil_to_addr(f, fil, lnum)
func_t *f;
fil_t *fil;
int lnum;
{
	lno_t *ln;

	if ((f->fu_flags & FU_NOSYM) || FU_LNOS(f) == NULL)
		return 0;
	
	for (ln = FU_LNOS(f); ln != NULL; ln = ln->ln_next) {
		if (ln->ln_fil == fil && ln->ln_num == lnum)
			return ln->ln_addr;
	}
	
	return 1;
}

#ifdef ARCH_MIPS
/*  There is no DS3100 version of get_startup_code() - the stuff is
 *  set up by skim_te_symtab() in st_te.c.
 */
#endif /* ARCH_MIPS */

/*  Determine the minimum displacement into the function code at which a
 *  breakpoint can be placed. This should be after the registers have
 *  been saved if they are saved. Return the address.
 *
 *  If the function has line number information, just return the address
 *  of the first line.
 */
int
get_min_bpt_addr(f, p_addr)
func_t *f;
taddr_t *p_addr;
{
	if (FU_LNOS(f) != NULL) {
		*p_addr = FU_LNOS(f)->ln_addr;
		return 0;
	}

	return st_get_min_bpt_addr(f, p_addr);
}

/*  Set *p_addr to the address corresponding to line lnum in function f.
 *  Return 0 for success, -1 and an error message otherwise.
 */
int
map_lnum_to_addr(f, fil, lnum, p_addr)
func_t *f;
fil_t *fil;
int lnum;
taddr_t *p_addr;
{
	taddr_t addr, min_addr;

	if ((addr = lnum_and_fil_to_addr(f, fil, lnum)) == 1) {
		errf("No executable code at line %d of %s", lnum,
							f->fu_fil->fi_name);
		return -1;
	}
	
	if (addr == 0) {
		errf("No line number information for %s", f->fu_name);
		return -1;
	}
	
	if (get_min_bpt_addr(f, &min_addr) != 0)
		return -1;

	*p_addr = (addr > min_addr) ? addr : min_addr;
	return 0;
}

/*  Open the source file for fil if not already open.
 */
bool
open_source_file(fil)
fil_t *fil;
{
	if (fil->fi_srcbuf == NULL) {
		fil->fi_srcbuf = srcpath_visit_file(fil->fi_symtab->st_apool,
						    fil->fi_path_hint,
						    fil->fi_name);
	}
		
	return fil->fi_srcbuf != NULL;	
}

bool
have_source_file(fil)
fil_t *fil;
{
	return fil->fi_srcbuf != NULL ||
	       srcpath_file_exists(fil->fi_path_hint, fil->fi_name);
}
