/* obj_bpt.c - breakpoint object handling */

/*  Copyright 1991 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_obj_bpt_c_sccsid[] = "@(#)obj_bpt.c	1.42 24 May 1995 (UKC)";

#include <mtrprog/ifdefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <unistd.h>

#include <local/wn.h>
#include <local/ukcprog.h>
#include <mtrprog/utils.h>
#include <local/obj/obj.h>
#include <local/edit/edit.h>

#include "objtypes.h"
#include "obj_util.h"
#include "ups.h"
#include "symtab.h"
#include "ci.h"
#include "ui.h"
#include "exec.h"
#include "target.h"
#include "breakpoint.h"
#include "st.h"
#include "data.h"
#include "obj_bpt.h"
#include "menudata.h"
#include "debug.h"
#include "tdr.h"
#include "config.h"
#include "srcbuf.h"
#include "reg.h"
#include "srcwin.h"
#include "state.h"
#include "util.h"
#include "expr.h"

typedef struct bpdescst {
	Editblock *editblock;
	fil_t *fil;
	int lnum;
	bool code_bad;
	bool data_needs_initialising;
	bool code_has_func_calls;
	bool want_save;
	parse_id_t parse_id;
	machine_t *machine;

	func_t *func;
	breakpoint_t *breakpoint;
} bpdesc_t;

typedef struct {
	FILE *ba_fp;
	char *ba_path;
} bptargs_t;

#define HAS_CODE(bd)	((bd)->parse_id != NULL || (bd)->code_bad)

static void bpfunc_draw PROTO((struct drawst *dets));
static int move_bpt PROTO((objid_t obj, char *new_text, int edit_lnum,
			   const char **p_display_string));
static void show_bpt_source PROTO((bpdesc_t *bd, bool beep_on_failure));
static void bpfunc_edit PROTO((struct drawst fdets));
static void bplnum_edit PROTO((struct drawst fdets));
static char *build_source_text PROTO((fil_t *fil, int lnum, const char *text));
static void update_editblock PROTO((bpdesc_t *bd, bool is_bpt, char *text));
static int set_initialise_data_flag PROTO((objid_t obj, fval_t unused_arg));
static int read_data PROTO((taddr_t addr, void *buf, size_t nbytes));
static int write_data PROTO((taddr_t addr, const void *buf, size_t nbytes));

static taddr_t get_bpt_addr PROTO((objid_t obj));
static int save_object_to_file PROTO((const char *path, FILE *fp,
					   bpdesc_t *bd));
static int get_bp_lnum PROTO((func_t *f, const char *lnumstr,
			      const char *offsetstr));
static bool get_code_text PROTO((FILE *fp, int *p_lnum, ebuf_t *eb));
static void load_breakpoints PROTO((void));
static int skip_to_close_curly PROTO(( FILE *fp, int *p_lnum));
static void handle_editblock_action PROTO((Editblock *editblock, 
                                           Editblock_action action));
static bool handle_editblock_change PROTO((Editblock *eb, const char *text,
					   bool force, size_t *p_error_point));
static bool bpfunc_quitfunc PROTO((objid_t obj, char *new_text, 
                                   const char **p_display_string));
static bool bplnum_quitfunc PROTO((objid_t obj, char *new_text, 
                                   const char **p_display_string));
static bool fil_and_lnum_to_bdobj PROTO((fil_t *fil, int lnum, objid_t *p_obj));
static bpdesc_t *make_bd PROTO((fil_t *fil, int lnum, func_t *func));
static void save_interpreted_code_to_file PROTO((const char *path, FILE *fp, 
                                                 bpdesc_t *bd));
static void save_breakpoint_to_file PROTO((const char *path, FILE *fp, 
                                           bpdesc_t *bd));

/*  Maximum length of a function name. If you change this you must also
 *  change the Bpt_format string to match.
 */
#define LEN_BP_FNAME 	32

const char Bpt_format[] = "%8cs:%[-]32cb line:%[B]5cn\n";

/*  Field numbers for breakpoints
 */
#define FN_BPT_TYPESTR	0
#define FN_BPT_FNAME	1
#define FN_BPT_LNUM	2
#define FN_BPT_LAST	3

/*  Flist terminator for breakpoint fields for set_all_fields().
 *  Can't use NULL as one of the fields is an int which can legitimately
 *  have the value 0.
 */
#define BPT_FLIST_TERMINATOR	((fval_t)-1)

fnamemap_t Bpt_fnamemap[] = {
	{ FN_BPT_FNAME,	"function-name",	FALSE,	bpfunc_quitfunc	},
	{ FN_BPT_LNUM,	"lnum",			FALSE,	bplnum_quitfunc	},
	{ 0,		NULL,			FALSE,	NULL		},
};

const char Bphead_format[] = "Breakpoints\n";

#define BPHEAD_OBJCODE	((objid_t)Bphead_format)

static const char Stop_keyword[] = "#stop";

fdef_t Bpt_fdefs[] = {
	{ 'b', bpfunc_draw, bpfunc_edit, NULL	},
	{ 'n', n_draw, bplnum_edit, NULL	},
	{ 0, NULL, NULL, NULL			},
};

static const char *Statefile_path = NULL;

void
set_breakpoint_statefile_path(path)
const char *path;
{
	Statefile_path = path;
}

static void
load_breakpoints()
{
	static char *lastpath = NULL;
	char *path;
	bool junk;

	if (lastpath == NULL)
		lastpath = strsave("");
	
	if (prompt_for_string("filename", "Load breakpoints from file: ",
			      lastpath, &path) != 0)
		return;

	if (strcmp(path, lastpath) != 0) {
		free(lastpath);
		lastpath = strsave(path);
	}

	read_config_file(path, FALSE, TRUE, TRUE, FALSE, &junk);
}

static bool
fil_and_lnum_to_bdobj(fil, lnum, p_obj)
fil_t *fil;
int lnum;
objid_t *p_obj;
{
	objid_t obj;

	for (obj = get_code(BPHEAD_OBJCODE, OBJ_CHILD);
	     obj != NULL;
	     obj = get_code(obj, OBJ_NEXT)) {
		bpdesc_t *bd;

		bd = (bpdesc_t *)obj;
		
		if (bd->fil == fil && bd->lnum == lnum) {
			*p_obj = obj;
			return TRUE;
		}
	}

	return FALSE;
}

int
save_all_breakpoints_to_file(path, fp)
const char *path;
FILE *fp;
{
	objid_t obj;

	for (obj = get_code(BPHEAD_OBJCODE, OBJ_CHILD);
	     obj != NULL;
	     obj = get_code(obj, OBJ_NEXT)) {
		bpdesc_t *bd;

		bd = (bpdesc_t *)obj;
		
		if (bd->want_save && save_object_to_file(path, fp, bd) != 0) {
			return -1;
		}
	}

	return 0;
}

static int
save_object_to_file(path, fp, bd)
const char *path;
FILE *fp;
bpdesc_t *bd;
{
	if (bd->breakpoint != NULL)
		save_breakpoint_to_file(path, fp, bd);
	else
		save_interpreted_code_to_file(path, fp, bd);

	fflush(fp);

	if (ferror(fp)) {
		failmesg("Error saving object to", "file", path);
		return -1;
	}

	return 0;
}

static void
save_interpreted_code_to_file(path, fp, bd)
const char *path;
FILE *fp;
bpdesc_t *bd;
{
	fprintf(fp, "code %s %d {\n", bd->fil->fi_name, bd->lnum);
	dump_text(srcbuf_get_editblock_text(bd->editblock), fp);
	fputs("}\n", fp);
}

static void
save_breakpoint_to_file(path, fp, bd)
const char *path;
FILE *fp;
bpdesc_t *bd;
{
	func_t *f;

	f = bd->func;
	
	fprintf(fp, "breakpoint %s", f->fu_name);
	
	if (f->fu_fil != NULL) {
		lno_t *ln;
		int n_code_lines;

		n_code_lines = 0;
		for (ln = FU_LNOS(f); ln != NULL; ln = ln->ln_next) {
			if (ln->ln_num == bd->lnum)
				break;
			++n_code_lines;
		}
		
		fprintf(fp, " %s %d [+%d]",
			f->fu_fil->fi_name, bd->lnum, n_code_lines);
	}
	
	if (HAS_CODE(bd)) {
		fputs(" {\n", fp);
		dump_text(srcbuf_get_editblock_text(bd->editblock), fp);
		fputc('}', fp);
	}

	fputc('\n', fp);
}

int
handle_add_breakpoint_command(cmd, args, nargs, from_statefile, ignore,
			      fp, p_lnum)
const char *cmd;
char **args;
int nargs;
bool from_statefile, ignore;
FILE *fp;
int *p_lnum;
{
	func_t *f;
	const char *funcname, *filename, *lnumstr, *offsetstr;
	bool ok, have_code;
	fil_t *fil;
	int lnum;
	objid_t obj;

	have_code = nargs > 1 && strcmp(args[nargs - 1], "{") == 0;

	if (have_code)
		--nargs;

	if (nargs < 1 || nargs > 4) {
		errf("Usage: breakpoint funcname [filename [lnum [offset]]] [{]");
		return -1;
	}

	funcname = args[0];
	filename = (nargs > 1) ? args[1] : NULL;
	lnumstr = (nargs > 2) ? args[2] : NULL;
	offsetstr = (nargs > 3) ? args[3] : NULL;
	
	if (filename == NULL) {
		if (find_func_by_name(funcname, &f) != 0) {
			skip_to_close_curly(fp, p_lnum);
			return -1;
		}

		fil = f->fu_fil;
		lnum = -1;
	}
	else {
		if ((fil = name_to_fil(filename)) == NULL ||
		    (f = name_and_fil_to_func(funcname, fil)) == NULL) {
			/*  Function may have moved to another file.
			 */
			if (find_func_by_name(funcname, &f) != 0) {
				skip_to_close_curly(fp, p_lnum);
				return -1;
			}
			fil = f->fu_fil;
		}

		if (lnumstr == NULL)
			lnum = -1;
		else
			lnum = get_bp_lnum(f, lnumstr, offsetstr);
	}

	if (ignore) {
		if (have_code && skip_to_close_curly(fp, p_lnum) != 0)
			return -1;
		return 0;
	}

	if (from_statefile && fil_and_lnum_to_bdobj(f->fu_fil, lnum, &obj))
		remove_object(obj, OBJ_SELF);
	
	if ((obj = add_breakpoint_object(f, fil, lnum)) == NULL) {
		if (have_code)
			skip_to_close_curly(fp, p_lnum);
		return -1;
	}

	((bpdesc_t *)obj)->want_save = from_statefile;

	if (have_code) {
		ebuf_t *eb;
		
		eb = ebuf_create(TRUE);

		ok = get_code_text(fp, p_lnum, eb) &&
		     change_bd_text(obj, ebuf_get(eb, (int *)NULL));

		if (!ok)
			remove_object(obj, OBJ_SELF);

		ebuf_free(eb);
	}
	else {
		ok = TRUE;
	}

	return ok ? 0 : -1;
}

int
handle_add_code_command(cmd, args, nargs, from_statefile, ignore, fp, p_lnum)
const char *cmd;
char **args;
int nargs;
bool from_statefile, ignore;
FILE *fp;
int *p_lnum;
{
	fil_t *fil;
	const char *filename, *lnumstr;
	int lnum;
	objid_t obj;
	ebuf_t *eb;
	char *text, *ends;
	
	if (nargs != 3 || strcmp(args[2], "{") != 0) {
		errf("Usage: code filename lnum {");
		return -1;
	}

	filename = args[0];
	lnumstr = args[1];
	
	if ((fil = name_to_fil(filename)) == NULL) {
		skip_to_close_curly(fp, p_lnum);
		return -1;
	}
	
	lnum = strtol(lnumstr, &ends, 10);

	if (ends == lnumstr || *ends != '\0') {
		errf("Line number `%s' is not a decimal integer", lnumstr);
		skip_to_close_curly(fp, p_lnum);
		return -1;
	}

	if (ignore)
		return skip_to_close_curly(fp, p_lnum);

	eb = ebuf_create(TRUE);
	
	if (!get_code_text(fp, p_lnum, eb)) {
		ebuf_free(eb);
		skip_to_close_curly(fp, p_lnum);
		return -1;
	}
	text = ebuf_get(eb, (int *)NULL);
	
	if (from_statefile && fil_and_lnum_to_bdobj(fil, lnum, &obj))
		remove_object(obj, OBJ_SELF);

	obj = add_interpreted_code(fil, lnum, strsave(text));
	ebuf_free(eb);
	
	if (obj != NULL)
		((bpdesc_t *)obj)->want_save = from_statefile;

	return (obj != NULL) ? 0 : -1;
}

static int
skip_to_close_curly(fp, p_lnum)
FILE *fp;
int *p_lnum;
{
	char *line;

	while ((line = fpgetline(fp)) != NULL) {
		if (strcmp(line, "}") == 0)
			return 0;
		++*p_lnum;
	}

	errf("Missing closing `}' in breakpoint code");
	return -1;
}

static bool
get_code_text(fp, p_lnum, eb)
FILE *fp;
int *p_lnum;
ebuf_t *eb;
{
	int code_lnum;
	char *line;

	code_lnum = *p_lnum;

	while ((line = fpgetline(fp)) != NULL) {

		++*p_lnum;

		if (strcmp(line, "}") == 0) {
			ebuf_add(eb, "", 1);
			return TRUE;
		}
		
		if (*line == '#')
			continue;
		if (*line == '\t')
			++line;

		ebuf_addstr(eb, line);
		ebuf_add(eb, "\n", 1);
	}

	*p_lnum = code_lnum;
	errf("Missing `}'");
	return FALSE;
}

static int
get_bp_lnum(f, lnumstr, offsetstr)
func_t *f;
const char *lnumstr, *offsetstr;
{
	int lnum;
	char *ends;
			
	lnum = strtol(lnumstr, &ends, 10);
	
	if (lnum < 0 || ends == lnumstr || *ends != '\0') {
		errf("Ignored illegal line number `%s'", lnumstr);
		lnum = -1;
	}

	if (offsetstr != NULL) {
		lno_t *ln;
		int i, count;
		char junkc;

		if (sscanf(offsetstr, "[+%d]%c", &count, &junkc) != 1)
			errf("Ignored malformed offset `%s'", offsetstr);
				
		ln = FU_LNOS(f);
		for (i = 0; i < count && ln != NULL; ++i)
			ln = ln->ln_next;

		if (ln != NULL)
			lnum = ln->ln_num;
		else
			errf("Ignored out of range offset `%s'", offsetstr);
	}

	return lnum;
}

const char *
bpt_getobjname(obj)
objid_t obj;
{
	static char *last = NULL;
	bpdesc_t *bd;

	bd = (bpdesc_t *)obj;

	if (bd->fil == NULL)
		return "[BLANK]";
	
	if (last != NULL)
		free(last);

	if (bd->breakpoint != NULL)
		last = strf("%s:%d", bd->func->fu_name, bd->lnum);
	else
		last = strf("file:%s:%d", bd->fil->fi_name, bd->lnum);

	return last;
}

static void
bpfunc_draw(dets)
register struct drawst *dets;
{
	wn_ttext(dets->dr_wn, (char *)dets->dr_fval, dets->dr_x, dets->dr_y,
						    dets->dr_fg, dets->dr_bg);
}

static taddr_t
get_bpt_addr(obj)
objid_t obj;
{
	bpdesc_t *bd;
	taddr_t addr;

	bd = (bpdesc_t *)obj;
	
	if (bd->lnum != 0) {
		addr = lnum_and_fil_to_addr(bd->func, bd->fil, bd->lnum);
	}
	else {
		if (get_min_bpt_addr(bd->func, &addr) != 0)
			panic("gmba failed in gba");
	}

	return addr;
}

/*  Build a source line for a breakpoint.  We duplicate any whitespace
 *  at the beginning of the source line following the breakpoint.
 */
static char *
build_source_text(fil, lnum, text)
fil_t *fil;
int lnum;
const char *text;
{
	size_t textsize, point, start, lim;
	char *obuf, *optr;
	Edit_buffer *buffer;
	
	if (!open_source_file(fil) ||
	    !srcbuf_file_lnum_to_point_range(fil->fi_srcbuf, lnum - 1,
					     &start, &lim)) {
		return strsave(text);
	}

	/*  Worst case is that the whole line is whitespace.
	 */
	textsize = strlen(text) + 1;
	optr = obuf = e_malloc((lim - start) + textsize);

	buffer = srcbuf_get_buffer(fil->fi_srcbuf);
	
	if (edit_find_char_forwards(buffer, start, lim, "! \t", &point)) {
		edit_copy_bytes(buffer, start, point, optr);
		optr += point - start;
	}
	
	memcpy(optr, text, textsize);
	return obuf;
}

void
recalculate_bpt_addrs(xp)
target_t *xp;
{
	objid_t obj;
	bpdesc_t *bd;

	obj = get_code(BPHEAD_OBJCODE, OBJ_CHILD);
	for (; obj != NULL; obj = get_code(obj, OBJ_NEXT)) {
		bd = (bpdesc_t *)obj;
		if (bd->breakpoint != NULL) {
			change_breakpoint_address(xp, bd->breakpoint,
						  get_bpt_addr(obj));
		}

		if (bd->code_has_func_calls) {
			ci_free_machine(bd->machine);
			bd->machine = recompile_code(bd->parse_id,
							(char *)NULL);
		}
	}
}

bool
change_bd_text(obj, text)
objid_t obj;
const char *text;
{
	bpdesc_t *bd;
	size_t junk;

	bd = (bpdesc_t *)obj;
	
	update_editblock(bd, bd->breakpoint != NULL, strsave(text));
	return handle_editblock_change(bd->editblock, text, FALSE, &junk);
}

static bool
handle_editblock_change(eb, text, force, p_error_pos)
Editblock *eb;
const char *text;
bool force;
size_t *p_error_pos;
{
	bpdesc_t *bd;
	compile_res_t *cr;
	block_t *block;
	char **lines;
	const char *start_text, *end_text;
	int nlines;
	lexinfo_t lxbuf;
			      
	bd = (bpdesc_t *)srcbuf_get_editblock_data(eb);
	
	td_record_bpt_code_edit((objid_t)bd, text);

	if (bd->func == NULL) {
		block = bd->fil->fi_block;
		start_text = "";
		end_text = "";
	}
	else {
		block = FU_BLOCKS(bd->func);
		while (block != NULL) {
			block_t *child;

			for (child = block->bl_blocks;
			     child != NULL;
			     child = child->bl_next) {
				if (child->bl_start_lnum <= bd->lnum &&
				    bd->lnum <= child->bl_end_lnum)
					break;
			}
		
			if (child == NULL)
				break;
			block = child;
		}

		if (block == NULL)
			block = bd->fil->fi_block;

		start_text = "void $start(void) {";
		end_text = "}";
	}
	
	/*  Make sure block vars and type information of fil are loaded.
	 */
	FI_VARS(bd->fil);
	
	if (bd->parse_id != NULL)
		ci_free_parse_id(bd->parse_id);
	
	if (bd->machine != NULL) {
		ci_free_machine(bd->machine);
		bd->machine = NULL;
	}

	lines = ssplit(text, "\0\n");
	for (nlines = 0; lines[nlines] != NULL; ++nlines)
		;
	
	cr = compile_code((const char **)lines, nlines, block, (char *)NULL,
					&lxbuf, start_text, end_text,
					Stop_keyword, Stop_funcname);

	bd->parse_id = cr->cr_parse_id;
	bd->machine = cr->cr_machine;
	bd->code_bad = cr->cr_machine == NULL || cr->cr_parse_id == NULL;
	bd->code_has_func_calls = cr->cr_code_has_func_calls;
	bd->want_save = TRUE;

	if (bd->code_bad) {
		int lnum;

		*p_error_pos = lxbuf.lx_cnum;
		
		for (lnum = 0; lnum < nlines && lnum < lxbuf.lx_lnum; ++lnum)
			*p_error_pos += strlen(lines[lnum]) + 1;

		/*  BUG: the following stuff should not really be necessary,
		 *       but I don't have time right now to investigate.
		 */
		if (*p_error_pos >= strlen(text)) {
			*p_error_pos = strlen(text);

			if (*p_error_pos != 0)
				--*p_error_pos;
		}

		return force;
	}
	
	return TRUE;
}

/*  Select or deselect a breakpoint object.  This function is registered
 *  as a callback for breakpoint editblock objects.
 */
static void
handle_editblock_action(editblock, action)
Editblock *editblock;
Editblock_action action;
{
	bpdesc_t *bd;

	bd = (bpdesc_t *)srcbuf_get_editblock_data(editblock);
	
	switch (action) {
	case EDL_SELECT:
		clear_selection();
		select_object((objid_t)bd, TRUE, OBJ_SELF);
		ensure_visible((objid_t)bd);
		break;
	case EDL_DESELECT:
		select_object((objid_t)bd, FALSE, OBJ_SELF);
		break;
	case EDL_REMOVE:
		bd->editblock = NULL;
		remove_object((objid_t)bd, OBJ_SELF);
		break;
	default:
		panic("unknown action in hea");
	}
}

/*  Make a new breakpoint object.
 */
objid_t
add_breakpoint_object(f, fil, lnum)
func_t *f;
fil_t *fil;
int lnum;
{
	fval_t fields[FN_BPT_LAST + 1];
	bpdesc_t *bd;
	const char *fname;
	taddr_t addr;
	target_t *xp;

	xp = get_current_target();
		
	if (f == NULL) {
		fname = "";
		fil = NULL;
	}
	else {
		if (lnum == -1) {
			if (get_min_bpt_addr(f, &addr) != 0)
				return NULL;
		}
		else {
			if (map_lnum_to_addr(f, fil, lnum, &addr) != 0)
				return NULL;
		}

		if (xp_addr_to_breakpoint(xp, addr) != 0) {
			errf("There is already a breakpoint at line %d of %s",
							      lnum, f->fu_name);
			return NULL;
		}

		addr_to_fil_and_lnum(f, addr, &fil, &lnum);
		
		fname = f->fu_name;
	}

	fields[FN_BPT_TYPESTR] = "function";
	fields[FN_BPT_FNAME] = (fval_t)fname;
	fields[FN_BPT_LNUM] = (fval_t)lnum;
	fields[FN_BPT_LAST] = BPT_FLIST_TERMINATOR;

	bd = make_bd(fil, lnum, f);
	
	td_set_obj_updating(OBJ_UPDATING_OFF);
	new_object((objid_t)bd, OT_BPT, BPHEAD_OBJCODE, OBJ_LAST_CHILD);
	set_all_fields((objid_t)bd, fields, BPT_FLIST_TERMINATOR);
	td_set_obj_updating(OBJ_UPDATING_ON);
	ensure_visible((objid_t)bd);

	if (bd->func != NULL) {
		bd->breakpoint = xp_add_breakpoint(xp, addr);
		update_editblock(bd, TRUE, (char *)NULL);
	}
	else {
		edit_field((objid_t)bd, FN_BPT_FNAME);
	}

	return (objid_t)bd;
}

static bpdesc_t *
make_bd(fil, lnum, func)
fil_t *fil;
int lnum;
func_t *func;
{
	bpdesc_t *bd;
	
	bd = (bpdesc_t *)e_malloc(sizeof(bpdesc_t));
	bd->breakpoint = NULL;
	bd->fil = fil;
	bd->func = func;
	bd->lnum = lnum;
	bd->editblock = NULL;
	bd->parse_id = NULL;
	bd->machine = NULL;
	bd->code_bad = FALSE;
	bd->data_needs_initialising = FALSE;
	bd->code_has_func_calls = FALSE;
	bd->want_save = TRUE;

	return bd;
}
		     
objid_t
add_interpreted_code(fil, lnum, text)
fil_t *fil;
int lnum;
char *text;
{
	objid_t obj;
	fval_t fields[4];
	bpdesc_t *bd;
	func_t *f;
	taddr_t addr;

	if ((f = lnum_to_func(fil, lnum)) != NULL &&
	    (addr = lnum_and_fil_to_addr(f, fil, lnum)) != 1 && addr != 0) {
		errf("There is executable code at line %d of %s",
		     lnum, fil->fi_name);
		return NULL;
	}
	
	if (fil_and_lnum_to_bdobj(fil, lnum, &obj)) {
		bd = (bpdesc_t *)obj;
		
		errf("There is already %s at line %d of %s",
		     (bd->breakpoint != NULL) ? "a breakpoint"
		     			      : "interpreted code",
		     lnum, fil->fi_name);
		return NULL;
	}

	
	fields[FN_BPT_TYPESTR] = "file";
	fields[FN_BPT_FNAME] = (fval_t)fil->fi_name;
	fields[FN_BPT_LNUM] = (fval_t)lnum;
	fields[FN_BPT_LAST] = BPT_FLIST_TERMINATOR;

	bd = make_bd(fil, lnum, (func_t *)NULL);
	
	td_set_obj_updating(OBJ_UPDATING_OFF);
	new_object((objid_t)bd, OT_BPT, BPHEAD_OBJCODE, OBJ_LAST_CHILD);
	set_all_fields((objid_t)bd, fields, BPT_FLIST_TERMINATOR);
	td_set_obj_updating(OBJ_UPDATING_ON);
	ensure_visible((objid_t)bd);

	update_editblock(bd, FALSE, text);

	return (objid_t)bd;
}

/*  Update the editblock for code or a breakpoint after a change of status
 *  or position.  If necessary, create the editblock.
 */
static void
update_editblock(bd, is_bpt, text)
bpdesc_t *bd;
bool is_bpt;
char *text;
{
	Editblock *editblock;
	char textbuf[sizeof(Stop_keyword) + 2];	/* word, semicolon, nl, nul */

	editblock = bd->editblock;

	if (bd->fil == NULL || bd->lnum == 0) {
		if (editblock != NULL) {
			srcbuf_remove_editblock(editblock, FALSE);
			bd->editblock = NULL;
		}
		return;
	}

	if (text == NULL) {
		if (is_bpt) {
			sprintf(textbuf, "%s;\n", Stop_keyword);
	
			text = build_source_text(bd->fil, bd->lnum, textbuf);
		}
		else {
			text = strsave("");
		}
	}
	else {
		bd->want_save = TRUE;
	}

	if (editblock != NULL) {
		srcbuf_remove_editblock(editblock, FALSE);
		editblock = NULL;
	}

	bd->editblock = srcwin_add_editblock(get_current_srcwin(),
						bd->fil, bd->lnum, text,
						handle_editblock_action,
						handle_editblock_change,
						(char *)bd);

	if (!is_bpt && *text == '\0') {
		size_t start, lim;
		
		srcbuf_get_editblock_point_range(bd->editblock, &start, &lim);
		
		srcwin_start_edit(get_current_srcwin(),
				  get_current_srcwin_region(),
				  bd->editblock, start);
	}
}

/*  Attempt to move a breakpoint after editing the function name
 *  or line number.
 */
static bool
move_bpt(obj, newstring, edit_lnum, p_display_string)
objid_t obj;
char *newstring;
int edit_lnum;
const char **p_display_string;
{
	fval_t fields[FN_BPT_LAST + 1];
	func_t *f;
	bpdesc_t *bd;
	const char *fname;
	fil_t *fil;
	int lnum;
	taddr_t addr;
	target_t *xp;

	xp = get_current_target();
	bd = (bpdesc_t *)obj;
	f = bd->func;

	if (edit_lnum) {
		lnum = atoi(newstring);
		fname = f->fu_name;
	}
	else {
		lnum = bd->lnum;
		fname = newstring;
	}

	get_all_fields(obj, fields);
	
	if (f == NULL || strcmp(fname, f->fu_name) != 0) {
		if (find_func_by_name(fname, &f) != 0)
			return FALSE;

		if (get_min_bpt_addr(f, &addr) != 0)
			return FALSE;
	}
	else {
		if (map_lnum_to_addr(f, bd->fil, lnum, &addr) != 0)
			return FALSE;
	}
	
	addr_to_fil_and_lnum(f, addr, &fil, &lnum);
	
	/*  We don't allow more than one breakpoint at a given address.
	 */
	if (xp_addr_to_breakpoint(xp, addr) != 0) {
		errf("There is already a breakpoint at line %d of %s",
							lnum, f->fu_name);
		return FALSE;
	}

	fields[FN_BPT_TYPESTR] = "function";
	fields[FN_BPT_FNAME] = f->fu_name;
	fields[FN_BPT_LNUM] = (fval_t)lnum;
	fields[FN_BPT_LAST] = BPT_FLIST_TERMINATOR;
	set_all_fields(obj, fields, BPT_FLIST_TERMINATOR);

	bd->fil = fil;
	bd->func = f;
	bd->lnum = lnum;
	bd->want_save = TRUE;

	/*  Move the actual breakpoint in the code to the new position.
	 */
	if (bd->breakpoint != NULL)
		xp_remove_breakpoint(xp, bd->breakpoint);
	bd->breakpoint = xp_add_breakpoint(xp, addr);

	update_editblock(bd, TRUE, (char *)NULL);

	if (edit_lnum) {
		static char buf[20];

		sprintf(buf, "%d", lnum);
		*p_display_string = buf;
	}
	else {
		*p_display_string = f->fu_name;
	}
	
	return TRUE;
}

bool
execute_bp_code(bp, fp, ap)
breakpoint_t *bp;
taddr_t fp, ap;
{
	objid_t obj;
	bpdesc_t *bd;
	ci_exec_result_t res;

	obj = get_code(BPHEAD_OBJCODE, OBJ_CHILD);
	for (; obj != NULL; obj = get_code(obj, OBJ_NEXT)) {
		if (((bpdesc_t *)obj)->breakpoint == bp)
			break;
	}

	if (obj == NULL)
		return TRUE;

	bd = (bpdesc_t *)obj;
	
	if (bd->code_bad) {
		errf("Bad breakpoint code");
		return TRUE;
	}
	if (bd->machine == NULL)
		return TRUE;
	
	if (bd->data_needs_initialising) {
		ci_initialise_machine(bd->machine, TRUE, FALSE,
						(char **)NULL, (char **)NULL);
		bd->data_needs_initialising = FALSE;
	}
	else {
		ci_initialise_machine(bd->machine, FALSE, FALSE,
						(char **)NULL, (char **)NULL);
	}

	res = ci_execute_machine(bd->machine, fp, ap, read_data, write_data,
							call_target_function);

	if (res != CI_ER_TRAP && res != STOP)
		errf("%s in breakpoint code", ci_exec_result_to_string(res));

	return res != CI_ER_TRAP;
}

static int
read_data(addr, buf, nbytes)
taddr_t addr;
void *buf;
size_t nbytes;
{
	return dread(get_current_target(), addr, buf, nbytes);
}

static int
write_data(addr, buf, nbytes)
taddr_t addr;
constvoidptr buf;
size_t nbytes;
{
	return dwrite(get_current_target(), addr, buf, nbytes);
}

static int
set_initialise_data_flag(obj, unused_arg)
objid_t obj;
fval_t unused_arg;
{
	((bpdesc_t *)obj)->data_needs_initialising = TRUE;
	return 0;
}

void
reinitialise_bpt_code_data()
{
	visit_objects(BPHEAD_OBJCODE, OBJ_CHILDREN, set_initialise_data_flag,
						(fval_t)0, (objid_t *)NULL);
}

void
remove_breakpoint_object(obj)
objid_t obj;
{
	bpdesc_t *bd;

	bd = (bpdesc_t *)obj;

	if (bd->editblock != NULL)
		srcbuf_remove_editblock(bd->editblock, FALSE);

	if (bd->breakpoint != NULL)
		xp_remove_breakpoint(get_current_target(), bd->breakpoint);

	if (bd->parse_id != NULL)
		ci_free_parse_id(bd->parse_id);
	if (bd->machine != NULL)
		ci_free_machine(bd->machine);
	
	free((char *)bd);
}

static void
show_bpt_source(bd, beep_on_failure)
bpdesc_t *bd;
bool beep_on_failure;
{
	if (bd->func == NULL) {
		show_source(bd->fil, bd->lnum);
	}
	else if (bd->lnum != 0 && bd->func->fu_fil != NULL) {
		show_source(bd->fil, bd->lnum);
	}
	else {
		errf("%s`%s' was not compiled with the -g flag",
					beep_on_failure ? "" : "\bNote: ",
					bd->func->fu_name);
	}
}

char *
bpt_format_obj(code)
objid_t code;
{
	const char *fname;
	int lnum;

	fname = get_field_value(code, FN_BPT_FNAME);
	lnum = (int)get_field_value(code, FN_BPT_LNUM);

	return strf("%s %d", fname, lnum);
}

int
pre_do_bpt(command, p_arg)
int command;
char **p_arg;
{
	bptargs_t *ba;
	const char *mode, *prompt;
	FILE *fp;
	char *path;
	
	if (command != MR_BPTMEN_SAVE) {
		*p_arg = NULL;
		return 0;
	}

	if (get_num_selected() == 1)
		prompt = "Save breakpoint to file: ";
	else
		prompt = "Save breakpoints to file: ";
	
	if (prompt_for_output_file(prompt, &path, &mode) != 0)
		return -1;

	if ((fp = fopen(path, mode)) == NULL) {
		failmesg("Can't create", "saved breakpoints file", path);
		return -1;
	}

	ba = (bptargs_t *)e_malloc(sizeof(bptargs_t));
	ba->ba_fp = fp;
	ba->ba_path = path;
	*p_arg = (char *)ba;
	return 0;
}

void
post_do_bpt(command, arg)
int command;
char *arg;
{
	bptargs_t *ba;
	
	if (command != MR_BPTMEN_SAVE)
		return;

	ba = (bptargs_t *)arg;

	if (fflush(ba->ba_fp) == EOF || ferror(ba->ba_fp)) {
		failmesg("Warning: Error writing to", "saved breakpoints file",
			 ba->ba_path);
	}

	fclose(ba->ba_fp);

	free(ba->ba_path);
	free((char *)ba);
}

/*  Process the return from the breakpoint menu enabling and disabling
 *  ACTIVE, RESTART etc. and displaying source.
 */
void
do_bpt(obj, command, arg)
objid_t obj;
int command;
char *arg;
{
	int oldstate;
	bptargs_t *ba;

	ba = (bptargs_t *)arg;

	oldstate = td_set_obj_updating(OBJ_UPDATING_OFF);
	
	switch (command) {
	case MR_BPTMEN_SOURCE:
		show_bpt_source((bpdesc_t *)obj, TRUE);
		break;
	case MR_BPTMEN_REMOVE:
		remove_object(obj, OBJ_SELF);
		break;
	case MR_BPTMEN_SAVE:
		save_object_to_file(ba->ba_path, ba->ba_fp, (bpdesc_t *)obj);
		break;
	default:
		panic("bad rv in db");
	}
	
	td_set_obj_updating(oldstate);
}

void
remove_symtab_breakpoints(st)
symtab_t *st;
{
	objid_t obj, next;
	int oldstate;

	oldstate = td_set_obj_updating(OBJ_UPDATING_OFF);
	for (obj = get_code(BPHEAD_OBJCODE, OBJ_CHILD); obj != NULL; obj = next) {
		next = get_code(obj, OBJ_NEXT);
		if (((bpdesc_t *)obj)->func->fu_symtab == st)
			remove_object(obj, OBJ_SELF);
	}
	td_set_obj_updating(oldstate);
}

/*  Add the breakpoint header to the display.  Called on startup from main().
 */
void
add_breakpoint_header(par)
objid_t par;
{
	new_object(BPHEAD_OBJCODE, OT_BPHEAD, par, OBJ_CHILD);
}

/*  Process the return from the breakpoint header menu, creating new
 *  breakpoints and such.
 */
/* ARGSUSED */
void
do_bps(unused_obj, command, arg)
objid_t unused_obj;
int command;
char *arg;
{
	bool junk;
	
	switch(command) {
	case MR_ADD_BREAKPOINT:
		add_breakpoint_object((func_t *)NULL, (fil_t *)NULL, 0);
		break;
	case MR_REMOVE_ALL_BREAKPOINTS:
		remove_object(BPHEAD_OBJCODE, OBJ_CHILDREN);
		break;
	case MR_RESTORE_BREAKPOINTS:
		if (Statefile_path == NULL)
			errf("No saved state file to load breakpoints from");
		else
			read_config_file(Statefile_path, TRUE, TRUE, TRUE,
					 FALSE, &junk);
		break;
	case MR_LOAD_BREAKPOINTS:
		load_breakpoints();
		break;
	default:
		panic("bad cmd in db");
	}
}

static bool
bpfunc_quitfunc(obj, new_text, p_display_string)
objid_t obj;
char *new_text;
const char **p_display_string;
{
	return move_bpt(obj, new_text, FALSE, p_display_string);
}

static bool
bplnum_quitfunc(obj, new_text, p_display_string)
objid_t obj;
char *new_text;
const char **p_display_string;
{
	return move_bpt(obj, new_text, TRUE, p_display_string);
}

static void
bplnum_edit(fdets)
struct drawst fdets;
{
	if (HAS_CODE((bpdesc_t *)fdets.dr_code)) {
		errf("Can't move a breakpoint that has interpreted code");
		wn_wait_for_release_of(fdets.dr_wn, B_ANY);
	}
	else {
		field_edit_start(&fdets, "line number",
				 strf("%d", (int)fdets.dr_fval));
	}
}

static void
bpfunc_edit(fdets)
struct drawst fdets;
{
	if (HAS_CODE((bpdesc_t *)fdets.dr_code)) {
		errf("Can't move a breakpoint that has interpreted code");
		wn_wait_for_release_of(fdets.dr_wn, B_ANY);
	}
	else {
		field_edit_start(&fdets, "function name", (char *)NULL);
	}
}
