/* gd_driver.c - target driver using a munged version of gdb */

/*  Copyright 1994 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_gd_driver_c_sccsid[] = "@(#)gd_driver.c	1.10 04 Jun 1995 (UKC)";

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <setjmp.h>
#include <signal.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <ctype.h>

#include <sys/time.h>
#include <sys/resource.h>

#include <local/ukcprog.h>
#include <mtrprog/hash.h>
#include <mtrprog/utils.h>

#include "ups.h"
#include "symtab.h"
#include "target.h"
#include "st.h"
#include "ci.h"
#include "as.h"
#include "ui.h"
#include "obj_target.h"
#include "obj_signal.h"
#include "obj_stack.h"
#include "obj_env.h"
#include "obj_bpt.h"

#undef PROTO

#define free dont_declare_free
#define realloc dont_declare_realloc
#define qsort dont_declare_qsort
#define strtok dont_declare_strtok
#define strerror dont_declare_strerror
#define error dont_declare_error
#define exit dont_declare_exit
#define malloc dont_declare_malloc

#include "gdb/upsgdb/defs.h"
#include "gdb/upsgdb/symtab.h"
#include "gdb/upsgdb/gdbcore.h"
#include "gdb/upsgdb/call-cmds.h"
#include "gdb/upsgdb/value.h"
#include "gdb/upsgdb/symfile.h"
#include "gdb/upsgdb/objfiles.h"
#include "gdb/upsgdb/target.h"
#include "gdb/upsgdb/breakpoint.h"
#include "gdb/upsgdb/inferior.h"
#include "gdb/upsgdb/value.h"

#undef free
#undef realloc
#undef qsort
#undef strtok
#undef strerror
#undef error
#undef exit
#undef malloc

typedef struct gd_stdata_s {
	struct objfile *st_objfile;
	hashtab_t *st_typehash;
} gd_stdata_t;

typedef struct gd_fidata_s {
	struct partial_symtab *fi_partial_symtab;
} gd_fidata_t;

#define GET_STD(st)	((gd_stdata_t *)(st)->st_data)
#define GET_FID(fil)	((gd_fidata_t *)(fil)->fi_data)

typedef struct {
	char *ia_path;
	char *ia_cmdline;
	const char **ia_envp;
} infargs_t;

void kill_command PARAMS ((char *arg, int from_tty));
void next_command PARAMS ((char *count_string, int from_tty));
void step_command PARAMS ((char *count_string, int from_tty));
void enable_command PARAMS ((char *bpnum_string, int from_tty));
void disable_command PARAMS ((char *bpnum_string, int from_tty));
void delete_command PARAMS ((char *bpnum_string, int from_tty));
void continue_command PARAMS ((char *proc_count_exp, int from_tty));
void finish_command PARAMS ((char *arg, int from_tty));
void detach_command PARAMS ((char *arg, int from_tty));
void breakpoint_1 PARAMS ((int, int));
void print_gdb_version PARAMS((FILE *));
CORE_ADDR allocate_space_in_inferior PARAMS((int));

void fputs_maybe_filtered PARAMS((const char *s, FILE *fp));
void error PARAMS((const char *cmd, ...));

extern struct breakpoint *breakpoint_chain;
extern jmp_buf error_return, quit_return;
extern unsigned char *signal_stop, *signal_print, *signal_program;

#undef PROTO
#define PROTO(args)         PARAMS(args)

typedef void (*Gdb_cmdfunc)PARAMS((char *arg, int from_tty));

static bool gd_match PROTO((const char *textpath,
			    text_block_t *tb, size_t nbytes));
static int gd_init_from_textfile PROTO((target_t *xp, int textfd,
					const char *corepath,
					bool user_gave_core,
					const char **p_cmdline));
static int gd_read_text PROTO((symtab_t *st, taddr_t addr, char *buf,
								size_t nbytes));
static taddr_t gd_get_addr_lim PROTO((func_t *f));
static void gd_close_symtab_data PROTO((symtab_t *st));
static var_t *gd_get_fi_vars PROTO((fil_t *fil));
static block_t *gd_get_fu_blocks PROTO((func_t *f));
static lno_t *gd_get_fu_lnos PROTO((func_t *f));
static const char *gd_disassemble_instruction PROTO((func_t *f, taddr_t addr,
				   		     const char *text,
						     const char **p_buf));
static bool gd_fil_may_have_matching_globals PROTO((fil_t *fil, const char *pat,
						    matchfunc_t matchf));
static void gd_kill PROTO((target_t *xp));
static int gd_is_attached PROTO((target_t *xp));
static tstate_t gd_get_state PROTO((target_t *xp));
static int gd_get_lastsig PROTO((target_t *xp));
static sigstate_t gd_get_sigstate PROTO((target_t *xp, int sig));
static Stack *gd_get_stack_trace PROTO((target_t *xp));
static int gd_read_fpval PROTO((target_t *xp, taddr_t addr, bool is_double,
							fpval_t *p_val));
static int gd_read_fpreg PROTO((target_t *xp, int regno, bool is_double,
							     fpval_t *p_val));
static int gd_readreg PROTO((target_t *xp, int regno, taddr_t *p_val));
static int gd_setreg PROTO((target_t *xp, int regno, taddr_t val));
static int gd_read_data PROTO((target_t *xp,
			       taddr_t addr, char *buf, size_t nbytes));
static int gd_write_data PROTO((target_t *xp,
				taddr_t addr, const char *buf, size_t nbytes));
static int gd_read_text_from_process PROTO((target_t *xp, taddr_t addr,
					    char *buf, size_t nbytes));
static int gd_call_func PROTO((target_t *xp, char *cma, taddr_t addr,
			       taddr_t *args, size_t *argsizes, int nargs,
			       taddr_t *p_res, const char **p_mesg));
static const char *gd_get_signal_tag PROTO((target_t *xp, int sig));
static taddr_t gd_funcptr_to_addr PROTO((symtab_t *st, taddr_t addr));
static taddr_t gd_addr_to_funcptr PROTO((symtab_t *st, taddr_t addr));
static breakpoint_t *gd_add_breakpoint PROTO((target_t *xp, taddr_t addr));
static int gd_remove_breakpoint PROTO((target_t *xp, breakpoint_t *bp));
static breakpoint_t *gd_addr_to_breakpoint PROTO((target_t *xp, taddr_t addr));
static taddr_t gd_get_breakpoint_addr PROTO((target_t *xp, breakpoint_t *bp));
static void add_symbols PROTO((alloc_pool_t *ap, fil_t *fil,
			       struct partial_symbol *symbase,
			       int symcount, bool global,
			       func_t **p_funcs));
static void load_gdb_symtab PROTO((fil_t *fil));
static int gdb_addr_to_lnum PROTO(( struct linetable *lt, CORE_ADDR addr));
static block_t *get_blocktree PROTO((alloc_pool_t *ap, hashtab_t *ht,
				     block_t *par_bl,
				     struct blockvector *bv, int bn,
				     struct linetable *lt, int *p_next_bn));
static void get_gdb_lnum_entries PROTO((alloc_pool_t *ap, func_t *f,
					struct linetable *lt,
					CORE_ADDR start_addr,
					CORE_ADDR end_addr));
static stopres_t gd_next PROTO((target_t *xp));
static stopres_t gd_step PROTO((target_t *xp));
static stopres_t gd_start PROTO((target_t *xp));
static stopres_t gd_cont PROTO((target_t *xp));
static bool sig_is_fatal PROTO((int sig));
static void push_gdb_block_vars PROTO((alloc_pool_t *ap, hashtab_t *ht,
				       block_t *bl, struct block *block,
				       bool global));
static type_t *parse_gdb_type PROTO((alloc_pool_t *ap, hashtab_t *ht,
				     struct type *gt));
static type_t *parse_gdb_aggr PROTO((alloc_pool_t *ap, hashtab_t *ht,
				     struct type *gt, bool is_struct));
static void enter_type PROTO((alloc_pool_t *ap, hashtab_t *ht,
			      struct type *gt, type_t *type));
static type_t *parse_gdb_enum PROTO((alloc_pool_t *ap, struct type *gt));
static bool match_symbol PROTO((struct partial_symbol *symbase, int symcount,
				matchfunc_t matchf, const char *pat));
static taddr_t gd_get_reg_addr PROTO((target_t *xp, Stack *stk, int reg));
static int do_command PROTO((Gdb_cmdfunc cfunc, const char *arg));
static int gd_initialise PROTO((ebuf_t *usage_eb, char **argv));
static void possibly_flatten_outer_block PROTO((func_t *f));
static symtab_t *objfile_to_st PROTO((struct objfile *objfile));
static int start_gdb_target PROTO((char *path, char *argline,
				   const char **envp));
static void stop_gdb_target PROTO((void));
static stopres_t run_gdb_target PROTO((Gdb_cmdfunc func, char *arg));
static void update_shared_library_symtabs PROTO((target_t *xp));
static tstate_t get_gdb_target_state PROTO((void));
static void gd_handle_signal_change PROTO((target_t *xp, int sig, bool ignore,
					   bool redraw, bool stop));
static int gd_enable_breakpoint PROTO((target_t *xp, breakpoint_t *bp));
static int gd_disable_breakpoint PROTO((target_t *xp, breakpoint_t *bp));
static const char *fmtnum PROTO((const char *fmt, unsigned long num));
static int change_bp_status PROTO((Gdb_cmdfunc cmdfunc, struct breakpoint *b));
static stopres_t do_next_or_step PROTO(( Gdb_cmdfunc cmdfunc));
static void get_symtab_and_line PROTO((struct frame_info *fi,
				       struct symtab **p_symtab, int *p_line));
static bool gd_do_debug_command PROTO((target_t *xp, const char *cmd,
				       size_t cmdlen, const char *arg));
static int gd_preinitialise PROTO((void));
static void gd_show_target_driver_info PROTO((bool name_only));
static void gd_detach PROTO((target_t *xp, int sig));
static stopres_t return_then_next_or_step PROTO((Gdb_cmdfunc cmdfunc,
						 struct frame_info *fi));
static int gd_get_min_bpt_addr PROTO((func_t *f, taddr_t *p_addr));
static CORE_ADDR gdb_skip_prologue PROTO((CORE_ADDR addr));
static const char *gdbcmd_to_name PROTO((Gdb_cmdfunc cfunc));
static void logcmd PROTO((FILE *fp, const char *cmd, const char *arg));
static void start_gdb_inferior PROTO((char *args, bool from_tty));
static void do_all_gdb_cleanups PROTO((void));
static int get_gdb_prev_frame PROTO((struct frame_info *fi,
				     struct frame_info **p_prev));

static int Sent_sigint_to_child;

static bool Want_gdb_cmdlog;
static bool Want_gdb_output;
static bool Disable_gdb;

xp_ops_t Gd_ops = {
	gd_preinitialise, gd_initialise, gd_show_target_driver_info,
	gd_match, gd_init_from_textfile,
	gd_start, gd_step, gd_next, gd_cont, gd_kill,
	NULL, NULL,
	NULL, NULL,
	gd_add_breakpoint, gd_remove_breakpoint,
	gd_enable_breakpoint, gd_disable_breakpoint,
	gd_addr_to_breakpoint, gd_get_breakpoint_addr,
	gd_is_attached, gd_detach,
	gd_get_state, gd_get_lastsig, NULL, gd_get_sigstate,
	gd_get_stack_trace, gd_get_reg_addr, gd_get_signal_tag,
	gd_read_fpval, gd_read_fpreg, gd_readreg, gd_setreg,
	NULL, NULL,
	NULL,
	gd_read_data, gd_write_data, gd_read_text_from_process,
	NULL, gd_call_func, gd_handle_signal_change, gd_do_debug_command
};

static sym_ops_t Gd_symops = {
	gd_read_text,
	gd_get_min_bpt_addr,
	gd_get_addr_lim,

	gd_funcptr_to_addr,
	gd_addr_to_funcptr,
		
	gd_fil_may_have_matching_globals,
	gd_close_symtab_data,

	NULL,
	NULL,
	NULL,
	NULL,
	NULL,

	gd_get_fu_lnos,
	gd_get_fu_blocks,
	gd_get_fi_vars,

	gd_disassemble_instruction,
	NULL,
};

#define CATCH_ERRORS(action) \
 { \
	 if (setjmp(error_return) != 0) { \
		 action; \
	 } \
	 memcpy(quit_return, error_return, sizeof(jmp_buf)); \
 }


void
wrap_here(s)
char *s;
{
}

void
fputs_maybe_filtered(s, fp)
const char *s;
FILE *fp;
{
	if (Want_gdb_output)
		fputs(s, stdout);
}

#ifdef __STDC__
void
error(const char *fmt, ...)
{
	char buf[100];
	char *s;
	va_list args;

	va_start(args, fmt);
	s = formf(buf, sizeof(buf), fmt, args);
	va_end(args);

	display_message(s);

	if (s != buf)
		free(s);

	return_to_top_level(RETURN_ERROR);
}
#else
void
error(va_alist)
va_dcl
{
	const char *fmt;
	char buf[100];
	char *s;
	va_list args;

	va_start(args);
	fmt = va_arg(args, const char *);
	s = formf(buf, sizeof(buf), fmt, args);
	va_end(args);

	display_message(s);

	if (s != buf)
		free(s);

	return_to_top_level(RETURN_ERROR);
}
#endif /* !__STDC__ */

static const char *
gdbcmd_to_name(cfunc)
Gdb_cmdfunc cfunc;
{
	if (cfunc == next_command)		return "next";
	if (cfunc == step_command)		return "step";
	if (cfunc == continue_command)		return "continue";
	if (cfunc == finish_command)		return "next";
	if (cfunc == kill_command)		return "kill";
	if (cfunc == detach_command)		return "detach";
	if (cfunc == enable_command)		return "enable";
	if (cfunc == disable_command)		return "disable";
	if (cfunc == exec_file_command)		return "exec-file";
	if (cfunc == symbol_file_command)	return "symbol-file";
	if (cfunc == core_file_command)		return "core-file";
	if (cfunc == attach_command)		return "attach";
	if (cfunc == break_command)		return "break";
	if (cfunc == delete_command)		return "delete";

	return "UNKNOWN-COMMAND";
}
	
static void
logcmd(fp, cmd, arg)
FILE *fp;
const char *cmd, *arg;
{
	if (arg == NULL)
		arg = "";
	
	fprintf(fp, "echo %s %s\\n\n", cmd, arg);
	fprintf(fp, "%s %s\n", cmd, arg);
}

int
print_insn(memaddr, stream)
CORE_ADDR memaddr;
GDB_FILE *stream;
{
	panic("print_insn called");
	return 0;
}
	
static int
do_command(cfunc, arg)
Gdb_cmdfunc cfunc;
const char *arg;
{
	if (Want_gdb_cmdlog && cfunc != start_gdb_inferior)
		logcmd(stdout, gdbcmd_to_name(cfunc), arg);
	
	CATCH_ERRORS(return -1);

	(*cfunc)((char *)arg, FALSE);

	do_all_gdb_cleanups();
	
	/*  If we get here, the command succeeded.
	 */
	return 0;
}

static void
do_all_gdb_cleanups()
{
	do_cleanups((struct cleanup *)NULL);
}

char *
tilde_expand(path)
char *path;
{
	return strsave(path);
}

struct value *
evaluate_expression(exp)
struct expression *exp;
{
	static struct value *dummy_value;
	
	panic("evaluate_expression called");
	return dummy_value;	/* to satisfy gcc */
}

char *
command_line_input(s, val, suffix)
char *s;
int val;
char *suffix;
{
	panic("command_line_input called");
	return NULL;		/* to satisfy gcc */
}

static int
gd_preinitialise()
{
	/*  This must happen before ups first calls malloc.
	 */
	init_malloc((PTR)NULL);

	return 0;
}

static int
gd_initialise(usage_eb, argv)
ebuf_t *usage_eb;
char **argv;
{
	static char dirbuf[1024];
	struct rlimit rlim;
#ifdef SET_STACK_LIMIT_HUGE
	extern int original_stack_limit;
#endif /* SET_STACK_LIMIT_HUGE */
	extern int caution;
	extern int current_display_number;
	enum target_signal ts;

	getcwd(dirbuf, sizeof(dirbuf));
	current_directory = dirbuf;
	initialize_all_files();

#ifdef SET_STACK_LIMIT_HUGE
	getrlimit(RLIMIT_STACK, &rlim);
	original_stack_limit = rlim.rlim_cur;
#endif /* SET_STACK_LIMIT_HUGE */

	current_display_number = -1;

	caution = 0;	/* Don't want any queries */

	Disable_gdb = extract_bool_arg(usage_eb, argv, "-nogdb");
	Want_gdb_output = extract_bool_arg(usage_eb, argv, "-gdboutput");
	Want_gdb_cmdlog = extract_bool_arg(usage_eb, argv, "-loggdbcmds");

	for (ts = TARGET_SIGNAL_HUP; ts < TARGET_SIGNAL_UNKNOWN; ++ts) {
		int sig;

		sig = target_signal_to_host(ts);
		
		if (ts != 0) {
			set_signal_attrs(sig,
					 !signal_program[ts],
					 signal_print[ts],
					 signal_stop[ts]);
		}
	}

	return 0;
}

static void
gd_show_target_driver_info(name_only)
bool name_only;
{
	bool save_output_flag;

	if (Disable_gdb)
		return;

	if (name_only) {
		puts("\tObject file support via a modified version of gdb");
		return;
	}

	save_output_flag = Want_gdb_output;
	Want_gdb_output = TRUE;
		
	puts("\
This version of ups uses a modified version of the GNU debugger gdb:");
	print_gdb_version(stdout);
	fputc('\n', stdout);

	Want_gdb_output = save_output_flag;
}

static void
gd_handle_signal_change(xp, sig, ignore, redraw, stop)
target_t *xp;
int sig;
bool ignore, redraw, stop;
{
	enum target_signal target_signal;

	target_signal = target_signal_from_host(sig);

	if (target_signal == TARGET_SIGNAL_UNKNOWN)
		return;

	signal_program[target_signal] = !ignore;
	signal_print[target_signal] = redraw;
	signal_stop[target_signal] = stop;
}

static bool
gd_do_debug_command(xp, cmd, cmdlen, arg)
target_t *xp;
const char *cmd;
size_t cmdlen;
const char *arg;
{
	static const char bpcmd[] = "dumpbps";
	bool old_output_flag;
	
	if (cmdlen != sizeof(bpcmd) - 1 || memcmp(cmd, bpcmd, cmdlen) != 0)
		return FALSE;

	old_output_flag = Want_gdb_output;
	Want_gdb_output = TRUE;
	breakpoint_1(-1, *arg != '\0');
	Want_gdb_output = old_output_flag;
	
	return TRUE;
}

static bool
gd_match(textpath, tb, nbytes)
const char *textpath;
text_block_t *tb;
size_t nbytes;
{
	return !Disable_gdb;
}

static int
gd_init_from_textfile(xp, textfd, corepath, user_gave_core, p_cmdline)
target_t *xp;
int textfd;
const char *corepath;
bool user_gave_core;
const char **p_cmdline;
{
	struct objfile *objfile;

	if (do_command(exec_file_command, xp->xp_textpath) != 0)
		return -1;
	
	/*  If the user is attempting to attach to a process do a quick check
	 *  that we have permission before the possibly lengthy work of
	 *  loading the symbol table.
	 */
	if (user_gave_core && is_number(corepath)) {
		int pid;

		pid = atoi(corepath);
		
		if (kill(pid, 0) != 0) {
			errf("Can't attach to process %d (%m)", pid);
			return -1;
		}
	}

	if (do_command(symbol_file_command, xp->xp_textpath) != 0)
		return -1;

	if (object_files == NULL || object_files->next != NULL)
		panic("#objfiles botch in iftf");
	objfile = object_files;

	if (user_gave_core) {
		Gdb_cmdfunc cmd;

		cmd = is_number(corepath) ? attach_command : core_file_command;

		if (do_command(cmd, corepath) != 0)
			return -1;
	}

	xp->xp_symtabs = objfile_to_st(objfile, TRUE);
	xp->xp_symtab_cache = NULL;
	xp->xp_data = NULL;

	xp->xp_words_big_endian = TARGET_BYTE_ORDER == BIG_ENDIAN;
	xp->xp_bits_big_endian = BITS_BIG_ENDIAN;

	if (attach_flag)
		update_shared_library_symtabs(xp);

	return 0;
}

static symtab_t *
objfile_to_st(objfile)
struct objfile *objfile;
{
	symtab_t *st;
	alloc_pool_t *ap;
	gd_stdata_t *std;
	struct partial_symtab *ps;
	fil_t *source_files, *fil;
	func_t *funcs, *f, *funclist;
	struct minimal_symbol *msym;
	addrlist_t *addrlist;
	functab_t *functab;
	CORE_ADDR min_addr, max_addr;
	int nfuncs;

	ap = alloc_create_pool();

	source_files = NULL;
	funcs = NULL;
	max_addr = 0;

	for (ps = objfile->psymtabs; ps != NULL; ps = ps->next) {
		fil_t *fil;
		gd_fidata_t *fid;
		
		if (ps->textlow == 0 && ps->texthigh == 0)
			continue;
		
		fil= ci_make_fil(ap, ps->filename, (char *)ps,
				 (block_t *)NULL, source_files);
		fid = (gd_fidata_t *)alloc(ap, sizeof(gd_fidata_t));
		fid->fi_partial_symtab = ps;
		fil->fi_data = (char *)fid;

		add_symbols(ap, fil,
			    objfile->global_psymbols.list + ps->globals_offset,
			    ps->n_global_syms, TRUE, &funcs);

		add_symbols(ap, fil,
			    objfile->static_psymbols.list + ps->statics_offset,
			    ps->n_static_syms, FALSE, &funcs);

		source_files = fil;

		if (ps->texthigh > max_addr)
			max_addr = ps->texthigh;
	}

	addrlist = NULL;
	
	for (msym = objfile->msymbols;
	     msym != NULL && SYMBOL_NAME(msym) != NULL; ++msym) {
		switch (MSYMBOL_TYPE(msym)) {
		case mst_file_text:
		case mst_text:
		case mst_solib_trampoline:
			funcs = ci_make_func(ap, SYMBOL_NAME(msym),
					     SYMBOL_VALUE_ADDRESS(msym),
					     st, (fil_t *)NULL, funcs,
					     (funclist_t *)NULL);
			funcs->fu_flags |= FU_NOSYM | FU_DONE_LNOS |
				           FU_DONE_BLOCKS;
			
			if (MSYMBOL_TYPE(msym) == mst_file_text)
				funcs->fu_flags |= FU_STATIC;

			break;

		default:
			insert_global_addr(ap, &addrlist, SYMBOL_NAME(msym),
					   SYMBOL_VALUE_ADDRESS(msym));
			break;
		}
	}

	min_addr = ~(CORE_ADDR)0;
	nfuncs = 0;
	for (f = funcs; f != NULL; f = f->fu_next) {
		if (f->fu_addr > max_addr)
			max_addr = f->fu_addr;
		if (f->fu_addr < min_addr)
			min_addr = f->fu_addr;
		++nfuncs;
	}

	make_funcinfo(ap, funcs, nfuncs, min_addr, max_addr, 0,
		      &funclist, &functab);

	for (fil = source_files; fil != NULL; fil = fil->fi_next)
		delete_duplicate_funcs(fil);

	std = (gd_stdata_t *)alloc(ap, sizeof(gd_stdata_t));
	std->st_typehash = hash_create_tab(ap, 256);
	std->st_objfile = objfile;

	st = make_symtab(ap, objfile->name, source_files, funclist,
			 &Gd_symops, (char *)std);

	st->st_functab = functab;
	st->st_addrlist = addrlist;
	
	return st;
}

static void
add_symbols(ap, fil, symbase, symcount, global, p_funcs)
alloc_pool_t *ap;
fil_t *fil;
struct partial_symbol *symbase;
int symcount;
bool global;
func_t **p_funcs;
{
	struct partial_symbol *psym;
	func_t *nextf;

	nextf = NULL;

	for (psym = symbase; psym < symbase + symcount; ++psym) {
		func_t *f;
		funclist_t *fl;
		
		switch (PSYMBOL_CLASS(psym)) {
		case LOC_BLOCK:
			fl = (funclist_t *)alloc(ap, sizeof(funclist_t));
			
			f = ci_make_func(ap, SYMBOL_NAME(psym),
					 SYMBOL_VALUE_ADDRESS(psym),
					 fil->fi_symtab, fil, *p_funcs, fl);

			fl->fl_func = f;
			fl->fl_next = fil->fi_funclist;
			fil->fi_funclist = fl;

			*p_funcs = f;

			if (!global)
				f->fu_flags |= FU_STATIC;

			break;

		default:
			break;
		}
	}
}

/*  Gdb 4.12 on the SunPRO C compiler produces a separate outer block for
 *  the arguments of a function.  We don't want to show this to the user,
 *  so try to spot the situation and merge the two outer blocks if
 *  necessary.  We guess by looking at the blocks rather than trying to
 *  work out whether we are looking at a cc binary or not, as that would
 *  involve making assumptions that are likely to change in future.
 */
static void
possibly_flatten_outer_block(f)
func_t *f;
{
	block_t *argbl, *locbl;
	var_t *v, *lastv;
	aggr_or_enum_def_t *ae, *lastae;
	typedef_t *td, *lasttd;

	argbl = f->fu__blocks;
	locbl = argbl->bl_blocks;

	if (argbl->bl_next != NULL || locbl == NULL || locbl->bl_next != NULL)
		return;

	for (v = argbl->bl_vars; v != NULL; v = v->va_next) {
		if (v->va_class != CL_ARG)
			return;
	}

	/*  Splice the arguments on the *end* of the locbl var list, as the
	 *  generic ups code reverses the order of the variables when
	 *  displaying them.
	 */
	lastv = NULL;
	for (v = locbl->bl_vars; v != NULL; v = v->va_next)
		lastv = v;
	if (lastv != NULL)
		lastv->va_next = argbl->bl_vars;

	/*  For completeness do the typedefs and aggrs.
	 */
	lastae = NULL;
	for (ae = locbl->bl_aggr_or_enum_defs; ae != NULL; ae = ae->ae_next)
		lastae = ae;
	if (lastae != NULL)
		lastae->ae_next = argbl->bl_aggr_or_enum_defs;

	lasttd = NULL;
	for (td = locbl->bl_typedefs; td != NULL; td = td->td_next)
		lasttd = td;
	if (lasttd != NULL)
		lasttd->td_next = argbl->bl_typedefs;

	f->fu__blocks = locbl;
	locbl->bl_parent = argbl->bl_parent;
}

static void
load_gdb_symtab(fil)
fil_t *fil;
{
	struct symtab *symtab;
	struct blockvector *bv;
	alloc_pool_t *ap;
	int bn, next_bn;
	funclist_t *fl;
	hashtab_t *ht;

	ap = fil->fi_symtab->st_apool;
	ht = GET_STD(fil->fi_symtab)->st_typehash;

	CATCH_ERRORS(panic("psymtab_to_symtab failed"));
	symtab = PSYMTAB_TO_SYMTAB(GET_FID(fil)->fi_partial_symtab);
	do_all_gdb_cleanups();
	bv = BLOCKVECTOR(symtab);

	fil->fi_path_hint = symtab->dirname;

	fil->fi_block = ci_make_block(ap, (block_t *)NULL);
	push_gdb_block_vars(ap, ht, fil->fi_block,
			    BLOCKVECTOR_BLOCK(bv, GLOBAL_BLOCK), TRUE);
	push_gdb_block_vars(ap, ht, fil->fi_block,
			    BLOCKVECTOR_BLOCK(bv, STATIC_BLOCK), FALSE);

	for (bn = FIRST_LOCAL_BLOCK; bn < BLOCKVECTOR_NBLOCKS(bv);
	     							bn = next_bn) {
		struct block *block;
		func_t *f;
		const char *name;

		block = BLOCKVECTOR_BLOCK(bv, bn);
		name = SYMBOL_NAME(BLOCK_FUNCTION(block));
		
		for (fl = fil->fi_funclist; fl != NULL; fl = fl->fl_next)
			if (strcmp(fl->fl_func->fu_name, name) == 0)
				break;
		if (fl == NULL)
			panic("func botch in lgs");
		f = fl->fl_func;

		f->fu__blocks = get_blocktree(ap, ht, fil->fi_block, bv, bn,
					      LINETABLE(symtab), &next_bn);
		possibly_flatten_outer_block(f);
		f->fu_symdata = (char *)block;
		f->fu_flags |= FU_DONE_BLOCKS;
	}

	/*  This is slow, but I'm pretending to myself that I'll come back
	 *  fix it later ...
	 */
	for (fl = fil->fi_funclist; fl != NULL; fl = fl->fl_next) {
		struct block *fblock;

		fblock = (struct block *)fl->fl_func->fu_symdata;
		if (fblock == NULL)
			continue;

		get_gdb_lnum_entries(ap, fl->fl_func, LINETABLE(symtab),
				     BLOCK_START(fblock), BLOCK_END(fblock));

	}

	fil->fi_flags |= FI_DONE_VARS;
}

static void
get_gdb_lnum_entries(ap, f, lt, start_addr, end_addr)
alloc_pool_t *ap;
func_t *f;
struct linetable *lt;
CORE_ADDR start_addr, end_addr;
{
	lno_t first, *last;
	int i, max_lnum;

	max_lnum = 0;

	last = &first;

	/*  Don't want line entries from before the function prologue.
	 *  BUG: should be using gdb functions to do line number lookup.
	 */
	SKIP_PROLOGUE(start_addr);

	if (lt != NULL) {
		for (i = 0; i < lt->nitems; ++i) {
			CORE_ADDR pc;
		
			pc = lt->item[i].pc;

			if (pc >= start_addr && pc < end_addr) {
				lno_t *ln;

				ln = (lno_t *)alloc(ap, sizeof(lno_t));
				ln->ln_addr = pc;
				ln->ln_num = lt->item[i].line;

				last->ln_next = ln;
				last = ln;

				if (ln->ln_num > max_lnum)
					max_lnum = ln->ln_num;
			}
		}
        }
	
	last->ln_next = NULL;

	f->fu__lnos = first.ln_next;
	f->fu_flags |= FU_DONE_LNOS;
	f->fu_max_lnum = max_lnum;
}

static int
gdb_addr_to_lnum(lt, addr)
struct linetable *lt;
CORE_ADDR addr;
{
	int i, prev_lnum;

	if (lt == NULL)
		return 0;
	
	prev_lnum = 0;
	
	for (i = 0; i < lt->nitems; ++i) {
		if (lt->item[i].pc > addr)
			break;
		prev_lnum = lt->item[i].line;
	}

	return prev_lnum;
}

static block_t *
get_blocktree(ap, ht, par_bl, bv, bn, lt, p_next_bn)
alloc_pool_t *ap;
hashtab_t *ht;
block_t *par_bl;
struct blockvector *bv;
int bn;
struct linetable *lt;
int *p_next_bn;
{
	struct block *block;
	block_t first, *last, *bl;
	int next_bn;

	block = BLOCKVECTOR_BLOCK(bv, bn++);
	
	bl = ci_make_block(ap, par_bl);
	bl->bl_start_lnum = gdb_addr_to_lnum(lt, BLOCK_START(block));
	bl->bl_end_lnum = gdb_addr_to_lnum(lt, BLOCK_END(block));

	push_gdb_block_vars(ap, ht, bl, block, FALSE);

	last = &first;
	
	for (; bn < BLOCKVECTOR_NBLOCKS(bv); bn = next_bn) {
		struct block *subblock;

		subblock = BLOCKVECTOR_BLOCK(bv, bn);
		if (BLOCK_SUPERBLOCK(subblock) != block)
			break;

		last->bl_next = get_blocktree(ap, ht, bl, bv, bn, lt, &next_bn);
		last = last->bl_next;
	}
	last->bl_next = NULL;

	bl->bl_blocks = first.bl_next;

	*p_next_bn = bn;

	return bl;
}

static void
push_gdb_block_vars(ap, ht, bl, block, global)
alloc_pool_t *ap;
hashtab_t *ht;
block_t *bl;
struct block *block;
bool global;
{
	int i;
	
	for (i = 0; i < BLOCK_NSYMS(block); ++i) {
		struct symbol *sym;
		var_t *v;
		class_t class;
		struct type *gt;
		typedef_t *td;
		const char *name;
		aggr_or_enum_def_t *ae;

		sym = BLOCK_SYM(block, i);
		name = SYMBOL_NAME(sym);
		gt = SYMBOL_TYPE(sym);

		switch (SYMBOL_NAMESPACE(sym)) {
		case STRUCT_NAMESPACE:
			ae = parse_gdb_type(ap, ht, gt)->ty_aggr_or_enum;
			ae->ae_next = bl->bl_aggr_or_enum_defs;
			bl->bl_aggr_or_enum_defs = ae;
			continue;
		case VAR_NAMESPACE:
			break;
		default:
			continue;
		}
		
		switch (SYMBOL_CLASS(sym)) {
		case LOC_TYPEDEF:
			td = ci_make_typedef(ap, name,
					     parse_gdb_type(ap, ht, gt));
			td->td_next = bl->bl_typedefs;
			bl->bl_typedefs = td;
			continue;
			
		case LOC_STATIC:
			class = global ? CL_EXT : CL_STAT;
			break;
		case LOC_REGISTER:
		case LOC_REGPARM:
			class = CL_REG;
			break;
		case LOC_ARG:
			class = CL_ARG;
			break;
		case LOC_LOCAL:
			class = CL_AUTO;
			break;
		default:
			continue;
		}

		v = ci_make_var(ap, SYMBOL_NAME(sym), class,
				parse_gdb_type(ap, ht, gt),
				SYMBOL_VALUE_ADDRESS(sym));
		v->va_next = bl->bl_vars;
		bl->bl_vars = v;
	}
}

static void
enter_type(ap, ht, gt, type)
alloc_pool_t *ap;
hashtab_t *ht;
struct type *gt;
type_t *type;
{
	struct type **p_gt;

	p_gt = (struct type **)alloc(ap, sizeof(struct type *));
	*p_gt = gt;
	hash_enter(ht, (char *)p_gt, sizeof(*p_gt), (char *)type);
}	

static type_t *
parse_gdb_aggr(ap, ht, gt, is_struct)
alloc_pool_t *ap;
hashtab_t *ht;
struct type *gt;
bool is_struct;
{
	aggr_or_enum_def_t *ae;
	typecode_t typecode;
	bool is_complete;
	int i;

	is_complete = (TYPE_FLAGS(gt) & TYPE_FLAG_STUB) == 0;

	/*  The IRIX C compiler (and possibly others) records undefined
	 *  structs as structs with no members.
	 */
	if (TYPE_NFIELDS(gt) == 0)
		is_complete = FALSE;

	if (is_struct)
		typecode = is_complete ? TY_STRUCT : TY_U_STRUCT;
	else
		typecode = is_complete ? TY_UNION : TY_U_UNION;

	ae = ci_make_aggr_or_enum_def(ap, TYPE_TAG_NAME(gt),
				      typecode, (type_t *)NULL);
	ae->ae_is_complete = is_complete ? AE_COMPLETE : AE_INCOMPLETE;
	ae->ae_size = TYPE_LENGTH(gt);
	
	enter_type(ap, ht, gt,ae->ae_type);
	
	ae->ae_aggr_members = NULL;
	for (i = 0; i < TYPE_NFIELDS(gt); ++i) {
		var_t *v;
		type_t *type;
		int bitpos;
		taddr_t addr;

		bitpos = TYPE_FIELD_BITPOS(gt, i);
		addr = bitpos / 8;
		type = parse_gdb_type(ap, ht, TYPE_FIELD_TYPE(gt, i));
		
		if (TYPE_FIELD_PACKED(gt, i)) {
			type = ci_make_bitfield_type(ap, type,
						     (int)(bitpos - addr * 8),
						     TYPE_FIELD_BITSIZE(gt, i));
		}
		
		v = ci_make_var(ap, TYPE_FIELD_NAME(gt, i),
				is_struct ? CL_MOS : CL_MOU, type, addr);
		v->va_next = ae->ae_aggr_members;
		ae->ae_aggr_members = v;
	}

	if (ae->ae_size == 0)
		ae->ae_size = -1;

	return ae->ae_type;
}

static type_t *
parse_gdb_enum(ap, gt)
alloc_pool_t *ap;
struct type *gt;
{
	aggr_or_enum_def_t *ae;
	bool is_complete;
	int i;
	enum_member_t first, *last;

	is_complete = (TYPE_FLAGS(gt) & TYPE_FLAG_STUB) == 0;

	ae = ci_make_aggr_or_enum_def(ap, TYPE_TAG_NAME(gt),
				      is_complete ? TY_ENUM : TY_U_ENUM,
				      (type_t *)NULL);
	ae->ae_is_complete = is_complete ? AE_COMPLETE : AE_INCOMPLETE;
	ae->ae_size = TYPE_LENGTH(gt);

	if (ae->ae_size == 0)
		ae->ae_size = sizeof(class_t); /* or any enum */
	
	last = &first;
	for (i = 0; i < TYPE_NFIELDS(gt); ++i) {
		enum_member_t *em;
		
		em = ci_make_enum_member(ap,
					 TYPE_FIELD_NAME(gt, i),
					 TYPE_FIELD_BITPOS(gt, i));
		last->em_next = em;
		last = em;
	}
	last->em_next = NULL;
	ae->ae_enum_members = first.em_next;

	return ae->ae_type;
}
	
static type_t *
parse_gdb_type(ap, ht, gt)
alloc_pool_t *ap;
hashtab_t *ht;
struct type *gt;
{
	dim_t *dim;
	type_t *type;
	typecode_t typecode;
	bool is_unsigned, wantbase;
	int len;

	if ((type = (type_t *)hash_lookup(ht, (char *)&gt, sizeof(gt))) != NULL)
		return type;
	
	switch (TYPE_CODE(gt)) {
	case TYPE_CODE_PTR:
		type = ci_make_type(ap, DT_PTR_TO);
		type->ty_qualifiers = 0;
		wantbase = TRUE;
		break;
		
	case TYPE_CODE_ARRAY:
		dim = (dim_t *)alloc(ap, sizeof(dim_t));
		dim->di_ltype = dim->di_htype = DT_CONSTANT;
		dim->di_low = 0;
		dim->di_high = (TYPE_LENGTH(gt) / TYPE_LENGTH(TYPE_TARGET_TYPE(gt))) - 1;
		dim->di_high_expr_id = NULL;
		dim->di_type = ci_code_to_type(TY_INT);
		type = ci_make_type(ap, DT_ARRAY_OF);
		type->ty_dim = dim;
		
		wantbase = TRUE;
		break;

	case TYPE_CODE_FUNC:
		type = ci_make_type(ap, DT_FUNC_RETURNING);
		type->ty_funcret = NULL;
		wantbase = TRUE;
		break;

	case TYPE_CODE_STRUCT:
	case TYPE_CODE_UNION:
		type = parse_gdb_aggr(ap, ht, gt,
				      TYPE_CODE(gt) == TYPE_CODE_STRUCT);
		wantbase = FALSE;
		break;

	case TYPE_CODE_ENUM:
		type = parse_gdb_enum(ap, gt);
		wantbase = FALSE;
		break;

	case TYPE_CODE_FLT:
		switch (TYPE_LENGTH(gt)) {
		case sizeof(float):
			typecode = TY_FLOAT;
			break;
		case sizeof(double):
			typecode = TY_DOUBLE;
			break;
		default:
			typecode = TY_INT_ASSUMED;
			break;
		}
		type = ci_code_to_type(typecode);
		wantbase = FALSE;
		break;

	case TYPE_CODE_VOID:
		type = ci_code_to_type(TY_VOID);
		wantbase = FALSE;
		break;
		
	case TYPE_CODE_INT:
		is_unsigned = TYPE_UNSIGNED(gt);
		len = TYPE_LENGTH(gt);

		/*  We can't use a switch here because some of the type sizes
		 *  might be the same (e.g. int and long on most machines).
		 */
		if (len == sizeof(char)) {
			typecode = is_unsigned ? TY_UCHAR : TY_CHAR;
		}
		else if (len == sizeof(short)) {
			typecode = is_unsigned ? TY_USHORT : TY_SHORT;
		}
		else if (len == sizeof(int)) {
			typecode = is_unsigned ? TY_UINT : TY_INT;
		}
		else if (len == sizeof(long)) {
			typecode = is_unsigned ? TY_ULONG : TY_LONG;
		}
		else {
			typecode = TY_INT_ASSUMED;
		}
		
		type = ci_code_to_type(typecode);
		wantbase = FALSE;
		break;

	default:
		type = ci_code_to_type(TY_INT_ASSUMED);
		wantbase = FALSE;
		break;
	}

	/*  Gdb gives a size of zero for undefined enums with the SunPRO
	 *  C compiler on Solaris.  There may be other cases as well, so
	 *  leave the type as unknown unless the claimed type size is positive.
	 */
	if (TYPE_LENGTH(gt) > 0)
		type->ty_size = TYPE_LENGTH(gt);

	enter_type(ap, ht, gt, type);
	
	if (wantbase)
		type->ty_base = parse_gdb_type(ap, ht, TYPE_TARGET_TYPE(gt));

	return type;
}

static breakpoint_t *
gd_add_breakpoint(xp, addr)
target_t *xp;
taddr_t addr;
{
	struct breakpoint *b, *new_breakpoint;

	if (do_command(break_command, fmtnum("*0x%lx", addr)) != 0) {
		/*  The generic code doesn't currently handle a failure to
		 *  add a breakpoint - maybe it should.
		 */
		panic("failed to add breakpoint");
		return NULL;	/* to satisfy gcc */
	}

	new_breakpoint = NULL;
	
	for (b = breakpoint_chain; b != NULL; b = b->next) {
		if (b->address == addr) {
			if (new_breakpoint != NULL)
				panic("dup bp in ab");
			new_breakpoint = b;
		}
	}
	
	if (new_breakpoint == NULL)
		panic("failed to add breakpoint");

	return (breakpoint_t *)new_breakpoint;
}

static int
gd_remove_breakpoint(xp, bp)
target_t *xp;
breakpoint_t *bp;
{
	return do_command(delete_command,
			  fmtnum("%ld", (((struct breakpoint *)bp)->number)));
}

static int
gd_enable_breakpoint(xp, bp)
target_t *xp;
breakpoint_t *bp;
{
	return change_bp_status(enable_command, (struct breakpoint *)bp);
}

static int
gd_disable_breakpoint(xp, bp)
target_t *xp;
breakpoint_t *bp;
{
	return change_bp_status(disable_command, (struct breakpoint *)bp);
}

static int
change_bp_status(cmdfunc, b)
Gdb_cmdfunc cmdfunc;
struct breakpoint *b;
{
	const char *arg;

	arg = (b != NULL) ? fmtnum("%ld", (unsigned long)b->number) : NULL;
	return do_command(cmdfunc, arg);
}

static const char *
fmtnum(fmt, num)
const char *fmt;
unsigned long num;
{
	static char buf[30];

	sprintf(buf, fmt, num);
	return buf;
}

static breakpoint_t *
gd_addr_to_breakpoint(xp, addr)
target_t *xp;
taddr_t addr;
{
	struct breakpoint *b;
	
	for (b = breakpoint_chain; b != NULL; b = b->next) {
		if (b->address == addr)
			return (breakpoint_t *)b;
	}
	
	return NULL;
}

static taddr_t
gd_get_breakpoint_addr(xp, bp)
target_t *xp;
breakpoint_t *bp;
{
	return ((struct breakpoint *)bp)->address;
}

CORE_ADDR
parse_and_eval_address_1(expptr)
char **expptr;
{
	char *ends;
	CORE_ADDR val;
	
	val = strtoul(*expptr, &ends, 0);

	if (ends == *expptr || *ends != '\0')
		panic("val botch in paea1");

	*expptr = ends;
	return val;
}

static void
get_symtab_and_line(fi, p_symtab, p_line)
struct frame_info *fi;
struct symtab **p_symtab;
int *p_line;
{
	struct symtab_and_line sal;
		
	if (fi == NULL) {
		*p_symtab = NULL;
		*p_line = 0;
		return;
	}

	CATCH_ERRORS(*p_symtab = NULL; *p_line = 0; return);
	sal = find_pc_line(fi->pc, fi->next != NULL &&
			   	   fi->next->signal_handler_caller == 0);
	do_all_gdb_cleanups();

	*p_symtab = sal.symtab;
	*p_line = sal.line;
}

static stopres_t
do_next_or_step(cmdfunc)
Gdb_cmdfunc cmdfunc;
{
	stopres_t stopres;
	struct frame_info *fi, *inner_fi;

	/*  Ups does not stop if it hits a breakpoint during a next or step.
	 */
	if (change_bp_status(disable_command, (struct breakpoint *)NULL) != 0)
		return SR_FAILED;
	
	fi = (struct frame_info *)get_current_stack_data();

	if (fi != NULL && (inner_fi = get_next_frame(fi)) != NULL)
		stopres = return_then_next_or_step(cmdfunc, inner_fi);
	else
		stopres = run_gdb_target(cmdfunc, (char *)NULL);

	change_bp_status(enable_command, (struct breakpoint *)NULL);

	return stopres;
}

static stopres_t
return_then_next_or_step(cmdfunc, fi)
Gdb_cmdfunc cmdfunc;
struct frame_info *fi;
{
	struct symtab *old_symtab;
	int old_line;
	stopres_t stopres;

	get_symtab_and_line(get_prev_frame(fi), &old_symtab, &old_line);
		
	select_frame(fi, -1);
	stopres = run_gdb_target(finish_command, (char *)NULL);

	if (stopres == SR_BPT) {
		struct symtab *new_symtab;
		int new_line;
			
		get_symtab_and_line(get_current_frame(),
				    &new_symtab, &new_line);

		if (new_symtab == old_symtab && new_line == old_line)
			stopres = run_gdb_target(cmdfunc, (char *)NULL);
	}

	return stopres;
}

static stopres_t
gd_next(xp)
target_t *xp;
{
	return do_next_or_step(next_command);
}

static stopres_t
gd_step(xp)
target_t *xp;
{
	return do_next_or_step(step_command);
}

static stopres_t
gd_cont(xp)
target_t *xp;
{
	return run_gdb_target(continue_command, (char *)NULL);
}

static stopres_t
run_gdb_target(func, arg)
Gdb_cmdfunc func;
char *arg;
{
	CORE_ADDR orig_fp;
	
	orig_fp = (func == next_command) ? get_current_frame()->frame : 0;
	
	for (;;) {
		abort_func_t oldfunc;
		int res;
		bool stop;
		CORE_ADDR pc;
		struct breakpoint *b;
		struct frame_info *current_fi;

		Sent_sigint_to_child = FALSE;
		
		indicate_target_running();
		oldfunc = set_user_abort_func(stop_gdb_target);

		res = do_command(func, arg);

		set_user_abort_func(oldfunc);
		indicate_target_stopped(get_gdb_target_state());
				
		if (res != 0)
			return SR_FAILED;

		if (func == start_gdb_inferior) {
			func = continue_command;
			arg = NULL;
		}
		
		if (Sent_sigint_to_child) {
			if (!user_wants_stop())
				continue;
			if (stop_signal == TARGET_SIGNAL_INT)
				return SR_USER;
		}

		if (!target_has_execution)
			return SR_DIED;

		if (stop_bpstat == NULL && stop_signal != TARGET_SIGNAL_TRAP)
			return SR_SIG;

		/*  If we do `next' in a recursive function gdb (at least on
		 *  Solaris 2.3) sometimes stops at the next line in an inner
		 *  recursive call of the function.  Try to spot this and fix
		 *  it up.
		 */
		current_fi = get_current_frame();
		if (func == next_command &&
		    current_fi->frame INNER_THAN orig_fp) {
			struct frame_info *fi, *inner_fi;

			inner_fi = NULL;
			for (fi = current_fi; fi != NULL;
			     			      fi = get_prev_frame(fi)) {
				if (fi->frame == orig_fp)
					break;
				inner_fi = fi;
			}

			if (fi != NULL && inner_fi != NULL)
				return return_then_next_or_step(func, inner_fi);
		}

		/*  If we have nexted or stepped off the end of main,
		 *  arrange to continue until the target exits.
		 */
		if (get_prev_frame(current_fi) == NULL) {
			func_t *f;

			f = addr_to_func(current_fi->pc);
			if (f != NULL && (strcmp(f->fu_name, "_start") == 0 ||
					  strcmp(f->fu_name, "__start") == 0)) {
				func = continue_command;
				continue;
			}
		}
		    
		pc = read_pc();
		for (b = breakpoint_chain; b != NULL; b = b->next)
			if (b->address == pc)
				break;
		
		if (b == NULL) {
			stop = TRUE;
		}
		else {
			taddr_t fp;

			fp = read_fp();
			stop = execute_bp_code((breakpoint_t *)b, fp, fp);
		}
			
		/* TODO: check result of calling target functions */
		
		if (stop || func != continue_command)
			return (stop_bpstat != NULL) ? SR_BPT : SR_SSTEP;
	}
}

static void
stop_gdb_target()
{
	if (inferior_pid != 0) {
		/*  Using SIGTRAP here is a bit dubious, but SIGINT doesn't
		 *  stop the target on IRIX.  Leave it this way for now, as
		 *  wait_for_inferior() seems not be fazed by an unexpected
		 *  SIGTRAP.
		 */
		kill(inferior_pid, SIGTRAP);
		Sent_sigint_to_child = TRUE;
	}
}

static stopres_t
gd_start(xp)
target_t *xp;
{
	stopres_t stopres;
	char *cmdline;
	struct objfile *objfile;
	infargs_t iabuf;

	if (setup_shellcmd(FALSE, &cmdline) != 0)
		return SR_DIED;

	/*  We get the path from the objfile rather than using xp_textpath
	 *  because fork_inferior() wants the absolute path, and xp_textpath
	 *  can be a relative pathname.
	 */
	objfile = GET_STD(xp->xp_symtabs)->st_objfile;

	iabuf.ia_path = objfile->name;
	iabuf.ia_cmdline = cmdline;
	iabuf.ia_envp = get_environment();
	stopres = run_gdb_target(start_gdb_inferior, (char *)&iabuf);

	if (stopres != SR_DIED)
		update_shared_library_symtabs(xp);

	return stopres;
}

static void
start_gdb_inferior(arg, from_tty)
char *arg;
bool from_tty;
{
	infargs_t *ia;
	
	ia = (infargs_t *)arg;
	
	if (Want_gdb_cmdlog)
		logcmd(stdout, "run", ia->ia_cmdline);
	
	target_create_inferior(ia->ia_path, ia->ia_cmdline,
			       (char **)ia->ia_envp);
}


/*  Tack any new shared library symbol tables on the end of our
 *  existing list of symbol tables.  In particular, the first time
 *  the target is run this pulls in the shared library symtabs.
 */
static void
update_shared_library_symtabs(xp)
target_t *xp;
{
	symtab_t first, *last, *st;
	struct objfile *objfile;
	
	last = &first;
	for (objfile = object_files; objfile != NULL; objfile = objfile->next) {
		for (st = xp->xp_symtabs; st != NULL; st = st->st_next) {
			if (GET_STD(st)->st_objfile == objfile)
				break;
		}

		if (st != NULL) {
			/* TODO: check for relocation here.
			 */
		}
		else {
			last->st_next = objfile_to_st(objfile, FALSE);
			last = last->st_next;
		}
	}
	last->st_next = NULL;

	for (st = xp->xp_symtabs; st != NULL; st = st->st_next)
		last = st;
	last->st_next = first.st_next;
}

static void
gd_kill(xp)
target_t *xp;
{
	do_command(kill_command, (const char *)NULL);
}

static int
gd_is_attached(xp)
target_t *xp;
{
	return attach_flag;
}

static void
gd_detach(xp, sig)
target_t *xp;
int sig;
{
	do_command(detach_command, fmtnum("%ld", (unsigned long)sig));
}

/*  BUG: duplicate of function in ao_proc.c
 */
static bool
sig_is_fatal(sig)
int sig;
{
	return sig == SIGILL || sig == SIGSEGV || sig == SIGBUS;
}

static tstate_t
gd_get_state(xp)
target_t *xp;
{
	return get_gdb_target_state();
}

static tstate_t
get_gdb_target_state()
{
	if (target_has_execution) {
		int signo;

		signo = target_signal_to_host(stop_signal);
		return sig_is_fatal(signo) ? TS_HALTED : TS_STOPPED;
	}
	else {
		return target_has_stack ? TS_CORE : TS_NOTR;
	}
}

static int
gd_get_lastsig(xp)
target_t *xp;
{
	int sig;
	
	if (!target_has_execution && core_bfd != NULL) {
		sig = bfd_core_file_failing_signal(core_bfd);
		if (sig <= 0)
			sig = 0;
	}
	else {
		if (stop_signal == TARGET_SIGNAL_TRAP)
			sig = 0;
		else
			sig = target_signal_to_host(stop_signal);
	}

	return sig;
}

static sigstate_t
gd_get_sigstate(xp, sig)
target_t *xp;
int sig;
{
	panic("gd_get_sigstate NYI");
	return 0;		/* to satisfy gcc */
}

static int
get_gdb_prev_frame(fi, p_prev)
struct frame_info *fi, **p_prev;
{
	CATCH_ERRORS(return -1);
	*p_prev = get_prev_frame(fi);
	do_all_gdb_cleanups();
	return 0;
}

static Stack *
gd_get_stack_trace(xp)
target_t *xp;
{
	struct frame_info *fi;
	Stack *list;

	list = NULL;

	fi = get_current_frame();

	while (fi != NULL) {
		func_t *f;
		Stack *stk;
		CORE_ADDR pc;
		struct frame_info *prev;

		if ((f = addr_to_func(fi->pc)) == NULL)
			f = make_badfunc();

		pc = fi->pc;
		if (fi->next != NULL && fi->next->signal_handler_caller == 0)
			--pc;

		stk = make_stk(f, fi->pc, addr_to_lnum(f, pc), list);
		stk->stk_fp = FRAME_LOCALS_ADDRESS(fi);
		stk->stk_ap = FRAME_ARGS_ADDRESS(fi);
		stk->stk_sp = fi->frame;
		stk->stk_data = (char *)fi;

		list = stk;
		
		if (get_gdb_prev_frame(fi, &prev) != 0)
			break;
		fi = prev;
	}
	
	if (fi != NULL)
		list->stk_bad = TRUE;

	return list;
}

static const char *
gd_get_signal_tag(xp, sig)
target_t *xp;
int sig;
{
	return signame(sig);
}

static int
gd_read_fpval(xp, addr, is_double, p_val)
target_t *xp;
taddr_t addr;
bool is_double;
fpval_t *p_val;
{
	if (is_double) {
		return gd_read_data(xp, addr,
				    (char *)&p_val->d, sizeof(double));
	}
	else {
		return gd_read_data(xp, addr,
				    (char *)&p_val->f, sizeof(float));
	}
}

static int
gd_read_fpreg(xp, regno, is_double, p_val)
target_t *xp;
int regno;
bool is_double;
fpval_t *p_val;
{
	struct value *value;
	
	CATCH_ERRORS(return -1);
	
	if (is_double) {
		value = value_from_register(builtin_type_double, regno,
					    (struct frame_info *)NULL);
		p_val->d = value_as_double(value);
	}
	else {
		value = value_from_register(builtin_type_float, regno,
					    (struct frame_info *)NULL);
		p_val->f = value_as_double(value);
	}

	do_all_gdb_cleanups();

	return 0;
}

static taddr_t
gd_funcptr_to_addr(st, addr)
symtab_t *st;
taddr_t addr;
{
	return addr;
}

static taddr_t
gd_addr_to_funcptr(st, addr)
symtab_t *st;
taddr_t addr;
{
	return addr;
}

static taddr_t
gd_get_reg_addr(xp, stk, reg)
target_t *xp;
Stack *stk;
int reg;
{
	if (stk == NULL)
		return 0;
	return find_saved_register((struct frame_info *)stk->stk_data, reg);
}

static int
gd_readreg(xp, regno, p_val)
target_t *xp;
int regno;
taddr_t *p_val;
{
	struct value *value;
	
	CATCH_ERRORS(return -1);

	switch (regno) {
	case UPSREG_PC:
		*p_val = read_pc();
		break;
	case UPSREG_SP:
		*p_val = read_sp();
		break;
	case UPSREG_FP:
	case UPSREG_AP:
		*p_val = read_fp();
		break;
	default:
		value = value_from_register(builtin_type_long, regno,
					    (struct frame_info *)NULL);
		*p_val = value_as_long(value);
		break;
	}

	do_all_gdb_cleanups();
	
	return 0;
}

static int
gd_setreg(xp, regno, val)
target_t *xp;
int regno;
taddr_t val;
{
	CATCH_ERRORS(return -1);
	
	switch (regno) {
	case UPSREG_PC:
		write_pc(val);
		break;
	case UPSREG_SP:
		write_sp(val);
		break;
	case UPSREG_FP:
	case UPSREG_AP:
		write_fp(val);
		break;
	default:
		write_register(regno, (LONGEST)val);
		break;
	}
	
	do_all_gdb_cleanups();
	
	return 0;
}

static int
gd_read_data(xp, addr, buf, nbytes)
target_t *xp;
taddr_t addr;
char *buf;
size_t nbytes;
{
	if (target_read_memory(addr, buf, (int)nbytes) != 0)
		return -1;
	return 0;
}

static int
gd_write_data(xp, addr, buf, nbytes)
target_t *xp;
taddr_t addr;
const char *buf;
size_t nbytes;
{
	if (target_write_memory(addr, (char *)buf, (int)nbytes) != 0)
		return -1;
	return 0;
}

static int
gd_read_text_from_process(xp, addr, buf, nbytes)
target_t *xp;
taddr_t addr;
char *buf;
size_t nbytes;
{
	panic("gd_read_text_from_process NYI");
	return -1;		/* to satisfy gcc */
}

static int
gd_call_func(xp, cma, funcaddr, args, argsizes, nargs, p_res, p_mesg)
target_t *xp;
char *cma;
taddr_t funcaddr;
taddr_t *args;
size_t *argsizes;
int nargs;
taddr_t *p_res;
const char **p_mesg;
{
	struct value *funcval, *realargs[40], *retval;
	struct type *type;
	int i;

	CATCH_ERRORS(return -1);
	
	if (nargs > sizeof realargs / sizeof *realargs)
		panic("nargs too large in gd_call_func");

	/*  Make copies of the targets of pointers and arrays.
	 *
	 *  These are indicated by a non zero size.
	 */
	for (i = 0; i < nargs; ++i) {
		int size;
		CORE_ADDR addr;

		size = argsizes[i];
		if (size > 0 && ci_is_ci_addr((machine_t *)cma, args[i],
					      (size_t)size)) {
			if ((addr = allocate_space_in_inferior(size)) == 0) {
				errf("No memory available in target process");
				return -1;
			}
			
			write_memory(addr, (char *)args[i], size);
		}
		else {
			addr = (CORE_ADDR)args[i];
		}

		realargs[i] = value_from_longest(builtin_type_int,
						 (LONGEST)addr);
	}

	/*  BUG: we assume all functions return int, just like in the
	 *       native code.  We need to rewrite this interface some time.
	 */
	type = lookup_pointer_type(builtin_type_int);
	type = lookup_function_type(type);
	type = lookup_pointer_type(type);
	funcval = value_from_longest(type, (LONGEST)funcaddr);

	retval = call_function_by_hand(funcval, nargs, realargs);
	*p_res = value_as_long(retval);

	do_all_gdb_cleanups();
	
	return 0;
}

static int
gd_read_text(st, addr, buf, nbytes)
symtab_t *st;
taddr_t addr;
char *buf;
size_t nbytes;
{
	panic("gd_read_text NYI");
	return -1;		/* to satisfy gcc */
}

static int
gd_get_min_bpt_addr(f, p_addr)
func_t *f;
taddr_t *p_addr;
{
	CATCH_ERRORS(return -1);
	*p_addr = gdb_skip_prologue(f->fu_addr);
	do_all_gdb_cleanups();
	return 0;
}

static CORE_ADDR
gdb_skip_prologue(addr)
CORE_ADDR addr;
{
	SKIP_PROLOGUE(addr);
	return addr;
}

static taddr_t
gd_get_addr_lim(f)
func_t *f;
{
	panic("gd_get_addr_lim NYI");
	return 0;		/* to satisfy gcc */
}

static bool
match_symbol(symbase, symcount, matchf, pat)
struct partial_symbol *symbase;
int symcount;
matchfunc_t matchf;
const char *pat;
{
	struct partial_symbol *psym;
	
	for (psym = symbase; psym < symbase + symcount; ++psym) {
		if (PSYMBOL_CLASS(psym) == LOC_STATIC &&
		    (*matchf)(SYMBOL_NAME(psym), pat))
			return TRUE;
	}

	return FALSE;
}

static bool
gd_fil_may_have_matching_globals(fil, pat, matchf)
fil_t *fil;
const char *pat;
matchfunc_t matchf;
{
	struct partial_symtab *ps;
	struct objfile *objfile;

	ps = GET_FID(fil)->fi_partial_symtab;
	objfile = GET_STD(fil->fi_symtab)->st_objfile;

	return match_symbol(objfile->global_psymbols.list + ps->globals_offset,
			    ps->n_global_syms, matchf, pat) ||
	       match_symbol(objfile->static_psymbols.list + ps->statics_offset,
			    ps->n_static_syms, matchf, pat);
}

static void
gd_close_symtab_data(st)
symtab_t *st;
{
	panic("gd_close_symtab_data NYI");
}

static var_t *
gd_get_fi_vars(fil)
fil_t *fil;
{
	if ((fil->fi_flags & FI_DONE_VARS) == 0)
		load_gdb_symtab(fil);

	return fil->fi_block->bl_vars;
}

static block_t *
gd_get_fu_blocks(f)
func_t *f;
{
	if ((f->fu_flags & FU_DONE_BLOCKS) == 0)
		load_gdb_symtab(f->fu_fil);

	return f->fu__blocks;
}

static lno_t *
gd_get_fu_lnos(f)
func_t *f;
{
	if ((f->fu_flags & FU_DONE_LNOS) == 0)
		load_gdb_symtab(f->fu_fil);
	
	return f->fu__lnos;
}

static const char *
gd_disassemble_instruction(f, addr, text, p_buf)
func_t *f;
taddr_t addr;
const char *text, **p_buf;
{
	panic("gd_disassemble_instruction NYI");
	return NULL;		/* to satisfy gcc */
}
