/* ao_shlib.c - SunOS 4.X shared library routines */

/*  Copyright 1993 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_st_shlib_c_sccsid[] = "@(#)ao_shlib.c	1.5 24 May 1995 (UKC)";

#include <mtrprog/ifdefs.h>
#include "ao_ifdefs.h"

#ifdef AO_TARGET

#ifdef OS_SUNOS_4
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <a.out.h>
#include <link.h>

#ifdef __STDC__
#include <unistd.h>
#endif

#include <local/ukcprog.h>
#include <mtrprog/utils.h>
#include <mtrprog/io.h>

#include "ups.h"
#include "symtab.h"
#include "target.h"
#include "st.h"
#include "ao_syms.h"
#include "ao_shlib.h"
#include "ao_symload.h"
#include "ao_symread.h"
#include "ao_symscan.h"
#include "ao_target.h"
#include "ao_text.h"
#include "ao_aout.h"
#include "data.h"

static int get_preload_shlib_list PROTO((alloc_pool_t *ap,
					 const char *textpath,
					 shlib_t **p_shlibs));
static int get_shlibs_and_global_addrs PROTO((target_t *xp, alloc_pool_t *ap,
				              symtab_t *st,
					      taddr_t addr_dynamic,
				              shlib_t **p_shlibs));
static const char **add_to_env PROTO((const char *s));
static int load_shlib_symtabs PROTO((shlib_t *shlibs, symtab_t **p_stlist,
				     target_t *xp));
static void free_symtab_cache PROTO((target_t *xp));
static symtab_t *get_symtab_from_cache PROTO((target_t *xp, const char *name));

#define NEXT_ST(st)	AO_STDATA(st)->st_next

int
make_symtab_cache(textpath, p_stcache)
const char *textpath;
symtab_t **p_stcache;
{
	shlib_t *shlibs;
	symtab_t *stlist;
	alloc_pool_t *ap;

	ap = alloc_create_pool();
	get_preload_shlib_list(ap, textpath, &shlibs);

	if (load_shlib_symtabs(shlibs, &stlist, (target_t *)NULL) != 0) {
		alloc_free_pool(ap);
		return -1;
	}

	alloc_free_pool(ap);

	*p_stcache = stlist;
	return 0;
}

bool
aout_next_symtab(xp, st, load_new, p_next_st)
target_t *xp;
symtab_t *st;
bool load_new;
symtab_t **p_next_st;
{
	iproc_t *ip;
	symtab_t *next_st;
	
	ip = GET_IPROC(xp);

	if (st == NULL) {
		next_st = ip->ip_main_symtab;
	}
	else if (st == ip->ip_main_symtab) {
		if (ip->ip_shlib_symtabs != NULL)
			next_st = ip->ip_shlib_symtabs;
		else
			next_st = ip->ip_symtab_cache;
	}
	else {
		next_st = NEXT_ST(st);
	}

	if (next_st == NULL)
		return FALSE;
	
	*p_next_st = next_st;
	return TRUE;
}

/*  Dig out the list of loaded shared libraries and run time linked
 *  global addresses.  This basically involves trundling down linked	
 *  lists in the target via dread().  This must be called after the
 *  target has started and the mapping been done.
 */
static int
get_shlibs_and_global_addrs(xp, ap, st, addr_dynamic, p_shlibs)
target_t *xp;
alloc_pool_t *ap;
symtab_t *st;
taddr_t addr_dynamic;
shlib_t **p_shlibs;
{
	struct link_dynamic ldbuf;
	struct link_dynamic_1 ld1buf;
	struct link_map lmbuf, *lm;
	struct ld_debug lddbuf;
#ifdef COFF_SUN386
	struct syment nm;
#define rtc_sp		rtc_un.rtc_sp_coff
#define n_corename	n_offset
#else
	struct nlist nm;
#define n_corename	n_un.n_name
#endif
	struct rtc_symb rtcbuf, *rtc;
	shlib_t *shlist, *sh;
	char buf[256];

	if (dread(xp, addr_dynamic, (char *)&ldbuf, sizeof(ldbuf)) != 0)
		return -1;
	
	/*  Get the list of addresses of shared data objects.
	 */
	if (dread(xp, (taddr_t)ldbuf.ldd, (char *)&lddbuf, sizeof(lddbuf)) != 0)
		return -1;
	for (rtc = lddbuf.ldd_cp; rtc != NULL; rtc = rtcbuf.rtc_next) {
		const char *modname, *fname;
		
		if (dread(xp, (taddr_t)rtc,
					(char *)&rtcbuf, sizeof(rtcbuf)) != 0)
			return -1;
		if (dread(xp, (taddr_t)rtcbuf.rtc_sp,
						(char *)&nm, sizeof(nm)) != 0)
			return -1;
		if (dgets(xp, (taddr_t)nm.n_corename, buf, sizeof(buf)) != 0)
			return -1;

		parse_fname(st->st_apool, buf, '\0', TRUE, &modname, &fname);
		insert_global_addr(st->st_apool, &st->st_addrlist,
				   fname, nm.n_value);
	}

	/*  Now dig out the list of loaded shared libraries.
	 */
	if (dread(xp, (taddr_t)ldbuf.ld_un.ld_1,
					(char *)&ld1buf, sizeof(ld1buf)) != 0)
		return -1;

	shlist = NULL;
	for (lm = ld1buf.ld_loaded; lm != NULL; lm = lmbuf.lm_next) {
		if (dread(xp, (taddr_t)lm, (char *)&lmbuf, sizeof(lmbuf)) != 0)
			return -1;
		if (dgets(xp, (taddr_t)lmbuf.lm_name, buf, sizeof(buf)) != 0)
			return -1;
		sh = (shlib_t *)alloc(ap, sizeof(shlib_t));
		sh->sh_name = alloc_strdup(ap, buf);
		sh->sh_addr = (taddr_t)lmbuf.lm_addr;
		sh->sh_next = shlist;
		shlist = sh;
	}

	*p_shlibs = shlist;
	return 0;
}

/*  Return a pointer to an array of strings which consists of the
 *  environment plus string s, which should be of the form
 *  "name=value".
 */
static const char **
add_to_env(s)
const char *s;
{
	extern const char **environ;
	const char **sptr, **envp, **src, **dst;

	for (sptr = environ; *sptr != NULL; ++sptr)
		;
	envp = dst = (const char **)e_malloc(sizeof(char *) * (sptr - environ + 2));
	src = environ;
	while (*src != NULL)
		*dst++ = *src++;
	*dst++ = s;
	*dst = NULL;
	return envp;
}

/*  Get the list of shared libraries that the textpath will load
 *  when it is run.  We do this in the same way as ldd(1), by
 *  running the target with LD_TRACE_LOADED_OBJECTS set in its
 *  environment and parsing the output.
 */
static int
get_preload_shlib_list(ap, textpath, p_shlibs)
alloc_pool_t *ap;
const char *textpath;
shlib_t **p_shlibs;
{
	int pid, wpid;
	int fds[2];
	char buf[256], name[256];
	char *pos;
	const char **envp;
	FILE *fp;
	shlib_t *sh, *shlist;

	envp = add_to_env("LD_TRACE_LOADED_OBJECTS=1");
	if (pipe(fds) != 0) {
		errf("Pipe failed: %s", get_errno_str());
		free((char *)envp);
		return -1;
	}
	if ((fp = fdopen(fds[0], "r")) == NULL) {
		errf("Fdopen failed: %s", get_errno_str());
		free((char *)envp);
		return -1;
	}
	if ((pid = vfork()) == -1) {
		errf("Vfork failed: %s", get_errno_str());
		fclose(fp);
		close(fds[1]);
		free((char *)envp);
		return -1;
	}
	if (pid == 0) {
		close(fds[0]);
		if (fds[1] != 1)
			dup2(fds[1], 1);
		execle(textpath, textpath, (char *)NULL, envp);
		failmesg("Can't exec", "", textpath);
		_exit(1);
	}

	free((char *)envp);
	close(fds[1]);

	shlist = NULL;
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		/*  We seem to get carriage returns as well as newlines
		 */
		if ((pos = strchr(buf, '\r')) != NULL)
			*pos = '\0';
		if ((pos = strchr(buf, '\n')) != NULL)
			*pos = '\0';
		if (sscanf(buf, "%*s => %s", name) == 1) {
			sh = (shlib_t *)alloc(ap, sizeof(shlib_t));
			sh->sh_name = alloc_strdup(ap, name);
			sh->sh_addr = 0;
			sh->sh_next = shlist;
			shlist = sh;
		}
	}
	fclose(fp);
	while ((wpid = wait((int *)NULL)) != pid && wpid != -1)
		;
	*p_shlibs = shlist;
	return 0;
}

/*  Free any symbol tables in the cache.  This is called just
 *  after startup of the target to flush any cached symbol
 *  tables that weren't in fact used.
 */
static void
free_symtab_cache(xp)
target_t *xp;
{
	symtab_t *st, *next;
	
	for (st = GET_IPROC(xp)->ip_symtab_cache; st != NULL; st = next) {
		next = NEXT_ST(st);
		close_symtab(st);
	}
}

/*  Look up the symbol table for name in the cache.  Remove
 *  it from the cache and return it if found, otherwise
 *  return NULL.
 *
 *  Name is the name of the executable or shared library file
 *  that the symbol table came from.
 */
static symtab_t *
get_symtab_from_cache(xp, name)
target_t *xp;
const char *name;
{
	iproc_t *ip;
	symtab_t *st, *prev;

	ip = GET_IPROC(xp);
	
	prev = NULL;
	for (st = ip->ip_symtab_cache; st != NULL; st = NEXT_ST(st)) {
		if (strcmp(name, st->st_path) == 0) {
			if (prev != NULL)
				NEXT_ST(prev) = NEXT_ST(st);
			else
				ip->ip_symtab_cache = NEXT_ST(st);
			NEXT_ST(st) = NULL;		/* for safety */
			return st;
		}
		prev = st;
	}
	return NULL;
}

/*  Load the symbol tables for the shared libraries of an object.
 *  This is called just after the target starts, because only
 *  then have the shared libraries been loaded and mapped.
 */
int
load_shared_library_symtabs(xp)
target_t *xp;
{
	static char dynamic[] = "__DYNAMIC";
	const char *sym0_name;
	nlist_t nm;
	taddr_t addr;
	shlib_t *shlibs;
	symtab_t *main_st, *stlist;
	ao_stdata_t *main_ast;
	alloc_pool_t *ap;
	iproc_t *ip;

	ip = GET_IPROC(xp);
	
	main_st = ip->ip_main_symtab;
	main_ast = AO_STDATA(main_st);

	if (ip->ip_main_symtab == NULL || ip->ip_shlib_symtabs != NULL)
		panic("shared lib botch");

	
	if (main_ast->st_dynamic) {
		sym0_name = symstring(main_ast->st_text_symio, 0);
		if (strcmp(sym0_name, dynamic) != 0) {
			errf("First symbol in %s is %s (expected %s)",
					main_st->st_path, sym0_name, dynamic);
			return -1;
		}
		getsym(main_ast->st_text_symio, 0, &nm);
		addr = nm.n_value;

		ap = alloc_create_pool();

		if (get_shlibs_and_global_addrs(xp, ap, main_st, addr,
						&shlibs) != 0) {
			alloc_free_pool(ap);
			return -1;
		}

		if (load_shlib_symtabs(shlibs, &stlist, xp) != 0) {
			alloc_free_pool(ap);
			return -1;
		}

		alloc_free_pool(ap);
		free_symtab_cache(xp);
		ip->ip_shlib_symtabs = stlist;
	}
	
	return 0;
}

/*  Load the shared libraries desribed in shlibs, and point *p_stlist
 *  at the loaded list.  Return 0 for success, -1 for failure.
 *  In the normal case all the shared libraries will be in the
 *  cache, either from the preload or from a previous run of the
 *  target.
 */
static int
load_shlib_symtabs(shlibs, p_stlist, xp)
shlib_t *shlibs;
symtab_t **p_stlist;
target_t *xp;
{
	symtab_t *stlist, *st;
	shlib_t *sh;
	int shlib_fd;
	taddr_t junk;

	stlist = NULL;
	for (sh = shlibs; sh != NULL; sh = sh->sh_next) {
		if (xp != NULL &&
		    (st = get_symtab_from_cache(xp, sh->sh_name)) != NULL) {
			change_base_address(st, sh->sh_addr);
		}
		else {
			if (!open_for_reading(sh->sh_name, "shared_library",
					      &shlib_fd) ||
			    !open_textfile(shlib_fd, sh->sh_name, &junk) ||
			    !aout_scan_symtab(sh->sh_name, shlib_fd,
					      sh->sh_addr, &st,
					      (func_t **)NULL))
				return -1;
		}
		NEXT_ST(st) = stlist;
		stlist = st;
	}
	*p_stlist = stlist;
	return 0;
}

/*  Unload shared library symbol tables.  This is called when the
 *  target dies.  We don't free the symbol tables at this point - we
 *  just put them in the cache in anticipation of using them again
 *  when the target is re-run.
 */
void
unload_shared_library_symtabs(xp)
target_t *xp;
{
	iproc_t *ip;

	ip = GET_IPROC(xp);
	
	ip->ip_symtab_cache = ip->ip_shlib_symtabs;
	ip->ip_shlib_symtabs = NULL;
}
#endif /* OS_SUNOS_4 */

#endif /* AO_TARGET */
