#ifndef TEST_FCAP_C

#define __SMP__

# include <linux/kernel.h>
# include <linux/module.h>
# include <linux/slab.h>

# include <linux/fs.h>
# include <linux/sched.h>
# include <linux/proc_fs.h>

#endif /* ndef TEST_FCAP_C */

/* TODO: need a spinlock to protect changes to the chain etc. */
/* TODO: need to think about liberating the allocated memory */

/* the non-hashed fields are a simple effort to catch when the file
   has changed -- without this module being notified. Before the
   kernel patch is complete, we need to instrument all parts of the
   kernel that modify the file, to announce their changes via a call
   to the vfs layer. */

struct file_index {

    /* linked list */
    struct file_index *next;

    /* reference */
    unsigned long fi_ino;
    kdev_t fi_dev;

    /* HACK: clues to verify that inode has not changed */

    umode_t fi_mode;
    uid_t fi_uid;
    gid_t fi_gid;
    time_t fi_mtime;
    __u32 fi_generation;

    /* pretty printing */
    char *full_page;
    char *filename;

    /* stored capabilities */

    kernel_cap_t fi_effective;
    kernel_cap_t fi_inheritable;
    kernel_cap_t fi_permitted;
};

#define FI_BINS           37
#define FI_BLKS           10

/*
 * global state info
 */

static unsigned saved_securebits;
static unsigned extra_ref_count = 0;  /* vague idea about knowing when its
					 safe to unload... (XXX - to be used)*/

static spinlock_t fcap_access = SPIN_LOCK_UNLOCKED;

static struct file_index **fi_tab = NULL;
static struct file_index *chain = NULL;

/* linked list allocation/removal */

static struct file_index *alloc_fi(void)
{
    struct file_index *remains;

    if (chain == NULL) {
	int i;

	chain = (struct file_index *) vmalloc(sizeof(struct file_index)
					      * FI_BLKS);
	memset(chain, 0, FI_BLKS * sizeof(*chain));

	for (i=0; ++i<FI_BLKS; chain[i-1].next = (chain+i));
    }

    remains = chain;
    chain = chain->next;
    remains->next = NULL;

    return remains;
}

static struct file_index *free_fi(struct file_index *fi)
{
    struct file_index *remains;

    /* clean up */
    remains = fi->next;
    memset(fi, 0, sizeof(*fi));
    fi->next = chain;
    chain = fi;

    return remains;
}

#define hash_fn(inode, dev)  (( (inode) - (dev) ) % FI_BINS)

/* hash array maintenance */

static struct file_index *locate_fi(struct inode *inode,
				    struct file_index ***fi_prev_p)
{
    struct file_index *fi, **fi_p;

    *fi_prev_p = &fi_tab[hash_fn(inode->i_ino, inode->i_dev)];

    for (fi = **fi_prev_p; fi ; fi = fi->next) {
	if ((fi->fi_ino == inode->i_ino)
	    && (fi->fi_dev == inode->i_dev)) {
	    return fi;
	}
	*fi_prev_p = &fi->next;
    }

    return NULL;
}

static void drop_fi(struct inode *inode)
{
    struct file_index *fi, **fi_ptr;

    fi = locate_fi(inode, &fi_ptr);
    if (fi) {
	/* liberate the path */
	free_page((unsigned long) fi->full_page);
	*fi_ptr = free_fi(fi);
    }
}

static struct file_index *find_fi(struct inode *inode)
{
    struct file_index *fi, **fi_ptr;

    /* just in case we have multiple entries -- a bug! */
    while ((fi = locate_fi(inode, &fi_ptr))) {
	if ((inode->i_mode == fi->fi_mode)
	    && (inode->i_uid == fi->fi_uid)
	    && (inode->i_gid == fi->fi_gid)
	    && (inode->i_mtime == fi->fi_mtime)
	    && (inode->i_generation == fi->fi_generation)) {

	    break;
	}

	*fi_ptr = free_fi(fi);
    }

    return fi;
}

static int add_fi(struct dentry *dentry, kernel_cap_t *effective,
		  kernel_cap_t *inheritable, kernel_cap_t *permitted)
{
    int replacement = 0, len;
    struct file_index *fi;
    struct inode *inode;

    inode = dentry->d_inode;

    if ((fi = find_fi(inode)) == NULL) {
	fi = alloc_fi();
	if (fi == NULL) {
	    printk(KERN_ERR "fcaps: out of memory for new capability set\n");
	    return -ENOMEM;
	}
	fi->full_page = (char *) __get_free_page(GFP_KERNEL);
	if (fi->full_page == NULL) {
	    printk(KERN_ERR "fcaps: no memory for path caching\n");
	    (void) free_fi(fi);
	    return -ENOMEM;
	}
    } else {
	replacement = 1;
    }

    fi->fi_ino = inode->i_ino;
    fi->fi_dev = inode->i_dev;

    fi->fi_mode = inode->i_mode;
    fi->fi_uid = inode->i_uid;
    fi->fi_gid = inode->i_gid;
    fi->fi_mtime = inode->i_mtime;
    fi->fi_generation = inode->i_generation;

    fi->fi_effective = *effective;
    fi->fi_inheritable = *inheritable;
    fi->fi_permitted = *permitted;

    fi->filename = d_path(dentry, fi->full_page, PAGE_SIZE);

    if (!replacement) {
	int h = hash_fn(fi->fi_ino, fi->fi_dev);
	fi->next = fi_tab[h];
	fi_tab[h] = fi;
    }
    
    return 0;
}

/*
 * main callback for identifying capabilities
 */

static int once1=0, once2=0;

static int cap_select_by_dentry(struct dentry *dentry, int operation,
				kernel_cap_t *effective,
				kernel_cap_t *inheritable,
				kernel_cap_t *permitted)
{
    if (IS_ERR(dentry->d_inode)) {
	return -EINVAL;
    }

    /* this module does not support preemption */
    if (operation & _CAP_FS_PREEMPT) {
	if (!once1) {
	    printk(KERN_NOTICE
		   "fcaps: this module does not support preemption\n");
	    once1 = 1;
	}
	return -ENOSYS;
    }


    switch (operation) {

    case _CAP_FS_SET:
    {
	int retval;

	spin_lock(&fcap_access);
	if (effective && inheritable && permitted) {
	    retval = add_fi(dentry, effective, inheritable, permitted);
	} else {
	    drop_fi(dentry->d_inode);
	    retval = 0;
	}
	spin_unlock(&fcap_access);

	return retval;
    }
    case _CAP_FS_GET:
    {
	struct file_index *fi;

	spin_lock(&fcap_access);	
	fi = find_fi(dentry->d_inode);
	if (fi) {
	    *effective = fi->fi_effective;
	    *inheritable = fi->fi_inheritable;
	    *permitted = fi->fi_permitted;
	}
	spin_unlock(&fcap_access);

	return 0;
    }
    default:

	if (!once2) {
	    printk(KERN_WARNING
		   "fcaps: invalid fcap capability call (%d)\n", operation);
	    once2 = 1;
	}
	/* fall through */

    }

    return -ENOSYS;
}

/*
 * The /proc/secure filesystem entries:
 *
 *      /proc/secure/capable
 */

/*
 * This function dumps the contents of the cache out to an effective
 * file.  Its a little tricky to be able to deal with being called a
 * number of times if the output buffer isn't large enough.
 */

static int read_cache(char *buffer, char **buffer_used_p, off_t position,
		      int buffer_length, int zero)
{
    int i, count;

    /* Spin through the hash array looking for filled bins */

    for (i=count=0; (buffer_length > 0) && (i < FI_BINS); ++i) {
	struct file_index *fi;

	spin_lock(&fcap_access);

	/* interrogate those that are filled */
	for (fi = fi_tab[i]; (buffer_length > 0) && fi; fi = fi->next) {
	    char l_buff[60];
	    int name, len;

	    for (name=1; (buffer_length > 0) && (name >= 0); --name) {
		char *l_buffer;

		if (name) {
		    l_buffer = fi->filename;
		} else {
		    l_buffer = l_buff;
		    sprintf(l_buffer, " [%.4x:%.8x] %.8x %.8x %.8x\n",
			    fi->fi_dev, fi->fi_ino,
			    fi->fi_effective, fi->fi_inheritable,
			    fi->fi_permitted);
		}

		len = strlen(l_buffer);

		if (position > len) {

		    position -= len;
		    len = 0;

		} else if (position >= 0) {

		    if (position) {
			len -= position;
		    }

		    if (len > buffer_length) {
			len = buffer_length;
		    }

		    memcpy(buffer, l_buffer+position, len);
		    count += len;
		    buffer += len;
		    buffer_length -= len;
		    position = 0;
		}
	    }
	}

	spin_unlock(&fcap_access);
    }

    return count;
}

#define CAPABLE_FILE_NAME "capable"

static struct proc_dir_entry fcaps_capable_file = {
    0,       /* Inode - kernel can fill in */
    sizeof(CAPABLE_FILE_NAME)-1,
    CAPABLE_FILE_NAME,
    S_IFREG | S_IRUGO,
    1,       /* number of links to this file */
    0, 0,    /* uid / gid */
    0,       /* size of file - dynamic */
    NULL,    /* no support for misc inode functions */
    read_cache,
    NULL,    /* no support for setting up the inode */
};

/* XXX - this is lifted from the parport module */


static struct proc_dir_entry *new_proc_entry(const char *name, mode_t mode,
					     struct proc_dir_entry *parent,
					     unsigned short ino)
{
    struct proc_dir_entry *ent;

    ent = kmalloc(sizeof(struct proc_dir_entry), GFP_KERNEL);
    if (!ent)
	return NULL;

    memset(ent, 0, sizeof(struct proc_dir_entry));
	
    if (mode == S_IFDIR)
	mode |= S_IRUGO | S_IXUGO;
    else if (mode == 0)
	mode = S_IFREG | S_IRUGO;

    ent->low_ino = ino;
    ent->name = name;
    ent->namelen = strlen(name);
    ent->mode = mode;

    if (S_ISDIR(mode)) {
	ent->nlink = 2;
    } else {
	ent->nlink = 1;
    }

    proc_register(parent, ent);
	
    return ent;
}

/* We'll put the "securebits" file and this "secure" directory into
   the kernel patch later. We'll retain the "capable" file in this
   module since this module is the one that has the data! */

#define PROC_SECURE 2001  /* XXX - this needs to get into <linux/proc_fs.h> */

struct proc_dir_entry *fcaps_proc_base = NULL;

/* create the filesystem */

static int fcaps_proc_init()
{
    fcaps_proc_base = new_proc_entry("secure", S_IFDIR,
				     &proc_root, PROC_SECURE);
    if (fcaps_proc_base == NULL) {
	printk(KERN_ERR "fcaps: unable to initialize proc/secure\n");
	return 0;
    }

    if (proc_register(fcaps_proc_base, &fcaps_capable_file)) {
	printk(KERN_ERR "fcaps: unable to regiseter proc/secure/"
	       CAPABLE_FILE_NAME "\n");
	return 0;
    }

    printk(KERN_NOTICE "fcaps: registering proc/secure\n");
    return 1;
}

/* remove the filesystem */
static void fcaps_proc_cleanup()
{
    if (fcaps_proc_base) {

	if (fcaps_capable_file.low_ino) {
	    proc_unregister(fcaps_proc_base, fcaps_capable_file.low_ino);
	    fcaps_capable_file.low_ino = 0;
	}

	proc_unregister(&proc_root, fcaps_proc_base->low_ino);
	kfree(fcaps_proc_base);
	fcaps_proc_base = NULL;

	printk(KERN_NOTICE "fcaps: unregistered proc/secure\n");
    }
}

/*
 * Initialize module
 */

int init_module()
{
    /* If we return a non zero value, it means that init_module failed and
     * the kernel module can't be loaded */

    (void) fcaps_proc_init();
    
    saved_securebits = securebits;   /* some level of sanity can be
					preserved after cleanup */

    fi_tab = (struct file_index **) vmalloc(sizeof(struct file_index *)
					    * FI_BINS);
    if (fi_tab == NULL) {
	printk(KERN_ERR
	       "fcaps: failed to allocate a hash header cache");
	return 1;
    }
    memset(fi_tab, 0, sizeof(struct file_index *) * FI_BINS);

    return !fs_capability_fn_register(cap_select_by_dentry);
}


/* Dropping the module.  This is pretty bad and unless the kernel is
   prepared for it, it will likely cause the system to crash - every
   exec will cause a panic.

   XXX - We do not release memory here - there is a memory leak. */

void cleanup_module()
{
    printk(KERN_NOTICE
	   "fcaps: returning securebits 0x%.8x to 0x%.8x\n",
	   securebits, saved_securebits);

    securebits = saved_securebits;

    fcaps_proc_cleanup();

    printk(KERN_NOTICE
	   "fcaps: if kernel's not prepared to lose its fcap fn,"
	   " your system will crash..!\n");

    (void) fs_capability_fn_register(NULL);
}
