#include "../../H/ugens.h"
#include "../../H/sfheader.h"
#include <stdio.h>

extern SFHEADER      sfdesc[NFILES];

/* iir -- creates an iir filter with up to 64 resonance peaks, specifiable
*  center frequency and bandwidth for each peak
*
*  subcommands:
*
*  setup()
*  p0 = center frequency 1 (hertz or oct.pc)
*  p1 = bandwidth 1 (or multiplier of cf if negative)
*  p2 = relative amplitude 1
*  <p3, p4, p5 ... up to 64 triples>
*
*  inputsig()
*  p0 = input skip
*  p1 = output skip
*  p2 = duration
*  p3 = amplitude multiplier
*  p4 = input channel (0 or 1)
*  p5 = stereo spread (0-1) [optional]
*  assumes function table 1 is the amplitude envelope
*
*  noise()
*  p0 = start
*  p1 = duration
*  p2 = amplitude
*  p3 = stereo spread (0-1) [optional]
*  assumes function table 1 is the amplitude envelope
*
*  buzzit()
*  p0 = start
*  p1 = duration
*  p2 = amplitude
*  p3 = pitch (hz or oct.pc)
*  p4 = stereo spread (0-1) [optional]
*  assumes function table 1 is the amplitude envelope
*  assumes function table 2 is a sine wave
*
*  pulseit()
*  p0 = start
*  p1 = duration
*  p2 = amplitude
*  p3 = pitch (hz or oct.pc)
*  p4 = stereo spread (0-1) [optional]
*  assumes function table 1 is the amplitude envelope
*
*/

float rsnetc[64][5],amp[64];
int nresons;


double
setup(p,n_args)
float p[];
int n_args;
{
	int i,j;
	float first,bw,cf;

	first = (p[0] < 15.0) ? cpspch(p[0]) : p[0];
	bw = p[1] < 0 ? -p[1] * first : p[1];
	rsnset(first,bw,1.,0.,rsnetc[0]);
	printf("centerfreq    bandwidth  relative amp\n");
	amp[0] = p[2];
	printf("%10.4f %10.4f %10.4f\n",first,bw,amp[0]);
	for(i=3,j=1; i<n_args; i += 3)  {
		if(p[i] < 0.) cf = -p[i] * first;
		else  cf = (p[i] < 15.0) ? cpspch(p[i]) : p[i];
		bw = p[i+1] < 0 ? -p[i+1] * cf : p[i+1];
		amp[j] = p[i+2]; 
		rsnset(cf ,bw ,1.,0.,rsnetc[j]);
		printf("%10.4f %10.4f %10.4f\n",cf,bw,amp[j]); 
		j++;
	}
	nresons = j;
	return((double)nresons);
}


double
inputsig(p,n_args)
float p[];
int n_args;
{
	int nsamps,inchan,chans,i,j;
	float in[2],out[2],val,aamp;
	float *amparr,amptabs[2];
	int lenamp,skip,cdown;
	extern int resetval;

	nsamps = setnote(p[0], p[2], 0);
	setnote(p[1], p[2], 1);

	amparr = floc(1);
	lenamp = fsize(1);
	tableset(p[2], lenamp, amptabs);

	for(i = 0; i < nresons; i++) rsnetc[i][3] = rsnetc[i][4] = 0;

	inchan = p[4];
	chans = sfchans(&sfdesc[1]);
	skip = SR/(float)resetval;
	cdown = 0;
	for(i = 0; i < nsamps; i++) {
		if(!cdown--) {
			aamp = tablei(i, amparr, amptabs) * p[3];
			cdown = skip;
			}

		if(!GETIN(in, 0)) break;

		out[0] = 0.0;
		for(j = 0; j < nresons; j++) {
			val = reson(in[inchan],rsnetc[j]);
			out[0] += val * amp[j];
			}

		out[0] *= aamp;
		if (chans > 1) {
			out[1] = (1.0 - p[5]) * out[0];
			out[0] *= p[5];
			}

		ADDOUT(out, 1);
		}

	endnote(1);
}


double
noise(p,n_args)
float p[];
int n_args;
{
	int nsamps,chans,i,j;
	float namp,sig,out[2],val,aamp;
	float *amparr,amptabs[2];
	int lenamp,skip,cdown;
	extern int resetval;

	nsamps = setnote(p[0], p[1], 1);

	amparr = floc(1);
	lenamp = fsize(1);
	tableset(p[1], lenamp, amptabs);

	for(i=0; i<nresons; i++) rsnetc[i][3] = rsnetc[i][4] = 0;

	chans = sfchans(&sfdesc[1]);
	skip = SR/(float)resetval;
	cdown = 0;
	for(i = 0; i < nsamps; i++) {
		if(!cdown--) {
			aamp = tablei(i, amparr, amptabs) * p[2];
			cdown = skip;
			}

		sig = rrand();

		out[0] = 0.0;
		for(j = 0; j < nresons; j++) {
			val = reson(sig, rsnetc[j]);
			out[0] += val * amp[j];
			}

		out[0] *= aamp;
		if (chans > 1) {
			out[1] = (1.0 - p[3]) * out[0];
			out[0] *= p[3];
			}

		ADDOUT(out, 1);
		}

	endnote(1);
}


double
buzzit(p,n_args)
float p[];
int n_args;
{
	int nsamps,chans,i,j;
	float hn,phase,si,sig,out[2],val,aamp;
	float *amparr,amptabs[2];
	int lenamp,skip,cdown;
	float *sinetable;
	int lensine;
	extern int resetval;

	nsamps = setnote(p[0], p[1], 1);

	amparr = floc(1);
	lenamp = fsize(1);
	tableset(p[1], lenamp, amptabs);

	sinetable = floc(2);
	lensine = fsize(2);
	si = p[3] < 15.0 ? cpspch(p[3])*(float)lensine/SR : p[3]*(float)lensine/SR;
	hn = (int)(0.5/(si/(float)lensine));

	for(i=0; i<nresons; i++) rsnetc[i][3] = rsnetc[i][4] = 0;
		
	phase = 0.0;
	chans = sfchans(&sfdesc[1]);
	skip = SR/(float)resetval;
	cdown = 0;
	for(i = 0; i < nsamps; i++) {
		if(!cdown--) {
			aamp = tablei(i, amparr, amptabs) * p[2];
			cdown = skip;
			}

		sig = buzz(1.0, si, hn, sinetable, &phase);

		out[0] = 0.0;
		for(j = 0; j < nresons; j++) {
			val = reson(sig, rsnetc[j]);
			out[0] += val * amp[j];
			}

		out[0] *= aamp;
		if (chans > 1) {
			out[1] = (1.0 - p[4]) * out[0];
			out[0] *= p[4];
			}

		ADDOUT(out, 1);
		}

	endnote(1);
}


double
pulseit(p,n_args)
float *p;
{
	int nsamps,chans,i,j;
	float pulse();
	float phase,si,sig,out[2],val,aamp;
	float *amparr,amptabs[2];
	int lenamp,skip,cdown;
	extern int resetval;

	nsamps = setnote(p[0], p[1], 1);

	amparr = floc(1);
	lenamp = fsize(1);
	tableset(p[1], lenamp, amptabs);

	si = p[3] < 15.0 ? cpspch(p[3])*512.0/SR : p[3] * 512.0/SR;

	for(i=0; i<nresons; i++) rsnetc[i][3] = rsnetc[i][4] = 0;

	phase = 512.0;
	chans = sfchans(&sfdesc[1]);
	skip = SR/(float)resetval;
	cdown = 0;
	for(i = 0; i < nsamps; i++) {
		if(!cdown--) {
			aamp = tablei(i, amparr, amptabs) * p[2];
			cdown = skip;
			}

		sig = pulse(1.0, si, &phase);

		out[0] = 0.0;
		for(j = 0; j < nresons; j++) {
			val = reson(sig, rsnetc[j]);
			out[0] += val * amp[j];
			}

		out[0] *= aamp;
		if (chans > 1) {
			out[1] = (1.0 - p[4]) * out[0];
			out[0] *= p[4];
			}

		ADDOUT(out, 1);
		}

	endnote(1);
}

float pulse(amp,si,phase)
float amp,*phase,si;
{

	*phase += si;
	if(*phase > 512.0) {
		*phase -= 512.0;
		return(amp);
		}
	else return(0.0);
}


int NBYTES = 32768;
 
profile()
{
	UG_INTRO("setup",setup);
	UG_INTRO("inputsig",inputsig);
	UG_INTRO("noise",noise);
	UG_INTRO("buzz",buzzit);
	UG_INTRO("pulse",pulseit);
	UG_INTRO("reset",reset);
}
